/*
  Copyright (C) 2011 Stefan Stäglich

  This file is part of pung.

  Pung is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Pung is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with pung.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <GL/gl.h>
#include "./Ball.h"
#include "./Pong.h"
#include "./PongObject.h"
#include "./Paddle.h"

// _____________________________________________________________________________
Paddle::Paddle(Pong* pong, SDLKey moveUpKey, SDLKey moveDownKey)
{
  _pong = pong;
  _xPos = 0;
  _xSize = _pong->_wres / 50;
  _ySize = _pong->_hres / 5;
  _xDiff = 0;
  _yDiff = 0;

  _moveUpKey = moveUpKey;
  _moveDownKey = moveDownKey;

  _score = 0;

  reset();
}

// _____________________________________________________________________________
void Paddle::pushBall(size_t ballId)
{
  // Coordinates of corner marks of the ball
  // A(x1, y1)
  // B(x1, y2)
  // C(x2, y1)
  // D(x2, y2)
  float x1 = _pong->_balls[ballId]->_xPos;
  float y1 = _pong->_balls[ballId]->_yPos;
  float x2 = _pong->_balls[ballId]->_xPos + _pong->_balls[ballId]->_xSize;
  float y2 = _pong->_balls[ballId]->_yPos + _pong->_balls[ballId]->_ySize;

  // One of the points inside paddle?
  bool AInsidePaddle = x1 >= _xPos && x1 <= _xPos+_xSize &&
    (y1 >= _yPos && y1 <= _yPos+_ySize);
  bool BInsidePaddle = x1 >= _xPos && x1 <= _xPos+_xSize &&
    (y2 >= _yPos && y2 <= _yPos+_ySize);
  bool CInsidePaddle = x2 >= _xPos && x2 <= _xPos+_xSize &&
    (y1 >= _yPos && y1 <= _yPos+_ySize);
  bool DInsidePaddle = x2 >= _xPos && x2 <= _xPos+_xSize &&
    (y2 >= _yPos && y2 <= _yPos+_ySize);

  // A ball corner inside paddle -> Ball::inpulse()
  if (AInsidePaddle || BInsidePaddle || CInsidePaddle || DInsidePaddle)
    _pong->_balls[ballId]->inpulse(_yDiff);
}

// _____________________________________________________________________________
void Paddle::move()
{
  // Touched barrier
  if (touchedBarrier())
  {
    _yDiff = 0;
    if (_yPos <= 0)
      _yPos = 1;
    else
      _yPos = _pong->_hres - _ySize-1;
  }
  else
    PongObject::move();
}

// _____________________________________________________________________________
void Paddle::keyEvent(const SDL_Event& event)
{
  switch (event.type)
  {
    case SDL_KEYDOWN:
      if (event.key.keysym.sym == _moveUpKey)
        _yDiff = -3 * _pong->_speed;
      if (event.key.keysym.sym == _moveDownKey)
        _yDiff = 3 * _pong->_speed;
      break;

    case SDL_KEYUP:
    if (event.key.keysym.sym == _moveUpKey ||
        event.key.keysym.sym == _moveDownKey)
      _yDiff = 0;
    break;

    default:
    break;
  }
}

// _____________________________________________________________________________
void Paddle::draw()
{
  PongObject::draw();

  // Current size of the score bar
  float scoreSize = _ySize * _score / _pong->_maxScore;

  // Draw the score bar on the paddle
  glBegin(GL_QUADS);
  // Uni red
  glColor3f(0.75, 0, 0.17);
  glVertex2f(_xPos, _yPos);
  glVertex2f(_xPos, _yPos + scoreSize);
  glVertex2f(_xPos + _xSize, _yPos + scoreSize);
  glVertex2f(_xPos + _xSize, _yPos);
  glEnd();
}

// _____________________________________________________________________________
void Paddle::reset()
{
  // Reset the paddle to the middle
  _yPos = (_pong->_hres / 2) - (_ySize / 2);
}
