/**************************************************************************************

        PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

#include <mustux.h>

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>

#include <iostream>


#define DEFAULT_BUS_ID 0
#define DEFAULT_BUFFER_SIZE 1024


void print_usage(char *programName);
int show_info(char *fileName);
int play_song(char *fileName, int busId);
int write_wav(char *prafFileName, char *wavFileName);
int write_praf(char *wavFileName, char *prafFileName);
int split_song(char *fileName, int splitPosition);
int write_blocks(PrafFile *inFile, PrafFile *outFile, int numberOfBlocks);


int main(int argc, char **argv)
{
  //  MustuxDebugger::set_debug_on();
  //  MustuxDebugger::set_flood_debug_on();

  if (argc < 2)
    {
      print_usage(argv[0]);
      return 0;
    }
  
  if (argv[1][0] == '-')
    switch(argv[1][1])
      {
      case 'i':
	if (argc < 3)
	  {
	    std::cout << "Option '-i' needs a PrafFile" << std::endl;
	    print_usage(argv[0]);
	    return -1;
	  }
	return show_info(argv[2]);
	break;

      case 'b':
	if (argc < 4)
	  {
	    std::cout << "Option '-b' needs a busId and a PrafFile" << std::endl;
	    print_usage(argv[0]);
	    return -1;
	  }
	return play_song(argv[3], atoi(argv[2]));
	break;

      case 'w':
	if (argc < 4)
	  {
	    std::cout << "Option '-w' needs a WavFile and a PrafFile" << std::endl;
	    print_usage(argv[0]);
	    return -1;
	  }
	return write_wav(argv[3], argv[2]);
	break;

      case 'p':
	if (argc < 4)
	  {
	    std::cout << "Option '-p' needs a PrafFile and a WavFile" << std::endl;
	    print_usage(argv[0]);
	    return -1;
	  }
	return write_praf(argv[3], argv[2]);
	break;

      case 's':
	if (argc < 4)
	  {
	    std::cout << "Option '-s' needs a split point" << std::endl;
	    print_usage(argv[0]);
	    return -1;
	  }
	return split_song(argv[3], atoi(argv[2]));
	break;
	
      default:
	std::cout << "Unkown option: -" << argv[1][1] << std::endl;
	print_usage(argv[0]);
	return -1;
      }
  else
    return play_song(argv[1], DEFAULT_BUS_ID);
}


void print_usage(char *programName)
{
  std::cout << "Usage:\t" << programName << "  [-i][-b busId][-w WavFile][-p PrafFile][-s SplitPoint] filename" << std::endl;
  std::cout << "Play back a PRAF file" << std::endl;
  std::cout << " -i:\t\tshow file infos and exit" << std::endl;
  std::cout << " -b busId:\tuse busId as output device (default: " << DEFAULT_BUS_ID << ")" << std::endl;
  std::cout << " -w WavFile:\twrite WAV file instead of playing back" << std::endl;
  std::cout << " -p praFFile:\twrite Praf file instead of playing back" << std::endl;
  std::cout << " -s SplitPoint:\tsplit song at SplitPoint" << std::endl;
}


int show_info(char *fileName)
{
  if (!fileName)
    return -1;

  PrafFile *inputFile = new PrafFile();

  if (inputFile->open(fileName) != 1)
    {
      std::cout << "Sorry, could not open file \"" << fileName << "\", check spelling and read permissions" << std::endl;
      return -1;
    }

  if (inputFile->read_header() != 1)
    {
      std::cout << "Sorry, this seems not to be a PRAF file" << std::endl;
      return -1;
    }

  inputFile->get_info();

  inputFile->close_file();

  return 0;
}


int play_song(char *fileName, int busId)
{
  if (!fileName)
    return -1;

  PrafFile *inputFile = new PrafFile();

  if (inputFile->open(fileName) != 1)
    {
      std::cout << "Sorry, could not open file \"" << fileName << "\", check spelling and read permissions" << std::endl;
      return -1;
    }

  if (inputFile->read_header() != 1)
    {
      std::cout << "Sorry, this seems not to be a PRAF file" << std::endl;
      return -1;
    }

  inputFile->get_info();

  MustuxAudioDeviceMapper::init();
  MustuxAudioDeviceMapper::probe_busses_valid_modes();
  if (MustuxAudioDeviceMapper::open_bus_out(busId, inputFile->rate, inputFile->bitDepth, inputFile->channels) < 0)
    {
      std::cout << "Sorry, invalid playback bus" << std::endl;
      return -1;
    }

  char *buffer = MustuxAudioDeviceMapper::get_bus_out_transfer_buffer(busId, inputFile->channels);
  MustuxAudioDeviceMapper::clean_bus_out(busId);
  int bufferSize = MustuxAudioDeviceMapper::get_bus_out_transfer_size(busId);

  while (inputFile->read_fragment(buffer, bufferSize))
    {
      MustuxAudioBusTransferResult* tr = MustuxAudioDeviceMapper::bus_out_transfer(busId);
      if (tr->xrun < 0)
	std::cout << "Warning: Buffer Underrun" << std::endl;
      MustuxAudioDeviceMapper::clean_bus_out(busId);
    }

  MustuxAudioDeviceMapper::close_bus_out(busId);
  inputFile->close_file();

  return 0;
}


int write_wav(char *prafFileName, char *wavFileName)
{
  if (!prafFileName || !wavFileName)
    return -1;

  PrafFile *prafFile = new PrafFile();

  if (prafFile->open(prafFileName) != 1)
    {
      std::cout << "Sorry, could not open file \"" << prafFileName << "\", check spelling and read permissions" << std::endl;
      return -1;
    }

  if (prafFile->read_header() != 1)
    {
      std::cout << "Sorry, this seems not to be a PRAF file" << std::endl;
      return -1;
    }

  WavFile *wavFile = new WavFile();

  if (wavFile->create(wavFileName) != 1)
    {
      std::cout << "Sorry, could not create file \"" << wavFileName << "\", check write permissions" << std::endl;
      return -1;
    }

  wavFile->write_header(prafFile->channels, prafFile->rate, prafFile->bitDepth);

  int bufferSize = DEFAULT_BUFFER_SIZE;
  char *buffer = (char *) malloc(sizeof(char) * bufferSize);

  while (int read = prafFile->read_fragment(buffer, bufferSize))
      wavFile->append_samples_from_buffer(buffer, read);

  wavFile->fix_audio_size();
  
  wavFile->close_file();
  prafFile->close_file();

  return 0;
}


int write_praf(char *wavFileName, char *prafFileName)
{
  if (!wavFileName || !prafFileName)
    return -1;

  WavFile *wavFile = new WavFile();

  if (wavFile->open(wavFileName) != 1)
    {
      std::cout << "Sorry, could not open file \"" << prafFileName << "\", check spelling and read permissions" << std::endl;
      return -1;
    }

  if (wavFile->read_header() != 1)
    {
      std::cout << "Sorry, this seems not to be a WAV file" << std::endl;
      return -1;
    }

  PrafFile *prafFile = new PrafFile();

  if (prafFile->create(prafFileName) != 1)
    {
      std::cout << "Sorry, could not create file \"" << wavFileName << "\", check write permissions" << std::endl;
      return -1;
    }

  prafFile->write_header(wavFile->channels, wavFile->rate, wavFile->bitDepth, 0, "no project", "no group", "no hardware", "no description");

  int bufferSize = DEFAULT_BUFFER_SIZE;
  char *buffer = (char *) malloc(sizeof(char) * bufferSize);

  while (int read = wavFile->read_fragment(buffer, bufferSize))
      prafFile->append_samples_from_buffer(buffer, read);

  prafFile->fix_audio_size();

  prafFile->close_file();
  wavFile->close_file();

  return 0;
}


//TODO: check if splitPoint is in the song at the begining !
//TODO: to be rewritten
int split_song(char *fileName, int splitPosition)
{
  if (!fileName)
    return -1;

  PrafFile *inFile = new PrafFile();

  if (inFile->open(fileName) != 1)
    {
      std::cout << "Sorry, could not open file \"" << fileName << "\", check spelling and read permissions" << std::endl;
      return -1;
    }

  if (inFile->read_header() != 1)
    {
      std::cout << "Sorry, this seems not to be a PRAF file" << std::endl;
      return -1;
    }

  int blockSize = inFile->channels + inFile->bitDepth / 8;
  int splitAdresse = splitPosition * blockSize;

  int dotPosition;
  for (dotPosition = strlen(fileName) -1; dotPosition >= 0 && fileName[dotPosition] != '.'; dotPosition--);

  char *newFileName = (char *) malloc((strlen(fileName) + 3) * sizeof(char));

  if (dotPosition >= 0)
    {
      for(int i = 0; i < dotPosition; i++)
	newFileName[i] = fileName[i];

      newFileName[dotPosition] = '_';
      newFileName[dotPosition + 1] = '1';

      for(int i = dotPosition; i <= strlen(fileName); i++)
	newFileName[i+2] = fileName[i];
    }
  else
    {
      dotPosition = strlen(fileName);

      for(int i = 0; i < dotPosition; i++)
	newFileName[i] = fileName[i];

      newFileName[dotPosition] = '_';
      newFileName[dotPosition + 1] = '1';
      newFileName[dotPosition + 2] = '\0';
    }

     

  PrafFile *outFile1 = new PrafFile();
  PrafFile *outFile2 = new PrafFile();

  if (outFile1->create(newFileName) != 1)
    {
      std::cout << "Sorry, could not create file \"" << newFileName << "\", check write permissions" << std::endl;
      return -1;
    }

  newFileName[dotPosition + 1] = '2';

  if (outFile2->create(newFileName) != 1)
    {
      std::cout << "Sorry, could not create file \"" << newFileName << "\", check write permissions" << std::endl;
      return -1;
    }

  outFile1->write_header(inFile->channels, inFile->rate, inFile->bitDepth, 0, inFile->projectLabel, inFile->groupLabel, inFile->hardwareLabel, inFile->description);
  outFile2->write_header(inFile->channels, inFile->rate, inFile->bitDepth, 0, inFile->projectLabel, inFile->groupLabel, inFile->hardwareLabel, inFile->description);

  std::cout <<  write_blocks(inFile, outFile1, splitPosition) << std::endl;
  std::cout <<  write_blocks(inFile, outFile2, -1) << std::endl;

  /*
  int read;
  int totalRead = 0;
  int bufferSize = DEFAULT_BUFFER_SIZE;
  char *buffer = (char *) malloc(sizeof(char) * bufferSize);

  while ((read = inFile->read_fragment(buffer, bufferSize)) && ((totalRead + bufferSize < splitAdresse)))
    {
      outFile1->append_samples_from_buffer(buffer, read);
      totalRead += read;
    }

  if (read)
    {
       outFile1->append_samples_from_buffer(buffer, (read > splitAdresse - totalRead) ? (splitAdresse - totalRead) : read);
       outFile2->append_samples_from_buffer(buffer + splitAdresse - totalRead, bufferSize - splitAdresse + totalRead);
    }

  while (read = inFile->read_fragment(buffer, bufferSize))
    {
       outFile2->append_samples_from_buffer(buffer, read);
    }
  */

  outFile1->fix_audio_size();
  outFile2->fix_audio_size();

  outFile1->close_file();
  outFile2->close_file();
  inFile->close_file();

  return 0;

}


// Write numberOfBlocks from inFile to outFile
//
// If outFile is NULL, will just read numberOfBlocks from inFile and discard it
// (usefull to skip a certain number of blocks from the file)
//
// If numberOfBlocks is -1, will write the remaining of inFile to outFile
//
// Returns the number of blocks really read/written
int write_blocks(PrafFile *inFile, PrafFile *outFile, int numberOfBlocks)
{
  if (!inFile)
    return -1;

  int blockSize = inFile->channels * inFile->bitDepth / 8;

  int bufferSize = DEFAULT_BUFFER_SIZE;
  char *buffer = (char *) malloc(sizeof(char) * bufferSize);

  int totalToRead = numberOfBlocks * blockSize;
  int totalRead = 0;
  int leftToRead = totalToRead;

  int toRead;
  int read;

  for (;;)
    {
      toRead = (leftToRead >= bufferSize || numberOfBlocks == -1) ? bufferSize : leftToRead;
      read = inFile->read_fragment(buffer, toRead);

      if (read && outFile)
	outFile->append_samples_from_buffer(buffer, read);

      leftToRead -= read;
      totalRead += read;

      if ((read != toRead) || (leftToRead == 0))
	break;
    }

  free(buffer);

  return totalRead / blockSize;
}
