/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/

#include "FilterChain.hh"
#include "Filters.hh"
#include "../ColorManager.hh"

#define PSHOW(X) printf(" X = %4.2f \n", (float) X );

FilterChain::FilterChain(Track* pAssocTrack)
	{
	PENTERCONS;
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		filterController[k] = (FilterController*) 0;
	currentFilterController=-1;
	assocTrack=pAssocTrack;
	draggingNode = false;
	isSelectingController = false;
	PEXITCONS;
	}


FilterChain::~FilterChain()
	{
	PENTERDES;
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		if (filterController[k])
			delete filterController[k];
	PEXITDES;
	}

void FilterChain::draw()
	{
	PENTER4;
	int baseY = assocTrack->real_baseY();
	int height = assocTrack->get_height();
	QPainter* painter = assocTrack->get_painter();

	if (assocTrack->is_active())
		{
		ColorManager::set_blur(ColorManager::CURVE_ACTIVE , false);
		ColorManager::set_blur(ColorManager::FILTERCONTROLLER_ACTIVE, false);
		ColorManager::set_blur(ColorManager::CURVE_NONACTIVE , false);
		ColorManager::set_blur(ColorManager::FILTERCONTROLLER_NONACTIVE, false);
		}
	else
		{
		ColorManager::set_blur(ColorManager::CURVE_ACTIVE , true);
		ColorManager::set_blur(ColorManager::FILTERCONTROLLER_ACTIVE, true);
		ColorManager::set_blur(ColorManager::CURVE_NONACTIVE , true);
		ColorManager::set_blur(ColorManager::FILTERCONTROLLER_NONACTIVE, true);
		}

	// draw the controllers queue
	painter->setPen(QColor(0,0,100));
	painter->setFont(QFont("Fixed",6));
	int by = baseY + height - 25;
	int y1 = by + 6;
	QColor cf;
	int j=0;
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		{
		if (filterController[k])
			{
			int bx = 15 + j*80;
			int x1 = bx - 10;
			painter->setPen(Qt::black);
			painter->moveTo(x1,   y1);
			painter->lineTo(x1+5, y1);
			painter->lineTo(x1+5, y1-4);
			painter->lineTo(x1+8, y1+4);
			painter->lineTo(x1+5, y1+11);
			painter->lineTo(x1+5, y1+7);
			painter->lineTo(x1,   y1+7);

			QString filterType = filterController[k]->get_type();
			if (filterController[k]->get_filter()->isBypassed)
				cf = CM_COLOR(FILTERCONTROLLER_NONACTIVE);
			else
				{
				if (k==currentFilterController)
					{
					filterController[k]->get_current_curve()->activate();
					cf = CM_COLOR(FILTERCONTROLLER_ACTIVE);
					}
				else
					{
					filterController[k]->get_current_curve()->deactivate();
					cf = CM_COLOR(FILTERCONTROLLER_NONACTIVE);
					}
				}
			QString currentCurveName = filterController[k]->get_current_curve()->get_type();
			painter->fillRect(bx,   by,    70, 19, cf);
			painter->setPen(Qt::black);
			painter->drawRect(bx,   by,    70, 19);
			painter->drawText(bx+3, by+9, filterType);
			painter->drawText(bx+3, by+17, currentCurveName);
			j++;
			}
		}

	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		{
		if (filterController[k])
			filterController[k]->draw();
		}
	}


FilterController* FilterChain::add_filter_controller(QString filterName)
	{
	PENTER;
	int fcIndex=-1;

	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		{
		if (!filterController[k])
			{
			fcIndex=k;
//			assocTrack->parentSong->recreate_mta();
			break;
			}
		}
	if (fcIndex<0)
		{
		PERROR("Max number of filter controllers reached for track %s", (const char*) assocTrack->get_name().latin1());
		PEXIT;
		return (FilterController*) 0;
		}
	PMESG("Allocating %s controller in slot %d",(const char*) filterName.latin1() ,fcIndex);
	if (filterName=="")
		{
		assocTrack->parentSong->get_default_lcd()->print("Ok... no filter added :-/",1);
		PEXIT;
		return (FilterController*) 0;
		}
	if (filterName=="Gain")
		{
		filterController[fcIndex]=new GainController(this);
		}
	else
	if (filterName=="Pan")
		{
		filterController[fcIndex]=new PanController(this);
		}
	/* FOR LATER
	else
	if (filterName=="Reverb")
		{
		Reverb* filter = new Reverb(44100,2);
		filterController[fcIndex]=new ReverbController(parentSong,drawArea,filter);
		}
	*/
	else
	if (filterName=="Noise Gate")
		{
		filterController[fcIndex]=new NoiseGateController(this);
		}
	/* FOR LATER
	else
	if (filterName=="Octaver")
		{
		Octaver* filter = new Octaver(44100,2);
		filterController[fcIndex]=new OctaverController(parentSong,drawArea,filter);
		}
	*/
	else
	if (filterName=="Delay")
		{
		filterController[fcIndex]=new DelayController(this);
		}
	/*
	else
	if (filterName=="Compressor")
		{
		Compressor* filter = new Compressor(44100,2);
		filterController[fcIndex]=new CompressorController(parentSong,drawArea,filter);
		}
	*/
	//else if (filterName=="...
	else
		{
		assocTrack->parentSong->get_default_lcd()->print("Filter/Controller not implemented yet",1);
		PEXIT;
		return (FilterController*) 0;
		}
	filterController[fcIndex]->setupFrontend->setCaption(filterController[fcIndex]->get_type()+" Controller Setup");
	set_current_filter_controller(fcIndex);
	assocTrack->recreate(); // should be recreate_mta
	PEXIT;
	return filterController[currentFilterController];
	}


int FilterChain::set_current_filter_controller(FilterController* fc)
	{
	PENTER2;
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		if (filterController[k] && filterController[k]==fc)
			{
			PEXIT2;
			return set_current_filter_controller(k);
			}
	return -1;
	PEXIT2;
	}


int FilterChain::set_current_filter_controller(int index)
	{
	PENTER2;
	if ((index>=0) && (index<MAX_FILTER_CONTROLLERS) && (filterController[index]))
		{
		for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
			if (filterController[k])
				filterController[k]->deactivate();
		currentFilterController = index;
		if (assocTrack->parentSong->get_editing_mode()==Song::EDITING_MODE_TRACK_CURVES)
			filterController[currentFilterController]->activate();
		PEXIT2;
		return 1;
		}
	PEXIT2;
	return -1;
	}


void FilterChain::select_controller(bool isBeginning)
	{
	isSelectingController = isBeginning;
	}




int FilterChain::remove_controller(FilterController* fc)
	{
	PENTER2;
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		if (filterController[k] && filterController[k]==fc)
			{
			PEXIT2;
			return remove_controller(k);
			}
	return -1;
	PEXIT2;
	}


int FilterChain::remove_controller(int index)
	{
	PENTER2;
	if (index==-1)
		index=currentFilterController;
	if (currentFilterController>=0)
		{
		delete (filterController[currentFilterController]);
		filterController[currentFilterController] = (FilterController*) 0;
		currentFilterController=-1;
		for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
			{
			if (filterController[k])
				{
				currentFilterController=k;
				filterController[currentFilterController]->activate();
				break;
				}
			}
		assocTrack->recreate_mta();
		PEXIT2;
		return 1;
		}
	PEXIT2;
	return -1;
	}


void FilterChain::followMouse(int x, int y)
	{
	PENTER4;
	// CONTROLLER SELECTION
	if (isSelectingController)
		{
		int ncs=0;
		for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
			if (filterController[k]) ncs++;
		if (ncs==0)
			{
			PEXIT4;
			return;
			}
		int index = x/(assocTrack->get_width()/ncs) + 1;
		//PMESG("Selecting Controller : " << index);
		int i=0;
		for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
			{
			if (filterController[k]) i++;
			if (i==index)
				{
				set_current_filter_controller(k);
				break;
				}
			}
		assocTrack->recreate_mta();
		PEXIT4;
		return;
		}

	// NODE/CURVE MANIPULATION
	if (
		(currentFilterController<0) ||
		(assocTrack->parentSong->get_editing_mode()!=Song::EDITING_MODE_TRACK_CURVES) ||
		((!draggingNode) && (!assocTrack->is_pointed()))
	    )
	    	{
		PEXIT4;
		return;
		}


	long long lpos = assocTrack->parentSong->xpos_to_block(x);
	if (lpos<0)
		{
		PEXIT4;
		return;
		}

	float cv = filterController[currentFilterController]->get_current_curve()->get_value_at(lpos);

	/* improveme
	MustuxLcd* lcd = assocTrack->parentSong->parentProject->get_default_lcd();
	lcd->clear(550,100,100,20);
	p->setPen(Qt::black);
	QString scv; scv.setNum(cv);
	p->setFont(QFont("Fixed",8));
	p->drawText(560,100,scv);
	lcd->update(550,100,100,20);
	*/

	if (draggingNode)
		{
		CurveNode* cn = filterController[currentFilterController]->get_current_curve()->get_current_curve_node();
		if (cn->prev) // This is to avoid changing the root node position
			{
			long long pos = assocTrack->parentSong->xpos_to_block( x - originX ) + originPos;
			if(cn->prev->pos > pos)
				pos=cn->prev->pos + 4;
			if (cn->next)
				if(cn->next->pos < pos)
					pos=cn->next->pos - 4;
			cn->set_pos(pos);
			}
		int half = assocTrack->get_height()/2;
		int dy = y - originY;
		float variation = -100.0 * ( ( (float) ( dy - half) / half ) + 1.0 );
		float v = originValue + variation;
		cn->set_value(v);
		assocTrack->recreate_mta();
		}
	else if (assocTrack->is_pointed())
		filterController[currentFilterController]->update_current_node(x);
	PEXIT4;
	}

void FilterChain::drag_node(bool isBeginning)
	{
	draggingNode = isBeginning;
	if ((draggingNode) && (currentFilterController>=0))
		{
		originX = assocTrack->get_mouse_x();
		originY = assocTrack->get_mouse_y();
		originPos   = filterController[currentFilterController]->get_current_curve()->get_current_curve_node()->pos;
		originValue = filterController[currentFilterController]->get_current_curve()->get_current_curve_node()->value;
		}
	}


void FilterChain::add_node()
	{
	PENTER3;
	if (currentFilterController>=0)
		{
		int x = assocTrack->get_mouse_x();
		int y = assocTrack->get_mouse_y();
		long long pos  = assocTrack->parentSong->xpos_to_block(x);
		int half = assocTrack->get_height()/2;
		float value = -100.0 * ((float) ((y - assocTrack->get_baseY()) - half) / half );
		filterController[currentFilterController]->get_current_curve()->add_node(pos, value);
		assocTrack->recreate_mta();
		}
	PEXIT3;
	}


void FilterChain::node_setup()
	{
	if (currentFilterController>=0)
		{
		filterController[currentFilterController]->node_setup();
		assocTrack->recreate_mta();
		}
	}


void FilterChain::filter_setup()
	{
	if (currentFilterController>=0)
		{
		filterController[currentFilterController]->setup();
		assocTrack->recreate_mta();
		}
	}

int FilterChain::process(char* data, int size)
	{
	PENTER4;
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		{
		if (filterController[k])
			filterController[k]->process(data,size);
		}
	PEXIT4;
	return 1;
	}

int FilterChain::cleanup()
	{
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		{
		if (filterController[k])
			filterController[k]->cleanup();
		}
	return 1;
	}


void FilterChain::activate()
	{
	PENTER2;
	if (currentFilterController>=0)
		filterController[currentFilterController]->activate();
	// maybe some things more here ...
	PEXIT2;
	}


void FilterChain::deactivate()
	{
	PENTER2;
	if (currentFilterController>=0)
		filterController[currentFilterController]->deactivate();
	PEXIT2;
	}



QString FilterChain::get_schema()
	{
	PENTER;
	QString schema="";
	for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
		{
		if (filterController[k])
			{
			schema.append("       <filtercontroller type=\""+filterController[k]->get_type()+"\">\n");
			schema.append(filterController[k]->get_schema());
			schema.append("       </filtercontroller>\n");
			}
		}
	PEXIT;
	return schema;
	}

//eof
