/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include <sys/stat.h>
#include <sys/types.h>
#include <math.h>
#include <errno.h>
#include <qregexp.h>
#include <qsplitter.h>
#include <qfiledialog.h>
#include <qtextstream.h>
#include <qvaluelist.h>
#include <qdatetime.h>
#include <mustux.h>
#include "Song.hh"

// ALL THESE JOG STUFF ARE A MESS. MUST A CLEAN ASAP. THERE MUST BE AN
// GENERALIZATION OF JOGS


static const int LOCATOR_HEIGHT = 30;
static const int SHUTTLE_SENSIBILITY = 80; // 1-100  : will be user configurable
static const int VERTICAL_SCROLL_SENSIBILITY = 80; // 1-100  : will be user configurable
static const int MOUSE_SENSIBILITY = 6;

Song::Song(int pMyNumber, Project* pParentProject, QWidget* parent, int pRate, int pBitDepth) : QWidget(parent)
	{
	PENTERCONS;
	myNumber = pMyNumber;
	QString sMyNumber = ""; sMyNumber.setNum( myNumber );
	parentProject=pParentProject;
	audioSourcesList = new AudioSourcesList();
	rate=pRate;
	bitDepth=pBitDepth;
	propertiesDialog = new SongPropertiesDialog(this);
	QBoxLayout *mainVBoxLayout = new QVBoxLayout( this, 2);
	QSplitter* splitter = new QSplitter(this);
	splitter->setOpaqueResize( false );
	QString pName1 = "Tracks Panels Area ( Song " + sMyNumber + " )";
	trackPanelsArea = new TrackPanelsArea( this , splitter, (const char*) pName1);// WAS leftBox);
	trackPanelsArea->setMinimumWidth(169);
	splitter->setResizeMode(trackPanelsArea, QSplitter::KeepSize);
	QString pName2 = "Tracks Area ( Song " + sMyNumber + " )";
	tracksArea = new TracksArea( this , splitter, (const char*) pName2 ); // WAS rightBox

	mainVBoxLayout->addWidget(splitter);
	mainVBoxLayout->activate();
	QString projDir = parentProject->get_root_dir();
	QString songNum(""); songNum.setNum(myNumber);
	rootDir = projDir + "/song_" + songNum;
	audioSourcesDir = rootDir + "/audiosources";
	captureDir = rootDir + "/capture";
	mtaSchemaFilename = rootDir + "/mta.schema";
	const char* cRootDir= (const char*) rootDir.ascii();
	if ( mkdir(cRootDir, S_IREAD | S_IWRITE | S_IEXEC)<0)
		{
		if (errno!=EEXIST)
			{
			PERROR("Cannot create dir %s" , (const char*) rootDir.ascii());
			}
		}
	const char* cAudioSourcesDir = (const char*) audioSourcesDir.ascii();
	if (mkdir(cAudioSourcesDir, S_IREAD | S_IWRITE | S_IEXEC)<0)
		{
		if (errno!=EEXIST)
			{
			PERROR("Cannot create dir %s ",(const char*) audioSourcesDir.ascii());
			}
		}
	const char* cCaptureDir = (const char*)captureDir.ascii();
	if (mkdir(cCaptureDir, S_IREAD | S_IWRITE | S_IEXEC)<0)
		{
		if (errno!=EEXIST)
			{
			PERROR("Cannot create dir %s",(const char*)  captureDir.ascii());
			}
		}
	title="Untitled";
	masterGain = 0.0f;
	artists = "No artists name yet";
	jogMode = JOG_NONE;
	gainingTrack = -1;
	hzoom = GlobalProperties::get_int("HZOOM_LEVEL");
	firstBlock = 0;
	lastBlock=0;
	workingBlock = 0;
	takeNumber=0;
	playBlocksOffset = 0;
	numTracks=0;
	editingMode = EDITING_MODE_NORMAL;
	activeTrackNumber=0;
	regionList = (MtaRegion*) 0;
	isSnapOn=true;
	changed=false;
	isSomePeakBuilding=false;
	peakBuildCount=0;

	for (int i=0; i<MAX_TRACKS; i++)
		track[i] = (Track*) 0;
	QValueList<int> lst;
	lst.append( 180 );
	lst.append( 450 );
	//splitter->setSizes( lst );
	mtaBaseY = 0;
	mixer = new Mixer(this);
	shuttleTimer = new QTimer(this);
	updateProgressTimer = new QTimer(this);
	BusMonitorRecreateTimer = new QTimer (this);
	connect( shuttleTimer , SIGNAL(timeout()), this, SLOT(update_shuttle()));
	connect( updateProgressTimer , SIGNAL(timeout()), this, SLOT(update_progress()));
	connect( BusMonitorRecreateTimer , SIGNAL(timeout()), this, SLOT(update_busmonitor()));
	filterSelector = new FilterSelector(tracksArea);
	busSelector = new BusSelector(tracksArea,mixer);
	cursorManager = new CursorManager(this);
	PEXITCONS;
	}



Song::~Song()
	{
	PENTERDES;
	for (int i=0; i<numTracks; i++)
		{
		delete track[i];
		delete trackPanel[i];
		}
	delete cursorManager;
	delete shuttleTimer;
	delete mixer;
	delete audioSourcesList;
	delete propertiesDialog;
	PEXITDES;
	}



int Song::build(int tracksToCreate)
	{
	PENTER;
	if (tracksToCreate==-1)
		tracksToCreate = GlobalProperties::get_int("DEFAULT_NUM_TRACKS");
	numTracks=0;
	for (int i=0; i<tracksToCreate; i++)
		{
		track[i]= new Track(this, i, "Unnamed",  LOCATOR_HEIGHT + i * Track::INITIAL_HEIGHT, Track::INITIAL_HEIGHT);
		trackPanel[i]= new TrackPanel(track[i]);
		track[i]->set_assoc_track_panel(trackPanel[i]);
		numTracks++;
		}
	init();
	PEXIT;
	return 1;
	}



int Song::load()
	{
	PENTER;
	QFile file1( rootDir + "/song.properties" );
	QString line;
	if ( file1.open( IO_ReadOnly ) )
		{
		QTextStream stream( &file1 );
		while (!stream.eof())
			{
			QString line = stream.readLine();
			line = line.stripWhiteSpace();
			if (line.find("title=")>=0)
				title=line.mid(line.find("title=")+6);
			if (line.find("artists=")>=0)
				artists=line.mid(line.find("artists=")+8);
			if (line.find("rate=")>=0)
				rate=line.mid(line.find("rate=")+5).toInt();
			if (line.find("bitDepth=")>=0)
				bitDepth=line.mid(line.find("bitDepth=")+9).toInt();
			if (line.find("activeTrackNumber=")>=0)
				activeTrackNumber=line.mid(line.find("activeTrackNumber=")+18).toInt();
			if (line.find("firstBlock=")>=0)
				firstBlock=(long long)line.mid(line.find("firstBlock=")+11).toDouble();
			if (line.find("workingBlock=")>=0)
				workingBlock=(long long)line.mid(line.find("workingBlock=")+13).toDouble();
			if (line.find("hzoom=")>=0)
				hzoom=line.mid(line.find("hzoom=")+6).toInt();
			if (line.find("editingMode=")>=0)
				editingMode=line.mid(line.find("editingMode=")+12).toInt();
			}
		PMESG("Reading Song Properties : title=%s rate=%d  bitDepth=%d",title.ascii(),rate,bitDepth);
		file1.close();
		}
	propertiesDialog->update();
	QFile f(mtaSchemaFilename);
	int ntrack=0;
	if ( f.open(IO_ReadOnly) )
		{
		QTextStream t( &f );
		while (!t.eof())
			{
			QString line = t.readLine();
			line = line.stripWhiteSpace();
			if (line.find("<regions>")>=0)
				{
				while (line.find("</regions>")<0)
					{
					line = t.readLine();
					line = line.stripWhiteSpace();
					if (line.find("<region ")>=0)
						{
						int x5 = line.find("begin=")+6;
						int x6 = line.find(" ", x5+1);
						QString sbb=line.mid(x5,x6-x5);
						int x7 = line.find("end=")+4;
						int x8 = line.find(" ", x7+1);
						QString sbe=line.mid(x7,x8-x7);
						long long bb = (long long) sbb.toDouble();
						long long be = (long long) sbe.toDouble();
						MtaRegion* m = new MtaRegion(bb,be);
						add_mta_region(m);
						}
					}
				}
			if (line.find("<track")>=0)
				{
				PMESG("Loading Track %d ",ntrack);
				track[ntrack]= new Track(this, ntrack, "Unnamed",  LOCATOR_HEIGHT  + ntrack * Track::INITIAL_HEIGHT, Track::INITIAL_HEIGHT);
				trackPanel[ntrack]= new TrackPanel(track[ntrack]);
				track[ntrack]->set_assoc_track_panel(trackPanel[ntrack]);
				QString tschema = line + "\n";
				while (line.find("</track")<0)
					{
					line = t.readLine();
					line = line.stripWhiteSpace();
					tschema = tschema + line + "\n";
					}
				track[ntrack]->set_schema(tschema);
				ntrack++;
				numTracks++;
				}
			}
		}
	// checks if sample rate and bitdepth are OK. If not, ask user to correct it.
	// this routine might go to recreate();...
	bool validMode = true;
	// IMPROVE-ME
	if ((rate<22050) || (bitDepth!=16)) // FOR FUTURE  : .. && pBitDepth!=20,24,32 ... and so on
		validMode=false;
	if (!validMode) // ask user about rate and bitDepth
		{
		propertiesDialog->raise();
		rate=propertiesDialog->get_rate();
		bitDepth=propertiesDialog->get_bit_depth();
		}
	init();
	PEXIT;
	return 1;
	}



int Song::save()
	{
	PENTER;
	PMESG("Saving Song MTA schema : %s",(const char*) mtaSchemaFilename.ascii());
	FILE* file = fopen((const char*)mtaSchemaFilename.ascii(),"w");
	if (!file)
		{
		PERROR("Cannot Save Song MTA schema #%d" ,myNumber);
		PEXIT;
		return -1;
		}
	QString mta_schema = get_mta_schema();
	char* data = (char*) mta_schema.ascii();
	fwrite( data, 1, mta_schema.length(), file );
	fclose( file );
	PMESG("Saving Song Properties : title=%s rate=%d  bitDepth=%d",title.ascii(),rate,bitDepth);
	QString p = rootDir + "/song.properties";
	FILE* file2 = fopen((const char*)p.ascii(),"w");
	if (!file2)
		{
		PERROR("Cannot Save Song Properties#%d" ,myNumber);
		PEXIT;
		return -1;
		}
	QString sRate; sRate.setNum(rate);
	QString sBitDepth; sBitDepth.setNum(bitDepth);
	QString sActiveTrackNumber; sActiveTrackNumber.setNum(activeTrackNumber);
	QString sFirstBlock; sFirstBlock.setNum((double)firstBlock);
	QString sWorkingBlock; sWorkingBlock.setNum((double)workingBlock);
	QString sHzoom; sHzoom.setNum(hzoom);
	QString sEditingMode; sEditingMode.setNum(editingMode);
	QString data2 = "title=" + title + "\nartists=" + artists + "\nrate=" + sRate + "\nbitDepth="+ sBitDepth+
			"\nactiveTrackNumber=" + sActiveTrackNumber + "\nfirstBlock=" + sFirstBlock +
			"\nworkingBlock=" + sWorkingBlock + "\nhzoom=" + sHzoom + "\neditingMode=" + sEditingMode +
			"\n"; // IMPROVE-ME
	fwrite( data2, 1, data2.length(), file2 );
	fclose( file2 );
	changed=false;
	recreate_mta();
	PEXIT;
	return 1;
	}


int Song::init()
	{
	PENTER;
	touch_track(activeTrackNumber);
	cursorManager->start();
	update_last_block();
	set_editing_mode();
	PEXIT;
	return 1;
	}


void Song::recreate()
	{
	PENTER3;
	recreate_tracks();
	parentProject->get_bus_monitor()->recreate();
	PEXIT3;
	}


void Song::recreate_tracks()
	{
	PENTER3;
	cursorManager->disable();
	cursorManager->set_work_at(workingBlock);
	for (int i=0; i<numTracks; i++)
		track[i]->recreate();
	recreate_locator();
	//Clear cursorManager area above the trackPanels
	trackPanelsArea->painter->fillRect(0, 0, trackPanelsArea->width(), LOCATOR_HEIGHT, Qt::black);
	cursorManager->enable();
	cursorManager->recreate();
	clear_root_space();
	PEXIT3;
	}


void Song::recreate_locator()
	{
        int wk=0;
        int yk;
	QPainter* painter = tracksArea->painter;
	painter->fillRect(0,0,tracksArea->width(),LOCATOR_HEIGHT,QColor(140,170,200));
        painter->setPen(QColor(0,50,100));
        painter->setFont( QFont( "Helvetica", 6) );
	if (jogMode==JOG_CREATE_REGION)
		{
		QPainter* painter = tracksArea->painter;
		int xs = block_to_xpos(origBlockL);
		int xe = tracksArea->get_mouse_x();
		if (xs<0) xs=0;
		if (xe > tracksArea->width()) xe=tracksArea->width();
		painter->fillRect(xs,0,xe-xs,LOCATOR_HEIGHT,QColor(55,100,150));
		tracksArea->update(xs,0,xe-xs,LOCATOR_HEIGHT);
		}

	long long lastb = firstBlock + tracksArea->width() * Peak::zoomStep[hzoom] + 1;
	int k=0;
	for (long long b = firstBlock - (firstBlock % 5000); b<lastb; b+=(Peak::zoomStep[hzoom]+1)*10)
		{
		if (b<firstBlock) continue;
		int x = block_to_xpos(b);
		painter->drawLine(x,0,x,10);
		if (++k>10)
			{
			QString(s); s.sprintf("%ld",(long) b);
			painter->drawLine(x,10,x,20);
			painter->drawText(x+3,25,s);
			k=0;
			}
                }
	int xl = block_to_xpos(lastBlock);
	if (xl<tracksArea->width())
		tracksArea->painter->fillRect(xl,0,tracksArea->width()-xl,LOCATOR_HEIGHT,Qt::gray);
	}

void Song::recreate_mta()	//Use this method if only the mta needs to be repainted
	{
	PENTER3;
	cursorManager->disable();
	cursorManager->set_work_at(workingBlock);
	for (int i=0; i<numTracks; i++)
		track[i]->recreate_mta();
	recreate_locator();
	clear_root_space();
	cursorManager->enable();
	PEXIT3;
	}


void Song::recreate_panels()
	{
	PENTER3;
	for (int i=0; i<numTracks; i++)
		trackPanel[i]->recreate();
	//Clear cursorManager area above the trackPanels
	trackPanelsArea->painter->fillRect(0, 0, trackPanelsArea->width(), LOCATOR_HEIGHT, Qt::black);
	PEXIT3;
	}

void Song::resizeEvent(QResizeEvent* e)
	{
	}



void Song::mouseMoveEvent(QMouseEvent* e)
	{
	PENTER4;
	int x = e->x();
	int y = e->y();
	if (!(jogMode==JOG_ZOOM || jogMode==JOG_VERTICAL_SCROLL))
		cursorManager->update(x,y);
	if (editingMode==EDITING_MODE_TRACK_CURVES)
		{
		track[activeTrackNumber]->filterChain->followMouse(x,y);
		}
	if (jogMode==JOG_NONE)
		{
		int ccmi = cursorManager->get_current_cursor_map_index();
		if (get_clip_under_XY(x,y))
			{
			if (ccmi!=CursorManager::CURSOR_FLOAT_OVER_CLIP)
				cursorManager->set_cursor(CursorManager::CURSOR_FLOAT_OVER_CLIP);
			}
		else
			{
			if (ccmi!=CursorManager::CURSOR_FLOAT_OVER_TRACK)
				cursorManager->set_cursor(CursorManager::CURSOR_FLOAT_OVER_TRACK);
			}
		PEXIT4;
		return;
		}
	switch (jogMode)
		{
		case JOG_ZOOM:
			update_jog_zoom(e->x(),e->y());
			break;
		case JOG_SHUTTLE:
			update_shuttle_factor(tracksArea->get_mouse_x());
			break;
		case JOG_EDGE:
			update_drag_edge(e->x(),e->y());
			break;
		case JOG_FADE_IN:
		case JOG_FADE_OUT:
		case JOG_FADE_BOTH:
		case JOG_CLIP_GAIN:
   			update_jog_fade(get_mouse_x(),get_mouse_y());
			break;
		case JOG_CREATE_REGION:
			update_jog_create_region();
			break;
		case JOG_CROP:
			update_jog_crop(e->x());
			break;
		case JOG_DRAG:
			if (workingClip)
				{
				if (!isCopying)
					{
					Track* tc = workingClip->parentTrack;
					int nx = snapped_x(e->x()-cursorManager->get_x_offset());
					workingClip->set_track_first_block(xpos_to_block(nx));
					tc->recreate_mta();
					}
				}
			break;
		/*
		case JOG_DRAG_BUSMONITOR:
			break;
		*/
		case JOG_GAIN:
			update_gain(e->y());
			trackPanel[gainingTrack]->set_gain();
			trackPanel[gainingTrack]->update();
			break;
		case JOG_PAN:
			update_pan(e->x());
			trackPanel[gainingTrack]->set_pan();
			trackPanel[gainingTrack]->update();
			break;
		case JOG_GAIN_AND_PAN:
			update_pan(e->x());
			update_gain(e->y());
			trackPanel[gainingTrack]->set_gain();
			trackPanel[gainingTrack]->set_pan();
			trackPanel[gainingTrack]->update();
			break;
		case JOG_VERTICAL_SCROLL:
			update_vertical_scroll(e->y());
			break;
		}
	PEXIT4;
	}



MustuxDrawable* Song::get_tracks_area()
	{
	return (MustuxDrawable*) tracksArea;
	}



MustuxDrawable* Song::get_track_panels_area()
	{
	return (MustuxDrawable*) trackPanelsArea;
	}

void Song::set_artists(QString pArtists)
	{
	artists = pArtists;
	}


QString Song::get_artists()
	{
	return artists;
	}

void Song::set_master_gain(float pMasterGain)
	{
	masterGain = pMasterGain;
	}

void Song::set_title(QString sTitle)
	{
	title=sTitle;
	}

QString Song::get_title()
	{
	return title;
	}

int Song::get_mynumber()
	{
	return myNumber;
	}

QString Song::get_capture_dir()
	{
	return captureDir;
	}



QString Song::get_sources_dir()
	{
	return audioSourcesDir;
	}



float Song::get_master_gain()
	{
	return masterGain;
	}



int Song::render(int mode, bool toFile)
	{
	PENTER;
	if (mode==RENDER_MODE_NORMAL)
		{
		if (toFile)
			{
			QString filename( QFileDialog::getSaveFileName( title + ".praf", "Praf/Wav Files (*.praf *.wav)", this ) );
			if ( !filename.isEmpty() )
				{
				int r = mixer->render_all(filename);
				if (r<0)
					PERROR("Rendering all tracks failed");
				PEXIT;
				return 1;
				}
			}
		else
			{
			//TODO render to standard location ~/protuxprojects/Project/rendered/song_x.praf
			}
		}
	// else if (mode==...) TODO
	PEXIT;
	return -1;
	}



int Song::import_audio(QString filename, QProgressBar *monitorProgressBar)
	{
	PENTER;
	cancelImportAudio = false;
	QString originalName = filename;
	QString pureName = originalName.mid(originalName.findRev('/')+1);
	if ( !filename.isEmpty() )
		{
#ifdef OGG_VORBIS_SUPPORT
		if ((filename.find("praf",0,false)>=0) || (filename.find("wav" ,0,false)>=0) ||(filename.find("ogg" ,0,false)>=0) )
#else
		if ((filename.find("praf",0,false)>=0) || (filename.find("wav" ,0,false)>=0) )
#endif
			{
			QString targetFilename = audioSourcesDir + "/" + pureName;
			PMESG("Copying %s  to %s", (const char*)originalName.ascii(),(const char*)targetFilename.ascii());
			FILE* fSource;
			FILE* fTarget;
			fSource = fopen(originalName.ascii(),"r");
			fTarget = fopen(targetFilename.ascii(),"w+");
			if ((!fSource) || (!fTarget))
				{
				PERROR("Cannot copy this audio source to audio sources dir");
				PEXIT;
				return -1;
				}
			size_t r;
			char buf[4096];
			if (monitorProgressBar) monitorProgressBar->setTotalSteps(100);
			int count=0;
			fseek(fSource,0,SEEK_END);
			long long total=ftell(fSource);
			rewind(fSource);
			parentProject->get_default_lcd()->print("Copying file ...", 3);
			while (!feof(fSource))
				{
				if(cancelImportAudio)
					{
				        QFile file(targetFilename);
				        if (!file.remove())
						{
				                PERROR("failed to remove file %s\n", targetFilename.latin1());
                				}
					cancelImportAudio = false;
					fclose(fSource);
					PMESG("Importing audio canceled!");
					PEXIT;
					return -1;
					}
				r = fread(buf,1,4096,fSource);
				fwrite(buf,1,r,fTarget);
				count+=4096;
				if (count % 5000 == 0)
					{
					int i=(int)(100.0*(double)count/total);
					if (monitorProgressBar)
						monitorProgressBar->setProgress(i);
					qApp->processEvents();
					}
				}
			fclose(fSource);
			fclose(fTarget);
			if (monitorProgressBar)	monitorProgressBar->reset();
			parentProject->get_default_lcd()->clear_line(3);
			if (!create_clip(audioSourcesDir, pureName))
				{
				PEXIT;
				return -1;
				}
			}
#ifdef OGG_VORBIS_SUPPORT
		else if ( (filename.find("mp3" ,0,false)>=0) )
#else
		else if ( (filename.find("mp3" ,0,false)>=0) || (filename.find("ogg" ,0,false)>=0) )
#endif
			{
			QString particle;
#ifdef OGG_VORBIS_SUPPORT
				particle="MP3";
#else
			if (filename.find("mp3" ,0,false)>=0)
				particle="MP3";
			else
				particle="OGG";
#endif
			QString key = particle + "_DECODE_COMMAND";
			char* cs = GlobalProperties::get((char*)key.ascii());
			QString decodeCommand(cs);
			if ((decodeCommand.find("%if")<0) || (decodeCommand.find("%of")<0))
				{
				QString msg = particle + "Decode Command is invalid. Please change it. It must use %if and %of to place input file and output file references , respectively. Exemple : mpg123 -w %of %if";
				PERROR(msg.ascii());
				PEXIT;
				return -1;
				}
			QString inFilename = originalName;
			QString outFilename;
#ifdef OGG_VORBIS_SUPPORT
				outFilename= audioSourcesDir + "/" + pureName.replace(QRegExp(".mp3"),".wav");
#else
			if (filename.find("mp3" ,0,false)>=0)
				outFilename= audioSourcesDir + "/" + pureName.replace(QRegExp(".mp3"),".wav");
			else
				outFilename= audioSourcesDir + "/" + pureName.replace(QRegExp(".ogg"),".wav");
#endif
			decodeCommand = decodeCommand.replace(QRegExp("%if"),"\""+inFilename+"\"");
			decodeCommand = decodeCommand.replace(QRegExp("%of"),"\""+outFilename+"\"");
			printf("\n");
			PWARN("Executing External command : %s\n",decodeCommand.ascii());
			printf("Begin of output :\n");
			printf("----------------------------------------\n");
			FILE* pipe_in;
			const char* ccommand = (const char*) decodeCommand.ascii();
			pipe_in = popen (ccommand, "r");
			if (!pipe_in)
				{
				// TODO Jan error message ...
				PEXIT;
				return -1;
				}
			char c;
			while (!feof(pipe_in))
				fread(&c,1,1,pipe_in);
			pclose (pipe_in);
			printf("----------------------------------------\n");
			printf("End of output.\n\n");
#ifdef OGG_VORBIS_SUPPORT
				pureName = pureName.replace(QRegExp(".mp3"),".wav");
#else
			if (filename.find("mp3" ,0,false)>=0)
				pureName = pureName.replace(QRegExp(".mp3"),".wav");
			else
				pureName = pureName.replace(QRegExp(".ogg"),".wav");
#endif
			if (!create_clip(audioSourcesDir, pureName))
				{
				PEXIT;
				return -1;
				}
			}
		else
			{
			PERROR("Unkown Audio Format");
			PEXIT;
			return -1;
			}
		}
	update_last_block();
	PEXIT;
	return 1;
	}


void Song::cancel_import_audio()
	{
	cancelImportAudio = true;
	}


Project* Song::get_parent_project()
	{
	return parentProject;
	}



int Song::get_active_track_number()
	{
	return activeTrackNumber;
	}



void Song::touch()
	{
	PENTER;
	touch_x(tracksArea->get_mouse_x());
	PEXIT;
	}



void Song::touch_x(int x)
	{
	PENTER;
	for (int i=0; i<numTracks; i++)
		{
		if ((track[i]->is_pointed()) || (trackPanel[i]->is_pointed()))
			{
			workingBlock = xpos_to_block(x);
			cursorManager->set_work_at(x);
			PMESG("Track %d touched : Activating it...",i);
			touch_track(i);
			break;
			}
		}
	PEXIT;
	}



void Song::touch_track(int trackNumber)
	{
	PENTER2;
	track[activeTrackNumber]->deactivate();
	track[activeTrackNumber]->recreate();
	activeTrackNumber=trackNumber;
	track[activeTrackNumber]->activate();
	track[activeTrackNumber]->recreate();
	PEXIT2;
	}


// DEPRECATED
void Song::touch_in_active_track_panel()
	{
	PENTER;
	touch_in_active_track(get_mouse_x());
	PEXIT;
	}



void Song::set_active_track(int trackNumber)
	{
	if ((trackNumber>numTracks) || (trackNumber<0))
		return;
	track[activeTrackNumber]->deactivate();
	activeTrackNumber=trackNumber;
	track[activeTrackNumber]->activate();
	}



TrackPanel* Song::get_active_track_panel()
	{
	return trackPanel[activeTrackNumber];
	}



TrackPanel* Song::get_track_panel_under_y(int y)
	{
	for (int t=0; t<numTracks; t++)
		if ( (y >= track[t]->real_baseY() ) && ( y <= track[t]->real_baseY() + track[t]->height) )
			return trackPanel[t];
	return (TrackPanel*) 0;
	}



Track* Song::get_track(int trackNumber)
	{
	if ((trackNumber<0) || (trackNumber>numTracks))
		return (Track*) 0;
	return track[trackNumber];
	}



void Song::jog_gain(int t, int mouseY)
	{
	PENTER;
	if ((t < 0) || (t > numTracks))
		{
		info("INVALID TRACK");
		PEXIT;
		return;
		}
	if (jogMode == JOG_NONE)
		{
		PMESG("Starting jog gain for track %d",t);
		jogMode = JOG_GAIN;
		gainingTrack = t;
		origY = mouseY;
		origGain = track[t]->get_gain();
		}
	else
		{
		PMESG("Finishing jog gain for track %d",t);
		jogMode = JOG_NONE;
		}
	PEXIT;
	}



void Song::jog_pan(int t, int mouseX)
	{
	PENTER;
	if ((t < 0) || (t > numTracks))
		{
		info("INVALID TRACK");
		PEXIT;
		return;
		}
	if (jogMode == JOG_NONE)
		{
		PMESG("Starting jog pan for track %d",t);
		jogMode = JOG_PAN;
		gainingTrack = t;
		origX = mouseX;
		origPan = track[t]->get_pan();
		}
	else
		{
		PMESG("Finishing jog pan for track %d",t);
		jogMode = JOG_NONE;
		}
	PEXIT;
	}



void Song::jog_gain_pan(int track, int mouseX, int mouseY)
	{
	PENTER;
	if ((track < 0) || (track > numTracks))
		{
		info("INVALID TRACK");
		PEXIT;
		return;
		}
	if (jogMode == JOG_NONE)
		{
		PMESG("Starting jog gain/pan for track %d",track);
		jogMode = JOG_GAIN_AND_PAN;
		gainingTrack = track;
		origX = mouseX;
		origY = mouseY;
		}
	else
		{
		PMESG("Finishing jog gain/pan for track %d",track);
		jogMode = JOG_NONE;
		gainingTrack = -1; // NOT NEEDED .... THIS LINE CAN BE REMOVED
		}
	PEXIT;
	}



void Song::update_gain(int y)
	{
	float h = (float) height() * 0.6;
	float ofy = (float) origY - y;
	float g = 2.0f * ( ofy / h ) * 12;
	float fg = g + origGain;
	track[gainingTrack]->set_gain( fg );
	}



void Song::update_pan(int x)
	{
	float w = (float) (width() * 0.6);
	float ofx = (float) origX - x;
	float p = -2.0f *  (ofx) / w ;
	float fp = p + origPan;
	track[gainingTrack]->set_pan( fp );
	}



AudioClip* Song::create_clip(QString audiofileRootdir, QString filename, QProgressBar *monitorProgressBar)
	{
	PENTER;
	Audio* audio = new Audio(this);
	if (audio->build(audiofileRootdir,filename,monitorProgressBar)<0)
		{
		PERROR("File not Found : %s%s",(const char*) audiofileRootdir.ascii(),(const char*)  filename.ascii());
		return (AudioClip*) 0;
		}
	AudioClip* clip = create_clip(audio);
	audioSourcesList->add(audio);
	PEXIT;
	return clip;
	}



AudioClip* Song::create_clip(Audio* au, bool isTake)
	{
	return create_clip(au, isTake, track[activeTrackNumber], workingBlock);
	}



AudioClip* Song::create_clip(Audio* au, bool isTake, Track* t, long long insertPos)
	{
	PENTER;
	AudioClip* c=0;
	c = t->add_clip(au, insertPos, isTake);
	if (!c)
		{
		PERROR("Cannot add clip into track %d",activeTrackNumber);
		}
	PEXIT;
	return c;
	}



int Song::add_track()
	{
	return add_track(numTracks);
	}



int Song::add_track(int newTrackNumber)
	{
	PENTER;
	if (newTrackNumber>=numTracks)
		newTrackNumber=numTracks;
	// else if it is before, then..
	// else if it is after, then...
	track[newTrackNumber]= new Track(this,newTrackNumber+1,"Unnamed", numTracks * Track::INITIAL_HEIGHT, Track::INITIAL_HEIGHT);
	trackPanel[newTrackNumber]= new TrackPanel(track[newTrackNumber]);
	track[newTrackNumber]->set_assoc_track_panel(trackPanel[newTrackNumber]);
	numTracks++;
	recreate();
	PEXIT;
	return 1;
	}



int Song::delete_track()
	{
	return delete_track(activeTrackNumber);
	}



int Song::delete_track(int trackNumber)
	{
	PENTER;
	if ((trackNumber<0) || (trackNumber>=numTracks))
		return -1;
	if (track[trackNumber]->clipList)
		{
		QString s1; s1.setNum(track[trackNumber]->get_total_clips());
		QString mesg = "There are "+s1+" clips on this tracks\nAre you sure you want to delete it ?";
		if ( QMessageBox::warning( this, "Delete Track", mesg,"&YES", "&CANCEL", 0 , 0, 1 ) != 0)
			return -1;
		}
	Track* tT = track[trackNumber];
	TrackPanel* tP = trackPanel[trackNumber];
	if (trackNumber==numTracks-1)
		{
		PDEBUG("apagando a ultima..." );
		track[numTracks]=(Track*) 0;
		trackPanel[numTracks]=(TrackPanel*) 0;
		activeTrackNumber=numTracks-2;
		numTracks--;
		}
	else
		{
		for (int i=trackNumber+1; i<numTracks; i++)
			{
			int b = track[i-1]->get_baseY();
			int h =track[i-1]->get_height();
			track[i-1]=track[i];
			trackPanel[i-1]=trackPanel[i];
			track[i-1]->set_baseY(b);
			track[i-1]->set_height(h);
			}
		track[numTracks-1]=(Track*)0;
		trackPanel[numTracks-1]=(TrackPanel*)0;
		activeTrackNumber=trackNumber - 1 >=0 ? trackNumber - 1 : 0;
		numTracks--;
		}
	delete tT;
	delete tP;
	recreate();
	PEXIT;
	return 1;
	}



int Song::split()
	{
	PENTER;
	touch();
	track[activeTrackNumber]->split_clip(workingBlock);
	PEXIT;
	return 1;
	}



int Song::mute()
	{
	PENTER;
	bool validPoint = false;
	int pointedTrack = -1;
	AudioClip* cli = 0 ;
	for (int i=0; i < numTracks; i++)
		{
		if (track[i]->is_pointed())
			{
			validPoint = true;
			pointedTrack = i;
			}
		}
	if (validPoint)
		{
		int pointedBlock = xpos_to_block(tracksArea->get_mouse_x());
		cli = track[pointedTrack]->get_clip_under(pointedBlock);
		}
	if (cli)
		{
		cli->set_muted(!cli->isMuted);
		track[pointedTrack]->recreate_mta();
		return 1;
		}
	// no clips pointed ? so check the trackpanels
	for (int i=0; i < numTracks; i++)
		{
		if (trackPanel[i]->is_pointed())
			{
			track[i]->isMuted=!track[i]->isMuted; // should be track[i]->toggle_mute();
			trackPanel[i]->recreate();
			break;
			}
		}
	PEXIT;
	return 1;
	}



int Song::solo()
	{
	for (int i=0; i < numTracks; i++)
		{
		if (trackPanel[i]->is_pointed())
			{
			track[i]->isSolo=!track[i]->isSolo;
			trackPanel[i]->recreate();
			}
		else
			{
			if(track[i]->isSolo)
				{
				track[i]->isSolo=false;
				trackPanel[i]->recreate();
				}
			}
		}
	return 1;
	}




void Song::set_output_spectrogram(Spectrogram* spectrogram)
	{
	//FOR LATER get_bus_monitor()->set_output_vu(spectrogram);
	}



int Song::toggle_go(int goMode)
	{
	PENTER;
	int r;
	if (mixer->get_status()==Mixer::GOING)
		{
		mixer->force_stop();
		info("STOP");
		recreate_mta();
		update_last_block();
		}
	else
		{
		update_last_block();
		if (parentProject->any_other_song_going())
			{
			PWARN("Another Mixer engine is already going !");
			}
		if (goMode == GO_MODE_NORMAL)
			r = mixer->go(workingBlock);
		else if (goMode == GO_MODE_REGIONS)
			{
			if (regionList)
				{
				r = mixer->go(regionList);
				info("GOING THRU REGIONS ...");
				PMESG("GOING (REGIONS) FROM %d ...", (int) regionList->beginBlock );
				}
			else
				{
				info("No region to play !");
				}
			}
		else if (goMode == GO_MODE_LOOP_REGIONS)
			{
			// TODO
			}
		if (r<0)
			{
			info("Mixer engine cannot be started !");
			PERROR("Mixer engine cannot be started");
			PEXIT;
			return -1;
			}
		cursorManager->set_mode(CursorManager::MODE_PLAY);
		if (any_track_armed()) changed=true;
		}
	parentProject->get_bus_monitor()->recreate();
	PEXIT;
	return 1;
	}


void Song::hzoom_out()
	{
	PENTER;
	if ( hzoom < Peak::MAX_ZOOM_LEVELS - 1  )
		{
		hzoom++;
		PMESG("HZOOM Now is %d",hzoom);
		show_zoom_info();
		}
	center();
	PEXIT;
	}


void Song::hzoom_in()
	{
	PENTER;
	if (hzoom > 0)
		{
		hzoom--;
		PMESG("HZOOM Now is %d",hzoom);
		show_zoom_info();
		}
	center();
	PEXIT;
	}



void Song::vzoom_out()
	{
	PENTER;
	int lasty=LOCATOR_HEIGHT;
	for (int i=0; i<numTracks; i++)
		{
		track[i]->vzoom_out(lasty);
		lasty+=track[i]->get_height();
		}
	recreate_tracks();
	PEXIT;
	}


void Song::vzoom_in()
	{
	PENTER;
	int lasty=LOCATOR_HEIGHT;
	for (int i=0; i<numTracks; i++)
		{
		track[i]->vzoom_in(lasty);
		lasty+=track[i]->get_height();
		}
	recreate_tracks();
	PEXIT;
	}


void Song::clear_root_space()
	{
	int lasty;
	if (numTracks>0)
		lasty = track[numTracks-1]->real_baseY() + track[numTracks-1]->get_height();
	else
		lasty =0;
	tracksArea->clear(0,lasty,tracksArea->width(),tracksArea->height()-lasty);
	trackPanelsArea->clear(0,lasty,trackPanelsArea->width(),trackPanelsArea->height()-lasty);
	}



void Song::set_hzoom_level(int level)
	{
	PENTER;
	if ( ( level > 0) && ( level < Peak::MAX_ZOOM_LEVELS - 1))
		{
		hzoom = level;
		PMESG("HZOOM Now is %d",hzoom);
		show_zoom_info();
		recreate_tracks();
		}
	PEXIT;
	}





void Song::jog_zoom(int mouseX, int mouseY)
	{
	PENTER;
	if (jogMode == JOG_NONE)
		{
		cursorManager->set_cursor(CursorManager::CURSOR_MAGIC_ZOOM);
		jogZoomTotalX = width();
		jogZoomTotalY = height();
		int x = mouseX;
		verticalJogZoomLastY = mouseY;
		baseJogZoomXFactor = hzoom - ((int) ( (float) (jogZoomTotalX - x) / jogZoomTotalX * 50 ) + 1 );
		lastJogZoomXFactor = -1;
		jogMode = JOG_ZOOM;
		}
	else
		{
		cursorManager->set_cursor(CursorManager::CURSOR_FLOAT);
		jogMode = JOG_NONE;
		}
	PEXIT;
	}



void Song::update_jog_zoom(int x, int y)
	{
	int jzxfactor = (int) ( (float) (jogZoomTotalX - x) / jogZoomTotalX * 50 ) + 1;
	if (jzxfactor==0) jzxfactor=1;
	if (jzxfactor != lastJogZoomXFactor)
		{
		lastJogZoomXFactor = jzxfactor;
		hzoom = jzxfactor + baseJogZoomXFactor;
		if ( hzoom < 0 )
			hzoom = 0;
		if ( hzoom > Peak::MAX_ZOOM_LEVELS -1 )
			hzoom = Peak::MAX_ZOOM_LEVELS - 1;
		center();
		}
	int vzy = y - verticalJogZoomLastY;
	if (vzy>10)
		{
		vzoom_in();
		verticalJogZoomLastY = verticalJogZoomLastY + 10;
		}
	else if (vzy<-10)
		{
		vzoom_out();
		verticalJogZoomLastY = verticalJogZoomLastY - 10;
		}
	show_zoom_info();
	}



void Song::shuttle()
	{
	PENTER;
	if (jogMode == JOG_NONE)
		{
		PMESG("Starting shuttle");
		jogMode = JOG_SHUTTLE;
		shuttleTimer->start(25);
		}
	else
		{
		PMESG("Stopping shuttle");
		shuttleTimer->stop();
		jogMode = JOG_NONE;
		}
	PEXIT;
	}



void Song::update_shuttle_factor(int x)
	{
	float f = (float) x / tracksArea->width();
	shuttleFactor = (int) (( f * 200.0f ) - 100.0f);
	//printf("f=%2.2f    shuttleFactor=%d\n",f,shuttleFactor);
	}



void Song::update_shuttle()
	{
	float maxAmount =  ((float) SHUTTLE_SENSIBILITY / 100.0f) * 30.0f;
	int amount = (int) ( (float)(shuttleFactor/100.f) * maxAmount );
	if (amount != 0)
		{
		if (shuttleFactor < 0)
			scroll_left (-amount);
		else
			scroll_right(amount);
		recreate_mta();
		}
	}



void Song::goto_begin()
	{
	PENTER;
	firstBlock = 0;
	workingBlock = 0;
	recreate_mta();
	PEXIT;
	}



void Song::goto_end()
	{
	PENTER;
	set_work_at(get_last_block());
	center();
	PEXIT;
	}



void Song::center()
	{
	PENTER2;
	int w = tracksArea->width();
	int half = w/2;
	long long minimumBlockToCenter =  w * Peak::zoomStep[hzoom] / 2;
	if ( workingBlock >= minimumBlockToCenter )
		{
		int x = block_to_xpos(workingBlock);
		if (x<half)
			scroll_left(half - x);
		else
			scroll_right(x-half);
		}
	recreate_mta();
	PEXIT2;
	}



void Song::jog_vertical_scroll(int mouseY)
	{
	if (jogMode==JOG_NONE)
		{
		verticalScrollTotalHeight=get_floorY();
		jogMode = JOG_VERTICAL_SCROLL;
		origY = mouseY;
		}
	else
		{
		jogMode = JOG_NONE;
		}
	}



void Song::update_vertical_scroll(int y)
	{
	float f;
	int amount;
	int vsy = origY - y;
	float limiter = 1.0f;
	if (vsy < -12 || vsy > 12)
		{
		float dy = (float)MOUSE_SENSIBILITY*(((float)vsy)/tracksArea->height());
		if (dy < 0.0f) dy *= -1.0f;
		if (dy > 0.6f) dy = 0.6f;
		limiter -= dy;
		}
        if (vsy < -MOUSE_SENSIBILITY)
                {
                f = (float) vsy / (tracksArea->height());
                amount = (int) (f * verticalScrollTotalHeight * VERTICAL_SCROLL_SENSIBILITY / 100.0f * limiter);
                mtaBaseY+=amount;
                origY +=MOUSE_SENSIBILITY;
                if (mtaBaseY < -(verticalScrollTotalHeight-tracksArea->height()) + 1)
                        mtaBaseY=-(verticalScrollTotalHeight-tracksArea->height() + 1);
                if (verticalScrollTotalHeight < tracksArea->height())
                        mtaBaseY = 0;
                recreate_tracks();
                }
        if (vsy > MOUSE_SENSIBILITY)
                {
                f = (float) vsy / (tracksArea->height());
                amount = (int) (f * verticalScrollTotalHeight * VERTICAL_SCROLL_SENSIBILITY / 100.0f * limiter);
                mtaBaseY+=amount;
                origY -=MOUSE_SENSIBILITY;
                if (mtaBaseY > 0)
                        mtaBaseY = 0;
                recreate_tracks();
                }
	}



void Song::scroll_right(int amount)
	{
	PENTER3;
	firstBlock += amount*Peak::zoomStep[hzoom];
	PEXIT3;
	}



void Song::scroll_left(int amount)
	{
	PENTER3;
	firstBlock -= amount*Peak::zoomStep[hzoom];
	if (firstBlock<0)
		firstBlock=0;
	PEXIT3;
	}



void Song::scroll_down()
	{
	PENTER3;
	mtaBaseY-=30;
	recreate();
	PEXIT3;
	}



void Song::scroll_up()
	{
	PENTER3;
	if (mtaBaseY+30>0)
		mtaBaseY=0;
	else
		mtaBaseY+=30;
	recreate();
	PEXIT3;
	}



void Song::touch_in_active_track()
	{
	PENTER;
	int x = tracksArea->get_mouse_x();
	touch_in_active_track(x);
	PEXIT;
	}



void Song::touch_in_active_track(int x)
	{
	PENTER;
	workingBlock = xpos_to_block(x);
	cursorManager->set_work_at(x);
	PEXIT;
	}



Track* Song::get_track_under_y(int y)
	{
	for (int t=0; t<numTracks; t++)
		if ( (y >= track[t]->real_baseY() ) && ( y <= track[t]->real_baseY() + track[t]->height) )
			return track[t];
	return (Track*) 0;
	}



int Song::select(int selectionType, int x, int y)
	{
	PENTER;
	switch (selectionType)
		{
		case SELECTION_ALL:
			for (int i=0; i < numTracks; i++)
				{
				AudioClip* c = track[i]->clipList;
				while (c)
					{
					c->select();
					c=c->next;
					}
				}
			recreate_mta();
			return 1;
		case SELECTION_NONE:
			for (int i=0; i < numTracks; i++)
				{
				AudioClip* c = track[i]->clipList;
				while (c)
					{
					c->deselect();
					c=c->next;
					}
				}
			recreate_mta();
			return 1;
		case SELECTION_INVERSE:
			for (int i=0; i < numTracks; i++)
				{
				AudioClip* c = track[i]->clipList;
				while (c)
					{
					if (c->selected())
						c->deselect();
					else
						c->select();
					c=c->next;
					}
				}
			recreate_mta();
			return 1;
		}
	AudioClip* cli = get_clip_under_XY(x,y);
	if (cli)
		switch (selectionType)
			{
			case SELECTION_SINGLE:
				for (int i=0; i < numTracks; i++)
					{
					AudioClip* c = track[i]->clipList;
					while (c)
						{
						c->deselect();
						c=c->next;
						}
					}
				cli->select();
				break;
			case SELECTION_ADD:
				cli->select();
				break;
			case SELECTION_REMOVE:
				cli->deselect();
				break;
			case SELECTION_REGION:
				break;
			case SELECTION_ADD_REGION:
				break;
			case SELECTION_REMOVE_REGION:
				break;
			case SELECTION_DIALOG:
				break;
			}
/* FOR LATER
else // pointing track ??
		{
		for (int i=0; i < numTracks; i++)
			if (trackPanel[i]->is_pointed())
				{
				//FOR LATERtrackPanel[i]->toggle_selected();
				break;
				}
		}
*/
	recreate_mta();
	PEXIT;
	return 1;
	}


int Song::drag(bool pIsCopying, int mouseX, int mouseY)
	{
	PENTER;
	if (jogMode == JOG_NONE)
		{
		jogMode = JOG_DRAG;
		isCopying=pIsCopying;
		bool validPoint = false;
		int pointedTrack = -1;
		AudioClip* cli = (AudioClip* ) 0 ;
		for (int i=0; i < numTracks; i++)
			if (track[i]->is_pointed())
				{
				validPoint = true;
				pointedTrack = i;
				}
		if (validPoint)
			{
			int x = mouseX;
			long long pointedBlock = xpos_to_block(x); //oops shouldnt it be long long ?
			cli = track[pointedTrack]->get_clip_under(pointedBlock);
			}
		if (cli)
			{
			workingClip = cli;
			int cx1 = block_to_xpos( cli->trackFirstBlock );
			int cx2 = block_to_xpos( cli->trackLastBlock );
			int cy1 = track[pointedTrack]->real_baseY();
			int origX = mouseX;
			int origY = mouseY;
			cursorManager->set_mode(CursorManager::MODE_DRAG,
						origX,
						origY,
						cx2 - cx1,
						track[pointedTrack]->height - 5,
						origX - cx1,
						origY - cy1
						);
			cursorManager->set_cursor(CursorManager::CURSOR_DRAG);
			}
		else
			workingClip = (AudioClip*) 0;
		}
	else
		{
		if (workingClip)
			{
			bool selected = workingClip->selected();
			int x = snapped_x(tracksArea->get_mouse_x()-cursorManager->get_x_offset());
			int y = mouseY;
			long long newInsertBlock = xpos_to_block( x );
			Track* originTrack = workingClip->parentTrack;
			Track* targetTrack = get_track_under_y(y);
			if (!targetTrack)
				{
				if (!isCopying)
					{
					PMESG("Deleting clip %p",workingClip);
					originTrack->delete_clip(workingClip);
					originTrack->recreate_mta();
					}
				}
			else
				{
				AudioClip* clip = targetTrack->add_clip(workingClip,newInsertBlock);
				if (!isCopying)
					{
					originTrack->delete_clip(workingClip);
					originTrack->recreate_mta();
					if (selected) clip->select();
					}
				targetTrack->recreate_mta();
				}
			cursorManager->set_mode(CursorManager::MODE_FLOAT);
			cursorManager->set_cursor(CursorManager::CURSOR_FLOAT);
			}
		jogMode = JOG_NONE;
		workingClip=(AudioClip*) 0;
		}
	update_last_block();
	recreate_locator();
	PEXIT;
	return true;
	}



int Song::drag_edge(int mode, int mouseX)
	{
	PENTER;
	if ( jogMode == JOG_NONE )
		{
		bool validPoint = false;
		int pointedTrack = -1;
		AudioClip* cli = 0 ;
		for (int i=0; i < numTracks; i++)
			if (track[i]->is_pointed())
				{
				validPoint = true;
				pointedTrack = i;
				}
		if (validPoint)
			{
			int pointedBlock = xpos_to_block(mouseX);
			cli = track[pointedTrack]->get_clip_under(pointedBlock);
			}
		if (cli)
			{
			jogMode = JOG_EDGE;
			workingClip = cli;
			int x = mouseX;
			if (mode==0) // [E]
				{
				int cxm = block_to_xpos( cli->trackFirstBlock + (cli->get_lenght()/2));
				if (x < cxm)
					draggingEdge = LEFT_EDGE;
				else
					draggingEdge = RIGHT_EDGE;
				}
			else if (mode==1) draggingEdge = LEFT_EDGE; // [EW]
			else if (mode==2) draggingEdge = RIGHT_EDGE; // [ER]
			else if (mode==3) draggingEdge = BOTH_EDGES; // [WR]
			if (draggingEdge==RIGHT_EDGE)
				cursorManager->set_x_offset(block_to_xpos( cli->trackLastBlock ) - x);
			else
				cursorManager->set_x_offset(block_to_xpos( cli->trackFirstBlock ) - x);
			}
		}
	else
		{
		track[activeTrackNumber]->recreate();
		jogMode = JOG_NONE;
		}
	update_last_block();
	PEXIT;
	return true;
	}



void Song::update_drag_edge(int x, int y)
	{
	PENTER4;
	long long b = xpos_to_block(x + cursorManager->get_x_offset());
	long long tfb = workingClip->trackFirstBlock;
	long long tlb = workingClip->trackLastBlock;
	long long sfb = workingClip->sourceFirstBlock;
	long long slb = workingClip->sourceLastBlock;
	long long totb = workingClip->audioSource->file->totalBlocks;
	bool recr = false;
	if (draggingEdge==LEFT_EDGE)
		{
		if (sfb + (b-tfb)>=0)
			{
			workingClip->set_left_edge(b);
			recr = true;
			}
		}
	else
	if (draggingEdge==RIGHT_EDGE)
		{
		if (slb + (b-tlb) <= totb )
			{
			workingClip->set_right_edge(b);
			recr = true;
			}
		}
	else
	if (draggingEdge==BOTH_EDGES)
		{
		long long w = tlb-tfb;
		if ((sfb + (b-tfb)>=0) && (slb + (b+w-tlb) <= totb ))
			{
			workingClip->set_left_edge(b);
			workingClip->set_right_edge(b+w);
			recr = true;
			}
		}
	if (recr)
		workingClip->parentTrack->recreate_mta();// same as track[pointedtrack]->recreate(); maybe could use this second since is more clear
	PEXIT4;
	}


int Song::jog_fade(bool in, bool out)
	{
	PENTER;
	if ( jogMode == JOG_NONE )
		{
		bool validPoint = false;
		int pointedTrack = -1;
		AudioClip* cli = 0 ;
		for (int i=0; i < numTracks; i++)
			if (track[i]->is_pointed())
				{
				validPoint = true;
				pointedTrack = i;
				}
		if (validPoint)
			{
			int pointedBlock = xpos_to_block(tracksArea->get_mouse_x());
			cli = track[pointedTrack]->get_clip_under(pointedBlock);
			}
		if (cli)
			{
			if (in && out)
				jogMode = JOG_FADE_BOTH;
			else if (in)
				jogMode = JOG_FADE_IN;
			else if (out)
				jogMode = JOG_FADE_OUT;
			else
				jogMode = JOG_CLIP_GAIN;
			workingClip = cli;
			origX = get_mouse_x();
			origY = get_mouse_y();
			origGain = workingClip->gain;
			origBlockL = workingClip->fadeInBlocks;
			origBlockR = workingClip->fadeOutBlocks;
			}
		}
	else
		{
		track[activeTrackNumber]->recreate();
		jogMode = JOG_NONE;
		}
	update_last_block();
	PEXIT;
	return 1;
	}



void Song::update_jog_fade(int x, int y)
	{
	PENTER4;
	if (jogMode==JOG_FADE_IN)
		{
		long long b = (x - origX) * Peak::zoomStep[hzoom];
		long long dbi = origBlockL + b;
		dbi = dbi < 0? 0:dbi;
		workingClip->set_fade_in(dbi);
		}
	else if (jogMode==JOG_FADE_OUT)
		{
		long long b = (origX - x) * Peak::zoomStep[hzoom];
		long long dbo = origBlockR + b;
		dbo = dbo < 0? 0:dbo;
		workingClip->set_fade_out(dbo);
		}
	else if (jogMode==JOG_FADE_BOTH)
		{
		long long b = (x - origX) * Peak::zoomStep[hzoom];
		long long dbi = origBlockL + b;
		dbi = dbi < 0? 0:dbi;
		long long dbo = origBlockR + b;
		dbo = dbo < 0? 0:dbo;
		workingClip->set_fade_in(dbi);
		workingClip->set_fade_out(dbo);
		}
	else if (jogMode==JOG_CLIP_GAIN)
		{
		int dy = (origY - y);
		float g = origGain + ( (float) dy / 200 );
		workingClip->set_gain(g);
		}
	workingClip->parentTrack->recreate_mta();
	PEXIT4;
	}


int Song::process_delete()
	{
	PENTER;
	AudioClip* cli = 0 ;
	for (int i=0; i < numTracks; i++)
		{
		bool someClipDeleted=false;
		cli=track[i]->clipList;
		while (cli)
			if (cli->selected())
				{
				AudioClip* c = cli->next;
				track[i]->delete_clip(cli, true);	//FIXME See Track:delete_clip for more details
				cli=c;
				someClipDeleted=true;
				}
			else
				cli=cli->next;
		if (someClipDeleted)
			recreate_tracks();
		}
	update_last_block();
	PEXIT;
	return 1;
	}



int Song::block_to_xpos(long long block)
	{
	int x= ((block - firstBlock)  / Peak::zoomStep[hzoom]);
	return x;
	}



long long Song::xpos_to_block(int xpos)
	{
	long long b = firstBlock + xpos * Peak::zoomStep[hzoom];
	return b;
	}



int Song::get_hzoom()
	{
	return hzoom;
	}



int Song::get_vzoom() // DEPRECATED
	{
	return 0;
	}



int Song::get_zoom_step()
	{
	return Peak::zoomStep[hzoom];
	}



long long Song::get_first_block()
	{
	return firstBlock;
	}



void Song::set_first_block(long long pos)
	{
	firstBlock = (pos >=0) ? pos : 0;
	}



void Song::set_work_at(long long pos)
	{
	workingBlock=pos;
	recreate_mta();
	}



int Song::get_playing_xpos()
	{
	long long lo = mixer->get_going_pos();
	int pxp = block_to_xpos(lo - playBlocksOffset);
	return pxp;
	}



long long int Song::get_playing_block()
	{
	return mixer->get_going_pos();
	}



void Song::disable_cursors()
	{
	cursorManager->disable();
	}



void Song::enable_cursors()
	{
	cursorManager->enable();
	}



QString Song::get_mta_schema()
	{
	QString output="";

	// regions
	output = output+"<regions>\n";
	MtaRegion* m = regionList;
	while (m)
		{
		QString sb; sb.setNum((double)m->beginBlock);
		QString se; se.setNum((double)m->endBlock);
		output = output+"     <region begin=" + sb +" end=" + se + ">\n";
		m=m->next;
		}
	output = output+"</regions>\n\n";

        // tracks
	for (int i=0; i < numTracks; i++)
		output = output + track[i]->get_schema();


	return output;
	}



int Song::get_total_tracks()
	{
	return numTracks;
	}



int Song::get_mouse_x() // maybe these are deprecated... ?
	{
	return mapFromGlobal(QCursor::pos()).x();
	}



int Song::get_mouse_y() // maybe these are deprecated... ?
	{
	return mapFromGlobal(QCursor::pos()).y();
	}



int Song::get_mta_baseY()
	{
	return mtaBaseY;
	}



MustuxLcd* Song::get_default_lcd()
	{
	return parentProject->get_default_lcd();
	}



void Song::info(QString message)
	{
	parentProject->get_parent_interface()->info(message);
	}



void Song::info(QString message, int line)
	{
	parentProject->get_parent_interface()->info(message, line);
	}



void Song::show_zoom_info()
	{
	static int const ZOOM_INFO_X = 5;
	static int const ZOOM_INFO_Y = 80;
	static int const ZOOM_INFO_W = 150;
	static int const ZOOM_INFO_H = 20;
	MustuxDrawable* dataBox = parentProject->get_parent_interface()->dataBox;
	dataBox->clear(ZOOM_INFO_X,ZOOM_INFO_Y,ZOOM_INFO_W,ZOOM_INFO_H+16);
	QPainter* p = dataBox->painter;
	p->setPen(QColor(0,0,20));
	p->setFont(QFont("Helvetica",10));
	QString sHzoom; sHzoom.setNum(Peak::zoomStep[hzoom]); sHzoom = "HZ 1:" + sHzoom;
	if (hzoom <= Peak::MAX_ZOOM_USING_SOURCEFILE)
		sHzoom = sHzoom + " Micro View";
	p->drawText(ZOOM_INFO_X,ZOOM_INFO_Y+15,sHzoom);
	dataBox->update(ZOOM_INFO_X,ZOOM_INFO_Y,ZOOM_INFO_W,ZOOM_INFO_H+16);
	}



int Song::get_editing_mode()
	{
	return editingMode;
	}


void Song::set_editing_mode()
	{
	if (editingMode == EDITING_MODE_TRACK_CURVES)
		{
		for (int i=0; i<numTracks; i++)
			track[i]->set_blur(true);
		cursorManager->set_mode(CursorManager::MODE_NONE);
		cursorManager->set_work_cursor_color(255,255,255);
		track[activeTrackNumber]->filterChain->activate();
		}
	if (editingMode == EDITING_MODE_NORMAL)
		{
		for (int i=0; i<numTracks; i++)
			track[i]->set_blur(false);
		editingMode = EDITING_MODE_NORMAL;
		cursorManager->set_mode(CursorManager::MODE_FLOAT);
		cursorManager->set_work_cursor_color(250,50,50);
		for (int i=0; i<numTracks; i++)
			track[i]->filterChain->deactivate();
		}
	cursorManager->enable();
	recreate_tracks();
	}


int Song::get_status()
	{
	return mixer->get_status();
	}



long long Song::get_working_block()
	{
	return workingBlock;
	}



long long Song::get_last_block()
	{
	return lastBlock;
	}



bool Song::any_track_armed()
	{
	for (int t=0; t<numTracks; t++)
		if (track[t]->armed())
			return true;
	return false;
	}



BusMonitor* Song::get_bus_monitor()
	{
	return parentProject->get_parent_interface()->get_bus_monitor();
	}



Track* Song::get_track_under_mouse()
	{
	for (int i=0; i < numTracks; i++)
		{
		if ((trackPanel[i]->is_pointed()) || (track[i]->is_pointed()))
			return track[i];
		}
	return (Track*) 0;
	}



Mixer* Song::get_mixer()
	{
	return mixer;
	}



int Song::get_rate()
	{
	return rate;
	}



int Song::get_bit_depth()
	{
	return bitDepth;
	}



int Song::get_block_size()
	{
	return (bitDepth/8)*MustuxAudioDeviceMapper::STEREO; // IMPROVEME
	}



void Song::change_settings(int pRate, int pBitDepht)
	{
	PENTER;
	rate=pRate;
	bitDepth=pBitDepht;
	save();
        update_properties();
	PEXIT;
	}

int Song::update_properties()
	{
	mixer->validate_buses(rate,MustuxAudioDeviceMapper::STEREO,bitDepth);
	recreate();
	}

int Song::get_floorY()
	{
	if (numTracks>0)
		{
		int ct = numTracks-1;
		int by = track[ct]->get_baseY();
		int h = track[ct]->get_height();
		int fy = by+h;
		return fy;
		}
	return 0;
	}



void Song::update_last_block()
	{
	PENTER;
	// find the last blockpos in MTA
	int numTracks = get_total_tracks();
	lastBlock=0;
	for (int i=0; i<numTracks; i++)
		{
		AudioClip* clip = track[i]->clipList;
		while (clip)
			{
			if (clip->trackLastBlock >= lastBlock)
				lastBlock = clip->trackLastBlock;
			clip=clip->next;
			}
		}
	PEXIT;
	}



AudioClip* Song::get_clip_under_XY(int x, int y)
	{
	Track* t = get_track_under_y(y);
	if (t)
		return t->get_clip_under(xpos_to_block(x));
	return (AudioClip*) 0;
	}



// THIS METHOD IS TOO BIG. IT IS NECESSARY SOME KIND OF CLEANING
int Song::process_jmb_action(MustuxJogMouseBoardMessage* jmbMesg)
	{
	PENTER2;
	cursorManager->disable();
	QString actionName = jmbMesg->label;
	int parameter = jmbMesg->parameter;
	int collNumber = jmbMesg->collectedNumber;
	bool isBeginning = jmbMesg->moment;
	int x = tracksArea->get_mouse_x();
	int y = tracksArea->get_mouse_y();
	if (actionName == "EDITING MODE : NORMAL")
		{
		editingMode = EDITING_MODE_NORMAL;
		set_editing_mode();
		PEXIT2;
		return 1;
		}
	else if (actionName == "EDITING MODE : TRACK CURVES")
		{
		editingMode = EDITING_MODE_TRACK_CURVES;
		set_editing_mode();
		PEXIT2;
		return 1;
		}

	// ACTIONS ALLOWED IN ALL EDITING MODES
	if (actionName == "CANCEL")
		{
		// REVISE-ME
		/*
		if (recorder && (recorder->get_status()!=Recorder::REC_NONE))
			{
			recorder->abort();
			info("Recorder aborted!");
			}
		*/
		}
	else if (actionName == "SAVE SONG")
		{
		save();
		info("Song Sucessfully Saved !");
		PEXIT2;
		return 1;
		}
	else if (actionName == "SONG PROPERTIES")
		{
		propertiesDialog->show();
		}
	else if (actionName == "SELECT BUS IN")
		{
		if (isBeginning)
			{
			busSelector->start(BusSelector::IN);
			volatileTrack = get_track_under_mouse();
			if (!volatileTrack)
				volatileTrack = track[activeTrackNumber];
			}
		else
			{
			busSelector->stop();
			int b = busSelector->get_selected_bus();
			if ((b>=0) && (mixer->valid_capture_bus(b)))
				{
				volatileTrack->set_bus_in(b);
				}
			else
				get_default_lcd()->print("INVALID CAPTURE BUS FOR THIS SONG",2);
			recreate();
			}
		}
	else if (actionName == "SELECT BUS OUT")
		{
		if (isBeginning)
			{
			busSelector->start(BusSelector::OUT);
			volatileTrack = get_track_under_mouse();
			if (!volatileTrack)
				volatileTrack = track[activeTrackNumber];
			}
		else
			{
			busSelector->stop();
			int b = busSelector->get_selected_bus();
			if ((b>=0) && (mixer->valid_playback_bus(b)))
				{
				volatileTrack->set_bus_out(b);
				}
			else
				get_default_lcd()->print("INVALID PLAYBACK BUS FOR THIS SONG",2);
			recreate();
			}
		}
	else if (actionName == "CAPTURE FROM BOTH CHANNELS")
		{
		volatileTrack = get_track_under_mouse();
		if (volatileTrack)
			{
			volatileTrack->set_which_channels(MustuxAudioDeviceMapper::STEREO);
			volatileTrack->recreate();
			}
		}
	else if (actionName == "CAPTURE ONLY FROM LEFT CHANNEL")
		{
		volatileTrack = get_track_under_mouse();
		if (volatileTrack)
			{
			volatileTrack->set_which_channels(MustuxAudioDeviceMapper::ONLY_LEFT_CHANNEL);
			volatileTrack->recreate();
			}
		}
	else if (actionName == "CAPTURE ONLY FROM RIGHT CHANNEL")
		{
		volatileTrack = get_track_under_mouse();
		if (volatileTrack)
			{
			volatileTrack->set_which_channels(MustuxAudioDeviceMapper::ONLY_RIGHT_CHANNEL);
			volatileTrack->recreate();
			}
		}
	else if (actionName == "TOUCH")
		{
		int trackNumber = collNumber - 1;
		if (trackNumber>=0) // there is a track number collected !!
			{
			if ((trackNumber<=numTracks) && (trackNumber>=0))
				touch_track(trackNumber);
			}
		else
			touch();
		}
	else if (actionName == "TOUCH AND CENTER")
		{
		int trackNumber = collNumber - 1;
		if (trackNumber>=0) // there is a track number collected !!
			{
			if ((trackNumber<=numTracks) && (trackNumber>=0))
				touch_track(trackNumber);
			}
		else
			touch();
		center();
		}
	else if (actionName == "ADD TRACK")
		{
		add_track();
		}
	else if (actionName == "DELETE TRACK")
		{
		delete_track();
		}
	else if (actionName == "HORIZONTAL ZOOM IN")
		{
		hzoom_in();
		}
	else if (actionName == "HORIZONTAL ZOOM OUT")
		{
		hzoom_out();
		}
	else if (actionName == "VERTICAL ZOOM IN")
		{
		vzoom_in();
		}
	else if (actionName == "VERTICAL ZOOM OUT")
		{
		vzoom_out();
		}
	else if (actionName == "ZOOM MAGIC KEY")
		{
		jog_zoom(x, y);
		}
	else if (actionName == "SCROLL RIGHT")
		{
		scroll_right(20);
		recreate_mta();
		}
	else if (actionName == "SCROLL LEFT")
		{
		scroll_left(20);
		recreate_mta();
		}
	else if (actionName == "SCROLL DOWN")
		{
		scroll_down();
		}
	else if (actionName == "SCROLL UP")
		{
		scroll_up();
		}
	else if (actionName == "SHUTTLE")
		{
		shuttle();
		}
	else if (actionName == "GO/STOP")
		{
		toggle_go(GO_MODE_NORMAL);
		}
	else if (actionName == "GO (REGIONS)")
		{
		toggle_go(GO_MODE_REGIONS);
		}
	else if (actionName == "GO (LOOP REGIONS)")
		{
		toggle_go(GO_MODE_LOOP_REGIONS);
		}
	else if (actionName == "CENTER OF VIEW")
		{
		center();
		}
	else if (actionName == "GO TO BEGIN")
		{
		goto_begin();
		}
	else if (actionName == "MUTE/UNMUTE")
		{
		mute();
		}
	else if (actionName == "SOLO/UNSOLO")
		{
		solo();
		}
	else if (actionName == "TOGGLE TRACK RECORDABLE")
		{
		Track* t = get_track_under_mouse();
		if (t)
			{
			if (t->armed())
				t->disarm();
			else
				t->arm();
			}
		}
	else if (actionName == "RESET CLIP GAIN")
		{
		AudioClip* c = get_clip_under_XY(x,y);
		if (c)
			c->set_gain(1.0);
		recreate_mta();
		}
	else if (actionName == "RESET CLIP FADE IN")
		{
		AudioClip* c = get_clip_under_XY(x,y);
		if (c)
			c->set_fade_in(0);
		recreate_mta();
		}
	else if (actionName == "RESET CLIP FADE OUT")
		{
		AudioClip* c = get_clip_under_XY(x,y);
		if (c)
			c->set_fade_out(0);
		recreate_mta();
		}
	else if (actionName == "RESET CLIP BOTH FADES")
		{
		AudioClip* c = get_clip_under_XY(x,y);
		if (c)
			{
			c->set_fade_in(0);
			c->set_fade_out(0);
			}
		recreate_mta();
		}
	else if (actionName == "GAIN") // CAN BE CLIP OR TRACK GAIN
		{
		AudioClip* cli = get_clip_under_XY(tracksArea->get_mouse_x(), tracksArea->get_mouse_y());
		if (cli)
			{
			info("CLIP GAIN");
			jog_fade(false,false);
			}
		else
			{
			info("TRACK GAIN");
			int trackNumber;
			if (collNumber>=0)
				trackNumber = collNumber - 1 ;
			else
				trackNumber = activeTrackNumber;
			jog_gain(trackNumber, y);
			}
		}
	else if (actionName == "TRACK PAN")
		{
		int trackNumber;
		if (collNumber>=0)
			trackNumber = collNumber - 1;
		else
			trackNumber = activeTrackNumber;
		jog_pan(trackNumber, x);
		}

	else if (actionName == "TRACK GAIN/PAN")
		{
		int track;
		if (collNumber>=0)
			track = collNumber*10 + parameter -1;
		else
			track = parameter-1;
		jog_gain_pan(track, x , y);
		}
	else if (actionName == "DOCK/UNDOCK BUSMONITOR")
		{
		if (parentProject->get_parent_interface()->is_busmonitor_docked())
			parentProject->get_parent_interface()->busmonitor_undock();
		else
			parentProject->get_parent_interface()->busmonitor_dock();
		}
	else if (actionName == "SHOW ONLY IN BUSES")
		{
		parentProject->get_parent_interface()->get_bus_monitor()->show_only_vus_in();
		}
	else if (actionName == "SHOW ONLY OUT BUSES")
		{
		parentProject->get_parent_interface()->get_bus_monitor()->show_only_vus_out();
		}
	else if (actionName == "SHOW BOTH IN/OUT BUSES")
		{
		parentProject->get_parent_interface()->get_bus_monitor()->show_both_vus();
		}
	else if (actionName == "SHOW/HIDE CARD INFO AREA")
		{
		parentProject->get_parent_interface()->get_bus_monitor()->set_view_card_info_area();
		}



	// NORMAL EDITING MODE RESTRICTED ACTIONS
	if (editingMode == EDITING_MODE_NORMAL)
		{
		if (actionName == "SPLIT CLIP")
			{
			split();
			}
		else if (actionName == "SELECT")
			{
			select(SELECTION_SINGLE,x,y);
			}
		else if (actionName == "ADD ITEM TO SELECTION")
			{
			select(SELECTION_ADD,x,y);
			}
		else if (actionName == "REMOVE ITEM FROM SELECTION")
			{
			select(SELECTION_REMOVE,x,y);
			}
		else if (actionName == "INVERT SELECTION")
			{
			select(SELECTION_INVERSE,x,y);
			}
		else if (actionName == "SELECT ALL")
			{
			select(SELECTION_ALL,x,y);
			}
		else if (actionName == "DESELECT ALL")
			{
			select(SELECTION_NONE,x,y);
			}
		else if (actionName == "ADD REGION TO SELECTION")
			{
			select(SELECTION_ADD_REGION,x,y);
			}
		else if (actionName == "REMOVE REGION FROM SELECTION")
			{
			select(SELECTION_REMOVE_REGION,x,y);
			}
		else if (actionName == "SELECTION DIALOG")
			{
			select(SELECTION_DIALOG,x,y);
			}
		else if (actionName == "DELETE CLIP")
			{
			process_delete();
			}
		else if (actionName == "DRAG AND DROP CLIP")
			{
			drag(false, x , y);
			}
		else if (actionName == "WORK ON NEXT EDGE")
			{
			long long w = lastBlock; //WAS almost 1 billion by Remon
			for (int i=0; i<numTracks; i++)
				{
				AudioClip* c=track[i]->get_clip_after(workingBlock);
				if ((c) && (c->trackFirstBlock<w))
					w=c->trackFirstBlock;
				}
			if (w != lastBlock)
				set_work_at(w);
			}
		else if (actionName == "WORK ON PREVIOUS EDGE")
			{
			long long w = 0;
			for (int i=0; i<numTracks; i++)
				{
				AudioClip* c=track[i]->get_clip_before(workingBlock);
				if ((c) && (c->trackFirstBlock >= w))
					w=c->trackFirstBlock;
				}
			set_work_at(w);
			}
		else if (actionName == "COPY+DRAG+PASTE CLIP")
			{
			drag(true, x, y);
			}
		else if (actionName == "CLIP FADE BOTH")
			{
			jog_fade(true,true);
			}
		else if (actionName == "FG")
			{
			info("CLIP FADE IN");
			jog_fade(true,false);
			}
		else if (actionName == "CLIP FADE OUT")
			{
			jog_fade(false,true);
			}
		else if (actionName == "EXPAND/CONTRACT")
			{
			drag_edge(0, x); // [E]
			}
		else if (actionName == "EXPAND LEFT")
			{
			drag_edge(1, x); // [EW]
			}
		else if (actionName == "EXPAND RIGHT")
			{
			drag_edge(2, x); // [ER]
			}
		else if (actionName == "EXPAND LEFT AND RIGHT")
			{
			drag_edge(3, x); // [WR]
			}
		else if (actionName == "VERTICAL SCROLL")
			{
			jog_vertical_scroll(y);
			}
		else if (actionName == "CREATE REGION START")
			{
			create_region_start(tracksArea->get_mouse_x());
			}
		else if (actionName == "CREATE REGION END")
			{
			create_region_end(tracksArea->get_mouse_x());
			}
		else if (actionName == "CREATE REGION")
			{
			jog_create_region();
			}
		else if (actionName == "DELETE REGION")
			{
			delete_region_under_x(tracksArea->get_mouse_x());
			}
		else if (actionName == "IN CROP")
			{
			jog_crop(true);
			}
		else
			{
			//info("Action not allowed in this mode");
			cursorManager->enable();
			PEXIT2;
			return 1;
			}
		}
	else if (editingMode == EDITING_MODE_TRACK_CURVES)
		{
		if (actionName == "FG")
			{
			info("ADD FILTER CONTROLLER");
			if (isBeginning)
				filterSelector->start();
			else
				{
				filterSelector->stop();
				QString filterType = filterSelector->get_selected_filter_type();
				//QString s = "Adding " + filterType + " Controller";
				//parentProject->get_parent_interface()->defaultLcd->roll_message(s);
				track[activeTrackNumber]->filterChain->add_filter_controller(filterType);
				recreate();
				}
			}
		else if (actionName == "SELECT FILTER CONTROLLER")
			{
			track[activeTrackNumber]->filterChain->select_controller(isBeginning);
			}
		else if (actionName == "REMOVE CURRENT FILTER CONTROLLER")
			{
			track[activeTrackNumber]->filterChain->remove_controller();
			}
		else if (actionName == "ADD NODE")
			{
			track[activeTrackNumber]->filterChain->add_node();
			}
		else if (actionName == "DRAG AND DROP NODE")
			{
			track[activeTrackNumber]->filterChain->drag_node(isBeginning);
			}
		else if (actionName == "FILTER SETUP")
			{
			track[activeTrackNumber]->filterChain->filter_setup();
			}
		else if (actionName == "NODE SETUP")
			{
			track[activeTrackNumber]->filterChain->node_setup();
			}
		}
	cursorManager->enable();
	changed=true;
	PEXIT2;
	return 1;
	}



AudioSourcesList* Song::get_asl()
	{
	return audioSourcesList;
	}



int Song::get_clips_count_for_audio(Audio* a)
	{
	int count=0;
	for (int i=0; i<numTracks; i++)
		{
		AudioClip* c = track[i]->clipList;
		while (c)
			{
			if (c->get_audio_source()==a)
				count++;
			c=c->next;
			}
		}
	return count;
	}



int Song::remove_all_clips_for_audio(Audio* a)
	{
	PENTER;
	int counter=0;
	for (int i=0; i < numTracks; i++)
		{
		Track* t = track[i];
		AudioClip* c = t->clipList;
		while (c)
			{
			AudioClip* c2 = c;
			c=c->next;
			if (c2->get_audio_source()==a)
				{
				t->delete_clip(c2);
				counter++;
				}
			}
		}
	PEXIT;
	return counter;
	}

void Song::toggle_snap()
	{
	isSnapOn=!isSnapOn;
	}

int Song::snapped_x(int x)
	{
	int nx = x;
	if (isSnapOn)
		{
		int wx = block_to_xpos(workingBlock);
		// TODO check if it is close to upper or lower nearest clip's edges
		if (nx<10) // check if it is close to track begin
			nx=0;
		if (abs(nx-wx)<10) // first try to snap to working block
			{
			PMESG("snapping !");
			nx=wx;
			}
		}
	return nx;
	}



bool Song::is_snap_on()
	{
	return isSnapOn;
	}



void Song::jog_crop(bool type)
	{
	if ( jogMode == JOG_NONE )
		{
		jogMode = JOG_CROP;
		origBlockL = xpos_to_block(tracksArea->get_mouse_x());
		origBlockR = -1;
		}
	else
		{
		origBlockR = xpos_to_block(tracksArea->get_mouse_x());
		PMESG2("Splitting clips at %ld and %ld",(long)origBlockL,(long)origBlockR);
		for (int t =0; t<numTracks; t++)
			{
			track[t]->split_clip(origBlockL);
			track[t]->split_clip(origBlockR);
			}
		long long hb = (origBlockL+origBlockR)/2;
		PMESG2("Deleting all clips in the half point (%ld)",(long)hb);
		for (int t =0; t<numTracks; t++)
			{
			AudioClip* cli = track[t]->get_clip_under(hb);
			if (cli) track[t]->delete_clip(cli);
			}
		long long mb = origBlockR - origBlockL;
		PMESG2("Moving all clips beyond %ld and moving left in %ld blocks\n",(long)origBlockL,(long)mb);
		for (int t =0; t<numTracks; t++)
			{
			AudioClip* cli = track[t]->get_clip_after(origBlockL);
			while (cli)
				{
				long long tfb = cli->trackFirstBlock;
				cli->set_track_first_block(tfb - mb);
				cli=cli->next;
				}
			}
		jogMode = JOG_NONE;
		recreate_tracks();
		}
	update_last_block();
	PEXIT;
	}

void Song::update_jog_crop(int x)
	{
	origBlockR = xpos_to_block(tracksArea->get_mouse_x());
	int x1 = block_to_xpos(origBlockL);
	int x2 = block_to_xpos(origBlockR);
	tracksArea->painter->fillRect(x1,0,x2,tracksArea->height(),QBrush(Qt::red,Qt::Dense5Pattern));
	}

void Song::create_region_start(int xpos)
	{
	//TODO
	}



void Song::create_region_end(int xpos)
	{
	//TODO
	}



void Song::jog_create_region()
	{
	if ( jogMode == JOG_NONE )
		{
		jogMode = JOG_CREATE_REGION;
		origBlockL = xpos_to_block(tracksArea->get_mouse_x());
		origBlockR = -1;
		}
	else
		{
		origBlockR = xpos_to_block(tracksArea->get_mouse_x());
		MtaRegion* m = new MtaRegion(origBlockL,origBlockR);
		add_mta_region(m);
		jogMode = JOG_NONE;
		recreate_tracks();
		}
	update_last_block();
	PEXIT;
	}



void Song::add_mta_region(MtaRegion* m)
	{
	PENTER;
	if (!regionList)
		{
		regionList=m;
		m->prev=0;
		m->next=0;
		}
	else
		{
		MtaRegion* r = regionList;
		MtaRegion* rx = 0;
		MtaRegion* rlast;
		while (r)
			{
			rlast=r;
			if (r->beginBlock > m->beginBlock)
				{
				rx=r;
				break;
				}
			r=r->next;
			}
		if (rx)
			{
			m->prev=rx->prev;
			m->next=rx;
			if (rx->prev) rx->prev->next=m;
			rx->prev=m;
			}
		else
			{
			rlast->next=m;
			m->prev=rlast;
			m->next=0;
			}
		if (!m->prev)
			regionList=m;
		}
	PEXIT;
	}

void Song::update_jog_create_region()
	{
	recreate_tracks();
	}


void Song::set_progress_timer(bool start)
	{
	if (start && !isSomePeakBuilding && !updateProgressTimer->isActive())
		{
		updateProgressTimer->start(200);
		isSomePeakBuilding = true;
		}
	if (start)
		peakBuildCount++;
	if (!start)
		{
		if (!(peakBuildCount < 0))
			peakBuildCount--;
		if (peakBuildCount == 0)
			{
			isSomePeakBuilding=false;
			if (updateProgressTimer->isActive())
				updateProgressTimer->stop();
			recreate_mta();
			}
		}
	}


void Song::delete_region_under_x(int xpos)
	{
	}


void Song::update_progress()
	{
	recreate_mta();
	}


void Song::update_busmonitor()
	{
	bool ab;
	for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_in(); b++)
		{
		if (MustuxAudioDeviceMapper::is_bus_allocated_for_playback(b))
			{
			ab = true;
			break;
			}
		}
	if (!ab)
		{
		parentProject->get_bus_monitor()->recreate();
		BusMonitorRecreateTimer->stop();
		}
	}

void Song::cleanupEventFilter()
	{
	// This empty method is here just to avoid the annoying warning "QObject::connect: No such slot Song::cleanupEventFilter()"
	// when running protux. If anybody knows how to avoid this without implementing this method, please, send an email to
	// one of protux developers. Thanks a lot.
	}

//eof
