/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/

#include "Project.hh"
#include <qtabwidget.h>
#include <qmessagebox.h>
#include <qfile.h>
#include <qtextstream.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>

Project::Project(QString pTitle, Interface* pParentInterface)
	: QWidget( (QWidget*) pParentInterface)
	{
	PENTERCONS;
	parentInterface = pParentInterface;
	resize(500,400);

	title = pTitle;
	currentSongId = -1;
	QBoxLayout* mainBox = new QVBoxLayout( this, 1);

	mainBox->setResizeMode(QLayout::FreeResize);

	songsContainer = new QWidgetStack(this);

	songsContainer->setFocusPolicy(QWidget::NoFocus);
	mainBox->addWidget(songsContainer);
	connect( songsContainer, SIGNAL( aboutToShow(QWidget*) ), this, SLOT(changed()));

	mainBox->activate();
	userDir = QString(getenv("HOME"));

	projsDir = userDir + "/protuxprojects";
	rootDir = projsDir + "/" + title;

	for (int i=0; i<MAX_SONGS_PER_PROJECT; i++)
		song[i]=(Song*) 0;

	numSongs=0;
	PEXITCONS;
	}

Project::~Project()
	{
	PENTERDES;
	for (int i=0; i<numSongs; i++)
		delete song[i];
	PEXITDES;
	}


int Project::create(int pNumSongs)
	{
	PENTER;
	PMESG("Creating new project %s  NumSongs=%d",(const char*)title.ascii(),pNumSongs);
	FILE* f;
	char* data;
	QString sdata;

	if (mkdir((const char*)rootDir.ascii(), S_IREAD | S_IWRITE | S_IEXEC)<0)
		{
		PERROR("Cannot create dir %s", (const char*) rootDir.ascii());
		PEXIT;
		return -1;
		}

	QString sfilename = rootDir + "/project.properties";
	char* filename = (char*) sfilename.ascii();
	f = fopen(filename,"w");
	if (!f)
		{
		PERROR("Cannot open %s for write", filename);
		PEXIT;
		return -1;
		}


	sdata = "Title=" + title + "\n";
	data = (char*) sdata.ascii();
	fwrite( data, 1, sdata.length(), f );

	sdata.setNum(pNumSongs);
	sdata="NumSongs="+sdata+"\n";
	data = (char*) sdata.ascii();
	fwrite( data, 1, sdata.length(), f );

	sdata = "Engineer=" + engineer + "\n";
	data = (char*) sdata.ascii();
	fwrite( data, 1, sdata.length(), f );

	fclose(f);

	numSongs=0; // I have to zero this because each add_song will increase it properly
	for (int i=0; i< pNumSongs; i++)
		{
		QString s; s.setNum(i+1);
		QString tabLabel = "SONG "+s;
		int rate = GlobalProperties::get_int("DEFAULT_SAMPLE_RATE");
		int bitDepth = GlobalProperties::get_int("DEFAULT_BIT_DEPTH");
		song[i] = new Song(numSongs+1, this, songsContainer, rate, bitDepth);
		songsContainer->addWidget( song[i] , i);
		song[i]->build(GlobalProperties::get_int("DEFAULT_NUM_TRACKS"));
		numSongs++;
		}
	set_current_song( numSongs-1 );
	parentInterface->info("NEW PROJECT CREATED");
	update_properties();
	PEXIT;
	return 1;
	}


int Project::load() // try to load the project by its title
	{
	PENTER;
	// check if the project exists
	int r=0;
	QString sNumSongs;
	QString sCurrentSongId;
	QFile f(rootDir+"/project.properties");
	if ( f.open(IO_ReadOnly) )
		{
		QTextStream stream( &f );
		while (!stream.eof())
			{
			QString line = stream.readLine();
			line = line.stripWhiteSpace();
			if (line.find("title=")>=0)
				title=line.mid(line.find("title=")+6);
			if (line.find("numSongs=")>=0)
				sNumSongs=line.mid(line.find("numSongs=")+9);
			if (line.find("engineer=")>=0)
				engineer=line.mid(line.find("engineer=")+9);
			if (line.find("currentSongId=")>=0)
				sCurrentSongId=line.mid(line.find("currentSongId=")+14);
			}
		f.close();
		PMESG("Loading project %s  NumSongs=%s",(const char*)title.ascii(),(const char*)sNumSongs.ascii());
		int pNumSongs = sNumSongs.toInt();
		currentSongId = sCurrentSongId.toInt();
		numSongs=0;
		for (int i=0; i< pNumSongs; i++)
			{
			QString s; s.setNum(i+1);
			QString tabLabel = "SONG "+s;
			int rate = 44100; // TODO read from song_`i`/song.properties
			int bitDepth = 16; // TODO read from song_`i`/song.properties
			song[i] = new Song(numSongs+1, this, songsContainer, rate, bitDepth);
			numSongs++;
			songsContainer->addWidget( song[i] , i );
			}
		parentInterface->info("PROJECT " + title + " LOADED");
		set_current_song(currentSongId);
		r=1;
		}
	else
		{
		parentInterface->info("PROJECT " + title + " CANNOT BE LOADED !!");
		r=-1;
		}
	PEXIT;
	return r;
	}


int Project::load_songs()
	{
	int r=1;
	for (int i=0; i<numSongs; i++)
		{
		if(song[i]->load() < 0)
			{
			PMESG("couldn't load Song: %d", i);
			r=0;
			}
		}
	//update_properties();	//is this really necessary?
	return r;
	}



void Project::set_title(QString pTitle)
	{
	title = pTitle;
	}

void Project::set_engineer(QString pEngineer)
	{
	engineer=pEngineer;
	}

QString Project::get_title()
	{
	return title;
	}

int Project::get_number_of_songs()
	{
	return numSongs;
	}


QString Project::get_engineer()
	{
	return engineer;
	}

bool Project::any_other_song_going()
	{
	for (int i=0; i<numSongs; i++)
		{
		if(song[i]->get_mixer()->get_status()==Mixer::GOING)
			{
			if(i != currentSongId)
				return true;
			}
		}
	return false;
	}


bool Project::has_changed()
	{
	for (int i=0; i<numSongs; i++)
		{
		if(song[i]->changed)
			return true;
		}
	return false;
	}


bool Project::is_some_peak_building()
	{
	for (int i=0; i<numSongs; i++)
		{
		if(song[i]->isSomePeakBuilding)
			return true;
		}
	return false;
	}


Song* Project::add_song()
	{
	return add_song(GlobalProperties::get_int("DEFAULT_SAMPLE_RATE"),
			GlobalProperties::get_int("DEFAULT_BIT_DEPTH"));
	}

Song* Project::add_song(int rate, int bitDepth)
	{
	PENTER;
	if (numSongs==MAX_SONGS_PER_PROJECT)
		{
		PEXIT;
		return (Song*) 0;
		}
	QString s; s.setNum(numSongs+1);
	QString tabLabel = "SONG "+s;
	song[numSongs] = new Song(numSongs+1, this, songsContainer , rate, bitDepth);
	songsContainer->addWidget( song[numSongs], numSongs );
	song[numSongs]->build(GlobalProperties::get_int("DEFAULT_NUM_TRACKS"));
	numSongs++;
	set_current_song(numSongs-1);
	PEXIT;
	return song[numSongs-1];
	}


int Project::set_current_song(Song* pSong)
	{
	PENTER;
	if (!pSong)
		{
		PERROR("Internal Error : Trying to set current song : invalid Song object reference" );
		}
	int res=-1;
	for (int i=0; i<numSongs; i++)
		{
		if (pSong==song[i])
		{
		res=i;
		currentSongId=i;
		show_info();
		songsContainer->raiseWidget(currentSongId);
		break;
		}
		}
	PEXIT;
	return res;
	}


void Project::set_current_song(int id)
	{
	PENTER;
	if ((id<0) || (id>=MAX_SONGS_PER_PROJECT) || (id>=numSongs))
		{
		PERROR("Internal Error : Trying to set current song : invalid index %d",id );
		}
	else
		{
		currentSongId=id;
		show_info();
		songsContainer->raiseWidget(currentSongId);
		}
	PEXIT;
	}



int Project::get_current_song_id()
	{
	return currentSongId;
	}


Song* Project::get_current_song()
	{
	if (currentSongId<0)
		return (Song*) 0;
	else
		return song[currentSongId];
	}

Song* Project::get_song(int id)
	{
	if ((currentSongId<0) || (id>MAX_SONGS_PER_PROJECT))
		return (Song*) 0;
	else
		return song[id];
	}

int Project::save()
	{
	PENTER;
	FILE* f;
	char* data;
	QString sdata;

	mkdir(rootDir, S_IREAD | S_IWRITE | S_IEXEC);
	QString sfilename = rootDir + "/project.properties";
	char* filename = (char*) sfilename.ascii();
	f = fopen(filename,"w");
	if (!f)
		{
		PERROR("Cannot open %s for write", filename);
		PEXIT;
		return -1;
		}
	QString sNumSongs; sNumSongs.setNum(numSongs);
	QString sCurrentSongId; sCurrentSongId.setNum(currentSongId);
	sdata = "title=" + title + "\nnumSongs=" + sNumSongs + "\nengineer=" + engineer + "\ncurrentSongId=" + sCurrentSongId + "\n";
	data = (char*) sdata.ascii();
	fwrite( data, 1, sdata.length(), f );

	fclose(f);

	for (int i=0; i< numSongs; i++)
		{
		song[i]->save();
		}
	PEXIT;
	return 0;
	}


QString Project::get_capture_dir()
	{
	return song[currentSongId]->get_capture_dir();
	}


QString Project::get_root_dir()
	{
	return rootDir;
	}

int Project::rename()
{
return 0;
}


int Project::render()
	{
	// TODO
	return 0;
	}

void Project::show_info()
	{
	QPainter* p = parentInterface->dataBox->painter;
	p->setFont(QFont("Helvetica",7));
	parentInterface->dataBox->clear();
	for (int i=0; i<numSongs; i++)
		{
		p->setPen(Qt::black);
		if (currentSongId==i)
			{
//			p->setPen(Qt::black);
//		else
//			p->setPen(QColor(160,170,150));
			p->drawRect(i*19+5,10,16,12);
			}
		QString ss; ss.setNum(i+1);
		p->drawText(i*19+9,10,20,15,Qt::AlignAuto,ss);
		}
	QString ss; ss.setNum(currentSongId+1);
	QString sinfo = ss + " - " + get_current_song()->get_title();
	p->setPen(Qt::black);
	p->setFont(QFont("Helvetica",10));
	p->drawText(5,35,"Project : "+title);
	p->drawText(5,50,"Current Song: "+sinfo);
	QString sRate; sRate.setNum(get_current_song()->get_rate());
	QString sBitDepth; sBitDepth.setNum(get_current_song()->get_bit_depth());
	QString s = "Rate:"+sRate + " Bit Depth: "+sBitDepth;
	p->drawText(5,65,s);
	parentInterface->dataBox->update();
	}


int Project::import_audio(QString filename,QProgressBar *monitorProgressBar )
	{
	PENTER;
	int r = song[currentSongId]->import_audio(filename,monitorProgressBar);
	PEXIT;
	return r;
	}

Interface* Project::get_parent_interface()
	{
	return parentInterface;
	}

void Project::changed()
	{
	//currentSongId = songsContainer->id(songsContainer->visibleWidget());	//Hey! This is a bad thing. About to show is called BEFORE the newly set widget is shown, so setting back currentSongId to the old value!!! And after all, currentSongId is set allready somewhere else!
	}


void Project::set_output_spectrogram(Spectrogram* pSpectrogram)
	{
	spectrogram=pSpectrogram;
	for (int i=0; i< numSongs; i++)
		{
		song[i]->set_output_spectrogram(spectrogram);
		}

	}

int Project::update_properties()
	{
	for (int i=0; i< numSongs; i++)
		{
		song[i]->update_properties();
		}
	}


MustuxLcd* Project::get_default_lcd()
	{
	return parentInterface->get_default_lcd();
	}

BusMonitor* Project::get_bus_monitor()
	{
	return parentInterface->get_bus_monitor();
	}


void Project::recreate()
	{
	PENTER2;
	//All this stuff is only to avoid layout mess!
	// The paramater 145 is: interface->hbox2->height + TOPPANEL_FIXED_HEIGHT. If you have a better idea, go ahead :-)
	setMinimumWidth(parentInterface->width()-10);
	setMinimumHeight(parentInterface->height()-143);
	//updateGeometry();
	show_info();
	//FIXME: this should be moved to dataBox stuff.
	get_current_song()->show_zoom_info();
	PEXIT2;
	}
//eof

