/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include <stdio.h>
#include <math.h>
#include <qprogressdialog.h>
#include <mustux.h>
#include "Mixer.hh"
#include "Track.hh"


/**
MIXER SIGNAL ROUTING (PUT ME ON DOCS...)

	AUDIO (FROM CLIPS) --->
	TRACK GAIN -->
	CLIP GAIN -->
	CLIP FADES GAIN -->
	TRACK PAN -->
	FILTER CHAIN ->
	BUS OUT 
*/

// FIXME There's a wrong concept in this class, mixer only works if a going-thread starts, but with an
// empty MTA there is no going-thread started. Fixed it by adding some conditions (if(emptyMTA) ...),
// but it should be reviewed.

// not being used right now , but it will be to pre-mix 10 or more fragments of audio at a time..
static const int DYNAMIC_SLICE_FACTOR = 10; // improve-me get from Global Properties
static const int SAMPLE_SIZE = sizeof(short);
static const int RENDER_BUFFER_SIZE=16384; // in bytes
static const int MIN_SAMPLE = -32768;
static const int MAX_SAMPLE =  32767;

static const int NOT_DEFINED=-1;
static const float DEFAULT_SUGGESTED_FPS = 20.0f; // WILL BE USER CONFIGURABLE SOON

// ----------------- PRIVATE CLASSES FOR MIXER --------------

/*!
* \brief GoThread : This is a private class used by Mixer engine to run
*        the mixer process under a thread.
*/
class GoThread : public QThread
	{
	public:
		GoThread(Mixer* pMixer)
			{
			mixer=pMixer;
			}

		void run()
			{
			// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("Go Thread : Starting ...\n");
			if ((mixer->goStartPos==-1) && (mixer->goEndPos==-1)) // region list mode !
				{
				MtaRegion* m = mixer->regionList;
				while (m)
					{
					printf("Going thru region [%ld,%ld]\n",(long) m->beginBlock, (long) m->endBlock);
					mixer->goStartPos = m->beginBlock;
					mixer->goEndPos = m->endBlock;
					mixer->goingPos = m->beginBlock;
					while (mixer->going()>0)
						{
						// printf("Running Go Thread ... \n");
						}
					m=m->next;
					}
				mixer->status=Mixer::STOPPED;
				}
			else
				while (mixer->going()>0)
					{
					// printf("Running Go Thread ... \n");
					}
			mixer->stop();
			// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("Go Thread : Finished ...\n");
			}
	private:
		Mixer* mixer;
	};


class ListenerThread : public QThread
	{
	public:
		ListenerThread(Mixer* pMixer)
			{
			mixer=pMixer;
			}

		void run()
			{
			printf("Listener Thread : Starting ...\n");
			while (mixer->listening()>0)
				{
				//printf("RecWhileGo Thread ... \n");
				}
			printf("Listener Thread : Finishing ...\n");
			}
	private:
		Mixer* mixer;
	};




#define MIX_SAMPLES( destSample, addingSample) newSample = *(destSample) + addingSample; newSample = newSample < MIN_SAMPLE ? MIN_SAMPLE : newSample; newSample = newSample > MAX_SAMPLE ? MAX_SAMPLE : newSample; *(destSample) = (short) newSample;


/*!
* \brief A Mixer is a Song's engine (1 to 1) which
*        can playback several audio chunks from several Tracks, at
*        the same time it can recorded on pre-specified tracks (the armed)
*        tracks.
*/
Mixer::Mixer(Song* pSong)
	{
	PENTERCONS;
	assocSong = pSong;
	status=STOPPED;
	goStartPos=0;
	goEndPos=-1;
	goThread = new GoThread(this);
	listenerThread = new ListenerThread(this);
	validate_buses(assocSong->get_rate(), MustuxAudioDeviceMapper::STEREO,assocSong->get_bit_depth());
	PEXITCONS;
	}



Mixer::~Mixer()
	{
	PENTERDES;
	PEXITDES;
	}



int Mixer::validate_buses(int rate, int whichChannels, int bitDepth)
	{
	PENTER;
	for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_out(); b++)
		validPlaybackBus[b]=MustuxAudioDeviceMapper::validate_bus_out(b, rate, whichChannels, bitDepth);
	for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_in(); b++)
		validCaptureBus[b]=MustuxAudioDeviceMapper::validate_bus_in(b, rate, whichChannels, bitDepth);
	PMESG("--- Summary of all valid bus for this project ( %d Hz , STEREO, %d bits ---",assocSong->get_rate(), assocSong->get_bit_depth());
	for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_out(); b++)
		{
		if (validPlaybackBus[b])
			{
			PMESG("         %s",(const char*) MustuxAudioDeviceMapper::get_bus_out_full_name(b).latin1());
			}
		}
	for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_in(); b++)
		{
		if (validCaptureBus[b])
			{
			PMESG("         %s",(const char*) MustuxAudioDeviceMapper::get_bus_in_full_name(b).latin1());
			}
		}
	PMESG("--- Buses not listed on this summary will be disable for this project ---");
	PEXIT;
	return 1;
	}



int Mixer::arm(int busIn, int whichChannels)
	{
	PENTER;
	// TODO : LG : sync IN fps, as it was done in OUT fps (LG:REALLY NECESSARY???)
	int r = MustuxAudioDeviceMapper::open_bus_in(busIn,
						assocSong->get_rate(),
						assocSong->get_bit_depth(),
						whichChannels);
	assocSong->parentProject->get_bus_monitor()->recreate();
	if (r==MustuxAudioDeviceMapper::ERROR_INVALID_BUS)
		{
		PERROR("Invalid Bus");
		}
	else if (r==MustuxAudioDeviceMapper::ERROR_BUSY_BUS)
		{
		PERROR("Busy bus");
		QString mesg = "IN-BUS ";
		QString sbus; sbus.setNum(busIn);
		mesg.append(sbus);
		mesg.append(" IS BUSY");
		assocSong->info(mesg,1);
		}
	else if (r<0)
		PERROR("Unknown error");
	if (r>=0)
		{
		status=ARMED;
		if (!listenerThread->running()) listenerThread->start();
		PMESG("CAPTURE  THREAD START : SUCESS");
		//BusMonitor* bm=assocSong->parentProject->get_bus_monitor();	//not used anymore?? By Remon
		}
	PEXIT;
	return r;
	}



int Mixer::disarm(int busIn)
	{
	PENTER;
	int r=0;
	if (!any_track_armed())
		{
		status=STOPPED;
		listenerThread->wait(); // "... it will stop soon, cause status now is STOPPED"
		}
	bool noMoreUsed=true;
	for (int t=0; t<assocSong->get_total_tracks(); t++)
		{
		Track* track = assocSong->get_track(t);
		if ((track->armed()) && (track->get_bus_in()==busIn))
			noMoreUsed=false;
		}
	if (noMoreUsed)
		{
		r = MustuxAudioDeviceMapper::close_bus_in(busIn);
		assocSong->parentProject->get_bus_monitor()->recreate();
		if (r==MustuxAudioDeviceMapper::ERROR_INVALID_BUS)
			{
			PERROR("Invalid Bus");
			}
		else if (r==MustuxAudioDeviceMapper::ERROR_BUSY_BUS)
			{
			PERROR("Busy bus");
			}
		// else ...
		}
	PEXIT;
	return r;
	}


int Mixer::go(MtaRegion* regionList)
	{
	this->regionList = regionList;
	go(-1,-1); // this forces then region list mode
	}

int Mixer::go(long long pStartPos, long long pEndPos)
	{
	PENTER;
	if ((!any_track_armed()) && (assocSong->get_last_block()==0)) // EMPTY MTA
		{
		PMESG("PLAYBACK THREAD START : FAILED : MTA SEEMS TO HAVE NO CONTENT !");
		PEXIT;
		return ERROR_EMPTY_MTA;
		}
	if (assocSong->parentProject->get_parent_interface()->projectManager->isPlaying)
		{
		assocSong->info("ProjectManager is still playing a file!",1);
		PEXIT;
		return -1;
		}
	goStartPos = pStartPos;
	goEndPos   = pEndPos;
	goingPos   = goStartPos;
	bool anyPlaybackBusOpen = false;
	emptyMTA=true;
	commonFpsOut = NOT_DEFINED;
	int commonTransferSize;

	for (int t=0; t<assocSong->get_total_tracks(); t++)
		{
		Track* track = assocSong->get_track(t);
		if (!track->clipList)
			continue;
		emptyMTA=false;
		int busOut = track->get_bus_out();
		int r;
		if (commonFpsOut==NOT_DEFINED)
			{
			r = MustuxAudioDeviceMapper::open_bus_out(busOut,
						assocSong->get_rate(),
						assocSong->get_bit_depth(),
						track->get_which_channels(),
						true,
						DEFAULT_SUGGESTED_FPS);
			commonFpsOut = MustuxAudioDeviceMapper::get_bus_out_fps(busOut);
			}
		else
			{
			r = MustuxAudioDeviceMapper::open_bus_out(busOut,
						assocSong->get_rate(),
						assocSong->get_bit_depth(),
						track->get_which_channels(),
						true,
						commonFpsOut);
			float fps = MustuxAudioDeviceMapper::get_bus_out_fps(busOut);
			if (commonFpsOut!=fps)
				{
				PWARN("Bus %d is working on %2.1f fps, and could not be sync'ed to the commont fps (%2.1f)",busOut,fps,commonFpsOut);
				PWARN("Common transfer size for OUT buses might be invalid now.");
				}
			}
		if (r>=0)
			{
			anyPlaybackBusOpen = true;
			commonTransferSize = MustuxAudioDeviceMapper::get_bus_out_transfer_size(busOut); // it will be overwritten everytime, but no problem, since it is supposed to be the same value
			}
		}
	if (emptyMTA)
		{
		listenPos = goStartPos;
		PMESG("MTA Seems to be empty. Performing only recording (No Playback Bus will be open)");
		}
	if ((!any_track_armed()) && (!anyPlaybackBusOpen) && !emptyMTA)
		{
		PERROR("PLAYBACK THREAD START : FAILED : None of required playback bus could be open. Try changing bus out on the tracks, please");
		assocSong->info("CANNOT GO :-( ... ");
		}
	else
		{
		status=GOING;
		goingStep =  commonTransferSize / assocSong->get_block_size(); // goingStep is in blocks
		if(!emptyMTA)
			if (!goThread->running())
				{
				goThread->start();
				PMESG("PLAYBACK  THREAD START : SUCESS");
				}
		}
	PEXIT;
	return 1;
	}




void Mixer::force_stop()
	{
	PENTER;
	if (any_track_armed())
		status = ARMED;
	else
		status = STOPPED;
	if(emptyMTA)
		stop();
	PEXIT;
	}




int Mixer::stop()
	{
	PENTER;
	int r = MustuxAudioDeviceMapper::close_all_buses_out();
	if (r<0)
		{
		PWARN("Cannot close one or more playback buses. Continuing anyway ...");
		}
	for (int t=0; t<assocSong->get_total_tracks(); t++)
		{
		Track* track = assocSong->get_track(t);
		if (track->cleanup_buffers() < 0)
			PWARN("Track %d cleanup_buffers returned an error!", t + 1);
		// ... if (status==ABORTED) track->discard_recorded_clip(); // (this abort is for later...)
		}
	assocSong->BusMonitorRecreateTimer->start(25);
	PEXIT;
	return 1;
	}



long long Mixer::get_going_pos()
	{
	if(!emptyMTA)
		return goingPos;
	return listenPos;
	}



int Mixer::going()
	{
	PENTER4;
	if ((status==STOPPED) || (status==ARMED))
		{
		PEXIT4;
		return -1; // returning -1 forces GO thread to stop
		}

	if ((!any_track_armed()) && (goingPos > assocSong->get_last_block()))
		{
		status = STOPPED;
		PEXIT4;
		return -1; // returning -1 forces GO thread to stop
		}

	if ( (goEndPos>0) && (goingPos > goEndPos))
		{
		PEXIT4;
		return -1;
		}

	// playback process (in all non-muted tracks)
	bool thereAreSolos = false;
	for (int t=0; t<assocSong->get_total_tracks(); t++)
		if (assocSong->get_track(t)->is_solo())
			{
			thereAreSolos=true;
			break;
			}
	for (int t=0; t<assocSong->get_total_tracks(); t++)
		{
		Track* track=assocSong->get_track(t);
		int busOut = track->get_bus_out();
		int busBufferBlockSize = MustuxAudioDeviceMapper::STEREO * SAMPLE_SIZE; // for later MustuxAudioDeviceMapper::get_bus_out_block_size(busOut);
		if (!MustuxAudioDeviceMapper::is_bus_allocated_for_playback(busOut))
			continue;
		if ( thereAreSolos && (!track->is_solo()))
			continue;
		if (track->is_muted())
			continue;
		// source and mix buffers
		char* busBuffer = MustuxAudioDeviceMapper::get_bus_out_transfer_buffer(busOut, track->get_which_channels());
		int busBufferSize = MustuxAudioDeviceMapper::get_bus_out_transfer_size(busOut);
		int maxSourceBufferSize = busBufferSize;
		char* sourceBuffer = new char[maxSourceBufferSize];
		int sat = mix_track_portion(track, sourceBuffer, maxSourceBufferSize, busBuffer, busBufferSize, busBufferBlockSize);
		if (sat>0) // Saturation (clipping)
			{
			PMESG("Detected %d over(s), please lower the output volume...", sat);
			// do something ...
			}
		delete sourceBuffer;
		}
	// The buses are transferred. Notice that this is done
	// syncronizedly, which is BAD. There WILL be a lag between each bus transfer.
	// It must be improved, because all buses must transferred at the SAME TIME!
	// Update (LG) : there is a threaded transfer support in MADM now, but it is
	// still experimental... The best thing to do is to perform the transfer at the
	// smallest fragment size possible.
	for (int b = 0; b<MustuxAudioDeviceMapper::get_total_buses_out(); b++)
		{
		if (MustuxAudioDeviceMapper::is_bus_allocated_for_playback(b))
			{
			MustuxAudioDeviceMapper::bus_out_transfer(b);
			MustuxAudioDeviceMapper::clean_bus_out(b);
			}
		}
	goingPos += goingStep;
	PEXIT4;
	return 1;
	}


int Mixer::listening() // should be called "listen"
	{
	PENTER4;
	// Capture process ( in all armed tracks)
	if (status==STOPPED)
		{
		printf("Listener Thread : status seems to be STOPPED. Finishing...\n");
		PEXIT4;
		return -1;
		}
	for (int t=0; t<assocSong->get_total_tracks(); t++)
		{
		Track* track=assocSong->get_track(t);
		if (!track->armed())
			continue;

		int busIn = track->get_bus_in();

		if ( MustuxAudioDeviceMapper::is_bus_allocated_for_capture(busIn) )
			{
			MustuxAudioBusTransferResult* tr = MustuxAudioDeviceMapper::bus_in_transfer(busIn);
			if ( tr->error==0 )
				{
				char* captureBuffer = MustuxAudioDeviceMapper::get_bus_in_transfer_buffer(busIn, track->get_which_channels());
				int captureBufferSize = MustuxAudioDeviceMapper::get_bus_in_transfer_size(busIn);
				if (status==GOING)
					{
					track->feed_recording_clip(captureBuffer,captureBufferSize);
					listenPos += captureBufferSize / assocSong->get_block_size(); // listenStep is in blocks
					}
				}
			else
				PERROR("Cannot read from bus ");
			}
		}
	PEXIT4;
	return 1;
	}


int Mixer::abort()
	{
	status = ABORTED;
	goThread->wait();
	PEXIT;
	return 1;
	}

int Mixer::get_status()
	{
	return status;
	}



int Mixer::render_all(QString filename)
	{
	PENTER;
	assocSong->update_last_block();
	long long renderStartPos=0;// IMPROVEME
	long long renderEndPos=assocSong->get_last_block();// IMPROVEME
	long long renderWidth = renderEndPos-renderStartPos;
	PMESG("Rendering Song : ");
	PMESG("   Output File : %s", (const char*)filename.ascii());
	PMESG("   Range       : %ld to %ld ", (long)renderStartPos, (long)renderEndPos);
	QString totalTimeInSmpte= assocSong->cursorManager->block_to_smpte(renderWidth,assocSong->rate);
	PMESG("   Total Time  : %s ", totalTimeInSmpte.ascii());
	bool thereAreSolos = false;
	for (int t=0; t<assocSong->get_total_tracks(); t++)
		if (assocSong->get_track(t)->is_solo())
			{
			thereAreSolos=true;
			break;
			}
	int renderBufferSize = RENDER_BUFFER_SIZE; // in bytes
	char* renderBuffer = new char[renderBufferSize];
	int sourceBufferSize = renderBufferSize;
	char* sourceBuffer = new char[sourceBufferSize];
	goStartPos = renderStartPos;
	goingPos = goStartPos;
	goingStep = sourceBufferSize / assocSong->get_block_size();
	long long renderAmount = 0 ;
	int i=0;
	QString s(filename);
	QProgressDialog progress( "Rendering...", "Abort Render", 100, this, "progress", TRUE );
	qApp->processEvents();

	MustuxAudioFileFormat* f;
	if (s.find(".praf",0,false)>0) // render to praf
		f = new PrafFile();
	else
		f = new WavFile();

	f->create((char*)filename.latin1());
	f->write_header(Audio::STEREO,
			assocSong->get_rate(),
			assocSong->get_bit_depth(),
			0,
			"RenderAllTest",
			"Test",
			"Test",
			"Output audio from RenderAll action");
	qApp->processEvents();
	while (goingPos<renderEndPos)
		{
		int prog =(int) (((double)renderAmount/(double)renderWidth)*100.0f);
		if (prog%2==0)
			{
			printf("--> Rendered %d %:\r",prog);
			progress.setProgress( prog );
			qApp->processEvents();
			}
		if ( progress.wasCancelled() )
			break;
		bzero(renderBuffer,renderBufferSize);
		for (int t=0; t<assocSong->get_total_tracks(); t++)
			{
			Track* track=assocSong->get_track(t);
			if (thereAreSolos && (!track->is_solo())) continue;
			if (track->is_muted()) continue;
			int sat = mix_track_portion(track, sourceBuffer, sourceBufferSize, renderBuffer, renderBufferSize, Audio::STEREO * SAMPLE_SIZE); // for now, only stereo renderings
			if (sat>0) // Saturation (clipping)
				{
				// do something ...
				}
			}
		f->append_samples_from_buffer( renderBuffer, renderBufferSize); // improve-me !!! the last chunck will have too much extra zeros...
		goingPos += goingStep;
		renderAmount += goingStep;
		}
	PMARK( "1");
	f->fix_audio_size();
	f->close_file();
	delete f;
	PMESG("Rendering Finished");
	progress.setProgress( 100 );
	qApp->processEvents();
	delete sourceBuffer;
	delete renderBuffer;
	PEXIT;
	return 1;
	}



bool Mixer::any_track_armed()
	{
	return assocSong->any_track_armed();
	}



bool Mixer::valid_playback_bus(int busId)
	{
	return validPlaybackBus[busId];
	}



bool Mixer::valid_capture_bus(int busId)
	{
	return validCaptureBus[busId];
	}




int Mixer::mix_track_portion( Track* track,
				char* sourceBuffer,
				int sourceBufferSize,
				char* mixBuffer,
				int mixBufferSize,
				int mixBufferBlockSize
				)
	{
	PENTER4;
	long long p;
	int newSample;
	bool monoSource;
	int overcounter=0;

	AudioClip* clip = track->get_clip_under(goingPos);
	if (!clip)
		clip = track->get_clip_after(goingPos);
	while ((clip) && (clip->trackFirstBlock < goingPos + goingStep))
		{
		p = ( ( clip->trackFirstBlock < goingPos ) ? goingPos : clip->trackFirstBlock );
		float clipGainFactor = clip->gain;
		if (!clip->is_muted())
			{
			long long sourceFirstBlockToMix = clip->sourceFirstBlock + ( goingPos - clip->trackFirstBlock );
			int mixBufferRelativeMixStartBlock = (int) p - goingPos;
			int lenght;
			if      ( (clip->trackFirstBlock < goingPos) && (clip->trackLastBlock > goingPos + goingStep) )
				lenght=goingStep;
			else if ( (clip->trackFirstBlock < goingPos)  && (clip->trackLastBlock <= goingPos+goingStep) )
				lenght=clip->trackLastBlock - goingPos;
			else if ( (clip->trackFirstBlock >= goingPos)  && (clip->trackLastBlock > goingPos+goingStep) )
				lenght = goingPos + goingStep - clip->trackFirstBlock;
			else
				lenght=clip->get_lenght();
			// get the source file
			Audio* audioSource = clip->audioSource;
			audioSource->file->block_seek(sourceFirstBlockToMix);
			// get the audio characteristics
			int numSamplesToReadFromSource;
			if (audioSource->file->channels == Audio::MONO)
				{
				monoSource = true;
				numSamplesToReadFromSource = lenght;
				}
			else
				{
				monoSource = false;
				numSamplesToReadFromSource = 2 * lenght;
				}

			// read the audio from the file
			int bytesRead = audioSource->file->read_fragment(sourceBuffer, numSamplesToReadFromSource * SAMPLE_SIZE); // bytesRead is not being used right now, but it can be in the future...
			char* pSourceBuffer = sourceBuffer;
			// calculate the destination point in the mixbuff, where the data will be mixed onto.
			char* pPreBuffer = mixBuffer + ( mixBufferRelativeMixStartBlock * mixBufferBlockSize);

			// retrieve track Gain and Pan
			float trackGain = track->get_gain();
			float trackGainFactor = (trackGain>=0.0f)? 1.0f + trackGain/6.0f : 1.0f / pow(2.0f,-1.0f*trackGain/6.0f);
			float panFactor = track->get_pan();
			float panL=1.0;
			float panR=1.0;
			panR = panFactor>=0 ? 1.0 : 1.0 + panFactor;
			panL = panFactor>=0 ? 1.0 - panFactor : 1.0;
			int sampleWithAppliedGain;
			int srcCounter=0;
			int channel=0;

			while (srcCounter < numSamplesToReadFromSource ) // && (playable))
				{
				float fadeAndCrossFadeGainFactor = clip->get_fade_factor_for(goingPos+srcCounter);
				short* sps = (short*) pSourceBuffer;
				short sample = *(sps);

				// APPLY GAINS, ACCORDDING TO THE SIGNAL ROUTING
				float fSampleWithAppliedGain = (float) sample * trackGainFactor * clipGainFactor * fadeAndCrossFadeGainFactor;

				// APPLY TRACK PAN (only if it is a stereo clip)
				if (!monoSource)
					{
					fSampleWithAppliedGain *= channel==1 ? panR : panL;
					sampleWithAppliedGain = (short) fSampleWithAppliedGain;
					short* spm = (short*) pPreBuffer;
					if(fSampleWithAppliedGain < MIN_SAMPLE || fSampleWithAppliedGain > MAX_SAMPLE)
						overcounter++;
					MIX_SAMPLES(spm,sampleWithAppliedGain);
					}
				else
					{
					// for mono source, handle left and right channel at ones, then jump to next sample
					float fSampleWithAppliedGainLeft  = fSampleWithAppliedGain * panL;
					float fSampleWithAppliedGainRight = fSampleWithAppliedGain * panR;

					//and apply at for left channel
					sampleWithAppliedGain = (short) fSampleWithAppliedGainLeft;
					short* spm = (short*) pPreBuffer;
					if(fSampleWithAppliedGain < MIN_SAMPLE || fSampleWithAppliedGain > MAX_SAMPLE)
						overcounter++;
					MIX_SAMPLES(spm,sampleWithAppliedGain);

					//and also for the right channel, by incrementing the position by SAMPLE_SIZE in pPreBuffer
					pPreBuffer+=SAMPLE_SIZE;
					spm = (short*) pPreBuffer;
					sampleWithAppliedGain = (short) fSampleWithAppliedGainRight;
					spm = (short*) pPreBuffer;
					if(fSampleWithAppliedGain < MIN_SAMPLE || fSampleWithAppliedGain > MAX_SAMPLE)
						overcounter++;
					MIX_SAMPLES(spm,sampleWithAppliedGain);

					//now we can jump right onto the next sample by making channel=1;
					channel = 1;
					}

				pSourceBuffer+=SAMPLE_SIZE;
				pPreBuffer+=SAMPLE_SIZE;
				srcCounter++;
				if (++channel>1) channel=0;
				}
			/*FOR LATER int lastSubClipBlock = relativeSliceBlock + lenght;
			if (lastSubClipBlock>maxPlayableBlockInSlice)
				maxPlayableBlockInSlice=lastSubClipBlock;*/
			}
		clip=clip->next;
		}
	// Process Filter Chain
	track->filterChain->process(mixBuffer,mixBufferSize);
	PEXIT4;
	return overcounter; // improve-me .. must return the amount overloaded.
	}
//eof

