/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include <mustux.h>
#include "BusMonitor.hh"
#include "Project.hh"

#include <stdlib.h>
#include <math.h>
#include <qframe.h>

#define MAX_SAMPLE 32768
#define SPACE_BETWEEN_BUSES 3
#define SPACE_BETWEEN_CHANNELS 2
#define VU_CHANNEL_WIDTH 7
#define TITLE_TO_BUS_DISTANCE 18
#define BUS_WIDTH 54
#define CARD_INFO_BOX_WIDTH 70
#define INDICATOR_WIDTH 25
#define LEVEL_CLEAR_COLOR QColor(60,70,60)

#define PERCENTUAL_OF_HEIGHT_USED_FOR_VUS 0.90f
#define MAX_CARDINFOAREA_HEIGHT 40

BusMonitor::BusMonitor(int minWidth, int minHeight, QWidget* parent, Interface* pAssocInterface)
	: QWidget( (QWidget*) parent, "", WStyle_Customize | Qt::WStyle_Tool )
	{
	PENTERCONS;
/*	printf("\n");
	printf("Testing  db to scale :    0dB = %2.1f\n",dB_to_scale_factor(0));
	printf("Testing  db to scale :   -6dB = %2.1f\n",dB_to_scale_factor(-6));
	printf("Testing  db to scale : -120dB = %2.1f\n",dB_to_scale_factor(-120));
	printf("Testing  scale to db:     0.0 = %2.1f\n",coefficient_to_dB(0.0f));
	printf("Testing  scale to db :    0.5 = %2.1f\n",coefficient_to_dB(0.5f));
	printf("Testing  scale to db :    1.0 = %2.1f\n",coefficient_to_dB(1.0f));
	printf("\n");
*/
	parentWidget = parent;
	assocInterface = pAssocInterface;
        QVBoxLayout* mainBoxLayout = new QVBoxLayout(this, 1);
	mainBoxLayout->setMargin(0);

 	// will be configurable in the future (LG)
	presetMark[0]=-3.0f;
	presetMark[1]=-6.0f;
	presetMark[2]=-9.0f;
	presetMark[3]=-18.0f;
	presetMark[4]=-24.0f;
	presetMark[5]=-48.0f;
	for (int k=0; k<BusMonitor::MAX_CHANNELS_PER_BUS; k++)
		for (int l=0; l<MustuxAudioDeviceMapper::get_total_buses_out(); l++)
			{
			tailDeltaYIn[l][k]=0.0f;
			tailDeltaYOut[l][k]=0.0f;
			}
	activeTail=false;

        hbox = new QWidget(this);
	hbox->setBackgroundColor(Qt::green);
	hboxlayout = new QHBoxLayout(hbox);
	mainBoxLayout->addWidget(hbox, 4);//just some stretch (4) to make it look nicer when resizing horizontally!
	setMinimumWidth(minWidth);
	setMinimumHeight(minHeight);
	setCaption("Bus Monitor");

	splitterbox = new QWidget (hbox);
	splitterlay = new QVBoxLayout(splitterbox);
	int totalBuses = MustuxAudioDeviceMapper::get_total_buses_in();
	int totalBusAreaWidth =  (totalBuses * (BUS_WIDTH + SPACE_BETWEEN_BUSES))+ INDICATOR_WIDTH;
	busInArea    = new SingleBusMonitor(totalBusAreaWidth, minHeight/2, splitterbox, this);
	busOutArea   = new SingleBusMonitor(totalBusAreaWidth, minHeight/2, splitterbox, this);
	cardInfoArea = new SingleBusMonitor(CARD_INFO_BOX_WIDTH, minHeight, hbox, this);
	//Start with no cardInfoArea visible!
	cardInfoArea->hide();
	busAreaSplitter = new QWidget(splitterbox);
	busAreaSplitter->setMaximumHeight(2);
	busAreaSplitter->setMinimumWidth(totalBusAreaWidth);

	splitterlay->add(busInArea);
	splitterlay->add(busAreaSplitter);
	splitterlay->add(busOutArea);

	busInArea->set_background_color(20,20,60);
	busOutArea->set_background_color(20,40,50);
	cardInfoArea->set_background_color(20,60,40);

	hboxlayout->addWidget(splitterbox, 4);

	onlyVUsOut = false;
	onlyVUsIn = false;
	showCardInfoArea = false;

	updateTimer = new QTimer(this);
	connect( updateTimer , SIGNAL(timeout()), this, SLOT(update()));

	// FOR LATER spectrogram=new Spectogram(...,splitterbox);
	PEXITCONS;
	}


BusMonitor::~BusMonitor()
	{
	PENTERDES;
	delete updateTimer;
	delete cardInfoArea;
	delete busOutArea;
	delete busInArea;
	PEXITDES;
	}


void BusMonitor::init()
	{
	PENTER;
	for (int j=0; j < MustuxAudioDeviceMapper::MAX_BUSES; j++)
		for (int i=0; i < MAX_CHANNELS_PER_BUS; i++)
			{
			levelOut[j][i] = 0;
			levelIn[j][i] = 0;
			}
	updateTimer->start(50);
	PEXIT;
	}

void BusMonitor::finish()
	{
	PENTER;
	updateTimer->stop();
	PEXIT;
	}

void BusMonitor::recreate()
	{
	PENTER3;
	if (!assocInterface || !assocInterface->get_current_project() || !assocInterface->get_current_project()->get_current_song() || !busInArea || !busOutArea || !cardInfoArea)
		{
		PEXIT3;
		return;
		}
	busInArea->clear();
	busOutArea->clear();
	cardInfoArea->clear();

	// input buses
	int pos = 0;
	if (!onlyVUsOut)
		{
		for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_in(); b++)
			{
			if (!assocInterface->get_current_project()->get_current_song()->get_mixer()->valid_capture_bus(b))
				continue;
			bool ab = MustuxAudioDeviceMapper::is_bus_allocated_for_capture(b);
			QString title=MustuxAudioDeviceMapper::get_bus_in_name(b);
			draw_bus_vu_base(busInArea,
					(pos++) * (BUS_WIDTH + SPACE_BETWEEN_BUSES),
					0,
					MustuxAudioDeviceMapper::is_stereo_bus_in(b) ? 2 : 1,
					title.left(title.find("-")-1),
					ab,
					b);
			}
		QString title = "IN";
		draw_indicators(busInArea, (pos * (BUS_WIDTH + SPACE_BETWEEN_BUSES)), 0, title);
		busInArea->update();
		}

	// output buses
	pos = 0;
	if (!onlyVUsIn)
		{
		for (int b=0; b<MustuxAudioDeviceMapper::get_total_buses_out(); b++)
			{
			if (!assocInterface->get_current_project()->get_current_song()->get_mixer()->valid_playback_bus(b))
				continue;
			bool ab = MustuxAudioDeviceMapper::is_bus_allocated_for_playback(b);
			QString title = MustuxAudioDeviceMapper::get_bus_out_name(b);
			draw_bus_vu_base(busOutArea,
					(pos++) * (BUS_WIDTH + SPACE_BETWEEN_BUSES),
					0,
					MustuxAudioDeviceMapper::is_stereo_bus_out(b) ? 2 : 1,
					title.left(title.find("-")-1),
					ab,
					b);
			}
		QString title = "OUT";
		draw_indicators(busOutArea, pos * (BUS_WIDTH + SPACE_BETWEEN_BUSES), 0, title);
		busOutArea->update();
		}

	// card info boxes
	int h = cardInfoArea->height();
	if (showCardInfoArea)
		{
		for (int c=0; c<MustuxAudioDeviceMapper::get_total_cards(); c++)
			{
			QPainter* p = cardInfoArea->painter;
			p->fillRect( c * CARD_INFO_BOX_WIDTH , 0 , CARD_INFO_BOX_WIDTH , h, QColor(60,70,60) );
			p->setFont(QFont("Helvetica",8));
			p->setPen(Qt::white);
			QRect r(c * CARD_INFO_BOX_WIDTH, 5,CARD_INFO_BOX_WIDTH,10);
			p->drawText(r, Qt::AlignCenter, MustuxAudioDeviceMapper::get_card_name(c));
			p->setPen(Qt::yellow);
			p->drawLine( c * CARD_INFO_BOX_WIDTH, 0, c * CARD_INFO_BOX_WIDTH, h );
			p->drawLine( (c+1) * CARD_INFO_BOX_WIDTH, 0 , (c+1)*CARD_INFO_BOX_WIDTH , h);
			}
		}
	cardInfoArea->update();
	PEXIT3;
	}


void BusMonitor::update()
	{
	PENTER4;
	if (isActiveWindow() && isTopLevel())
		assocInterface->setActiveWindow();

	if (!MustuxAudioDeviceMapper::is_any_bus_allocated())
		{
		if (activeTail)
			draw_active_tails();
		PEXIT4;
		return;
		}

	// update out levels
	for (int busOut=0; busOut<MustuxAudioDeviceMapper::get_total_buses_out(); busOut++)
		{
		if (!MustuxAudioDeviceMapper::is_bus_allocated_for_playback(busOut))
			continue;
		int channels = MustuxAudioDeviceMapper::is_stereo_bus_out(busOut) ? 2 : 1;
		float* lvl = MustuxAudioDeviceMapper::get_bus_out_levels(busOut);
		for (int k=0; k<channels; k++)
			levelOut[busOut][k]=lvl[k]; // levels must store the real sample values // WAS float f = lvl[k]; levelOut[busOut][k] = (int)((f / MAX_SAMPLE) * maxVULevel);
		}

	// update in levels
	for (int busIn=0; busIn<MustuxAudioDeviceMapper::get_total_buses_out(); busIn++)
		{
		if (!MustuxAudioDeviceMapper::is_bus_allocated_for_capture(busIn))
			continue;
		int channels = MustuxAudioDeviceMapper::is_stereo_bus_in(busIn) ? 2 : 1;
		float* lvl = MustuxAudioDeviceMapper::get_bus_in_levels(busIn);
		for (int k=0; k<channels; k++)
			levelIn[busIn][k] = lvl[k];
		}


	// draw playback buses levels
	int pos=0;
	for (int busOut = 0; busOut < MustuxAudioDeviceMapper::get_total_buses_out(); busOut++)
		{
		if (!assocInterface->get_current_project()->get_current_song()->get_mixer()->valid_playback_bus(busOut))
			continue;
		if (MustuxAudioDeviceMapper::is_bus_allocated_for_playback(busOut))
			{
			int channels = MustuxAudioDeviceMapper::is_stereo_bus_out(busOut) ? 2 : 1;
			draw_bus_vu_peaks(busOutArea,
				pos * (BUS_WIDTH + SPACE_BETWEEN_BUSES),
				0,
				channels,
				MustuxAudioDeviceMapper::PLAYBACK,
				levelOut[busOut],
				tailDeltaYOut[busOut]);
			}
		pos++;
		}

	// draw capture buses levels
	pos = 0;
	for (int busIn = 0; busIn < MustuxAudioDeviceMapper::get_total_buses_in(); busIn++)
		{
		if (!assocInterface->get_current_project()->get_current_song()->get_mixer()->valid_capture_bus(busIn))
			continue;
		if (MustuxAudioDeviceMapper::is_bus_allocated_for_capture(busIn))
			{
			int channels = MustuxAudioDeviceMapper::is_stereo_bus_in(busIn) ? 2 : 1;
			draw_bus_vu_peaks(busInArea,
				pos * (BUS_WIDTH + SPACE_BETWEEN_BUSES),
				0,
				channels,
				MustuxAudioDeviceMapper::CAPTURE,
				levelIn[busIn],
				tailDeltaYIn[busIn]);
			}
		pos++;
		}
	PEXIT4;
	}


void BusMonitor::draw_indicators(SingleBusMonitor* bus, int bx, int by, QString title)
	{
        QPainter* p = bus->painter;
	int fontSize = 9;
	int fontSpacing = 2;
	int maxh =  bus->height();
	int x = bx + 5;
	int y = (maxh/2) - (title.length()*(fontSize+fontSpacing) + 2)/2;

	// draw the base rectangle and indicator title
	p->fillRect( bx , by , INDICATOR_WIDTH , maxh , QColor(60,70,60) );
	p->setPen(Qt::white);
	p->setFont(QFont("Helvetica",fontSize));
	for (int i=0; i<title.length(); i++)
		{
		QRect r(bx + 5, y, fontSize+2, fontSize+fontSpacing);
		p->drawText (r, Qt::AlignCenter, title.mid(i,1));
		y += fontSize + 4;
		}
	}

void BusMonitor::draw_bus_vu_base(SingleBusMonitor* bus,
				int bx,
				int by,
				int channels,
				QString title,
				bool isActiveBus,
				int busId)
	{
	PENTER3;
        QPainter* p = bus->painter;
	int maxh =  bus->height();
	int bottony = by + maxh;
	int levelRange = (int) (maxh * PERCENTUAL_OF_HEIGHT_USED_FOR_VUS);
	int vuBottonYoffset = (maxh - levelRange)/2;

	// draw the base rectangle and bus title
	p->fillRect( bx , by , BUS_WIDTH , maxh , QColor (140+(isActiveBus?100:0),160,120) );
	p->setPen(Qt::blue);
	p->setFont(QFont("Helvetica",8));
	// draw bus number
	QString busNumber; busNumber.setNum(busId+1);
        p->setFont( QFont( "Helvetica", 9, QFont::Bold) );
        p->setPen(QColor(255,255,255));
        p->drawText(5,bottony - levelRange + 9, busNumber);
        p->setPen(QColor(0,0,225));
        p->drawText(6,bottony - levelRange + 10, busNumber);
	//draw bus title
	p->setPen(Qt::black);
        p->setFont(QFont("Helvetica",6));
        p->save();
        p->rotate(-90);
        p->drawText(- maxh + vuBottonYoffset, bx + 12, title);
        p->restore();


	p->setFont(QFont("Helvetica",7));

        // clear channels
	for (int k=0; k<channels; k++)
		p->fillRect(bx + TITLE_TO_BUS_DISTANCE + k*(VU_CHANNEL_WIDTH+SPACE_BETWEEN_CHANNELS),bottony - levelRange - vuBottonYoffset,VU_CHANNEL_WIDTH,levelRange,LEVEL_CLEAR_COLOR);

	p->setPen(Qt::black);
	int markX = bx + TITLE_TO_BUS_DISTANCE + ( channels * VU_CHANNEL_WIDTH );
	p->drawText(markX + 5, bottony - vuBottonYoffset - levelRange + 5,"0" ); // 0 is always drawn
	for (int j=0; j<MAX_PRESET_MARKS; j++)
		{
		if ((levelRange<80) && (j==1 ||j==3 || j==4)) continue; // skip some of them if visible range is not enough big
		if ((levelRange<170) && (j==4)) continue; // skip this one if visible range is not enough big
		int deltaY = (int) ( dB_to_scale_factor(presetMark[j])  * levelRange );
		QString spm; spm.sprintf("%2.0f",presetMark[j]);
		p->drawText(markX + 3, bottony - deltaY + 2 - vuBottonYoffset, spm);
		}
	PEXIT3;
	}


void BusMonitor::draw_bus_vu_peaks(SingleBusMonitor* bus,
				int bx,
				int by,
				int channels,
				int stream,
				float* levelBuffer,
				float* tailDeltaY)

	{
	PENTER4;
	QPainter* fastPainter = bus->fastPainter;

	int maxh =  bus->height();
	int bottony = by + maxh;
	int levelRange = (int) (maxh * PERCENTUAL_OF_HEIGHT_USED_FOR_VUS);
	int vuBottonYoffset = (maxh - levelRange)/2;

	// check overload
	if (levelBuffer) // for tail draws , levelBuffer will be null on purpose, that's why there is this test
		{
		activeTail=true;
		for (int k=0; k<channels; k++)
			{
			if (levelBuffer[k]>=MAX_SAMPLE)
				{
				float saturationFactor = levelBuffer[k] / MAX_SAMPLE;
				float saturationDB = coefficient_to_dB(saturationFactor);
				QRect r( bx + TITLE_TO_BUS_DISTANCE + k*(VU_CHANNEL_WIDTH+SPACE_BETWEEN_CHANNELS) , by + 1, VU_CHANNEL_WIDTH , (maxh-levelRange)/2);
				fastPainter->fillRect(r, Qt::red);
				fastPainter->setPen(Qt::black);
				QString ssat; ssat.setNum(saturationDB,'g',1);
				fastPainter->setFont( QFont( "Helvetica", 6) );
				fastPainter->drawText(r,Qt::AlignAuto,ssat);
				}
			}
		}
	// draw levels
	for (int k=0; k<channels; k++)
		{
		int thisLevelDeltaY;
		if (levelBuffer)
			thisLevelDeltaY = (int) ((levelBuffer[k]/MAX_SAMPLE)*levelRange);
		else
			thisLevelDeltaY = 0;

		// vu smooth fall
		float sf = tailDeltaY[k];
		sf *= 0.60f;
		sf = ( sf < 1 ) ? 0.0 : sf;
		tailDeltaY[k] = sf;
		if (thisLevelDeltaY < sf)
			thisLevelDeltaY = (int) sf;
		else
			tailDeltaY[k] = (float) thisLevelDeltaY;

		int x = bx + TITLE_TO_BUS_DISTANCE + k*(VU_CHANNEL_WIDTH+SPACE_BETWEEN_CHANNELS);

		// First clear where is not used
		fastPainter->fillRect(x , bottony - vuBottonYoffset - levelRange,
				      VU_CHANNEL_WIDTH,
				      levelRange - thisLevelDeltaY + 1,
				      LEVEL_CLEAR_COLOR);

		// now the new level
		int hlr = levelRange/2;
		for (int i = 0; i < thisLevelDeltaY; i++)
			{
			int cR = (int) ( (i < hlr ) ? ((float)i/hlr) * 255 : 255);
			int cG = (int) ( (i < hlr ) ? 255 : 255 * ((float)(levelRange-i)/hlr) );
			cG = cG > 255 ? 255 : cG;
			cR = cG < 0   ?   0 : cR;
			fastPainter->setPen(QColor(cR, cG, 0));
			int y = bottony - vuBottonYoffset - i;
			fastPainter->drawLine(x,y,x + VU_CHANNEL_WIDTH -1,y);
			}
		}
	PEXIT4;
	}



void BusMonitor::draw_active_tails()
	{
	PENTER3;
	// draw out "Tail" peaks (MAYBE it should be done also for input...)
	activeTail=false;
	int pos=0;
	for (int busOut = 0; busOut < MustuxAudioDeviceMapper::get_total_buses_out(); busOut++)
		{
		if (!assocInterface->get_current_project()->get_current_song()->get_mixer()->valid_playback_bus(busOut))
			continue;
		int channels = 2;// IMPROVEME MustuxAudioDeviceMapper::is_stereo_bus_out(busOut) ? 2 : 1;
		bool thereIsTail=false;
		for (int k=0; k<channels; k++)
			{
			if (tailDeltaYOut[busOut][k]>1.0f)
				thereIsTail=true;
			}
		if (thereIsTail)
			{
			activeTail=true;
			draw_bus_vu_peaks(busOutArea,
				pos * (BUS_WIDTH + SPACE_BETWEEN_BUSES),
				0,
				channels,
				MustuxAudioDeviceMapper::PLAYBACK,
				(float*) 0,
				tailDeltaYOut[busOut]);
			}
		pos++;
		}
	PEXIT3;
	}


void BusMonitor::show_both_vus()
	{
	PENTER3;
	if (!onlyVUsOut && !onlyVUsIn)
		return;
	if (onlyVUsOut)
		splitterlay->remove(busOutArea);
	if (onlyVUsIn)
		splitterlay->remove(busInArea);
	splitterlay->add(busInArea);
	splitterlay->add(busAreaSplitter);
	splitterlay->add(busOutArea);
	busOutArea->show();
	busAreaSplitter->show();
	busInArea->show();
	onlyVUsOut = false;
	onlyVUsIn = false;
	recreate();
	PEXIT3;
	}


void BusMonitor::show_only_vus_in()
	{
	PENTER3;
	if (onlyVUsIn)
		return;
	if (onlyVUsOut)
		{
		onlyVUsIn = true;
		onlyVUsOut = false;
		splitterlay->remove(busOutArea);
		splitterlay->remove(busAreaSplitter);
		busOutArea->hide();
		busAreaSplitter->hide();
		splitterlay->add(busInArea);
		busInArea->show();
		}
	else
		{
		onlyVUsIn = true;
		splitterlay->remove(busOutArea);
		splitterlay->remove(busAreaSplitter);
		busOutArea->hide();
		busAreaSplitter->hide();
		}
	recreate();
	PEXIT3;
	}


void BusMonitor::show_only_vus_out()
	{
	PENTER3;
	if (onlyVUsOut)
		return;
	if (onlyVUsIn)
		{
		onlyVUsOut = true;
		onlyVUsIn = false;
		splitterlay->remove(busInArea);
		splitterlay->remove(busAreaSplitter);
		busInArea->hide();
		busAreaSplitter->hide();
		splitterlay->add(busOutArea);
		busOutArea->show();
		}
	else
		{
		onlyVUsOut = true;
		splitterlay->remove(busInArea);
		splitterlay->remove(busAreaSplitter);
		busInArea->hide();
		busAreaSplitter->hide();
		}
	recreate();
	PEXIT3;
	}


void BusMonitor::set_view_card_info_area()
	{
	PENTER3;
	if (showCardInfoArea)
		{
		showCardInfoArea = false;
		hboxlayout->remove(cardInfoArea);
		cardInfoArea->hide();
		}
	else
		{
		showCardInfoArea = true;
		hboxlayout->addWidget(cardInfoArea, 0, Qt::AlignRight);
		cardInfoArea->show();
		}
	assocInterface->busMonitor->setMinimumWidth(width()-CARD_INFO_BOX_WIDTH);
	repaint();
	recreate();
	PEXIT3;
	}


void BusMonitor::resizeEvent(QResizeEvent* e)
	{
	PENTER3;
	recreate();
	//FIXME: This is a temporary fix, it should be moved to dataBox related stuff. (DataBox becomes empty after Busmonitor::resizeEvent!
	assocInterface->projectManager->currentProject->show_info();
	//FIXME: Same for this one!
	assocInterface->projectManager->currentProject->get_current_song()->show_zoom_info();
	PEXIT3;
	}


void BusMonitor::enterEvent ( QEvent* e )
	{
	//?
	}


void BusMonitor::leaveEvent ( QEvent* e)
	{
	assocInterface->setActiveWindow();
	}

// This is for VU : db = 20 * log ( sample / MaxSample )
float BusMonitor::dB_to_scale_factor (float dB)
	{
	// examples :
	// dB = 0    will return 1.0
	// db = -6.0 will return 0.5
	// db = -inf will return 0.0
	return dB > -120.0f ? pow (10.0f, dB * 0.05f) : 0.0f;
	}


float BusMonitor::coefficient_to_dB (float coeff)
	{
	// examples :
	// coeff = 1.0 will return 0 dB
	// coeff = 0.5 will return -6 dB
	// coeff = 0.0 will return -infinite dB
	if (coeff < 0.000001f)		//Should be (coeff == 0), but this will do...
		return (-120.0f);	//Should be minus infinity, but it will do for busMonitor purposes
	return 20.0f * log10 (coeff);
	}



// Auxiliary class.. put in .hh
SingleBusMonitor::SingleBusMonitor(int minw, int minh, QWidget* parent, BusMonitor* bm)
	: MustuxDrawable(minw,minh,parent)
	{
	assocBusMonitor = bm;
	}

SingleBusMonitor::~SingleBusMonitor()
	{
	// these are necessary because, somehow, BusMonitor::recreate is called BEFORE this
	// destructor, causing busInArea,busOutArea and CardInfoAre to be invalid and crash
	assocBusMonitor->busInArea=0;
	assocBusMonitor->busOutArea=0;
	assocBusMonitor->cardInfoArea=0;
	}

void SingleBusMonitor::resizeEvent(QResizeEvent* e)
	{
	PENTER3;
	MustuxDrawable::resizeEvent(e);
	assocBusMonitor->recreate();
	PEXIT3;
	}

//eof

