/**************************************************************************************

        MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

#ifndef MUSTUXTUNER_HH
#define MUSTUXTUNER_HH


        /**
        MustuxTuner is an engine to detect a pitch of a sound,
        and also a frontend for visually trace it. It can be
        used to trace the pitch of vocal capturing, or as a Guitar Tuner
        MustuxTuner uses a original algorithm called PHE - Progressive Harmonic Elimination
        that finds the fundamental frequency with a FFT followed for a o(n) search
        of lowest harmonic. the strongest 5 harmonics are found, and each one
        is zero-padded , until the most-at-left one (supposely the fundamental)
        be found. Tests show more than 99.5% of correct results.

        Tips :

        For REAL TIME Pitch detection : Valorize the lower latency !
        Best values are sampleRate = 44100 and fftsize >= 4096 . Latency is +/- 0.1s.
        Problems are : Freq Resolution ( +/- 10Hz)

        For NON-REAL TIME Pitch detection : Valorize the lower freq resolution
        Best values are sampleRate = 11025 and fftSize = 16386. Freq Res < 1Hz
        Problems are : slower processing

        */


#include <qwidget.h>
#include <qtimer.h>
#include <qpushbutton.h>
#include <qthread.h>

class MustuxFft;
class MustuxDrawable;
class MustuxTuner;

class ThListenLoop : public QThread
        {
        public:
                ThListenLoop(MustuxTuner* pTuner);
                void run();
        private:
                MustuxTuner* tuner;
        };

class MustuxTuner : public QWidget
        {
        Q_OBJECT
        public:
                static const int NUMBER_OF_HARMONICS_TO_FIND = 3;

                MustuxTuner(int pChannels, bool realTimeFlag);
                ~MustuxTuner();

                /**
                * Must be called by graphical clients so they can
                * use a nice graphic interface for tunning.
                * If not called, the MustuxTuner will work as a non-graphical
                * component, and can be used in a text based application
                */
                void init_frontend();

                /**
                * Returns the minimal diference of frequencies detectable
                */
                float get_freq_resolution();

                /**
                * Analyse the audio data , detects its fundamental pitch, and
                * the nearest temperated pitch
                */
                int perform_pitch_analisys(short* buf, int bufSize);

                /* Return the fundamental frequency (pitch) of the fragment in Hz
                * DOES NOT WORK IF CALLED WITHOUT A PREVIOUS perform_pitch_analisys
                * in this case, it will return 0
                */
                float get_fundamental_frequency();

                /**
                * Return the level of the fundamental pitch detected with get_freq();
                * DOES NOT WORK IF CALLED WITHOUT A PREVIOUS perform_pitch_analisys
                * in this case, it will return 0
                */
                float get_fundamental_frequency_level();

                /**
                * Return the nearest temperated frequency
                * DOES NOT WORK IF CALLED WITHOUT A PREVIOUS perform_pitch_analisys
                * in this case, it will return 0
                */
                float get_nearest_temperated_frequency();


                /**
                * Return the note name for nearest temperated pitch.
                * DOES NOT WORK IF CALLED WITHOUT A PREVIOUS perform_pitch_analisys
                * in this case, it will return 0
                */
                QString get_note();

                /**
                * Start listen from the input bus.
                */
                int start_listen();

                /**
                * listen (read) a single audio fragment from the input bus, so it can be analysed
                */
                int listen_frag();

                /**
                * Forces the listen thread to stop
                */
                void force_stop_listen();

                /**
                * Stop the listen thread
                */
                int stop_listen();

                /*
                * Show information about the analyser
                */
                void show_info();

                // TODO Should be private
                char* listenBuffer;
                int listenBufferSize;
                bool listening;

        public slots:
                void update();
                void toggle_start_stop();

        protected:
                void resizeEvent(QResizeEvent* e);
                void paintEvent(QPaintEvent* e);

        private:
                MustuxFft* mustuxFft;
                QTimer* updateTimer;
                QPushButton* buttonStartStop;
                bool listenerStarted;
                bool forceFragDuplication;
                
                int sampleRate;
                int channels;
                int bitDepth;
                int fragSize;
                int fftSize;
                int mustuxbus;
                bool realTimeMode;
                float freqRes;
                float freqTable[120];
                float semitoneFactor;
                float referenceA5;
                float fundamentalFrequency;
                float nearestTempFreq;
                int nearestNote;
                QString nearestNoteName;
                int fundamentalFrequencyIndex;
                float fundamentalFrequencyLevel;
                int fpsCursor;
                int framesCounter;
                int updateCounter;
                bool frontend;

                int lastY;
                int lastYref;
                int Y;
                int Yref;
                int traceX;
                float noiseGateLevel;
                float freq;
                float refFreq;
                MustuxDrawable* traceBox;
                float freqLevel;
                QString note;

                ThListenLoop* thListenLoop;

        };

#include "MustuxFft.hh"
#include "MustuxDrawable.hh"

#endif

//eof

