/**************************************************************************************

        MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

#include "MustuxTuner.hh"
#include "MustuxDebugger.hh"
#include "MustuxAudioDeviceMapper.hh"
#include <qlayout.h>
#include <qapplication.h>
#include <stdlib.h>

// REVIEW ME ENTIRELY !!!

static const QColor COLOR_TRACEBOX_BG(120,230,180);
static const QString noteName[] = {
                "C0", "Db0", "D0", "Eb0", "E0", "F0", "F#0", "G0", "Ab0", "A0", "Bb0", "B0",
                "C1", "Db1", "D1", "Eb1", "E1", "F1", "F#1", "G1", "Ab1", "A1", "Bb1", "B1",
                "C2", "Db2", "D2", "Eb2", "E2",  "F2", "F#2", "G2", "Ab2", "A2", "Bb2", "B2",
                "C3", "Db3", "D3", "Eb3", "E3", "F3", "F#3", "G3", "Ab3", "A3", "Bb3", "B3",
                "C4", "Db4", "D4", "Eb4", "E4", "F4", "F#4", "G4", "Ab4", "A4", "Bb4", "B4",
                "C5", "Db5", "D5", "Eb5", "E5", "F5", "F#5", "G5", "Ab5", "A5", "Bb5", "B5",
                "C6", "Db6", "D6", "Eb6", "E6", "F6", "F#6", "G6", "Ab6", "A6", "Bb6", "B6",
                "C7", "Db7", "D7", "Eb7", "E7", "F7", "F#7", "G7", "Ab7", "A7", "Bb7", "B7",
                "C8", "Db8", "D8", "Eb8", "E8", "F8", "F#8", "G8", "Ab8", "A8", "Bb8", "B8",
                "C9", "Db9", "D9", "Eb9", "E9", "F9", "F#9", "G9", "Ab9", "A9", "Bb9", "B9"
        };


// UNCOMMENT TO HAVE A FLOOD IN YOUR SCREEN :-)
// #define DEBUG_WITH_PRINTF

MustuxTuner::MustuxTuner(int pChannels, bool realTimeFlag) : QWidget ( (QWidget*) 0 )
        {
        PENTERCONS;
        sampleRate = 44100;
        channels = pChannels;
        bitDepth = 16;
        realTimeMode = realTimeFlag;
        forceFragDuplication = false;
        MustuxAudioDeviceMapper::init();

        // init fft
        if (realTimeMode)
                {
                mustuxbus=0; // FOR LATER = MustuxAudioDeviceMapper::find_bus(sampleRate,bitDepth);
                listenBufferSize = MustuxAudioDeviceMapper::get_bus_in_transfer_size(mustuxbus);
                fftSize = listenBufferSize/channels;
                }
        else
                fftSize = MAX_FFT_SIZE;

        // check if fftSize is power of 2, if not, uses an lower value which is power of 2. Must be lower to avoid being larger than listenBufferSize
        if      ((fftSize>1024)  && (fftSize<2048))  fftSize=1024;
        else if ((fftSize>2048)  && (fftSize<4096))  fftSize=2048;
        else if ((fftSize>4096)  && (fftSize<8192))  fftSize=4096;
        else if ((fftSize>8192)  && (fftSize<16384)) fftSize=8192;
        else if ((fftSize>16384) && (fftSize<32768)) fftSize=16384;

        if (forceFragDuplication)
                fftSize*=2;

        mustuxFft = new MustuxFft(fftSize, MustuxFft::HAMMING_WINDOW, sampleRate);
        freqRes = (float) sampleRate / fftSize;
        //PMESG("CONFIG:  SR="<<sampleRate<<"  fftSize="<<fftSize<< "  listenBufferSize="<<listenBufferSize*(forceFragDuplication?2:1)<< " (forceFragDuplication="<<forceFragDuplication << ") chs=" << channels << " BD=" << bitDepth << " --> fRes="<<freqRes);

        // builds the piano Do 0 - Si 9 (c0 - b9) temperated scale
        semitoneFactor = 1.059463094359;
        referenceA5 = 440.0;
        float f = referenceA5;
        for (int i=68; i>=0; i--)
                {
                freqTable[i] = f / semitoneFactor;
                f=freqTable[i];
                }
        freqTable[69]=referenceA5;
        f = referenceA5;
        for (int i=70; i<110; i++)
                {
                freqTable[i] = f * semitoneFactor;
                f=freqTable[i];
                }

        nearestNote = -1;
        fundamentalFrequency = -1;
        fundamentalFrequencyIndex = -1;
        fundamentalFrequencyLevel = -1.0;
        frontend = false;
        listening = false;

        thListenLoop=new ThListenLoop(this);
        PEXITCONS;
        }

MustuxTuner::~MustuxTuner()
        {
        delete mustuxFft;
        // FOR LATER : MustuxAudioDeviceMapper::unconfigure();
        }

void MustuxTuner::init_frontend()
        {
        PENTER;
        frontend = true;
        setCaption("Mustux Tuner");
        QVBoxLayout* vlay=new QVBoxLayout(this);
        traceBox = new MustuxDrawable(700,300,this);
        traceBox->fastPainter->setFont(QFont("Fixed",12));
        vlay->addWidget(traceBox);

        buttonStartStop = new QPushButton( this );
        buttonStartStop->setText( "START");
        connect( buttonStartStop, SIGNAL(clicked()), this, SLOT(toggle_start_stop()) );
        vlay->addWidget(buttonStartStop);
        vlay->activate();

        updateTimer = new QTimer(this);
        connect( updateTimer , SIGNAL(timeout()), this, SLOT(update()));

        resize (700,300);
        fpsCursor=0;
        PEXIT;
        }


void MustuxTuner::resizeEvent(QResizeEvent* e)
        {
        traceBox->clear();
        show_info();
        }

void MustuxTuner::paintEvent(QPaintEvent* e)
        {
        show_info();
        }


float MustuxTuner::get_freq_resolution()
        {
        return freqRes;
        }


int MustuxTuner::perform_pitch_analisys(short* buf, int bufSize)
        {
        PENTER;
        if (bufSize<fftSize)
                {
                PERROR("bufSize is smaller than fftSize : bufSize=%d  fftSize=%d",bufSize ,fftSize);
                PEXIT;
                return -1;
                }

        // initialize attributes
        nearestNote = -1;
        fundamentalFrequency = -1;
        fundamentalFrequencyIndex = -1;
        fundamentalFrequencyLevel = -1.0;
        int rPoint=-1;
        int lPoint=-1;

        // prepare the FFT
        float* data = new float[fftSize];
        float* levels = new float[fftSize/2]; // the output (frequency domain) levels array ( wich size is always fftSize/2 )

        for (int k=0; k<fftSize; k++)
                data[k] = (float) buf[k*channels];  // k*channels ensures that only one channel will be used to fft

        // performs the FFT - Oh Saint FFT ! What would I be without you ...
        mustuxFft->perform(data,levels);

        //PCHECK_BUFFER_MAX( levels, fftSize/2);

#ifdef DEBUG_WITH_PRINTF
        printf("\n--- SPECTRUM : ");
        for (int i = 1; i < 60; i++)
                printf("%2.0f ",levels[i]);
        printf("\n");
#endif

        // Now , The Ambicious Progressive Harmonic Extraction Algorithm :-)
        // This acts like an genetic algorithm, using elitist strategy
        // The "selected" frequencies are those who seems to be most representatives
        // but only 5 generations are created (we need to find 5 hamornics)
        int freqIndexStack[NUMBER_OF_HARMONICS_TO_FIND];
        float levelStack[NUMBER_OF_HARMONICS_TO_FIND];
        for (int i=0; i<NUMBER_OF_HARMONICS_TO_FIND; i++)
                {
                freqIndexStack[i]=0;
                levelStack[i]=0.0;
                }

        // MINIMUM and MAXIMUM frequency range...
        float maxFreqToCheck = 2637.02; // An E8 // IMPROVEME : make me a parameter
        float minFreqToCheck = 65.4063; // An C3 // IMPROVEME : make me a parameter
        int mfci = (int) (maxFreqToCheck / freqRes);
        int maxFreqIndexToCheck =  ( fftSize/2 > mfci ) ? mfci : fftSize/2;
        int minFreqIndexToCheck = (int) (minFreqToCheck / freqRes);
        float noiseLine = mustuxFft->get_level_noise_line();
        //PDEBUG("minfi=" <<  minFreqIndexToCheck << " maxfi=" << maxFreqIndexToCheck << " nl="<<noiseLine);

        // collect the NUMBER_OF_HARMONICS_TO_FIND greatest harmonic in the spectrum
        for (int stackIndex=0; stackIndex<NUMBER_OF_HARMONICS_TO_FIND; stackIndex++)
                {
                float harmonicLevel = -1.0;
                int harmonicFreqIndex = -1;

                // where is the max level ?
                for (int i = minFreqIndexToCheck; i < maxFreqIndexToCheck; i++ )
                        if (levels[i] > harmonicLevel)
                                {
                                harmonicLevel = levels[i];
                                harmonicFreqIndex = i;
                                }
#ifdef DEBUG_WITH_PRINTF
                printf("   HARM #%d=%d (%4.1fHz)\t",stackIndex,harmonicFreqIndex,harmonicFreqIndex*freqRes);
#endif
                // where does the peak begin and end in this harmonic ?
                for (int i = harmonicFreqIndex; (levels[i]>noiseLine) && (i<maxFreqIndexToCheck); i++) rPoint = i;
                for (int i = harmonicFreqIndex; (levels[i]>noiseLine) && (i>minFreqIndexToCheck); i--) lPoint = i;
#ifdef DEBUG_WITH_PRINTF
                printf("   PR(%d,%d)\t",lPoint,rPoint);
#endif

                if ((lPoint<0) || (rPoint<0)) // inconsistent harmonic
                        {
                        freqIndexStack[stackIndex]=-1;
                        levelStack[stackIndex]=-1.0;
#ifdef DEBUG_WITH_PRINTF
                        printf("   INCONSISTENT",harmonicLevel);
#endif
                        }
                else // zero padding : we eliminate this harmonic so it wont be found again ....
                        {
                        freqIndexStack[stackIndex]=harmonicFreqIndex;
                        levelStack[stackIndex]=harmonicLevel;
#ifdef DEBUG_WITH_PRINTF
                        printf("   L=%3.1f",harmonicLevel);
#endif
                        for (int i = lPoint; i<rPoint; i++) levels[i]=0;
                        }
#ifdef DEBUG_WITH_PRINTF
                printf("\n");
#endif
                }

        // Decision by position : The fundamental frequency is frequency for the minimum
        // frequency index.
        int tempIndex=999999;
        float tempLevel=-1.0;
        for (int i=0; i<NUMBER_OF_HARMONICS_TO_FIND; i++)
                {
                if (freqIndexStack[i]<tempIndex)
                        {
                        tempIndex=freqIndexStack[i];
                        tempLevel=levelStack[i];
                        }
                }
        fundamentalFrequencyIndex = tempIndex;
        fundamentalFrequencyLevel = tempLevel;

        // Did I find a fundamentalFrequencyIndex ?
        if (fundamentalFrequencyIndex<0)
                {
                fundamentalFrequency = 0;
                nearestNoteName = "?";
#ifdef DEBUG_WITH_PRINTF
                printf("*** INVALID NOTE");
#endif
                }
        else // Fine. So what is the fundamental frequency, then ?
                {
                fundamentalFrequency = (float) fundamentalFrequencyIndex * freqRes;

                // and what is the nearest piano note for it ?
                for (int i=1; i<110; i++)
                        {
                        float deltaDown = (freqTable[i] - freqTable[i-1])/2;
                        float deltaUp   = (freqTable[i+1] - freqTable[i])/2;
                        if ((fundamentalFrequency>=freqTable[i]-deltaDown) && (fundamentalFrequency <=freqTable[i]+deltaUp))
                                {
                                nearestTempFreq=freqTable[i];
                                nearestNote=i;
                                break;
                                }
                        }
                nearestNoteName = noteName[nearestNote];
#ifdef DEBUG_WITH_PRINTF
                printf("*** FFI=%d (%4.1fHz - %s) L=%3.1f \n",fundamentalFrequencyIndex,fundamentalFrequency,nearestNoteName,fundamentalFrequencyLevel);
#endif
                }

        delete data;
        delete levels;
        PEXIT;
        return 1;
        }

float MustuxTuner::get_fundamental_frequency()
        {
        return fundamentalFrequency;
        }


float MustuxTuner::get_fundamental_frequency_level()
        {
        return fundamentalFrequencyLevel;
        }


float MustuxTuner::get_nearest_temperated_frequency()
        {
        return nearestTempFreq;
        }


QString MustuxTuner::get_note()
        {
        return nearestNoteName;
        }


int MustuxTuner::start_listen()
        {
        PENTER;
        int r = MustuxAudioDeviceMapper::open_bus_in(mustuxbus, sampleRate, bitDepth, channels);
        if ( r<0)
                {
                PERROR("Cannot Open Audio device for Record");
                PEXIT;
                return -1;
                }
        listenBuffer = MustuxAudioDeviceMapper::get_bus_in_transfer_buffer(mustuxbus,MustuxAudioDeviceMapper::STEREO); // BE SURE about this BOTH_CHANNELS
        listening = true;
        thListenLoop->start();

        if (frontend)
                {
                lastY=-1;
                lastYref=-1;
                Y=0;
                Yref=0;
                traceX=0;
                noiseGateLevel = 5.0;// from 0 to 100
                updateTimer->start(100);
                }
        return 1;
        }



void MustuxTuner::force_stop_listen()
        {
        listening = false;
        thListenLoop->wait();
        }



int MustuxTuner::stop_listen()
        {
        int r=1;
        if (MustuxAudioDeviceMapper::close_bus_in(mustuxbus)<0)
                {
                PERROR("Cannot close audio device. Continuing anyway...");
                r=-1;
                }
        if (frontend)
                updateTimer->stop();
        delete listenBuffer;
        return r;
        }


void MustuxTuner::update() // I am called at each 25 ms
        {
        // DOES NOTHING. THIS IS NEEDED JUST TO PROCESS ONE QT EVENT ITERATION
        updateCounter++;
        fpsCursor++;
        if (updateCounter==40) // its been past exactly 1 second
                {
                show_info();
                updateCounter=0;
                fpsCursor=0;
                framesCounter=0;
                }

        QString s;
        traceBox->fastPainter->fillRect(10,10,500,30,COLOR_TRACEBOX_BG);
        if (freqLevel>noiseGateLevel)
                {
                lastY = Y;
                lastYref = Yref;
                Y = traceBox->height() - (int) ((freq+100)/600 * 200);
                Yref = traceBox->height() - (int) ((refFreq+100)/600 * 200);
                traceBox->fastPainter->setPen(Qt::red);
                traceBox->fastPainter->drawLine(traceX-1, lastY, traceX, Y);
                traceBox->fastPainter->setPen(Qt::green);
                traceBox->fastPainter->drawLine(traceX-1, lastYref, traceX, Yref);

                QString s1; s1.setNum(freq);
                QString s2; s2.setNum(refFreq);
                QString s3(note);
                traceBox->fastPainter->setPen(Qt::black);
                s = "Heard:" + s1 + " Hz  Ref:" + s2 + " Hz ( " + s3 +" )";
                traceBox->fastPainter->drawText(20,29,s);

                traceX++; if (traceX>traceBox->width())
                        {
                        traceBox->fastPainter->fillRect(0,0,traceBox->width(),traceBox->height(),Qt::black);
                        traceX=0;
                        }
                }
        else
                {
                traceBox->fastPainter->setPen(Qt::black);
                traceBox->fastPainter->drawText(20,29,"LISTENING ... ");
                }
        framesCounter++;
        traceBox->fastPainter->fillRect(150,traceBox->height()-40,50,20,Qt::red);
        traceBox->fastPainter->fillRect(150+fpsCursor,traceBox->height()-40+1,10,18,Qt::blue);
        }


void MustuxTuner::toggle_start_stop()
        {
        if (listening)
                {
                force_stop_listen();
                buttonStartStop->setText( "START");
                traceBox->fastPainter->fillRect(10,10,500,30,COLOR_TRACEBOX_BG);
                }
        else
                {
                framesCounter=0;
                updateCounter=0;
                start_listen();
                buttonStartStop->setText( "STOP");
                }
        }

void MustuxTuner::show_info()
        {
        traceBox->fastPainter->setPen(Qt::blue);
        int baseY = traceBox->height() - 40;
        traceBox->fastPainter->fillRect(10,baseY,250,20,Qt::black);

        QString s1; s1.setNum(freqRes,'f',1); s1 = "Res:" + s1 + " Hz";
        traceBox->fastPainter->drawText(10,baseY+12,s1);

        QString s2; s2.setNum(framesCounter); s2=s2+" fps";
        traceBox->fastPainter->drawText(120,baseY+12,s2);
        }


int MustuxTuner::listen_frag()
        {
        PENTER;
        MustuxAudioBusTransferResult* tr = MustuxAudioDeviceMapper::bus_in_transfer(mustuxbus);
        if ( tr->error != 0 )
                {
                PERROR("Cannot read from audio device");
                listening = false;
                return -1;
                }


        if (forceFragDuplication) // DOES NOT WORK. THE IDEA IS GOOD BUT I NEED TO
                                  // TRY SOME BETTER DUPLICATION TECNIQUE
                {
                char* dupBuf = new char[listenBufferSize*2];
                // raw duplication with simple interpolation
                memcpy(dupBuf,listenBuffer,listenBufferSize);
                memcpy(dupBuf+listenBufferSize,listenBuffer,listenBufferSize);
                for (int i=-20;  i<20; i++)
                        dupBuf[listenBufferSize+i]=(char)(dupBuf[listenBufferSize+i]*((float)abs(i)/20)); // interpolation
                
                perform_pitch_analisys((short*)dupBuf,listenBufferSize*2);
                delete dupBuf;
                }
        else
                perform_pitch_analisys((short*)listenBuffer,listenBufferSize);

        freq = get_fundamental_frequency();
        freqLevel = get_fundamental_frequency_level();
        refFreq = get_nearest_temperated_frequency();
        note = get_note();

#ifdef DEBUG_WITH_PRINTF
        printf("level = %3.1f \n",freqLevel);
#endif
        PEXIT;
        return 1;
        }


ThListenLoop::ThListenLoop(MustuxTuner* pTuner)
        {
        tuner = pTuner;
        }


void ThListenLoop::run()
        {
        msleep(100);
        while (tuner->listening)
                {
                if (tuner->listen_frag()<0)
                        break;
                }
        tuner->stop_listen();
        }



// NOTE / FREQUENCY REFERENCE TABLE
/*

C0       = 8.17577
Db0      = 8.66193
D0       = 9.177
Eb0      = 9.72269
E0       = 10.3008
F0       = 10.9134
F#0      = 11.5623
G0       = 12.2498
Ab0      = 12.9782
A0       = 13.75
Bb0      = 14.5676
B0       = 15.4338

C1       = 16.3516
Db1      = 17.3239
D1       = 18.354
Eb1      = 19.4454
E1       = 20.6017
F1       = 21.8267
F#1      = 23.1246
G1       = 24.4997
Ab1      = 25.9565
A1       = 27.4999
Bb1      = 29.1352
B1       = 30.8676

C2       = 32.7031
Db2      = 34.6478
D2       = 36.708  ---> Recommended minimum note detection limit for bass
Eb2      = 38.8908
E2       = 41.2034
F2       = 43.6535
F#2      = 46.2492
G2       = 48.9993
Ab2      = 51.913
A2       = 54.9999
Bb2      = 58.2704
B2       = 61.7353

C3       = 65.4063 ---> Recommended Minimum note detection limit for voice (bass)
Db3      = 69.2956
D3       = 73.4161
Eb3      = 77.7816
E3       = 82.4068
F3       = 87.307
F#3      = 92.4985
G3       = 97.9987
Ab3      = 103.826
A3       = 110
Bb3      = 116.541
B3       = 123.471
C4       = 130.813
Db4      = 138.591
D4       = 146.832
Eb4      = 155.563
E4       = 164.814
F4       = 174.614
F#4      = 184.997
G4       = 195.998
Ab4      = 207.652
A4       = 220
Bb4      = 233.082
B4       = 246.942
C5       = 261.625
Db5      = 277.183
D5       = 293.665
Eb5      = 311.127
E5       = 329.628
F5       = 349.228
F#5      = 369.994
G5       = 391.995
Ab5      = 415.305
A5       = 440
Bb5      = 466.164
B5       = 493.883
C6       = 523.251
Db6      = 554.365
D6       = 587.33
Eb6      = 622.254
E6       = 659.255
F6       = 698.457
F#6      = 739.989
G6       = 783.991
Ab6      = 830.61
A6       = 880
Bb6      = 932.328
B6       = 987.767
C7       = 1046.5
Db7      = 1108.73
D7       = 1174.66
Eb7      = 1244.51
E7       = 1318.51
F7       = 1396.91
F#7      = 1479.98
G7       = 1567.98
Ab7      = 1661.22
A7       = 1760
Bb7      = 1864.66
B7       = 1975.54
C8       = 2093.01
Db8      = 2217.46
D8       = 2349.32
Eb8      = 2489.02
E8       = 2637.02 ---> Recommended Maximum note detection limit for voice
F8       = 2793.83
F#8      = 2959.96
G8       = 3135.97
Ab8      = 3322.44
A8       = 3520.01
Bb8      = 3729.32
B8       = 3951.07
C9       = 4186.02
Db9      = 4434.93

*/


