/**************************************************************************************

        MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

#ifndef MUSTUXJOGMOUSEBOARD_HH
#define MUSTUXJOGMOUSEBOARD_HH

// These two is just to make the handlers names more coherent with JMB's terminology
#define JMB_PRESS_EVENT_HANDLER keyPressEvent
#define JMB_RELEASE_EVENT_HANDLER keyReleaseEvent
#define JMB_EVENT QKeyEvent

#include <qevent.h>

#include "MustuxEngine.hh"

class MustuxInterface;

#include "MustuxJogMouseBoardAction.hh"

/** MustuxJogMouseBoard is one of the most special features of MustuxLib
 *  and the core of Mustux Project philosophy. A Jog Mouse Board - for short JMB -
 *  is a engine that is able to recognize actions in the keyboard combined with
 *  mouse movements in order to avoid the traditional mouse-point-and-click paradigm
 *  of GUI applications. <br>
 *  In the traditional mouse-click paradigm , an user can perform 4 basic actions :
 *  click , double-click, click-and-drag, and right click.
 *  With JMB, there are at 16 kinds of actions multiplied by 101 keys of a standard keyboard.
 *  This means that once automated, an user can perform operations in MustuxApplications much
 *  faster than a traditional GUI environment. This is the base of Mustux Project paradigm and
 *  thats why some applications such Protux is beign called "the Emacs of the Audio".
 */
class MustuxJogMouseBoard : public MustuxEngine
        {
        Q_OBJECT
        public:
                /** create and build a JMB engine
                */
                MustuxJogMouseBoard(MustuxInterface* pInterface);
                ~MustuxJogMouseBoard();

                /** attach the JMB engine to an application (that will use and handle JMB events)
                *  Also, initialize the actions map, that is specified in a special file "jmb.map"
                *  (See MustuxJogMouseBoardEvent.hh for jmb.map details)
                */
                int init(QString mapFilename);

                /** activate the actions recognition engine and start responding to them
                */
                void activate();

                /** Suspends the engine . No action will be handled until activate() is called again
                */
                void suspend();

                /** finalize the engine and unbuild the actions map
                */
                int finalize();

                /** Initialize the JMB map using mapFilename as the map */
                int init_map(QString mapFilename);

                /** internal method. Not meant to be called by clients */
                void reset();


                /** internal method. Not meant to be called by clients */
                void set_clear_time(int time);

                /** set the hold sensitiveness */
                void set_hold_sensitiveness(int factor);

                /** set the double press interval in ms */
                void set_double_fact_interval(int time);

                /** return the nth action name */
                QString get_action_name(int n);

                /** return the nth action parameter */
                int get_action_parameter(int n);


                /** return lock the JMB engine */
                void lock();

                /** return unlock the JMB engine */
                void unlock();

                /** return return true JMB engine is locked*/
                bool is_locked();


        protected :

                static const int STACK_SIZE = 4;
                static const int MAX_ACTIONS = 300;
                static const int JMB_EVENT_PRESS = 1;
                static const int JMB_EVENT_RELEASE = 2;

                static QString actionTypeLabel[16];

                static const int FKEY = 0;                 // <K>    - press one key fast
                static const int FKEY2 = 1;                // <KK>   - press two keys fast, together
                static const int HKEY = 2;                 // [K]    - Hold one key
                static const int HKEY2 = 3;                // [KK]   - Hold two keys, together
                static const int D_FKEY = 4;               // <<K>>  - double press one key fast
                static const int D_FKEY2 = 5;              // <<KK>> - double press two keys fast, together
                static const int FHKEY = 6;                // <[K]>  - press fast and then hold one key
                static const int FHKEY2 = 7;               // <[KK]> - press fast and then hold two keys together
                static const int S_FKEY_FKEY = 8;          // >K>K   - press one key and then other key fast and sequentially
                static const int S_FKEY_FKEY2 = 9;         // >K>KK  - press one key and then two other keys fast and sequentially
                static const int S_FKEY2_FKEY = 10;        // >KK>K  - press two keys and then another one fast and sequentially
                static const int S_FKEY2_FKEY2 = 11;       // >KK>KK  - press two keys and then another one fast and sequentially
                static const int S_FKEY_HKEY = 12;         // >K[K]  - press fastly one key and then hold another key, sequentially
                static const int S_FKEY_HKEY2 = 13;        // >K[KK] - press fastly one key and then hold two ther ones, sequentially
                static const int S_FKEY2_HKEY = 14;        // >KK[K] - press two keys together fastly and then hold a third one, sequentially
                static const int S_FKEY2_HKEY2 = 15;       // >KK[KK] - press two keys together fastly and then hold a third one, sequentially


                MustuxJogMouseBoardAction map[MAX_ACTIONS];
                MustuxInterface* assocInterface;
                int stackIndex;
                int eventType[STACK_SIZE];
                int eventStack[STACK_SIZE];
                long eventTime[STACK_SIZE];
                int actionKey1A;
                int actionKey2A;
                int actionKey1B;
                int actionKey2B;
                int pressEventCounter; // that avoid more than 2 press events in a same fact
                int pairOf2;
                int pairOf3;

                // JMB settings
                int clearTime;
                int assumeHoldTime;
                int doubleFactWaitTime;

                QTimer* holdTimer;
                QTimer* clearOutputTimer;
                QTimer* secondChanceTimer;
                int mapSize;
                bool active;
                bool isHolding;
                bool isPressEventLocked;
                bool isHoldingOutput;
                bool isFirstFact;
                bool is_fake( int keyval);
                bool locked;
                bool isDoubleKey;

                int fact1_k1;
                int fact1_k2;
                int fact2_k1;
                int fact2_k2;
                int fact1Type;
                int fact2Type;
                int wholeAction;
                int wholeActionType;

                void push_event(int pType,  int pKey);
                void press_checker();
                void release_checker();
                int identify_first_fact();
                int identify_first_and_second_facts_together();
                void push_fact( int k1 ,  int k2);
                void give_a_chance_for_second_fact();
                void dispatch_action(int action);
                void dispatch_hold();
                void finish_hold();

                void conclusion();
                void hold_output();

                friend class MustuxInterface;

        private:
                int uppercase(int k);
                int get_collected_number();
                void stop_collecting();
                int catch_press(JMB_EVENT *);
                int catch_release(JMB_EVENT *);

                QString sCollectedNumber;
                int collectedNumber;
                void check_number_collection();
                int collectedDigit;
                bool isCollecting;

        public slots:
                void assume_hold();
                void quit_second_chance();
                void clear_output();
                void JMB_PRESS_EVENT_HANDLER   ( JMB_EVENT * );
                void JMB_RELEASE_EVENT_HANDLER ( JMB_EVENT * );


        };

#include "MustuxInterface.hh"



#endif


/*
JMB TERMINOLOGY

AN ACTION IS A FULL SEQUENCE OF KEYS. FOR EXAMPLE <<K>> OR >K>K OR >K[K] or <<KK>> , and so on
A FACT is A FULL SEQUENCE OF EVENTS , SO THE DOWN MOVEMENT OF A KEY IN THE KEYBOARD PLUS ITS RELEASE IS A KEY
A EVENT IS EITHER THE MOVEMENT DOWN OR THE MOVEMENT UP OF A KEY IN THE KEYBOARD

                                  ACTION
                                 /     \
                                /       \
                              FACT       FACT
                              /\         /\
                             /  \       /  \
                            /    \     /    \
                        EVENT  EVENT EVENT EVENT

FOR EXAMPLE , A >K[K] IS

                                  >K[K]
                                 /     \
                                /       \
                          NORMAL FACT  HOLD FACT
                              /\           \
                             /  \           \
                            /    \           \
                        PRESS RELEASE       PRESS

*/

//eof


