/**************************************************************************************

        MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

#include <stdio.h>
#include <qtimer.h>
#include <stdlib.h>

#include "MustuxJogMouseBoard.hh"
#include "MustuxJogMouseBoardAction.hh"
#include "MustuxJogMouseBoardMessage.hh"
#include "MustuxDebugger.hh"
#include <qtextstream.h>
#include <qfile.h>
#include <qdatetime.h>

#define SKIP do fread(&c, 1, 1, mapFile); while (((c==0x20) || (c==0x09)) && (!feof(mapFile)));
#define KEYBOARD_CODE key
#define MAX_TIME_DIFFERENCE_FOR_DOUBLE_KEY_CONSIDERATION 50

MustuxJogMouseBoard::MustuxJogMouseBoard(MustuxInterface* pInterface) :  MustuxEngine( pInterface)
        {
        PENTERCONS;
        assocInterface = pInterface;
        clearTime = 2000;
        assumeHoldTime = 200; // it will wait a release for 100 ms. Otherwise it will assume a hold
        doubleFactWaitTime = 200;
        collectedDigit=0;
        collectedNumber = -1;
        sCollectedNumber = "-1";
        locked=false; // THIS SHOULD CALL everythingLocked
        PEXITCONS;
        }

MustuxJogMouseBoard::~MustuxJogMouseBoard()
        {
        PENTERDES;
        PEXITDES;
        }


int MustuxJogMouseBoard::init(QString mapFilename)
        {
        PENTER2;
        activate();
        clearOutputTimer=0;
        reset();
        init_map(mapFilename);
        holdTimer =  new QTimer(this);
        connect( holdTimer, SIGNAL(timeout()), this, SLOT(assume_hold()) );
        secondChanceTimer = new QTimer(this);
        connect( secondChanceTimer, SIGNAL(timeout()), this, SLOT(quit_second_chance()) );
        clearOutputTimer = new QTimer();
        connect( clearOutputTimer, SIGNAL(timeout()), this, SLOT(clear_output()) );
        PEXIT2;
        return 1;
        }


void MustuxJogMouseBoard::activate()
        {
        PENTER3;
        assocInterface->get_default_lcd()->clear_line(1);
        assocInterface->get_default_lcd()->clear_line(2);
        assocInterface->get_default_lcd()->clear_line(3);
        assocInterface->get_default_lcd()->clear_line(4);
        assocInterface->get_default_lcd()->print("JMB ACTIVE",4);
        isFirstFact=true;
        active=true;
        PEXIT3;
        }


void MustuxJogMouseBoard::suspend()
        {
        PENTER3;
        assocInterface->get_default_lcd()->print("JMB SUSPENDED",4);
        active=false;
        PEXIT3;
        }


int MustuxJogMouseBoard::finalize()
        {
        PENTER3;
        delete holdTimer;
        delete secondChanceTimer;
        delete clearOutputTimer;
        PEXIT3;
        return 1;
        }


// ----------------------- JMB ENGINE : EVENT LEVEL HANDLING ---------------------

// reset the event stack and the press 'stack' (not exactly a stack)
void MustuxJogMouseBoard::reset()
        {
        PENTER3;
        isFirstFact = true;
        isDoubleKey = false;
        fact1_k1 = 0;
        fact1_k2 = 0;
        isHolding = false;
        isPressEventLocked = false;
        stackIndex = 0;
        pressEventCounter = 0;
        fact2_k1 = 0;
        fact2_k2 = 0;
        wholeAction = -1;
        for (int i=0; i < STACK_SIZE; i++)
                {
                eventType[i] = 0;
                eventStack[i] = 0;
                eventTime[i] = 0;
                }
        PEXIT3;
        }




void MustuxJogMouseBoard::JMB_PRESS_EVENT_HANDLER ( JMB_EVENT * e)
        {
        if (active)
                {
                if (!e->isAutoRepeat())
                        catch_press(e);
                }
        e->ignore();
        }

void MustuxJogMouseBoard::JMB_RELEASE_EVENT_HANDLER ( JMB_EVENT * e)
        {
        if (active)
                {
                if (!e->isAutoRepeat())
                        catch_release(e);
                }
        e->ignore();
        }


// Everthing starts here. Catch event takes anything happen in the keyboard
// and pushes it into a stack.
int MustuxJogMouseBoard::catch_press( JMB_EVENT * e )
        {
        PENTER3;
        int k = uppercase(e->KEYBOARD_CODE());
        if (!isPressEventLocked)
                {
                if (pressEventCounter < 2)
                        {
                        pressEventCounter++;
                        push_event(JMB_EVENT_PRESS, k);
                        press_checker();
                        if (!holdTimer->isActive())
                                holdTimer->start( assumeHoldTime, true ); // single shot timer
                        }
                else
                        {
                        isPressEventLocked = true;
                        }
                }
        PEXIT3;
        return 1;
        }

int MustuxJogMouseBoard::catch_release( JMB_EVENT * e)
        {
        PENTER3;
        int k = uppercase(e->KEYBOARD_CODE());
        if (!is_fake(k))
                {
                push_event(JMB_EVENT_RELEASE, k);
                release_checker();
                }
        PEXIT3;
        return 1;
        }


// This pushed an event to the stack
void MustuxJogMouseBoard::push_event(  int pType,  int pCode )
        {
        PENTER3;
        if (stackIndex < STACK_SIZE)
                {
                eventType[stackIndex] = pType;
                eventStack[stackIndex] = pCode;
                QTime currTime = QTime::currentTime();
                long ts=currTime.msec() + (currTime.second() * 1000) + (currTime.minute() * 1000 * 60);
                eventTime[stackIndex] = ts;
                PMESG3("Pushing EVENT %d (%s) key=%d at %d",stackIndex,( pType==JMB_EVENT_PRESS ? "PRESS" : "RELEASE" ),pCode,ts);
                stackIndex++;
                for (int j=0; j<STACK_SIZE; j++)
                        {
                        PMESG4("eventStack[%d]=%d %s at timestamp=%d\n",j,eventStack[j],(eventType[j]==JMB_EVENT_PRESS?"P":"R"),eventTime[j]);
                        }
                }
        PEXIT3;
        }


void MustuxJogMouseBoard::assume_hold() // no release so far ? so consider it a hold...
        {
        PENTER3;
        PMESG3("No release so far (waited %d ms). Assuming this is a hold and dispatching it",assumeHoldTime);
        isHolding = true;
        dispatch_hold();
        PEXIT3;
        }

int MustuxJogMouseBoard::uppercase(int k)
        {
        /*cout << "k=" << k << " , char(k)=" << char(k) << endl;
        if ((k>90) && (k<106))
                k-=32;
        */
        return k;
        }

// this is a intermediate step to push. Only non-fake events are allowed to be pushed
bool MustuxJogMouseBoard::is_fake(int codeVal)
        {
        // A fake event is an event which is generated by a ignored JMB_EVENT_PRESS
        bool b =   ( codeVal!=eventStack[0] )
                && ( codeVal!=eventStack[1] )
                && ( codeVal!=eventStack[2] )
                && ( codeVal!=eventStack[3] );
        return b;
        }

// this is the method that detects wether a fact is double key (KK) or single key (K)
// by measuring the time difference between the first 2 presses (in case event[0] AND
// event[1] are JMB_EVENT_PRESS'es
void MustuxJogMouseBoard::press_checker()
        {
        PENTER3;
        if (stackIndex==2) // first two events happend
                {
                PMESG3("Checking if 2 first events are PRESS'es and what is the time diff between them");
                if (     ( eventType[0]==JMB_EVENT_PRESS )
                      && ( eventType[1]==JMB_EVENT_PRESS )
                   )
                        {
                        isDoubleKey = ( eventTime[1] - eventTime[0] < MAX_TIME_DIFFERENCE_FOR_DOUBLE_KEY_CONSIDERATION );
                        }
                if (isDoubleKey)
                        {
                        PMESG3("Detected 2 initial PRESS almost together. It is a double key (KK) !");
                        }
                }
        PEXIT3;
        }

// This is the one who consolidate a fact. It can detect
// if the fact is K or KK, and also calls the push_fact handler
void MustuxJogMouseBoard::release_checker()
        {
        PENTER3;
        if (isHolding)
                {
                finish_hold();
                PEXIT3;
                return;
                }
        // If it is not fake, then lets take a look in the pairs
        // Remember that event 0 is always JMB_EVENT_PRESS
        if (eventType[1]==JMB_EVENT_RELEASE)
                {
                // event 0 and event 1 are a pair because event 1 must be the release of 0
                // since the event 1 was a release, then the first fact is finished
                // Now we must push_fact, but only if it is a release very fast (that
                // didnt cause a HOLD.
                if (!isHolding)
                        push_fact(eventStack[0],0);
                PEXIT3;
                return;
                }
        // event 1 is ALSO a JMB_EVENT_PRESS. so we must check for a double key
        if (isDoubleKey) // ( very short time difference between 2 first presses)
                {
                // Now we dont know if this is the event 2 or event 3
                if (stackIndex==3)
                        {
                        // So we are processing event (??), which is a release
                        // So this can be release of event 0 or 1 , so..
                        if (eventStack[2]==eventStack[0])
                                pairOf2 = 0;
                        else
                                pairOf2 = 1;
                        // the event is not finished yet but it must be avoided
                        // any reentrance (any new JMB_EVENT_PRESS)
                        isPressEventLocked = true;
                        PEXIT3;
                        return;
                        }
                // we are in the event 3 . the last!
                // the event is finished , surely. But we must check the pairs
                // we already know who event 2 is pair of. So ...
                if (pairOf2 == 0)
                        pairOf3 = 1;
                else
                        pairOf3 = 0;
                // The event is finished, and we know the pairs. So..
                // I must push_fact, but only if it is a release very fast (that
                // didnt cause a HOLD.
                if (!isHolding)
                        push_fact(eventStack[0],eventStack[1]);
                }
        else
                {
                // although the sequence was P-P-R-R (press, press, release, release)
                // the time difference between two first presses was too long
                // so it CAN'T be a double key (KK).
                // So the only possible explanation is that this is a >K>K action
                // where user typed too fast, so the release of 1st P happend AFTER the
                // 2nd press. In this case, I have to assume this is a >K>K
                // action, and dispatch TWO facts.
                // Now forcing prematurally a 2 facts by splitting and syncronizing
                // these events into 2 different facts. Probably this is a >K>K
                // of course, this  >K>K assumption mentioned above can be made ONLY if it is the first fact.
                if (isFirstFact)
                        {
                        PMESG("Inital double key too slow. It must be a premature >K>K. Dispatching 2 individual <K> facts.");
                        int f1_k1=eventStack[0];
                        int f2_k1=eventStack[1];
                        push_fact (f1_k1,0);
                        push_fact (f2_k1,0);
                        }
                }

        PEXIT3;
        }


// ----------------------------- JMB ENGINE : PRESS LEVEL HANDLING -------------------------


void MustuxJogMouseBoard::push_fact(int k1,int k2)
        {
        PENTER3;
        PMESG3("Pushing FACT : k1=%d k2=%d",k1,k2);
        assocInterface->get_default_lcd()->clear_line(1);
        holdTimer->stop(); // quit the holding check..
        if (isFirstFact)
                {
                // this is the first fact
                PMESG3("First fact detected !");
                // first try to find some action like k1k2
                fact1_k1 = k1;
                fact1_k2 = k2;

                // first check if this fact is just a collected number
                check_number_collection();
                if (isCollecting)
                        {
                        // another digit was collected. Fine , go on jmb job..
                        reset();
                        PEXIT3;
                        return;
                        }
                int action = identify_first_fact();
                if (action < 0)
                        {
                        PMESG3("First fact alone does not match any action. Waiting for a second fact...");
                        // Action not identified. Maybe is part of a double fact action. so...
                        give_a_chance_for_second_fact();
                        PEXIT3;
                        return;
                        }
                PMESG3("First fact matches action %d",action);
                // there is a single-fact action which matches this !! now must check if is not an immediate action
                if (!map[action].isInstantaneous)
                        {
                        PMESG3("Although this could be an SINGLE PRESS Action, it is not protected, so...");
                        // action is not an immediate action, so...
                        give_a_chance_for_second_fact();
                        PEXIT3;
                        return;
                        }
                // Action exists AND it is a immediate action. So
                // forces it to be a single fact action
                PMESG3("This is protected (immediate) action. It'll be treated as <K>");
                dispatch_action(action);
                conclusion();
                }
        else // ok . We are in the second fact.
                {
                secondChanceTimer->stop();
                fact2_k1 = k1;
                fact2_k2 = k2;
                if (fact2_k1!=0)
                        {
                        // this is the second press
                        PMESG3("Second fact detected !");
                        }
                // try to complement the first press.
                wholeAction = identify_first_and_second_facts_together();
                if (wholeAction >= 0)
                        {
                        PMESG3("First and second facts together matches with action %d !! Dispatching it...",wholeAction );
                        if (locked)
                                {
                                assocInterface->get_default_lcd()->print("Jog Mouse Board is LOCKED !",0);
                                PEXIT3;
                                return;
                                }
                        dispatch_action(wholeAction);
                        }
                else
                        {
                        PMESG3("Apparently, first and second facts together do not match any action. Sorry :-(");
                        QString s = "Unknown "; s = s.append(actionTypeLabel[wholeActionType]); s = s.append(" Action !");
                        assocInterface->get_default_lcd()->print(s,0);
                        }
                conclusion();
                }
        PEXIT3;
        }





int MustuxJogMouseBoard::identify_first_fact()
        {
        PENTER3;
        fact1Type = 0;
        // First we need to know the first fact type.
        if (fact1_k2==0) // <K> or [K]
                {
                if (!isHolding)
                        {
                        PMESG3("Detected <K>");
                        fact1Type = FKEY;
                        }
                else
                        {
                        PMESG3("Detected [K]");
                        fact1Type = HKEY;
                        }
                }
        else // <KK> or [KK]
                {
                if (!isHolding)
                        {
                        PMESG3("Detected <KK>");
                        fact1Type = FKEY2;
                        }
                else
                        {
                        PMESG3("Detected [KK]");
                        fact1Type = HKEY2;
                        }
                }

        // Fact 1 Type identified .
        PMESG3("Now, is there a SINGLE FACT action which matches %s and %d,%d",actionTypeLabel[fact1Type].ascii(),fact1_k1,fact1_k2);
        for (int i=0; i<mapSize; i++)
                        {
                        int ap1k1 = map[i].fact1_key1;
                        int ap1k2 = map[i].fact1_key2;
                        int ap2k1 = map[i].fact2_key1;
                        int ap2k2 = map[i].fact2_key2;
                        PMESG4("COMPARING %d,%d,%d,%d \tWITH %d,%d,%d,%d",ap1k1,ap1k2,ap2k1,ap2k2,fact1_k1,fact1_k2,fact2_k1,fact2_k2)
                        if (map[i].type != fact1Type )
                                continue;
                        if (
                                (
                                ((ap1k1==fact1_k1) && (ap1k2==fact1_k2))
                                ||
                                ((ap1k1==fact1_k2) && (ap1k2==fact1_k1))
                                )
                                &&
                                ( ap2k1 == 0 )
                                &&
                                ( ap2k2 == 0 )
                           )
                                {
                                // 'i' is a candidate for first press
                                PMESG3("Found a match : action %d",i);
                                PEXIT3;
                                return i;
                                }
                        }
        PMESG3("No single fact candidate action found. Keep going, since a 2nd fact might come soon");
        PEXIT3;
        return -1;
        }





// This is called whenever a second fact happens right after the first
int MustuxJogMouseBoard::identify_first_and_second_facts_together()
        {
        PENTER3;
        PMESG3("Adding a 2nd fact %d,%d  to the 1st one  %d,%d",fact2_k1,fact2_k2,fact1_k1,fact1_k2);
        if (fact2_k1!=0)
                {
                if (!isHolding)
                        {
                        if (fact1_k2==0) // first press is <K> (I know that its not [K] because if it was I'd never reach identify_first_and_second_facts_together())
                                {
                                if (fact2_k2==0)
                                        if (fact1_k1 == fact2_k1)
                                                {
                                                PMESG3("Whole action is a <<K>>");
                                                wholeActionType = D_FKEY;  // <<K>>
                                                }
                                        else
                                                {
                                                PMESG3("Whole action is a >K>K");
                                                wholeActionType = S_FKEY_FKEY; // >K>K
                                                }
                                else
                                        {
                                        PMESG3("Whole action is a >K>KK");
                                        wholeActionType = S_FKEY_FKEY2;  // >K>KK
                                        }
                                }

                        else
                                {
                                if (fact2_k2==0)
                                        {
                                        PMESG3("Whole action is a >KK>K");
                                        wholeActionType = S_FKEY2_FKEY;  // >KK>K
                                        }
                                else
                                        {
                                        if (
                                        ((fact1_k1==fact2_k1) && (fact1_k2==fact2_k2)) ||
                                        ((fact1_k1==fact2_k2) && (fact1_k2==fact2_k1))
                                        )
                                                {
                                                PMESG3("Whole action is a <<KK>>");
                                                wholeActionType = D_FKEY2;
                                                }
                                        else
                                                {
                                                PMESG3("Whole action is a >KK>KK");
                                                wholeActionType = S_FKEY2_FKEY2;
                                                }
                                        }
                                }
                        }
                else
                        {
                        if (fact1_k2==0) // first press is <K> (I know that its not [K] because if it was I'd never reach identify_first_and_second_facts_together())
                                {
                                if (fact2_k2==0)
                                        if (fact1_k1 == fact2_k1)
                                                {
                                                PMESG3("Whole action is a <[K]>");
                                                wholeActionType = FHKEY;
                                                }
                                        else
                                                {
                                                PMESG3("Whole action is a >K[K]");
                                                wholeActionType = S_FKEY_HKEY;
                                                }
                                else
                                        {
                                        PMESG3("Whole action is a >K[KK]");
                                        wholeActionType = S_FKEY_HKEY2;
                                        }
                                }
                        else
                                {
                                if (fact2_k2==0)
                                        {
                                        PMESG3("Whole action is a KK[K]");
                                        wholeActionType = S_FKEY2_HKEY;
                                        }
                                else
                                        {
                                        PMESG3("Whole action is a >KK[KK]");
                                        wholeActionType = S_FKEY2_HKEY2;
                                        }
                                }
                        }
                }
        else
                {
                PMESG3("Second fact is null (0,0). Assuming wholeActionType is %s",actionTypeLabel[fact1Type].ascii());
                wholeActionType = fact1Type;
                }

        // whole action type identified .
        PMESG3("Searching for a %s action that matches %d,%d,%d,%d ",actionTypeLabel[wholeActionType].ascii(),fact1_k1,fact1_k2,fact2_k1,fact2_k2);
        for (int i=0; i<mapSize; i++)
                        {
                        if ( map[i].type != wholeActionType ) continue;
                        int ap1k1 = map[i].fact1_key1;
                        int ap1k2 = map[i].fact1_key2;
                        int ap2k1 = map[i].fact2_key1;
                        int ap2k2 = map[i].fact2_key2;
                        PMESG4("COMPARING %d,%d,%d,%d  \tWITH  %d,%d,%d,%d",ap1k1,ap1k2,ap2k1,ap2k2,fact1_k1,fact1_k2,fact2_k1,fact2_k2)
                        if (
                                (
                                ((ap1k1==fact1_k1) && (ap1k2==fact1_k2))
                                ||
                                ((ap1k1==fact1_k2) && (ap1k2==fact1_k1))
                                )
                                &&
                                (
                                ((ap2k1==fact2_k1) && (ap2k2==fact2_k2))
                                ||
                                ((ap2k1==fact2_k2) && (ap2k2==fact2_k1))
                                )
                        )
                                {
                                // 'i' is a candidate the whole action
                                PMESG3("Found a match : action %d",i);
                                PEXIT3;
                                return i;
                                }
                        }
        PMESG3("No candidates found :-(");
        PEXIT3;
        return -1;
        }


void MustuxJogMouseBoard::give_a_chance_for_second_fact()
        {
        PENTER3;
        PMESG3("Waiting %d ms for second fact ...",doubleFactWaitTime );
        secondChanceTimer->start( doubleFactWaitTime );
        isFirstFact=false;
        isHolding = false;
        isPressEventLocked = false;
        stackIndex = 0;
        pressEventCounter = 0;
        fact2_k1 = 0;
        fact2_k2 = 0;
        wholeAction = -1;
        for (int i=0; i < STACK_SIZE; i++)
                {
                eventType[i] = 0;
                eventStack[i] = 0;
                eventTime[i] = 0;
                }
        PEXIT3;
        }


void MustuxJogMouseBoard::quit_second_chance()
        {
        PENTER3;
        secondChanceTimer->stop();
        if (!isHolding) // if it is holding, there is no need to push a new fact
                {
                PMESG3("No second fact (waited %d ms) ... Forcing a null second fact",doubleFactWaitTime);
                push_fact(0,0); // no second press performed, so I am adding a pair of Zeros as second press and keep going...
                }
        PEXIT3;
        }


// ----------------------------- JMB ENGINE : ACTION LEVEL HANDLING -------------------------
void MustuxJogMouseBoard::dispatch_action(int action)
        {
        PENTER3;
        assocInterface->get_default_lcd()->print(map[action].name,0);
        PMESG3("Processing Action %d ( %s ) ",action, (const char*) map[action].name.latin1() );
        MustuxJogMouseBoardMessage* m = new MustuxJogMouseBoardMessage(
                                                        map[action].name,
                                                        false,
                                                        action,
                                                        (QWidget*) 0,
                                                        0,
                                                        0,
                                                        get_action_parameter(action),
                                                        get_collected_number()
                                                        );
        assocInterface->process_jmb_action(m);
        delete m;
        PEXIT3;
        }



// This is called by
void MustuxJogMouseBoard::dispatch_hold()
        {
        PENTER3;
        if (clearOutputTimer!=0)
                {
                clearOutputTimer->stop();
                isHoldingOutput=false;
                }
        assocInterface->get_default_lcd()->clear_line(1);

        wholeAction = -1;
        if (isFirstFact)
                {
                fact1_k1 = eventStack[0];
                fact1_k2 = eventStack[1];
                wholeAction = identify_first_fact(); // I can consider first press the last because there is nothing after a []
                }
        else
                {
                fact2_k1 = eventStack[0];
                fact2_k2 = eventStack[1];
                wholeAction = identify_first_and_second_facts_together();
                }

        if (wholeAction>=0)
                {
                PMESG3("Processing HOLD Action %d ( %s ) ",wholeAction,(const char*) map[wholeAction].name.latin1());
                MustuxJogMouseBoardAction jmbe = map[wholeAction];
                assocInterface->get_default_lcd()->print(jmbe.name,0);
                if ((jmbe.useX) && (jmbe.useY))
                        {
                        assocInterface->get_default_lcd()->print("Move Mouse U/D/L/R",1);
                        }
                else if (jmbe.useX)
                        {
                        assocInterface->get_default_lcd()->print("Move Mouse Left/Right",1);
                        }
                else if(jmbe.useY)
                        {
                        assocInterface->get_default_lcd()->print("Move Mouse Up/Down",1);
                        }
                PMESG3("Processing hold action %d ",wholeAction );
                MustuxJogMouseBoardMessage* m = new MustuxJogMouseBoardMessage(
                                                        map[wholeAction].name,
                                                        true,
                                                        wholeAction,
                                                        (QWidget*) 0,
                                                        0,
                                                        0,
                                                        get_action_parameter(wholeAction),
                                                        get_collected_number()
                                                        );

                assocInterface->process_jmb_action(m);
                }
        else
                {
                QString s = "Unknown "; s = s.append(actionTypeLabel[fact1Type]); s = s.append(" Action !");
                assocInterface->get_default_lcd()->print(s,0);
                }
        stop_collecting();
        // note that we dont call conclusion here :)
        PEXIT3;
        }




void MustuxJogMouseBoard::finish_hold()
        {
        PENTER3;
        assocInterface->get_default_lcd()->clear_line(1);
        PMESG3("Finishing hold action %d",wholeAction);
        if (wholeAction>=0)
                {
                MustuxJogMouseBoardMessage* m =
                                new MustuxJogMouseBoardMessage(
                                                                map[wholeAction].name,
                                                                false,
                                                                wholeAction,
                                                                (QWidget*) 0,
                                                                0,
                                                                0,
                                                                get_action_parameter(wholeAction),
                                                                get_collected_number()
                                                                );

                assocInterface->process_jmb_action(m);
                }
        conclusion();
        PEXIT3;
        }


void MustuxJogMouseBoard::conclusion()
        {
        PENTER3;
        assocInterface->get_default_lcd()->print("Action Finished",4);
        reset();
        hold_output();
        PEXIT3;
        }





void MustuxJogMouseBoard::hold_output()
        {
        PENTER3;
        if (!isHoldingOutput)
                {
                clearOutputTimer->start(clearTime);
                isHoldingOutput=true;
                }
        PEXIT3;
        }



int MustuxJogMouseBoard::init_map(QString mapFilename)
        {
        PENTER2;
        mapSize=0;
        QFile mapFile(mapFilename);
        if ( mapFile.open(IO_ReadOnly) )
                {
                PMESG2("INITIALIZING JMB MAP ... ");
                int index=0;

                QTextStream t( &mapFile );        // use a text stream
                QString line;
                while ( !t.eof() )
                        {
                        line = t.readLine();
                        int x = line.find("#");
                        if (x>=0) line = line.left(x);
                        line = line.simplifyWhiteSpace();
                        if (line.length()==0)
                                continue;

                        bool useX=false,useY=false;
                        int actionTypeCode=0, nKeys=0;
                        int keyCode[4];
                        int parameter;

                        int x2=line.find("\"",1);
                        QString label = line.mid(1,x2-1);
                        line=line.mid(x2+2);
                        int x3=line.find(" ");
                        QString actionType = line.left(x3);
                        line=line.mid(x3+1);

                        if      ( strcmp(actionType,"<K>")    == 0 )  { actionTypeCode = FKEY;          nKeys = 1; }
                        else if ( strcmp(actionType,"<KK>")   == 0 )  { actionTypeCode = FKEY2;         nKeys = 2; }
                        else if ( strcmp(actionType,"[K]")    == 0 )  { actionTypeCode = HKEY;          nKeys = 1; }
                        else if ( strcmp(actionType,"[KK]")   == 0 )  { actionTypeCode = HKEY2;         nKeys = 2; }
                        else if ( strcmp(actionType,"<<K>>")  == 0 )  { actionTypeCode = D_FKEY;        nKeys = 1; }
                        else if ( strcmp(actionType,"<<KK>>") == 0 )  { actionTypeCode = D_FKEY2;       nKeys = 2; }
                        else if ( strcmp(actionType,"<[K]>")  == 0 )  { actionTypeCode = FHKEY;         nKeys = 1; }
                        else if ( strcmp(actionType,"<[KK]>") == 0 )  { actionTypeCode = FHKEY2;        nKeys = 2; }
                        else if ( strcmp(actionType,">K>K")   == 0 )  { actionTypeCode = S_FKEY_FKEY;   nKeys = 2; }
                        else if ( strcmp(actionType,">K>KK")  == 0 )  { actionTypeCode = S_FKEY_FKEY2;  nKeys = 3; }
                        else if ( strcmp(actionType,">KK>K")  == 0 )  { actionTypeCode = S_FKEY2_FKEY;  nKeys = 3; }
                        else if ( strcmp(actionType,">KK>KK") == 0 )  { actionTypeCode = S_FKEY2_FKEY2; nKeys = 4; }
                        else if ( strcmp(actionType,">K[K]")  == 0 )  { actionTypeCode = S_FKEY_HKEY;   nKeys = 2; }
                        else if ( strcmp(actionType,">K[KK]") == 0 )  { actionTypeCode = S_FKEY_HKEY2;  nKeys = 3; }
                        else if ( strcmp(actionType,">KK[K]") == 0 )  { actionTypeCode = S_FKEY2_HKEY;  nKeys = 3; }
                        else if ( strcmp(actionType,">KK[KK]")== 0 )  { actionTypeCode = S_FKEY2_HKEY2; nKeys = 4; }

                        int x4=line.find(" ");
                        QString sUseX = line.left(x4);
                        if (sUseX=="0") useX=false; else useX=true;
                        line=line.mid(x4+1);

                        int x5=line.find(" ");
                        QString sUseY = line.left(x5);
                        if (sUseX=="0") useY=false; else useY=true;
                        line=line.mid(x5+1);

                        // get nKeys hex codes
                        for (int nk=0; nk<4; nk++) keyCode[nk]=0;
                        for (int nk=0; nk<nKeys; nk++)
                                {
                                int x6=line.find(" ");
                                QString sTag = line.left(x6);
                                line=line.mid(x6+1);
                                int keycode=strtol((char*)sTag.latin1(),0,16);
                                keyCode[nk]=keycode;
                                }

                        // get the parameter
                        QString sParameter = line;
                        parameter = sParameter.toInt();

                        // Fix the keyCode positions
                        if      (( actionTypeCode == D_FKEY  )     ||
                                ( actionTypeCode == FHKEY  ))        { keyCode[2]=keyCode[0]; }
                        else if (( actionTypeCode == S_FKEY_FKEY ) ||
                                ( actionTypeCode == S_FKEY_HKEY ))   { keyCode[2]=keyCode[1]; keyCode[1]=0; }
                        else if (( actionTypeCode == D_FKEY2 )     ||
                                ( actionTypeCode == FHKEY  ))        { keyCode[2]=keyCode[0]; keyCode[3]=keyCode[1]; }
                        PMESG3("ADDED action: %s : %s  keys=%d,%d,%d,%d  par=%d", (const char*) label.latin1(),  (const char*)  actionType.latin1(), keyCode[0],keyCode[1],keyCode[2],keyCode[3], parameter);
                        map[index++].set(label, actionTypeCode, keyCode[0], keyCode[1], keyCode[2], keyCode[3], useX, useY, false, parameter);
                        }

                mapSize=index;
                PMESG2("Optimizing map for best performance ...");
                int optimizedActions=0;
                for (int i=0; i<mapSize; i++)
                        {
                        int c1A=map[i].fact1_key1;
                        int c2A=map[i].fact1_key2;

                        if ( (map[i].type==FKEY) )
                                {
                                bool alone = true;
                                for (int j = 0; j < mapSize; j++)
                                        {
                                        if (j==i) continue;
                                        int tt    = map[j].type;
                                        int t1A   = map[j].fact1_key1;
                                        if  (
                                                (t1A==c1A)  &&
                                                (
                                                        (tt==D_FKEY)       ||
                                                        (tt==FHKEY)        ||
                                                        (tt==S_FKEY_FKEY)  ||
                                                        (tt==S_FKEY_FKEY2) ||
                                                        (tt==S_FKEY_HKEY)  ||
                                                        (tt==S_FKEY_HKEY2)
                                                )
                                            )
                                        alone=false;
                                        }
                                if (alone)
                                        {
                                        PMESG3("Setting <K> action %s as protected (Instantaneous response)",(const char*) map[i].name.latin1());
                                        map[i].set_instantaneous(true);
                                        optimizedActions++;
                                        }
                                else
                                        map[i].set_instantaneous(false);

                                }
                        else if ((map[i].type==FKEY2))
                                {
                                bool alone = true;
                                for (int j = 0; j < mapSize; j++)
                                        {
                                        if (j==i) continue;
                                        int tt    = map[j].type;
                                        int t1A   = map[j].fact1_key1;
                                        int t2A   = map[j].fact1_key2;
                                        if (
                                        ((t1A==c1A) && (t2A==c2A)) &&
                                        (
                                        (tt==HKEY)         ||
                                        (tt==D_FKEY2)      ||
                                        (tt==S_FKEY2_FKEY) ||
                                        (tt==S_FKEY2_FKEY2)||
                                        (tt==S_FKEY2_HKEY) ||
                                        (tt==S_FKEY2_HKEY2)
                                        )
                                        )
                                        alone=false;
                                        }
                                if (alone)
                                        {
                                        PMESG3("Setting <KK> action %s for instantaneous response", map[i].name.ascii());
                                        map[i].set_instantaneous(true);
                                        optimizedActions++;
                                        }
                                else
                                        map[i].set_instantaneous(false);
                                }
                        }
                PMESG2("JMB MAP INITIALIZED! %d actions registered ( %d optimized) .", mapSize, optimizedActions);
                mapFile.close();
                }
        else
                {
                PERROR("Cannot open jmb.map");
                }
        PEXIT2;
        return 1;
        }






// callbacks


void MustuxJogMouseBoard::clear_output()
        {
        PENTER3;
        if ((isHoldingOutput))
                {
                // UNCOMMENT ME ASAP output->clear();
                clearOutputTimer->stop();
                isHoldingOutput=false;
                // UNCOMMENT ME ASAP output->draw_all_grids();
                for (int k=0; k<4; k++)
                        {
                        assocInterface->get_default_lcd()->clear_line(k);
                        assocInterface->get_default_lcd()->draw_line_grid(k);
                        }
                assocInterface->get_default_lcd()->print("Protux JMB Active",4);
                }
        PEXIT3;
        }


// Settings

void MustuxJogMouseBoard::set_clear_time(int time)
        {
        clearTime = time;
        }

void MustuxJogMouseBoard::set_hold_sensitiveness(int htime)
        {
        assumeHoldTime=htime;
        }

void MustuxJogMouseBoard::set_double_fact_interval(int time)
        {
        doubleFactWaitTime = time;
        }


QString MustuxJogMouseBoard::get_action_name(int a)
        {
        return map[a].name;
        }

int MustuxJogMouseBoard::get_action_parameter(int a)
        {
        return map[a].parameter;
        }



// Number colector
void MustuxJogMouseBoard::check_number_collection()
        {
        PENTER3;
        if ((fact1_k1 >= 0x30) && (fact1_k1 <= 0x39))
                {
                if (!isCollecting)
                        {
                        PMESG3("Starting number collection...");
                        sCollectedNumber="";
                        }
                isCollecting = true;
                collectedDigit++;
                sCollectedNumber.append( QChar(fact1_k1) ); // it had a ",1" complement after fact1_k1... why?
                PMESG3("Collected %s so far...",sCollectedNumber.ascii() ) ;
                QString sn = "NUMBER " + sCollectedNumber;
                collectedNumber = atoi(sCollectedNumber);
                assocInterface->get_default_lcd()->print(sn,3);
                }
        else
                stop_collecting();
        PEXIT3;
        }

int MustuxJogMouseBoard::get_collected_number()
        {
        PENTER3;
        int n = collectedNumber ;
        collectedNumber = -1; // collectedNumber has a life of only one get.
        PEXIT3;
        return n;
        }

void MustuxJogMouseBoard::stop_collecting()
        {
        PENTER3;
        isCollecting=false;
        collectedDigit = 0;
        collectedNumber = atoi(sCollectedNumber);
        sCollectedNumber ="-1";
        assocInterface->get_default_lcd()->print("Number Collection Finished",3);
        PEXIT3;
        }


void MustuxJogMouseBoard::lock()
        {
        PENTER3;
        locked=true;
        PMESG3("now locked=%d",locked);
        PEXIT3;
        }

void MustuxJogMouseBoard::unlock()
        {
        PENTER3;
        locked=false;
        PMESG3("now locked=%d",locked);
        PEXIT3;
        }

bool MustuxJogMouseBoard::is_locked()
        {
        return locked;
        }

QString MustuxJogMouseBoard::actionTypeLabel[16] = {"<K>","<KK>","[K]","[KK]","<<K>>","<<KK>>","<[K]>","<[KK]>",">K>K",">K>KK",">KK>K",">KK>KK",">K[K]",">K[KK]",">KK[K]",">KK[KK]" };
//eof
