/**************************************************************************************

        MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

#include "MustuxFft.hh"
#include "MustuxDebugger.hh"
#include <math.h>
#include <stdio.h>



unsigned MustuxFft::numberOfBitsNeeded ( unsigned param )
        {
        unsigned i;
        if ( param < 2 )
                {
                PERROR("%d is too small! ",param );
                return 0;
                }
        for ( i=0; ; i++ )
                {
                if ( param & (1 << i) )
                return i;
                }
        }



int MustuxFft::reverseBits ( unsigned index, unsigned numBits )
        {
        unsigned i, rev;
        for ( i=rev=0; i < numBits; i++ )
                {
                rev = (rev << 1) | (index & 1);
                index >>= 1;
                }
        return rev;
        }




MustuxFft::MustuxFft(int pSize, int pWinType, int pSampleRate)
        {
        PENTERCONS;
        size=pSize;
        winType=pWinType;
        sampleRate=pSampleRate;

        switch( winType )
                {
                case BLACKMANN_WINDOW:  // Blackman
                        {
                        for( int n = 0; n < size; n++ )
                                {
                                windowCoef[n] = 0.42 - 0.5*cos(PI2*n/(size-1)) +
                                        0.08*cos(2.*PI2*n/(size-1));
                                }
                        break;
                        }
                case BARTLETT_WINDOW:  // Bartlett
                        {
                        for( int n = 0; n <= (size-1)/2; n++ )
                                {
                                windowCoef[n] = 2.*n/(size-1);
                                }
                        for( int n = (size-1)/2; n < size; n++ )
                                {
                                windowCoef[n] = 2. - 2.*n/(size-1);
                                }
                        break;
                        }
                case HAMMING_WINDOW:  // Hamming
                        {
                        for( int n = 0; n < size; n++ )
                                {
                                windowCoef[n] = 0.54 - 0.46*cos( PI2*n/(size-1) );
                                }

                        break;
                        }
                case VON_HANN_WINDOW:  // von Hann (sometimes improperly called Hanning)
                        {
                        for( int n = 0; n < size; n++ )
                                {
                                windowCoef[n] = 0.5*(1.0 - cos(PI2*n/(size-1)) );
                                }
                        break;
                        }
                case BLACKMAN_HARRIS_WINDOW: // 4 term Blackman-Harris
                        {
                        for( int n = 0; n < size; n++ )
                                {
                                windowCoef[n] = 0.35875 - 0.48829 * cos( PI2 *      (float) (n + 0.5 ) / size ) +
                                                          0.14128 * cos( PI2 * 2. * (float) (n + 0.5 ) / size ) -
                                                          0.01168 * cos( PI2 * 3. * (float) (n + 0.5 ) / size );
                                }
                        break;
                        }
                }

        noiseLine = NORMALIZATION_FACTOR / 10; // also empiric. Must be parametrized
        PEXITCONS;
        }


MustuxFft::~MustuxFft()
        {
        PENTERDES;
        PEXITDES;
        }


void MustuxFft::perform(float* data_re, float* outputLevels)
        {

        // apply the smooth window on input data
        for (int i=0; i<size; i++) data_re[i]=data_re[i] * windowCoef[i];


        // Part of this code was adapted from FFT code by Don Cross <dcross@intersrv.com>

        unsigned numBits;    // Number of bits needed to store indices
        int k;

        float angleNumerator = 2.0 * PI;
        float tr, ti;

        // Uncomment the following line if you want reverse fft
        // angleNumerator = - angleNumerator;

        numBits = numberOfBitsNeeded ( size );

        // Do simultaneous data copy and bit-reversal ordering into outputs...

        for ( int i=0; i < size; i++ )
                {
                int j = reverseBits ( i, numBits );
                buf_re[j] = data_re[i];
                buf_im[j] = 0;
                }

        int blockEnd = 1;
        for ( int blockSize = 2; blockSize <= size; blockSize <<= 1 )
                {
                float delta_angle = angleNumerator / (float)blockSize;
                float sm2 = sin ( -2 * delta_angle );
                float sm1 = sin ( -delta_angle );
                float cm2 = cos ( -2 * delta_angle );
                float cm1 = cos ( -delta_angle );
                float w = 2 * cm1;
                float ar[3], ai[3];

                for ( int i=0; i < size; i += blockSize )
                        {
                        ar[2] = cm2;
                        ar[1] = cm1;

                        ai[2] = sm2;
                        ai[1] = sm1;

                        for ( int j=i, n=0; n < blockEnd; j++, n++ )
                                {
                                ar[0] = w*ar[1] - ar[2];
                                ar[2] = ar[1];
                                ar[1] = ar[0];

                                ai[0] = w*ai[1] - ai[2];
                                ai[2] = ai[1];
                                ai[1] = ai[0];

                                k = j + blockEnd;
                                tr = ar[0]*buf_re[k] - ai[0]*buf_im[k];
                                ti = ar[0]*buf_im[k] + ai[0]*buf_re[k];

                                buf_re[k] = buf_re[j] - tr;
                                buf_im[k] = buf_im[j] - ti;

                                buf_re[j] += tr;
                                buf_im[j] += ti;
                                }
                        }

                blockEnd = blockSize;
                }

                for ( int i=0; i < size; i++ )
                        {
                        buf_re[i] /= size;
                        buf_im[i] /= size;
                        }

        dcComponent = (buf_re[0]/size)*(buf_re[0]/size);

        for (int k = 1; k < size/2; k++)
                {
                outputLevels[k] =  sqrt((buf_re[k]*buf_re[k]) + (buf_im[k]*buf_im[k])) / NORMALIZATION_FACTOR; // - dcComponent;
                }

        }


float MustuxFft::index_to_frequency ( int index )
        {
        float f;
        if ( ( index < 0 ) || ( index > size/2 )   )
                return -1.0;
        f =   (float) index / (float) size;
        return f*sampleRate;
        }

int MustuxFft::frequency_to_index ( float freq)
        {
        if (  ( freq < 0 ) || ( freq > sampleRate/2 ))
                return -1;
        float dindex = (float) size * ( (float) freq / (float) sampleRate );
        return (int) dindex;
        }


float MustuxFft::get_level_noise_line()
        {
        return noiseLine;
        }


void MustuxFft::set_level_noise_line(float val)
        {
        val = val > 100.0 ? 100.0 : val;
        val = val < 0.0   ? 0.0   : val;
        noiseLine = val;
        }

//eof

