/**************************************************************************************

        MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/

/**
 *  MustuxDebugger is a collection of macros that makes easier the job of debugging
 *  a Mustux Application.
 *  The list of macros is :
 *  PENTER - Outputs a message when entering a method in level 1. Used in the FIRST line of a method;
 *  PEXIT  - Outputs a message when leaving a method in level 1. Used in the LAST line of a method,
 *           except if the last line is a return statement (in this case is put immediately before
 *           the return statement
 *  PENTER2 - Same as PENTER for levels 1  and 2
 *  PEXIT2  - Same as PEXIT for levels 1 and 2
 *  PENTER3 - Same as PENTER for levels 1 2 and 3
 *  PEXIT3  - Same as PEXIT for levels 1 2 and 3
 *  PENTER4 - Same as PENTER for levels 1 2 3 and 4
 *  PEXIT4  - Same as PEXIT for levels 1 2 3 and 4
 *  PMESG(message) - Outputs a message in level 1
 *  PMESG2(message) - Outputs a message in level 1 and 2
 *  PENTERCONS - Outputs a message when entering a constructor in levels 2, 3, and 4. Similar to PENTER
 *  PEXITCONS  - Outputs a message when leaving a constructor in levels 2, 3, and 4. Similar to PEXIT
 *  PENTERDES  - Outputs a message when entering a destructor in levels 2, 3, and 4. Similar to PENTER
 *  PEXITDES   - Outputs a message when leaving a destructor in levels 2, 3, and 4. Similar to PEXIT
 *  Same can be done for PENTERCONS2, PEXITCONS2, PENTERCONS3... and so on...
 */

#ifndef MUSTUXDEBUGGER_HH
#define MUSTUXDEBUGGER_HH

#include <qstring.h>

//Debugging Macros
#define CHANGE_COLOR_BLACK   printf("%c[0;31;40m",27)
#define CHANGE_COLOR_RED     printf("%c[0;31;40m",27)
#define CHANGE_COLOR_GREEN   printf("%c[0;32;40m",27)
#define CHANGE_COLOR_ORANGE  printf("%c[0;33;40m",27)
#define CHANGE_COLOR_BLUE    printf("%c[0;34;40m",27)
#define CHANGE_COLOR_MAGENTA printf("%c[0;35;40m",27)
#define CHANGE_COLOR_CYAN    printf("%c[0;36;40m",27)
#define CHANGE_COLOR_WHITE   printf("%c[0;37;40m",27)
#define CHANGE_COLOR_YELLOW  printf("%c[0;33;40m",27)

#define SHOW_WHERE printf("%s",__FILE__); printf("::"); printf("%s",__FUNCTION__); printf("\n");
#define PCHECK_BUFFER_MAX( buf, size)  float pcbmax=0.0; for (int pcbk=0; pcbk<size; pcbk++) if (buf[pcbk]>pcbmax) pcbmax=buf[pcbk]; printf("      MAX=%5.1f\n",pcbmax);

#define PENTER_REAL(lvl)        { using namespace MustuxDebugger; if (get_debug_level()>=lvl) { if (is_logging()) { QString output = get_tabs() + "ENTERING " + QString(__FILE__) + "::" + QString(__FUNCTION__) + "\n"; log(output); more_tabs(); } else { fill_tabs(); CHANGE_COLOR_GREEN; printf("ENTERING "); CHANGE_COLOR_WHITE; SHOW_WHERE; more_tabs(); } } }
#define PEXIT_REAL(lvl)         { using namespace MustuxDebugger; if (get_debug_level()>=lvl) { if (is_logging()) { less_tabs(); QString output = get_tabs() + "LEAVING " + QString(__FILE__) + "::" + QString(__FUNCTION__) + "\n"; log(output);  } else { less_tabs(); fill_tabs(); CHANGE_COLOR_BLUE; printf("LEAVING "); CHANGE_COLOR_WHITE; SHOW_WHERE;   } } }
#define PENTERCONS_REAL(lvl)    { using namespace MustuxDebugger; if (get_debug_level()>=lvl) { if (is_logging()) { QString output = get_tabs() + "ENTERING " + QString(__FUNCTION__) + " (CONSTRUCTOR)\n"; log(output); more_tabs(); } else { fill_tabs(); CHANGE_COLOR_GREEN; printf("ENTERING "); CHANGE_COLOR_WHITE; printf("%s",__FUNCTION__); CHANGE_COLOR_CYAN; printf(" (CONSTRUCTOR)"); CHANGE_COLOR_WHITE; printf("\n"); more_tabs(); } } }
#define PEXITCONS_REAL(lvl)     { using namespace MustuxDebugger; if (get_debug_level()>=lvl) { if (is_logging()) { less_tabs(); QString output = get_tabs() + "LEAVING " + QString(__FUNCTION__) + " (CONSTRUCTOR)\n";  log(output); } else { less_tabs(); fill_tabs(); CHANGE_COLOR_BLUE; printf("LEAVING "); CHANGE_COLOR_WHITE;  printf("%s",__FUNCTION__); CHANGE_COLOR_CYAN; printf(" (CONSTRUCTOR)"); CHANGE_COLOR_WHITE; printf("\n");  } } }
#define PENTERDES_REAL(lvl)     { using namespace MustuxDebugger; if (get_debug_level()>=lvl) { if (is_logging()) { QString output = get_tabs() + "ENTERING " + QString(__FUNCTION__) + " (DESTRUCTOR)\n"; log(output);  more_tabs(); } else { fill_tabs(); CHANGE_COLOR_GREEN; printf("ENTERING "); CHANGE_COLOR_WHITE; printf("%s",__FUNCTION__); CHANGE_COLOR_CYAN; printf(" (DESTRUCTOR)"); CHANGE_COLOR_WHITE; printf("\n"); more_tabs();  } } }
#define PEXITDES_REAL(lvl)      { using namespace MustuxDebugger; if (get_debug_level()>=lvl) { if (is_logging()) { less_tabs(); QString output = get_tabs() + "LEAVING " + QString(__FUNCTION__) + " (DESTRUCTOR)\n";   log(output); } else { less_tabs(); fill_tabs(); CHANGE_COLOR_BLUE; printf("LEAVING "); CHANGE_COLOR_WHITE; printf("%s",__FUNCTION__); CHANGE_COLOR_CYAN; printf(" (DESTRUCTOR)"); CHANGE_COLOR_WHITE; printf("\n");    } } }
#define PMESG(args...)          { using namespace MustuxDebugger; if (get_debug_level()>=BASIC)       { if (is_logging())  { QString x; x.sprintf(args); QString output = get_tabs() + "[ " + x + " ]\n"; log(output); } else { fill_tabs(); CHANGE_COLOR_MAGENTA; printf("[ "); printf(args); printf(" ]"); CHANGE_COLOR_WHITE; printf("\n"); } } }
#define PMESG2(args...)         { using namespace MustuxDebugger; if (get_debug_level()>=FLOOD)       { if (is_logging())  { QString x; x.sprintf(args); QString output = get_tabs() + "[ " + x + " ]\n"; log(output); } else { fill_tabs(); CHANGE_COLOR_MAGENTA; printf("[ "); printf(args); printf(" ]"); CHANGE_COLOR_WHITE; printf("\n"); } } }
#define PMESG3(args...)         { using namespace MustuxDebugger; if (get_debug_level()>=SUPER_FLOOD) { if (is_logging())  { QString x; x.sprintf(args); QString output = get_tabs() + "[ " + x + " ]\n"; log(output); } else { fill_tabs(); CHANGE_COLOR_MAGENTA; printf("[ "); printf(args); printf(" ]"); CHANGE_COLOR_WHITE; printf("\n"); } } }
#define PMESG4(args...)         { using namespace MustuxDebugger; if (get_debug_level()>=ALL)         { if (is_logging())  { QString x; x.sprintf(args); QString output = get_tabs() + "[ " + x + " ]\n"; log(output); } else { fill_tabs(); CHANGE_COLOR_MAGENTA; printf("[ "); printf(args); printf(" ]"); CHANGE_COLOR_WHITE; printf("\n"); } } }

#define PMESG_START(args...)    { using namespace MustuxDebugger; if (get_debug_level()>=BASIC) { if (is_logging()) { QString x; x.sprintf(args); QString output = get_tabs() + "[ " + x; log(output); } else { fill_tabs(); CHANGE_COLOR_MAGENTA; printf("[ "); printf(args); } } }
#define PMESG_END(args...)      { using namespace MustuxDebugger; if (get_debug_level()>=BASIC) { if (is_logging()) { QString x; x.sprintf(args); QString output = x + " ]\n"; log(output); } else { printf(args); printf(" ]"); CHANGE_COLOR_WHITE; printf("\n"); } } }
#define PMESG2_START(args...)   { using namespace MustuxDebugger; if (get_debug_level()>=FLOOD) { if (is_logging()) { QString x; x.sprintf(args); QString output = get_tabs() + "[ " + x; log(output); } else { fill_tabs(); CHANGE_COLOR_MAGENTA; printf("[ "); printf(args); } } }
#define PMESG2_END(args...)     { using namespace MustuxDebugger; if (get_debug_level()>=FLOOD) { if (is_logging()) { QString x; x.sprintf(args); QString output = x + " ]\n"; log(output); } else { printf(args); printf(" ]"); CHANGE_COLOR_WHITE; printf("\n"); } } }

#define PDEBUG(args...)         { using namespace MustuxDebugger; if (is_logging()) { QString x; x.sprintf(args); QString output = "DEBUG : " + QString(__FILE__) + "::" + QString(__FUNCTION__) + ":" + x + "\n"; log(output); } else { CHANGE_COLOR_GREEN; printf("DEBUG : ");printf("%s",__FILE__); printf("::"); printf("%s",__FUNCTION__); printf(":"); printf(args); CHANGE_COLOR_WHITE; printf("\n"); } }
#define PERROR(args...)         { using namespace MustuxDebugger; if (is_logging()) { QString x; x.sprintf(args); QString output = "\n *** Error in " + QString(__PRETTY_FUNCTION__) + "\n" + x + "\n\n"; } else {  printf("\n"); CHANGE_COLOR_RED; printf("*** Error in "); printf("%s",__PRETTY_FUNCTION__); printf("\n"); printf(args); CHANGE_COLOR_WHITE; printf("\n\n"); } }
#define PERROR2(args...)         { using namespace MustuxDebugger; if (is_logging()) { QString x; x.sprintf(args); QString output = "\n *** Error in " + QString(__PRETTY_FUNCTION__) + "\n" + x + "\n\n"; } else if (get_debug_level()>=FLOOD) {  printf("\n"); CHANGE_COLOR_RED; printf("*** Error in "); printf("%s",__PRETTY_FUNCTION__); printf("\n"); printf(args); CHANGE_COLOR_WHITE; printf("\n\n"); } }
#define PWARN(args...)          { using namespace MustuxDebugger; if (is_logging()) { QString x; x.sprintf(args); QString output = "WARNING: " + x + "\n"; log(output); } else { CHANGE_COLOR_YELLOW; printf("WARNING: "); printf(args); CHANGE_COLOR_WHITE; printf("\n"); } }
#define PWARN2(args...)         { using namespace MustuxDebugger; if (get_debug_level()>=FLOOD) { if (is_logging()) { QString x; x.sprintf(args); QString output = "WARNING: " + x + "\n"; log(output); } else { CHANGE_COLOR_YELLOW; printf("WARNING: "); printf(args); CHANGE_COLOR_WHITE; printf("\n"); } } }
#define PMARK(args...)          { using namespace MustuxDebugger; if (is_logging()) { QString x; x.sprintf(args); QString output = "--> Reached Mark : " + x + "\n"; log(output); } else { CHANGE_COLOR_YELLOW; printf("--> Reached Mark : "); printf(args); CHANGE_COLOR_WHITE; printf("\n"); } }



#define PENTER                  PENTER_REAL(MustuxDebugger::BASIC)
#define PEXIT                   PEXIT_REAL(MustuxDebugger::BASIC)
#define PENTER2                 PENTER_REAL(MustuxDebugger::FLOOD)
#define PEXIT2                  PEXIT_REAL(MustuxDebugger::FLOOD)
#define PENTER3                 PENTER_REAL(MustuxDebugger::SUPER_FLOOD)
#define PEXIT3                  PEXIT_REAL(MustuxDebugger::SUPER_FLOOD)
#define PENTER4                 PENTER_REAL(MustuxDebugger::ALL)
#define PEXIT4                  PEXIT_REAL(MustuxDebugger::ALL)
#define PENTERCONS              PENTERCONS_REAL(MustuxDebugger::BASIC)
#define PEXITCONS               PEXITCONS_REAL(MustuxDebugger::BASIC)
#define PENTERDES               PENTERDES_REAL(MustuxDebugger::BASIC)
#define PEXITDES                PEXITDES_REAL(MustuxDebugger::BASIC)
#define PENTERCONS2             PENTERCONS_REAL(MustuxDebugger::FLOOD)
#define PEXITCONS2              PEXITCONS_REAL(MustuxDebugger::FLOOD)
#define PENTERDES2              PENTERDES_REAL(MustuxDebugger::FLOOD)
#define PEXITDES2               PEXITDES_REAL(MustuxDebugger::FLOOD)
#define PENTERCONS3             PENTERCONS_REAL(MustuxDebugger::SUPER_FLOOD)
#define PEXITCONS3              PEXITCONS_REAL(MustuxDebugger::SUPER_FLOOD)
#define PENTERDES3              PENTERDES_REAL(MustuxDebugger::SUPER_FLOOD)
#define PEXITDES3               PEXITDES_REAL(MustuxDebugger::SUPER_FLOOD)
#define PENTERCONS4             PENTERCONS_REAL(MustuxDebugger::ALL)
#define PEXITCONS4              PEXITCONS_REAL(MustuxDebugger::ALL)
#define PENTERDES4              PENTERDES_REAL(MustuxDebugger::ALL)
#define PEXITDES4               PEXITDES_REAL(MustuxDebugger::ALL)


namespace MustuxDebugger
        {
        static const int OFF = 0;   // no debug output at all
        static const int BASIC = 1; // only level 1 calls
        static const int FLOOD = 2; // level 1 , level 2 and constructors/destructor calls
        static const int SUPER_FLOOD = 3; // all previous plus low level JMB messages
        static const int ALL = 4; // all messages (including in timer loops)

        /**
        * Used internally by MustuxDebugger. Align the output with the level of execution in a given
        * moment
        */
        void fill_tabs();

        /**
        * Used internally by MustuxDebugger. Get a " " (space) sequence whch aligns the output with the level of execution in a given
        * moment,
        */
        QString get_tabs();


        /**
        * Used internally by MustuxDebugger. Increase one level of execution in output messages
        * moment
        */
        void more_tabs();

        /**
        * Used internally by MustuxDebugger. Decrease one level of execution in output messages
        */
        void less_tabs();

        /**
         * Set the debug level
         */
        void set_debug_level(int l);

        /**
        * Used internally by MustuxDebugger. Returns true if debugOn flag is true.
        */
        int get_debug_level();

        /**
         * create a log file "fn" under home dir and enable copy of all debugging
         * messagem to this file.
         */
        void create_log(QString fn);

        /** close the log file
         */
        void close_log();

        /**
         * Used internally by MustuxDebugger. Feed the log file.
         */
        void log(QString msg);

        /**
         * Used internally to check if output is stdout or a log file
         */
        bool is_logging();
        }





#endif



