/**************************************************************************************

        PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/


#ifndef MUSTUXAUDIOFILEFORMATS_HH
#define MUSTUXAUDIOFILEFORMATS_HH

#include <stdio.h>
#include <qstring.h>

// remove/comment line below if you dont want internal ogg support
// #define OGG_VORBIS_SUPPORT

#ifdef OGG_VORBIS_SUPPORT
        #include <vorbis/codec.h>
        #include <vorbis/vorbisenc.h>
        #include <vorbis/vorbisfile.h>
#endif


/**
 *       The base class for PrafFile, WavFile and OggFile
 */
class MustuxAudioFileFormat
        {

        public:

                MustuxAudioFileFormat(int pHeaderSize);
                virtual ~MustuxAudioFileFormat();

                static const int HEADER_LABEL_SIZE = 16;
                static const int HEADER_DESCRIPTION_SIZE = 56;
                static const int HEADER_SIZE = 256;
                static const int HEADER_RESERVED_SIZE = 128;

                QString get_info();

                virtual int open(QString pFilename);
                virtual int create(QString pFilename);
                virtual int copy_to(QString pFilename);
                virtual int rewind_all();
                virtual int rewind_audio();
                virtual void go_botton();

                /** locate the file at a raw/exact current position given in bytes */
                virtual int byte_seek(long long bytePos);

                /** locate the file at a given block position (audio block position),
                 *  which is independent of blocksize, channels, rate or bitdepht
                 */
                virtual int block_seek(long long blockPos);

                /** returns the raw/exact current position of file in bytes */
                virtual long long byte_tell();

                /** returns the block position (audio block position) of file,
                 *  which is independent of blocksize, channels, rate or bitdepht
                 */
                virtual long long block_tell();

                /** locate the file right after its headernnels, rate or bitdepht
                 */
                virtual int skip_header();

                /** read the file header
                 */
                virtual int read_header();

                virtual int write_header(int pChannels=2, int pRate=44100, int pBitDepth=16,
                                          long long pAudioSize = 0,
                                          QString pProjectLabel = 0,
                                          QString pGroupLabel = 0 ,
                                          QString pHardwareLabel = 0,
                                          QString pDescription = 0 );
                virtual int append_samples_from_file(MustuxAudioFileFormat* sourceFile);
                virtual int append_samples_from_buffer(char* buffer, int bufferSize);

                /** read a fragment of audio from the file containing 'size' bytes
                 *  returns number of sucessfully fragments read*/
                virtual int read_fragment(char* frag, int size);

                /** write a fragment of audio from the file containing 'size' bytes
                 *  returns number of sucessfully fragments written */
                virtual int write_fragment(char* frag, int size);

                virtual int fix_audio_size();
                virtual int close_file();
                virtual bool eof();
                virtual int get_best_frag_size();

                // public attributes
                QString filename;
                int channels;
                int rate;
                int bitDepth;
                int blockSize;
                int headerSize;
                int bytesPerSec;
                int sampleSize;
                long long audioSize;
                long long fileSize;
                long long totalBlocks;
                QString projectLabel;
                QString groupLabel;
                QString hardwareLabel;
                QString description;

                /** returns a descriptor for the phisical file.
                 * ATTENTION : Some classes (Ogg, for example)
                 * overrides it and return null to forbids you
                 * to have access to the phisical file.
                 */
                FILE* get_file();


        protected:
                FILE* file;
                char formatLabel[4];
        };


/**
 * The native Protux Audio Format. It is a RAW-like format, but with
 * some reserved schema for specific Protux data and a fixed-size header
 */
class PrafFile : public MustuxAudioFileFormat
        {
        public:
                PrafFile();
                ~PrafFile();

                int open(QString pFilename);
                int create(QString pFilename);
                int copy_to(QString pFilename);
                int rewind_all();
                int rewind_audio();
                void go_botton();
                int byte_seek(long long bytePos);
                int block_seek(long long blockPos);
                long long byte_tell();
                long long block_tell();
                int skip_header();
                int read_header();
                int write_header(int pChannels=2, int pRate=44100, int pBitDepth=16,
                                          long long pAudioSize = 0,
                                          QString pProjectLabel = 0,
                                          QString pGroupLabel = 0 ,
                                          QString pHardwareLabel = 0,
                                          QString pDescription = 0 );
                int append_samples_from_file(MustuxAudioFileFormat* sourceFile);
                int append_samples_from_buffer(char* buffer, int bufferSize);
                int read_fragment(char* frag, int size);
                int write_fragment(char* frag, int size);
                int fix_audio_size();
                int close_file();
                bool eof();
                int get_best_frag_size();

                static const int PRAF_HEADER_SIZE = HEADER_SIZE;

        private:
                static const int FRAG_SIZE = 4096;
                };



/**
 * The standard WAV format. Only the audio data is supported, and the headers size
 * simplified.
 */
class WavFile : public MustuxAudioFileFormat
        {
        public:
                WavFile();
                ~WavFile();

                int open(QString pFilename);
                int create(QString pFilename);
                int copy_to(QString pFilename);
                int rewind_all();
                int rewind_audio();
                void go_botton();
                int byte_seek(long long bytePos);
                int block_seek(long long blockPos);
                long long byte_tell();
                long long block_tell();
                int skip_header();
                int read_header();
                int write_header(int pChannels=2, int pRate=44100, int pBitDepth=16,
                                          long long pAudioSize = 0,
                                          QString pProjectLabel = 0,
                                          QString pGroupLabel = 0 ,
                                          QString pHardwareLabel = 0,
                                          QString pDescription = 0 );
                int append_samples_from_file(MustuxAudioFileFormat* sourceFile);
                int append_samples_from_buffer(char* buffer, int bufferSize);
                int read_fragment(char* frag, int size);
                int write_fragment(char* frag, int size);
                int fix_audio_size();
                int close_file();
                bool eof();
                int get_best_frag_size();


        private :
                static const int FRAG_SIZE = 4096;
    };


#ifdef OGG_VORBIS_SUPPORT
/**
 * The OGG VORBIS format. It is intended to be transparent, so client do not
 * worry about compression stuff. They access the data normally, using the
 * same interface.
 * NOTE : libmustux offers READ-ONLY support to ogg vorbis. Writing support would
 *        require ogg encoding capabilities, which is not intended for a while
 */
class OggFile : public MustuxAudioFileFormat
    {
    public:
                OggFile();
                ~OggFile();

                int open(QString pFilename);
                int create(QString pFilename);
                int copy_to(QString pFilename);
                int rewind_all();
                int rewind_audio();
                void go_botton();
                int byte_seek(long long bytePos);
                int block_seek(long long blockPos);
                long long byte_tell();
                long long block_tell();
                int skip_header();
                int read_header();
                int write_header(int pChannels=2, int pRate=44100, int pBitDepth=16,
                                          long long pAudioSize = 0,
                                          QString pProjectLabel = 0,
                                          QString pGroupLabel = 0 ,
                                          QString pHardwareLabel = 0,
                                          QString pDescription = 0  );
                int append_samples_from_file(MustuxAudioFileFormat* sourceFile);
                int append_samples_from_buffer(char* buffer, int bufferSize);
                int read_fragment(char* frag, int size);
                int write_fragment(char* frag, int size);
                int fix_audio_size();
                int close_file();
                bool eof();
                int get_best_frag_size();
                FILE* get_file(); // overrided


        private :
                int feed_ring(char* buf, int size);
                int fill_ring();
                OggVorbis_File vorbisFile;
                int currentSection;
                static const int OGG_READ_BUFFER_SIZE = 4096;
                char OGG_READ_BUFFER[OGG_READ_BUFFER_SIZE];
                int excessSize;
                char EXCESS_OGG_READ_BUFFER[OGG_READ_BUFFER_SIZE];
                char* ringBuffer;
                int ringBufferSize;
                char* rbS;
                char* rbE;
                char* ringBufferEnd;
                bool realEof;
                void show_ring_status();
    };
#endif


#endif
