/**************************************************************************************

        PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
        AUTHOR : See AUTHORS file for details

        This software is distributed under the terms of the GNU General Public License
        as specified in the COPYING file.

***************************************************************************************/
#include <math.h>
#include <string.h>
#include "MustuxAudioFileFormats.hh"
#include "MustuxDebugger.hh"

#define WAV_COMMON_HEADER_SIZE 44

MustuxAudioFileFormat::MustuxAudioFileFormat(int pHeaderSize)
        {
        PENTERCONS2;
        headerSize = pHeaderSize;
        fileSize = 0;
        audioSize = 0;
        totalBlocks = 0;
        rate=0;
        channels=0;
        bitDepth=0;
        blockSize=0;
        bytesPerSec=0;
        sampleSize=0;
        projectLabel="";
        groupLabel="";
        hardwareLabel="";
        description="";
        PEXITCONS2;
        }


MustuxAudioFileFormat::~MustuxAudioFileFormat()
        {
        PENTERDES2;
        PEXITDES2;
        }




QString MustuxAudioFileFormat::get_info()
        {
        double seconds;
        int  minutes, hours;
        seconds = 8. * audioSize / (channels * rate * bitDepth);
        seconds -= (minutes = ((int) seconds) / 60) * 60;
        minutes -= (hours = minutes / 60) * 60;
        QString sRate; sRate.setNum(rate);
        QString sBitDepth; sBitDepth.setNum(bitDepth);
        QString sSize; sSize.setNum((long)audioSize); // OOOPS ... downcasting to long may fail..
        QString sAudioSize; sAudioSize.setNum((long)audioSize); // OOOPS ... downcasting to long may fail..
        QString sTotalBlocks; sTotalBlocks.setNum((long)totalBlocks); // OOOPS ... downcasting to long may fail..
        QString sh; QString sm; QString ss;
        sh.setNum(hours); sm.setNum(minutes);ss.setNum((int)seconds);
        QString sTime = sh + "h," + sm + "m," + ss + "s";
        QString s = "filename=" + filename + " " +
                    "rate=" + sRate + " channels=" + (channels == 1 ? "MONO, " : "STEREO") +" bitDepht=" + sBitDepth + "\n" +
                    "audioSize=" + sAudioSize + " fileSize=" + sSize + " totalBlocks=" + sTotalBlocks + " totalTime=" + sTime  + "\n" +
                    "projectLabel=" +  projectLabel +" " +
                    "groupLabel=" + groupLabel + " " +
                    "hardwareLabel=" +hardwareLabel + "  " +
                    "descriptionLabel="+ description + " ";
        return s;
        }



int MustuxAudioFileFormat::open(QString pFilename)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::rewind_all()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int  MustuxAudioFileFormat::rewind_audio()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }

void MustuxAudioFileFormat::go_botton()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        }

int MustuxAudioFileFormat::skip_header()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::close_file()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::create(QString pFilename)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::copy_to(QString pFilename)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::read_header()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::write_header(int pChannels, int pRate, int pBitDepth,
                                          long long pAudioSize,
                                          QString pProjectLabel,
                                          QString pGroupLabel,
                                          QString pHardwareLabel,
                                          QString pDescription)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::append_samples_from_file(MustuxAudioFileFormat* sourceFile)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::append_samples_from_buffer(char* buffer, int bufferSize)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::block_seek(long long blockPos)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }

int MustuxAudioFileFormat::byte_seek(long long blockPos)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }

long long MustuxAudioFileFormat::block_tell()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }

long long MustuxAudioFileFormat::byte_tell()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::read_fragment(char* frag, int size)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::write_fragment(char* frag, int size)
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


int MustuxAudioFileFormat::fix_audio_size()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }


bool MustuxAudioFileFormat::eof()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return false;
        }

int MustuxAudioFileFormat::get_best_frag_size()
        {
        PENTER2;
        PERROR2("Oops. I am virtual ! I should not be here...");
        PEXIT2;
        return -1;
        }

FILE* MustuxAudioFileFormat::get_file()
        {
        return file;
        }


//############################################################
//                       PRAF
//############################################################


PrafFile::PrafFile()
        : MustuxAudioFileFormat(PRAF_HEADER_SIZE)
        {
        PENTERCONS2;
        PEXITCONS2;
        }


PrafFile::~PrafFile()
        {
        PENTERDES2;
        PEXITDES2;
        }

int PrafFile::open(QString pFilename)
        {
        PENTER2;
        filename=pFilename;
        file = fopen(filename,"r");
        if (!file)
                {
                PERROR2("Cannot open %s",filename.ascii());
                PEXIT2;
                return -1;
                }
        // get the filesize. Isnt there a better way to do that?
        fseek(file, 0, SEEK_END);
        fileSize = ftell(file);
        rewind(file);
        PEXIT2;
        return 1;
        }



int PrafFile::rewind_all()
        {
        rewind(file);
        }


int PrafFile::rewind_audio()
        {
        rewind_all();
        skip_header();
        }


void PrafFile::go_botton()
        {
        fseek(file, 0, SEEK_END);
        }


int PrafFile::skip_header()
        {
        fseek(file, headerSize, SEEK_SET);
        return 1; // improve-me
        }


int PrafFile::block_seek(long long blockPos)
        {
        long long bp = headerSize + blockPos * blockSize;
        return fseek(file, bp, SEEK_SET);
        }

int PrafFile::byte_seek(long long bytePos)
        {
        return fseek(file, bytePos, SEEK_SET);
        }

long long PrafFile::block_tell()
        {
        long long blockPos = (ftell(file) - headerSize) / blockSize;
        return blockPos;
        }

long long PrafFile::byte_tell()
        {
        return ftell(file);
        }

int PrafFile::close_file()
        {
        int r = fclose(file);
        return r;
        }


int PrafFile::create(QString pFilename)
        {
        PENTER2;
        filename=pFilename;
        file = fopen(filename,"w+");
        if (!file)
                {
                PEXIT2;
                return -1;
                }
        PEXIT2;
        return 1;
        }


int PrafFile::copy_to(QString filename)
        {
        PENTER2;
        PrafFile* newFile = new PrafFile();
        if (newFile->open(filename)<0)
                {
                PERROR2("Cannot copy PRAF file to %s", filename.ascii());
                PEXIT2;
                return -1;
                }
        newFile->write_header(channels, rate, bitDepth, audioSize,projectLabel,groupLabel,hardwareLabel,description);
        char buf[FRAG_SIZE];
        rewind_audio();
        int size;
        while (!feof(file))
            {
            size = read_fragment(buf, FRAG_SIZE);
            if (newFile->write_fragment(buf, size)<0)
                {
                //error
                }
            }
        newFile->close_file();
        delete newFile;
        PEXIT2;
        return 1;
        }


int PrafFile::read_header()
        {
        PENTER2;
        rewind_all();		//WAS: rewind_audio(); Ehm, we shouldn't do that, should we :-) By Remon
        fread(&(formatLabel), 4, 1 , file);
        if ((formatLabel[0]!='P') ||
                (formatLabel[1]!='R') ||
                (formatLabel[2]!='A') ||
                (formatLabel[3]!='F'))
                {
                PERROR2("This seems not to be a PRAF file.");
                PEXIT2;
                return -1;
                }
        fread(&(channels), 4, 1 , file);
        fread(&(rate), 4, 1 , file);
        fread(&(bitDepth) , 4, 1 , file);
        fread(&(audioSize) , 8, 1 , file);
        char cProjectLabel[HEADER_LABEL_SIZE+1];
        char cGroupLabel[HEADER_LABEL_SIZE+1];
        char cHardwareLabel[HEADER_LABEL_SIZE+1];
        char cDescription[HEADER_DESCRIPTION_SIZE+1];
        char cReserved[HEADER_RESERVED_SIZE+1];
        fread(&(cProjectLabel), HEADER_LABEL_SIZE, 1 , file);
        fread(&(cGroupLabel), HEADER_LABEL_SIZE, 1 , file);
        fread(&(cHardwareLabel), HEADER_LABEL_SIZE, 1 , file);
        fread(&(cDescription), HEADER_DESCRIPTION_SIZE, 1 , file);
        fread(&(cReserved), HEADER_RESERVED_SIZE,1, file);
        cProjectLabel[HEADER_LABEL_SIZE]=0;
        cGroupLabel[HEADER_LABEL_SIZE]=0;
        cHardwareLabel[HEADER_LABEL_SIZE]=0;
        cDescription[HEADER_DESCRIPTION_SIZE]=0;
        projectLabel=QString(cProjectLabel);
        groupLabel=QString(cGroupLabel);
        hardwareLabel=QString(cHardwareLabel);
        description=QString(cDescription);
        blockSize = channels * (bitDepth / 8);
        bytesPerSec = rate * blockSize;
        sampleSize = blockSize / channels;
        totalBlocks = audioSize / blockSize;
        PMESG2("%s",get_info().ascii());
        PEXIT2;
        return 1;
        }



int PrafFile::write_header(int pChannels, int pRate, int pBitDepth,
                                          long long pAudioSize,
                                          QString pProjectLabel,
                                          QString pGroupLabel,
                                          QString pHardwareLabel,
                                          QString pDescription)
        {
        PENTER2;
        formatLabel[0] = 'P';
        formatLabel[1] = 'R';
        formatLabel[2] = 'A';
        formatLabel[3] = 'F';
        projectLabel = pProjectLabel;
        groupLabel = pGroupLabel;
        hardwareLabel = pHardwareLabel;
        description = pDescription;
        char cProjectLabel[HEADER_LABEL_SIZE+1];
        char cGroupLabel[HEADER_LABEL_SIZE+1];
        char cHardwareLabel[HEADER_LABEL_SIZE+1];
        char cDescription[HEADER_DESCRIPTION_SIZE+1];
        char cReserved[HEADER_RESERVED_SIZE+1];
        for (int i=0; i<HEADER_LABEL_SIZE; i++) cProjectLabel[i] = 0;
        for (int i=0; i<HEADER_LABEL_SIZE; i++) cGroupLabel[i] = 0;
        for (int i=0; i<HEADER_LABEL_SIZE; i++) cHardwareLabel[i] = 0;
        for (int i=0; i<HEADER_DESCRIPTION_SIZE; i++) cDescription[i] = 0;
        for (int i=0; i<HEADER_RESERVED_SIZE; i++) cReserved[i] = 0;
        // TODO .. copy projectLabel and so on into cProjectLabel and so on...
        channels = pChannels;
        rate = pRate;
        bitDepth = pBitDepth;
        audioSize = pAudioSize;
        rewind_all();		//WAS: rewind_audio(); Ehm, we shouldn't do that, should we :-) By Remon
        fwrite(&(formatLabel), 4, 1 , file);
        fwrite(&(channels), 4, 1 , file);
        fwrite(&(rate), 4, 1 , file);
        fwrite(&(bitDepth) , 4, 1 , file);
        fwrite(&(audioSize) , 8, 1 , file);
        fwrite(&(cProjectLabel), HEADER_LABEL_SIZE, 1 , file);
        fwrite(&(cGroupLabel), HEADER_LABEL_SIZE, 1 , file);
        fwrite(&(cHardwareLabel), HEADER_LABEL_SIZE, 1 , file);
        fwrite(&(cDescription), HEADER_DESCRIPTION_SIZE, 1 , file);
        fwrite(&(cReserved), HEADER_RESERVED_SIZE, 1, file);
        blockSize = channels * (bitDepth / 8);
        bytesPerSec = rate * blockSize;
        sampleSize = blockSize / channels;
        totalBlocks = audioSize / blockSize;
	PEXIT2;
        return 1;
        }


int PrafFile::append_samples_from_file(MustuxAudioFileFormat* f)
        {
        PENTER2;
        char buf[FRAG_SIZE];
        int i=0;
        f->rewind_audio();
        while (!f->eof())
            {
            int size = f->read_fragment(buf, FRAG_SIZE);
            if (f->write_fragment(buf, size) < 0)
                {
                // error
                }
            i++;
            PMESG("read %d buffers", ++i );
            }
        PEXIT2;
        return 1;
        }


int PrafFile::append_samples_from_buffer(char* buffer, int bufferSize)
        {
        size_t size ;
        if (fseek(file, 0, SEEK_END)<0)
                {
                PERROR2("Cannot to go file botton to append samples");
                return -1;
                }
        size = fwrite(buffer, 1 , bufferSize, file);
        if ((int) size < bufferSize)
                {
                PERROR2("Could not append the whole buffer to file");
                return -1;
                }
        return 1;
        }


int PrafFile::fix_audio_size()
        {
	PMESG3("Fixing PRAF file size ...");
        fseek(file, 0, SEEK_END);
        fileSize = ftell(file);
        audioSize =  fileSize - headerSize;
        totalBlocks = audioSize / blockSize;
        rewind(file);
	fseek(file, 16, SEEK_SET); // check PRAF's format specification to understand this "16"
        fwrite(&audioSize, 8, 1, file); // // check PRAF's format specification to understand this "8"
	return 1;
        }


int PrafFile::read_fragment(char* frag, int size)
        {
        return fread(frag,1,size,file);
        }


int PrafFile::write_fragment(char* frag, int size)
        {
        return fwrite(frag,1,size,file);
        }


bool PrafFile::eof()
        {
        return ( feof(file) ? true : false );
        }


int PrafFile::get_best_frag_size()
        {
        return FRAG_SIZE;
        }



//############################################################
//                            WAV
//############################################################


WavFile::WavFile()
        : MustuxAudioFileFormat(WAV_COMMON_HEADER_SIZE) // common wav header size
        {
        PENTERCONS2;
        PEXITCONS2;
        }


WavFile::~WavFile()
        {
        PENTERDES2;
        PEXITDES2;
        }


int WavFile::open(QString pFilename)
        {
        PENTER2;
        filename=pFilename;
        file = fopen(filename,"r");
        if (!file)
                {
                PERROR2("Cannot open %s",filename.ascii());
                PEXIT2;
                return -1;
                }
        // get the filesize. Isnt there a better way to do that?
        fseek(file, 0, SEEK_END);
        fileSize = ftell(file);
        rewind(file);
        PEXIT2;
        return 1;
        }

int WavFile::create(QString pFilename)
        {
        audioSize=0;
        filename=pFilename;
        file = fopen(filename,"w+");
        if (!file)
                return -1;
        write_header(2,44100,16);
        return 1;
        }


int WavFile::read_header()
        {
        PENTER2;
        char  audioType[4];
        char  riffType[4];
        char  useLess[8];
        short compressionCode;
        long  wFileSize;
        long  wBytesPerSec;
        short wBlockSize;
        long  wRate;
        short wBitDepth;
        short wChannels;
        rewind_all();
        fread(&(audioType), 4, 1 , file);
        fread(&(wFileSize), sizeof(long), 1 , file);
        // no need to make filesize = wFileSize since filesize already got the real file size in the constructor.
        fread(&(riffType), 4, 1 , file);
        if (strncmp(audioType,"RIFF",4)!=0)
                {
                PERROR2("File seems not to be a WAV/RIFF file");
                PEXIT2;
                return -1;
                }
        fread(&(useLess),  8, 1 , file);
        fread(&(compressionCode), sizeof(short), 1, file);
        fread(&(wChannels), sizeof(short), 1 , file);
        fread(&(wRate), sizeof(long), 1 , file);
        fread(&(wBytesPerSec), sizeof(long), 1 ,file);
        fread(&(wBlockSize), sizeof(short), 1 , file);
        fread(&(wBitDepth), sizeof(short), 1 , file);
        rate = wRate;
        bitDepth = wBitDepth;
        channels = wChannels;
        audioSize = fileSize - headerSize;
        blockSize = channels * (bitDepth / 8);
        bytesPerSec = rate * blockSize;
        sampleSize = blockSize / channels;
        totalBlocks = audioSize / blockSize;
        PMESG2("%s",get_info().ascii());
        PEXIT2;
        return 1;
        }


int WavFile::write_header(int pChannels, int pRate, int pBitDepth,
                                          long long pAudioSize,
                                          QString pProjectLabel,
                                          QString pGroupLabel,
                                          QString pHardwareLabel,
                                          QString pDescription)
        {
        PENTER2;
        char audioType[4];
        char riffType[4];
        char useLess[8];
        short compressionCode;
        rate = pRate;
        bitDepth = pBitDepth;
        bytesPerSec = rate * channels * (bitDepth/8);
        blockSize = channels * (bitDepth/8);
        channels=pChannels;
        audioSize=0;
        rewind_all();		//WAS: rewind_audio(); Ehm, we shouldn't do that, should we :-) By Remon
        audioType[0] = 'R';
        audioType[1] = 'I';
        audioType[2] = 'F';
        audioType[3] = 'F';
        fwrite(&(audioType), 4, 1 , file);
        long wFileSize = audioSize + headerSize;
        fwrite(&(wFileSize), sizeof(long), 1 , file);
        riffType[0] = 'W';
        riffType[1] = 'A';
        riffType[2] = 'V';
        riffType[3] = 'E';
        fwrite(&(riffType), 4, 1 , file);
        useLess[0] = 'f';
        useLess[1] = 'm';
        useLess[2] = 't';
        useLess[3] = ' ';
        useLess[4] = 16;
        useLess[5] = 0;
        useLess[6] = 0;
        useLess[7] = 0;
        fwrite(&(useLess),  8, 1 , file);
        compressionCode=1;
        fwrite(&(compressionCode), sizeof(short), 1, file);
        short wChannels = (short) channels;
        fwrite(&(wChannels), sizeof(short), 1 , file);
        long wRate = (long) rate;
        fwrite(&(wRate), sizeof(long), 1 , file);
        long wBytesPerSec = (long) bytesPerSec;
        fwrite(&(wBytesPerSec), sizeof(long), 1 ,file);
        short wBlockSize = (short) blockSize;
        fwrite(&(wBlockSize), sizeof(short), 1 , file);
        short wBitDepth = (short) bitDepth;
        fwrite(&(wBitDepth), sizeof(short), 1 , file);
        useLess[0] = 'd';
        useLess[1] = 'a';
        useLess[2] = 't';
        useLess[3] = 'a';
	long dataSize = wFileSize - 36;
        fwrite(&(useLess),  4, 1 , file);
	fwrite(&(dataSize), 4, 1 , file);
        PEXIT2;
        return 1;
        }



int WavFile::copy_to(QString filename)
        {
        // TODO
        return -1;
        }


int WavFile::skip_header()
        {
        fseek(file, headerSize, SEEK_SET);
        return 1; // improve-me
        }


int WavFile::rewind_all()
        {
        rewind(file);
        }


int WavFile::rewind_audio()
        {
        rewind_all();
        skip_header();
        }

void WavFile::go_botton()
        {
        fseek(file,0,SEEK_END);
        }

int WavFile::append_samples_from_file(MustuxAudioFileFormat* f)
        {
        // TODO
        return -1;
        }

int WavFile::block_seek(long long blockPos)
        {
        long long bp = headerSize + blockPos * blockSize;
        return fseek(file, bp, SEEK_SET);
        }

int WavFile::byte_seek(long long bytePos)
        {
        return fseek(file, bytePos, SEEK_SET);
        }

long long WavFile::block_tell()
        {
        long long blockPos = (ftell(file) - headerSize) / blockSize;
        return blockPos;
        }

long long WavFile::byte_tell()
        {
        return ftell(file);
        }

int WavFile::append_samples_from_buffer(char* buffer, int bufferSize)
        {
        size_t size ;
        if (fseek(file, 0, SEEK_END)<0)
                {
                PERROR2("Cannot to go file botton to append samples");
                return -1;
                }
        size = fwrite(buffer, 1 , bufferSize, file);
        if ((int) size < bufferSize)
                {
                PERROR2("Could not append the whole buffer to file");
                return -1;
                }
        return 1;
        }


int WavFile::fix_audio_size()
        {
	PENTER3;
	PMESG3("Fixing WAV file size ...");
	fseek(file, 0, SEEK_END);
        fileSize = ftell(file);
	audioSize =  fileSize - headerSize;
        totalBlocks = audioSize / blockSize;
	// FOR LATER if (fileSize > 2Giga) PERROR (Wav supports file only up to 2 gbytes);
	long wFileSize = audioSize + headerSize - 8;
	rewind(file);
	fseek(file, 4, SEEK_SET);
	fwrite(&(wFileSize), sizeof(long), 1 , file);
	long dataSize = wFileSize - 36;
	fseek(file, 40, SEEK_SET);
	fwrite(&(dataSize), sizeof(long), 1 , file);
	PMESG("Fixed audioSize to %ld", wFileSize );
	PEXIT3;
	return 1;
        }


int WavFile::read_fragment(char* frag, int size)
        {
        return fread(frag,1,size,file);
        }


int WavFile::write_fragment(char* frag, int size)
        {
        return fread(frag,1,size,file);
        }


int WavFile::close_file()
        {
        PENTER2;
        int r = fclose(file);
        PEXIT2;
        return r;
        }


bool WavFile::eof()
        {
        return ( feof(file) ? true : false );
        }


int WavFile::get_best_frag_size()
        {
        return FRAG_SIZE;
        }


#ifdef OGG_VORBIS_SUPPORT
//############################################################
//                            OGG
//############################################################
OggFile::OggFile()
        : MustuxAudioFileFormat(0) // TODO what is the "header" size for an ogg ?
        {
        PENTERCONS;
        excessSize=0;
        realEof=false;
        ringBuffer = (char*) 0;
        PEXITCONS;
        }


OggFile::~OggFile()
        {
        PENTERDES;
        if (ringBuffer)
                delete ringBuffer;
        PEXITDES;
        }


int OggFile::create(QString pFilename)
        {
        PENTER2;
        PERROR2("Mustux OggVorbis support is READ-ONLY !");
        PEXIT2;
        return -1;
        }


int OggFile::open(QString pFilename)
        {
        PENTER2;
        filename=pFilename;
        file = fopen(filename,"r");
        if (!file)
                {
                PERROR2("Cannot open %s",pFilename.ascii());
                PEXIT2;
                return -1;
                }
        if (ov_open(file, &vorbisFile, NULL, 0) < 0)
                {
                PERROR2("Input does not appear to be an Ogg bitstream.");
                PEXIT2;
                return -1;
                }
        // check if it is seekable
        if (!ov_seekable(&vorbisFile))
                {
                PERROR2("Ogg file is not seekable!");
                PEXIT2;
                return -1;
                }
        PEXIT2;
        return 1;
        }


int OggFile::rewind_all()
        {
        // TODO
        }


int OggFile::rewind_audio()
        {
        rewind_all();
        skip_header();
        }

void OggFile::go_botton()
        {
        // TODO
        }

int OggFile::skip_header()
        {
        // TODO
        }


int OggFile::read_header()
        {
        PENTER2;
        vorbis_info *vi = ov_info ( &vorbisFile , -1 );
        rate = (int) vi->rate ;
        bitDepth = 16; // FIXME !!
        channels = vi->channels;
        totalBlocks = (long long) ov_pcm_total(&vorbisFile,-1);
        bytesPerSec = rate * channels * (bitDepth / 8);
        blockSize = channels * (bitDepth / 8);
        sampleSize = blockSize / channels;
        audioSize = totalBlocks * blockSize;
        if (ringBuffer)
                delete ringBuffer;
        ringBufferSize = rate*blockSize;
        ringBuffer = new char[rate*blockSize]; // 1 second of audio
        rbS = ringBuffer + (ringBufferSize/2);
        rbE = rbS; // when rbS==rbE, it means the ring is empty
        ringBufferEnd = ringBuffer + ringBufferSize;
        excessSize=0;
        realEof=false;
        fill_ring();
        PEXIT2;
        return 1;
        }


int OggFile::write_header(int pChannels, int pRate, int pBitDepth,
                          long long   newAudioSize,
                          QString newProjectLabel,
                          QString newGroupLabel,
                          QString newHardwareLabel,
                          QString newDescription)
        {
        PENTER2;
        PERROR2("Mustux OggVorbis support is READ-ONLY !");
        PEXIT2;
        return -1;
        }



int OggFile::copy_to(QString filename)
        {
        // TODO
        return -1;
        }


int OggFile::append_samples_from_file(MustuxAudioFileFormat* f)
        {
        PENTER2;
        PERROR2("Mustux OggVorbis support is READ-ONLY !");
        PEXIT2;
        return -1;
        }


int OggFile::append_samples_from_buffer(char* buffer, int bufferSize)
        {
        PENTER2;
        PERROR2("Mustux OggVorbis support is READ-ONLY !");
        PEXIT2;
        return -1;
        }


int OggFile::fix_audio_size()
        {
        PENTER2;
        PERROR2("Mustux OggVorbis support is READ-ONLY !");
        PEXIT2;
        return -1;
        }


int OggFile::block_seek(long long blockPos)
        {
        int r=ov_pcm_seek(&vorbisFile, (ogg_int64_t) blockPos);
        if (r==OV_ENOSEEK)
                {
                PERROR2("Bitstream is not seekable");
                }
        else if (r==OV_EINVAL)
                {
                PERROR2("Invalid argument value");
                }
        else if (r==OV_EREAD)
                {
                PERROR2("A read from media returned an error");
                }
        else if (r==OV_EFAULT)
                {
                PERROR2("Internal Ogg Stream Logic fault; indicates a bug or heap/stack corruption");
                }
        else if (r==OV_EBADLINK)
                {
                PERROR2("Invalid stream section supplied to libvorbisfile, or the requested link is corrupt");
                }
        return r;
        }

long long OggFile::block_tell()
        {
        return (long long) ov_pcm_tell(&vorbisFile);
        }


//---------

int OggFile::byte_seek(long long bytePos)
        {
        return block_seek ( bytePos / blockSize ) ;
        }


long long OggFile::byte_tell()
        {
        return (long long) ov_pcm_tell(&vorbisFile) * blockSize;
        }

//----------

int OggFile::read_fragment(char* buf, int bufSize)
        {
        PENTER3;
        //printf("Consuming %d bytes ... \n" ,bufSize);
        int bytesRead=0;
        if (rbS < rbE)
                {
                if  (rbS + bufSize <  rbE )
                        {
                        memcpy(buf,rbS,bufSize);
                        rbS += bufSize;
                        bytesRead = bufSize;
                        }
                else // overrun !
                        {
                        memcpy(buf,rbS,rbE - rbS);
                        bytesRead = rbE - rbS;
                        rbS = rbE;
                        //printf("\tOverrun !\n");
                        }
                }
        else
                {
                if ( rbS+bufSize < ringBufferEnd )
                        {
                        memcpy(buf,rbS,bufSize);
                        rbS += bufSize;
                        bytesRead = bufSize;
                        }
                else
                        {
                        bytesRead = ringBufferEnd - rbS;
                        memcpy(buf,rbS,bytesRead);
                        rbS = ringBuffer;
                        buf += bytesRead;
                        int bytesYetToRead = bufSize - bytesRead;
                        if  (rbS + bytesYetToRead <  rbE )
                                {
                                memcpy(buf,rbS,bytesYetToRead);
                                rbS += bytesYetToRead;
                                bytesRead += bytesYetToRead;
                                }
                        else // overrun !
                                {
                                memcpy(buf,rbS,rbE - rbS);
                                rbS = rbE;
                                bytesRead += (rbE - rbS);
                                //printf("\tOverrun !\n");
                                }
                        }
                }

        //show_ring_status();
        fill_ring();
        PEXIT3;
        return bytesRead;
        }




int OggFile::fill_ring()
        {
        int totalBytesFeeded = 0;
        if (excessSize>0)
                {
                //printf("Compensating excess : ");
                if (feed_ring(EXCESS_OGG_READ_BUFFER,excessSize)>0)
                        {
                        //printf("Still excess ??!? ( exceed %d bytes even after the compensation )\n",excessSize);
                        }
                }
        while (excessSize==0)
                {
                int bytesRead = (int) ov_read(&vorbisFile,OGG_READ_BUFFER,OGG_READ_BUFFER_SIZE,0,2,1,&currentSection);
                if (bytesRead == 0)
                        {
                        realEof = true;
                        break;
                        }
                if (bytesRead < 0)
                        {
                        PERROR2("Cannot read from file");
                        return -1;
                        }
                int bytesFeeded = feed_ring(OGG_READ_BUFFER,bytesRead);
                totalBytesFeeded += bytesFeeded;
                realEof = false;
                }
        return totalBytesFeeded;
        }



int OggFile::feed_ring(char* buf, int bufSize)
        {
        //printf("Feeding %d bytes into ring ...\n" ,bufSize);
        if (rbE >= rbS)
                {
                if (rbE + bufSize < ringBufferEnd)
                        {
                        //printf("case 1\n");
                        memcpy(rbE,buf,bufSize);
                        rbE += bufSize;
                        excessSize=0;
                        }
                else
                        {
                        int b1 = ringBufferEnd - rbE;
                        memcpy(rbE,buf,b1);
                        //printf("\t Looping back...");
                        rbE = ringBuffer;
                        char* pORB = buf + b1;
                        int b2 = bufSize - b1;
                        if  (rbE + b2 < rbS )
                                {
                                //printf("case 2\n");
                                memcpy(rbE,pORB,b2);
                                rbE += b2;
                                excessSize=0;
                                }
                        else // end of ring
                                {
                                int remainBytes = (rbS-rbE)/2; // fill only half of remaining bytes, so it never loop-reaches the rbS
                                excessSize = b2 - remainBytes;
                                //printf(" \t Can't feed %d bytes. Feeding only %d and saving %d for later...",b2,remainBytes,excessSize);
                                memcpy(rbE,pORB,remainBytes);
                                rbE += remainBytes;
                                memcpy(EXCESS_OGG_READ_BUFFER,pORB+remainBytes,excessSize);
                                }
                        }
                }
        else
                {
                if (rbE + bufSize < rbS)
                        {
                        memcpy(rbE,buf, bufSize);
                        rbE += bufSize;
                        excessSize=0;
                        }
                else
                        {
                        int remainBytes = (rbS - rbE)/2;// fill only half of remaining bytes, so it never loop-reaches the rbS
                        excessSize = bufSize - remainBytes;
                        //printf(" \t Can't feed %d bytes. Feeding only %d and saving %d for later...",bufSize,remainBytes,excessSize);
                        memcpy(rbE , buf , remainBytes);
                        rbE += remainBytes;
                        memcpy(EXCESS_OGG_READ_BUFFER , buf + remainBytes , excessSize);
                        }
                }
        //show_ring_status();
        return excessSize;
        }


void OggFile::show_ring_status()
        {
        int avSize = (rbS<rbE) ? ringBufferSize - (rbE-rbS) : rbS-rbE;
        printf("Ring \t[ %d Free |  %d Used ] ",avSize,ringBufferSize-avSize);
        if (rbS<rbE)
                printf("\t  (.....)  \n");
        else if (rbS>rbE)
                printf("\t....) (....\n");
        else
                printf("\t.....O.....\n");
        }


int OggFile::write_fragment(char* frag, int size)
        {
        PENTER2;
        PERROR2("Mustux OggVorbis support is READ-ONLY !");
        PEXIT2;
        return -1;
        }

int OggFile::close_file()
        {
        PENTER2;
        int r = ((ov_clear(&vorbisFile)==0)?1:-1);
        PEXIT2;
        return r;
        }

bool OggFile::eof()
        {
        return realEof;
        }

int OggFile::get_best_frag_size()
        {
        // IMPROVE_ME
        return OGG_READ_BUFFER_SIZE;
        }

FILE* OggFile::get_file()
        {
        // clients cannot access the phisical ogg file, since it is
        // makes no sense to access encripted audio data directly
        PERROR2("Trying to access the internal encripted ogg file.");
        return (FILE*) 0;
        }


#endif

//eof



