/**************************************************************************************

	MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/

#ifndef MUSTUXAUDIODEVICEMAPPER_HH
#define MUSTUXAUDIODEVICEMAPPER_HH

#define ALSA_PCM_NEW_HW_PARAMS_API
#include <alsa/asoundlib.h>

class Bus;

class MustuxAudioBusTransferResult
	{
	public :
		int error;
		QString message;
		int bytesTransferred;
		int xrun; // 0=no xrun ; -1=underrun; 1=overrun
		void clean()
			{
			error=0;
			xrun=0;
			bytesTransferred=0;
			message="";
			}

	};

namespace MustuxAudioDeviceMapper
	{
	static const int MAX_CARDS = 10;
	static const int MAX_BUSES = 20;
	static const int MONO = 1; // MONO mode
	static const int STEREO = 2; // STEREO mode
	static const int ONLY_LEFT_CHANNEL = 3; // STEREO mode (but uses only LEFT)
	static const int ONLY_RIGHT_CHANNEL = 4; // STEREO mode (but uses only RIGHT)

	static const int PLAYBACK = 0;
	static const int CAPTURE = 1;
	static const int ERROR_INVALID_CARD = -10;
	static const int ERROR_INVALID_BUS = -11;
	static const int ERROR_INVALID_RATE_CHANNELS_OR_BITDEPTH = -12;
	static const int ERROR_BUSY_BUS = -13;
	static const int ERROR_CANNOT_INIT_MAPPER = -14;
	static const int ERROR_CANNOT_OPEN_PLAYBACK_BUS = -15;
	static const int ERROR_CANNOT_OPEN_CAPTURE_BUS = -16;
	static const int ERROR_CANNOT_OPEN_BUS = -17;
	static const int ERROR_MONO_BUS = -18;
	static const int ERROR_BUS_DOES_NOT_SUPPORT_THIS_BIT_DEPTH   = -21;
	static const int ERROR_BUS_DOES_NOT_SUPPORT_THIS_CHANNELS    = -22;
	static const int ERROR_BUS_DOES_NOT_SUPPORT_THIS_SAMPLE_RATE = -23;
	static const int ERROR_8_BITS_BANNED_IN_MUSTUX_APPLICATIONS  = -25;
	static const int ERROR_CANNOT_DEFINE_FORMAT = -26;
	static const int ERROR_CANNOT_SET_PARAMS_ANY = -28;
	static const int ERROR_UNDERRUN = -29;
	static const int ERROR_OVERRUN = -30;

	/*! \brief Detect all cards, all playback and capture buses,
	*         map them into a tree, find their ALSA descriptors
	*         and check if them are available.
	*         Each bus will be assigned with a descriptor and a index.
	*         You can access the buses only by the index. If you have
	*         3 buses in and 2 out you will have buses in 0, 1,and 2 and
	*         buses out 0 and 1.
	* \return 1 if initialization successful
	*/
	int init();

	/* \brief Probe all busses to get their valid mode(s). Valid mode(s)
	*	can change when a user changes the alsa-descriptor. Only
	*	call this when the alsa-descriptor has changed!
	* \return 1 if succesfull, -1 when a bus(ses) returns "invalid mode"
	*/
	int probe_busses_valid_modes();

	/** \brief List all detected buses to stdout. Usually used
	*	for debug purposes.
	*/
	void list_buses();


	void set_plughw_use(bool flag);
	bool get_plughw_use();

	int get_total_cards();
	int get_total_buses_out();
	int get_total_buses_in();

	QString get_card_name(int cardId);
	QString get_card_full_name(int cardId);
	QString get_bus_in_name(int busId);
	QString get_bus_in_full_name(int busId);
	QString get_bus_out_name(int busId);
	QString get_bus_out_full_name(int busId);
	QString get_bus_out_descriptor(int busId);
	QString get_bus_in_descriptor(int busId);

	int get_bus_out_index_for_descriptor(QString desc);
	int get_bus_in_index_for_descriptor(QString desc);

	bool validate_bus_out(int busId, int pRate, int pChannels, int pBitDepht);
	bool validate_bus_in(int busId, int pRate, int pChannels, int pBitDepht);

	bool is_stereo_bus_out(int busId);
	bool is_stereo_bus_in(int busId);
	bool is_bus_allocated_for_playback(int busId);
	bool is_bus_allocated_for_capture(int busId);
	bool is_any_bus_allocated();

	int open_bus_in (int busId, int pRate, int pBitDepht, int whichChannels, bool block=true, float suggestedFps=20.0);
	int open_bus_out(int busId, int pRate, int pBitDepht, int whichChannels, bool block=true, float suggestedFps=20.0);

	char* get_bus_out_transfer_buffer(int busId, int whichChannels);
	char* get_bus_in_transfer_buffer(int busId, int whichChannels);

	int get_bus_out_transfer_size(int busId);
	int get_bus_in_transfer_size(int busId);

	float get_bus_in_fps(int busId);
	float get_bus_out_fps(int busId);

	/** Return an array of floats from -1.0 to 0.0 correspondind
	*  to the audio levels present in current transfer bus.
	*  The size of array is the number of channels for the given bus.
	*  The client is responsable for converting it to dB levels.
	*/
	float* get_bus_in_levels(int busId);
	float* get_bus_out_levels(int busId);

	/** Transfer the bus buffer to the hardware
	*  Normally, this function will transfer the whole bus.
	*  But in case you need to force to transfer a specific
	*  number of bytes, you can pass it throught bytesToTrasnfer
	*  parameter. Normally, you can just ignore it.
	*
	*  returns the number of bytes actually transferred.
	*/
	MustuxAudioBusTransferResult* bus_out_transfer(int busId, int bytesToTransfer=-1);
	MustuxAudioBusTransferResult* bus_in_transfer(int busId);

	int clean_bus_out(int busId);
	int clean_bus_in(int busId);

	int close_bus_in (int busId);
	int close_bus_out(int busId);

	int close_all_buses_out();
	int close_all_buses_in();
	int close_all_buses();

	QString get_error_mesg(int err);

	int find_bus_out(int rate, int bitDepth, int channels);
	int find_bus_in(int rate, int bitDepth, int channels);

	snd_pcm_format_t MustuxAudioDeviceMapper::get_format(int bitDepht);
	};

#include "MustuxAudioBus.hh"

#endif


