/**************************************************************************************

	MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/

#include <stdio.h>
#include <qstring.h>
#include <qfile.h>
#include <qtextstream.h>
#include <sched.h>
#include "MustuxDebugger.hh"
#include "MustuxAudioDeviceMapper.hh"


namespace MustuxAudioDeviceMapper
	{
	QString cardName[MAX_CARDS];
	QString cardShortName[MAX_CARDS];
	Bus* busIn[MustuxAudioDeviceMapper::MAX_BUSES];
	Bus* busOut[MustuxAudioDeviceMapper::MAX_BUSES];
	int totalCards;
	int totalBusesIn;
	int totalBusesOut;
	bool initialized=false;
	bool tryPlugHw=false;
	}


int MustuxAudioDeviceMapper::init()
	{
	PENTER;
	if (initialized)
		{
		PEXIT;
		return 1;
		}
	// Detect the number of cards;
	totalCards = 0;
	QFile fCards("/proc/asound/cards");
	if ( fCards.open(IO_ReadOnly) )
		{
		// Create the list of Card objects
		QTextStream t( &fCards);
		QString line;
		int cardIndex=0;
		while ( !t.eof() )
			{
			line = t.readLine();
			int x = line.find("]:");
			if (x>=0)
				{
				QString s2 = line.mid(x+3).simplifyWhiteSpace();
				PMESG("Found ALSA supported Card : %s",(const char*) s2.ascii());
				cardName[cardIndex] =s2;
				cardShortName[cardIndex] = s2.left(s2.find("-")-1);
				cardIndex++;
				}
			}
		totalCards = cardIndex;
		fCards.close();
		}
	else
		{
		PERROR("Cannot open /proc/asound/cards for reading ( maybe busy or no read permission ? )");
		PEXIT;
		return MustuxAudioDeviceMapper::ERROR_CANNOT_INIT_MAPPER;
		}
	// detect all buses
	totalBusesIn=0;
	totalBusesOut=0;
	int busInCountForThisCard=0;
	int busOutCountForThisCard=0;
	int previousBusInCi=-1;
	int previousBusOutCi=-1;
	QFile fDevices("/proc/asound/devices");
	if ( fDevices.open(IO_ReadOnly) )
		{
		QTextStream t( &fDevices);
		QString line;
		while ( !t.eof() )
			{
			line = t.readLine();
			int x = line.find("]: digital audio");
			if (x>=0)
				{
				int px = line.find("[");
				int px2 =line.find("-");
				int px3 = line.find("]");
				QString sCardNumber=line.mid(px+1,px2-px-1);
				int ci=sCardNumber.toInt();
				QString sBusNumber=line.mid(px2+2,px3-px2-2);
				// THE NEXT LINE WILL BE CORRECT IF THE ANSWER FOR THIS QUESTION IS TRUE :
				// "For a descriptor hw:X,Y, will there always be a directory /proc/asound/cardX/pcmYc or /proc/asound/cardX/pcmYp ?"
				QString desc ="hw:"+sCardNumber+","+sBusNumber; // the "plug" is added dinamically depending on MustuxAudioDeviceMapper::get_plughw_use(). Dont add it here.
				if (line.find("capture")>0)
					{
					if (ci!=previousBusInCi)
						{
						busInCountForThisCard=0;
						previousBusInCi=ci;
						}
					Bus* b=new Bus(ci, busInCountForThisCard++, MustuxAudioDeviceMapper::CAPTURE, desc);
					if (totalBusesIn>MustuxAudioDeviceMapper::MAX_BUSES)
						return -1;
					busIn[totalBusesIn++]=b;
					}
				else
					{
					if (ci!=previousBusOutCi)
						{
						busOutCountForThisCard=0;
						previousBusOutCi=ci;
						}
					Bus* b=new Bus(ci, busOutCountForThisCard++, MustuxAudioDeviceMapper::PLAYBACK, desc);
					if (totalBusesOut>MustuxAudioDeviceMapper::MAX_BUSES)
						return -1;
					busOut[totalBusesOut++]=b;
					}
				}
			}
		fDevices.close();
		}
	else
		{
		PERROR("Cannot open /proc/asound/devices for reading ( do u have read permission ? )");
		PEXIT;
		return MustuxAudioDeviceMapper::ERROR_CANNOT_INIT_MAPPER;
		}
	if (totalCards<0)
		PWARN("No ALSA Supported cards found :-(");
	struct sched_param param;
	param.sched_priority = 70;
	if( sched_setscheduler( 0, SCHED_RR, &param ) == 0 )
		{
		PMESG("Using Real Time Scheduling policy with priority %d\n", param.sched_priority);
		}
	else
		{
		PWARN("Cannot set Real Time scheduling. Please, retry as root, or with setuid (be aware of the security risks involved).");
		}
	initialized=true;
	PEXIT;
	return 1;
	}


int MustuxAudioDeviceMapper::probe_busses_valid_modes()
	{
	PENTER2;
	if (totalBusesIn==0)
		{
		PMESG2("     No CAPTURE buses found :-(");
		}
	else
		{
		PMESG2("Probing CAPTURE Buses ------------------------------------------------");
		for (int j=0; j<totalBusesIn; j++)
			{
			Bus* b=busIn[j];
			b->probe_valid_modes();
			}
		}
	if (totalBusesOut==0)
		{
		PMESG2("     No PLAYBACK buses found :-(");
		}
	else
		{
		PMESG2("Probing PLAYBACK Buses ------------------------------------------------");
		for (int j=0; j<totalBusesOut; j++)
			{
			Bus* b= busOut[j];
			b->probe_valid_modes();
			}
		}
	PEXIT2;
	}


void MustuxAudioDeviceMapper::list_buses()
	{
	printf("\n");
	printf("PLAYBACK BUSES :\n");
	printf("Mustux Bus ID\tDescriptor\tBus Name\n");
	printf("======================================================================\n");
	for (int b=0; b<get_total_buses_out(); b++)
		printf("  %d \t\t%s \t\t%s\n",b,(const char*) get_bus_out_descriptor(b).ascii(),(const char*) MustuxAudioDeviceMapper::get_bus_out_full_name(b).ascii());
	printf("\n");
	printf("CAPTURE BUSES :\n");
	printf("Mustux Bus ID\tDescriptor\tBus Name\n");
	printf("======================================================================\n");
	for (int b=0; b<get_total_buses_in(); b++)
		printf("  %d \t\t%s \t\t%s\n",b,(const char*) get_bus_in_descriptor(b).ascii(),(const char*) MustuxAudioDeviceMapper::get_bus_in_full_name(b).ascii());
	printf("\n");
	}

int MustuxAudioDeviceMapper::open_bus_in(int busId, int pRate, int pBitDepht, int whichChannels, bool block, float suggestedFps)
	{
	PENTER2;
	if ((busId<0) || (busId>MAX_BUSES)) return -1;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		PEXIT2;
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	int r = bus->open(pRate, pBitDepht, whichChannels, block, suggestedFps);
	PEXIT2;
	return r;
	}




int MustuxAudioDeviceMapper::open_bus_out(int busId, int pRate, int pBitDepht, int whichChannels, bool block, float suggestedFps)
	{
	PENTER2;
	if ((busId<0) || (busId>MAX_BUSES)) return -1;
	Bus* bus=busOut[busId];
	if (!bus)
		{
		PEXIT2;
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	int r = bus->open(pRate, pBitDepht, whichChannels, block, suggestedFps);
	PEXIT2;
	return r;
	}




MustuxAudioBusTransferResult* MustuxAudioDeviceMapper::bus_out_transfer(int busId, int bytesToTransfer)
	{
	if ((busId<0) || (busId>MAX_BUSES))
		return (MustuxAudioBusTransferResult*) 0;
	Bus* bus=busOut[busId];
	if (!bus)
		return (MustuxAudioBusTransferResult*) 0;
	return bus->transfer(bytesToTransfer);
	}



MustuxAudioBusTransferResult* MustuxAudioDeviceMapper::bus_in_transfer(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES))
		return (MustuxAudioBusTransferResult*) 0;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		return (MustuxAudioBusTransferResult*) 0;
		}
	return bus->transfer();
	}



int MustuxAudioDeviceMapper::clean_bus_out(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return -1;
	Bus* bus=busOut[busId];
	if (!bus)
		{
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	bus->clean_buffer();
	return 1;
	}



int MustuxAudioDeviceMapper::clean_bus_in(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return -1;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	bus->clean_buffer();
	return 1;
	}



int MustuxAudioDeviceMapper::close_bus_in(int busId)
	{
	PENTER;
	if ((busId<0) || (busId>MAX_BUSES))
		{
		PERROR("Invalid Bus %d",busId);
		PEXIT;
		return -1;
		}
	Bus* bus=busIn[busId];
	if (!bus)
		{
		PEXIT;
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	int r = bus->close();
	PEXIT;
	return r;
	}



int MustuxAudioDeviceMapper::close_bus_out(int busId)
	{
	PENTER;
	if ((busId<0) || (busId>MAX_BUSES))
		{
		PERROR("Invalid Bus %d",busId);
		PEXIT;
		return -1;
		}
	Bus* bus=busOut[busId];
	if (!bus)
		{
		PEXIT;
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	int r = bus->close();
	PEXIT;
	return r;
	}



bool MustuxAudioDeviceMapper::is_bus_allocated_for_playback(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return false;
	Bus* bus=busOut[busId];
	if (!bus)
		{
		return false;
		}
	int b = bus->allocated_for_playback();
	return b;
	}



bool MustuxAudioDeviceMapper::is_bus_allocated_for_capture(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return false;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		return false;
		}
	int b = bus->allocated_for_capture();
	return b;
	}



bool MustuxAudioDeviceMapper::is_any_bus_allocated()
	{
	bool any=false;
	for (int i=0; i<totalBusesOut; i++)
		{
		if (busOut[i]->allocated_for_playback())
			any=true;
		}
	for (int i=0; i<totalBusesIn; i++)
		{
		if (busIn[i]->allocated_for_capture())
			any=true;
		}
	return any;
	}



int MustuxAudioDeviceMapper::close_all_buses()
	{
	PENTER;
	close_all_buses_out();
	close_all_buses_in();
	PEXIT;
	return 1;
	}



int MustuxAudioDeviceMapper::close_all_buses_out()
	{
	PENTER;
	for (int i=0; i<totalBusesOut; i++)
		busOut[i]->close();
	PEXIT;
	return 1;
	}



int MustuxAudioDeviceMapper::close_all_buses_in()
	{
	PENTER;
	for (int i=0; i<totalBusesIn; i++)
		busIn[i]->close();
	PEXIT;
	return 1;
	}



int MustuxAudioDeviceMapper::get_bus_out_transfer_size(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return -1;
	Bus* bus=busOut[busId];
	if (!bus)
		{
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	int r = bus->get_transfer_size();
	return r;
	}



int MustuxAudioDeviceMapper::get_bus_in_transfer_size(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return -1;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	int r = bus->get_transfer_size();
	return r;
	}



char* MustuxAudioDeviceMapper::get_bus_out_transfer_buffer(int busId, int whichChannels)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return (char*) 0;
	Bus* bus=busOut[busId];
	if (!bus)
		{
		return (char*) 0;
		}
	char* buf = bus->get_transfer_buffer(whichChannels);
	return buf;
	}



char* MustuxAudioDeviceMapper::get_bus_in_transfer_buffer(int busId, int whichChannels)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return (char*) 0;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		return (char*) 0;
		}
	char* buf = bus->get_transfer_buffer(whichChannels);
	return buf;
	}



QString MustuxAudioDeviceMapper::get_bus_in_name(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return "INVALID BUS";
	Bus* bus=busIn[busId];
	if (!bus)
		return "INVALID BUS";
	return bus->get_short_name();
	}



QString MustuxAudioDeviceMapper::get_bus_in_full_name(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return "INVALID BUS";
	Bus* bus=busIn[busId];
	if (!bus)
		return "INVALID BUS";
	return bus->get_full_name();
	}



QString MustuxAudioDeviceMapper::get_bus_out_name(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return "INVALID BUS";
	Bus* bus=busOut[busId];
	if (!bus)
		return "INVALID BUS";
	return bus->get_short_name();
	}



QString MustuxAudioDeviceMapper::get_bus_out_full_name(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return "INVALID BUS";
	Bus* bus=busOut[busId];
	if (!bus)
		return "INVALID BUS";
	return bus->get_full_name();
	}



QString MustuxAudioDeviceMapper::get_bus_out_descriptor(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return "INVALID BUS";
	Bus* bus=busOut[busId];
	if (!bus)
		return "INVALID BUS";
	return bus->get_descriptor();
	}



QString MustuxAudioDeviceMapper::get_bus_in_descriptor(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return "INVALID BUS";
	Bus* bus=busIn[busId];
	if (!bus)
		return "INVALID BUS";
	return bus->get_descriptor();
	}



int MustuxAudioDeviceMapper::get_total_cards()
	{
	return totalCards;
	}



QString MustuxAudioDeviceMapper::get_card_name(int cardId)
	{
	if ((cardId > totalCards) || (cardId<0))
		return "INVALID";
	return cardShortName[cardId];
	}



QString MustuxAudioDeviceMapper::get_card_full_name(int cardId)
	{
	if ((cardId > totalCards) || (cardId<0))
		return "INVALID";
	return cardName[cardId];
	}



int MustuxAudioDeviceMapper::get_total_buses_out()
	{
	return totalBusesOut;
	}



int MustuxAudioDeviceMapper::get_total_buses_in()
	{
	return totalBusesIn;
	}



bool MustuxAudioDeviceMapper::validate_bus_out(int busId, int pRate, int pChannels, int pBitDepht)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return false;
	Bus* bus=busOut[busId];
	if (!bus)
		{
		return false;
		}
	int b = bus->valid_mode(pRate, pBitDepht, pChannels);
	return b;
	}



bool MustuxAudioDeviceMapper::validate_bus_in(int busId, int pRate, int pChannels, int pBitDepht)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return false;
	Bus* bus=busIn[busId];
	if (!bus)
		{
		return false;
		}
	int b = bus->valid_mode(pRate, pBitDepht, pChannels);
	return b;
	}



bool MustuxAudioDeviceMapper::is_stereo_bus_out(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return false;
	Bus* bus=busOut[busId];
	if (!bus)
		return false;
	return bus->is_stereo();
	}



bool MustuxAudioDeviceMapper::is_stereo_bus_in(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return false;
	Bus* bus=busIn[busId];
	if (!bus)
		return false;
	return bus->is_stereo();
	}



int MustuxAudioDeviceMapper::get_bus_out_index_for_descriptor(QString desc)
	{
	for (int i=0; i<totalBusesOut; i++)
		if (busOut[i]->get_descriptor()==desc)
			return i;
	return -1;
	}



int MustuxAudioDeviceMapper::get_bus_in_index_for_descriptor(QString desc)
	{
	for (int i=0; i<totalBusesIn; i++)
		if (busIn[i]->get_descriptor()==desc)
			return i;
	return -1;
	}



float MustuxAudioDeviceMapper::get_bus_in_fps(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return -1.0;
	Bus* bus=busIn[busId];
	if (!bus)
		return -1.0;
	return bus->get_fps();
	}



float MustuxAudioDeviceMapper::get_bus_out_fps(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return -1.0;
	Bus* bus=busOut[busId];
	if (!bus)
		return -1.0;
	return bus->get_fps();
	}



snd_pcm_format_t MustuxAudioDeviceMapper::get_format(int bitDepht)
	{
	switch (bitDepht)
		{
		case 16 : return SND_PCM_FORMAT_S16_LE;
		case 20 : return SND_PCM_FORMAT_S20_3LE;
		case 24 : return SND_PCM_FORMAT_S24_LE;
		case 32 : return SND_PCM_FORMAT_S32_LE;
		case 64 : return SND_PCM_FORMAT_FLOAT64_LE;
		}
	return SND_PCM_FORMAT_UNKNOWN;
	}



bool MustuxAudioDeviceMapper::get_plughw_use()
	{
	return tryPlugHw;
	}



void MustuxAudioDeviceMapper::set_plughw_use(bool flag)
	{
	tryPlugHw = flag;
	}



float* MustuxAudioDeviceMapper::get_bus_out_levels(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return (float*) 0;
	Bus* bus=busOut[busId];
	if (!bus)
		return (float*) 0;
	return bus->get_levels();
	}

float* MustuxAudioDeviceMapper::get_bus_in_levels(int busId)
	{
	if ((busId<0) || (busId>MAX_BUSES)) return (float*) 0;
	Bus* bus=busIn[busId];
	if (!bus)
		return (float*) 0;
	return bus->get_levels();
	}

int MustuxAudioDeviceMapper::find_bus_out(int rate, int bitDepth, int channels)
	{
	//TODO
	return -1;
	}

int MustuxAudioDeviceMapper::find_bus_in(int rate, int bitDepth, int channels)
	{
	//TODO	
	return -1;
	}


QString MustuxAudioDeviceMapper::get_error_mesg(int err)
	{
	switch (err)
		{
		case ERROR_INVALID_CARD :
			return "ERROR_INVALID_CARD";
		case ERROR_INVALID_BUS :
			return "ERROR_INVALID_BUS";
		case ERROR_INVALID_RATE_CHANNELS_OR_BITDEPTH :
			return "ERROR_INVALID_RATE_CHANNELS_OR_BITDEPTH";
		case ERROR_BUSY_BUS :
			return "ERROR_BUSY_BUS";
		case ERROR_CANNOT_INIT_MAPPER :
			return "ERROR_CANNOT_INIT_MAPPER";
		case ERROR_CANNOT_OPEN_PLAYBACK_BUS :
			return "ERROR_CANNOT_OPEN_PLAYBACK_BUS";
		case ERROR_CANNOT_OPEN_CAPTURE_BUS :
			return "ERROR_CANNOT_OPEN_CAPTURE_BUS";
		case ERROR_CANNOT_OPEN_BUS :
			return "ERROR_CANNOT_OPEN_BUS";
		case ERROR_MONO_BUS :
			return "ERROR_MONO_BUS";
		case ERROR_BUS_DOES_NOT_SUPPORT_THIS_BIT_DEPTH :
			return "ERROR_BUS_DOES_NOT_SUPPORT_THIS_BIT_DEPTH";
		case ERROR_BUS_DOES_NOT_SUPPORT_THIS_CHANNELS :
			return "ERROR_BUS_DOES_NOT_SUPPORT_THIS_CHANNELS";
		case ERROR_BUS_DOES_NOT_SUPPORT_THIS_SAMPLE_RATE :
			return "ERROR_BUS_DOES_NOT_SUPPORT_THIS_SAMPLE_RATE";
		case ERROR_8_BITS_BANNED_IN_MUSTUX_APPLICATIONS :
			return "ERROR_8_BITS_BANNED_IN_MUSTUX_APPLICATIONS";
		case ERROR_CANNOT_DEFINE_FORMAT :
			return "ERROR_CANNOT_DEFINE_FORMAT";
		case ERROR_CANNOT_SET_PARAMS_ANY :
			return "ERROR_CANNOT_SET_PARAMS_ANY";
		default:
			return "UNKNOWN ERROR";
		}
	}
//eof



/* ALSA FORMATS

*** 16 bits

SND_PCM_FORMAT_S16 Signed 16 bit CPU endian
SND_PCM_FORMAT_U16 Unsigned 16 bit CPU endian
SND_PCM_FORMAT_S16_LE Signed 16 bit Little Endian
SND_PCM_FORMAT_S16_BE Signed 16 bit Big Endian
SND_PCM_FORMAT_U16_LE Unsigned 16 bit Little Endian
SND_PCM_FORMAT_U16_BE Unsigned 16 bit Big Endian

*** 20 bits

SND_PCM_FORMAT_S20_3LE Signed 20bit Little Endian in 3bytes format
SND_PCM_FORMAT_S20_3BE Signed 20bit Big Endian in 3bytes format
SND_PCM_FORMAT_U20_3LE Unsigned 20bit Little Endian in 3bytes format
SND_PCM_FORMAT_U20_3BE Unsigned 20bit Big Endian in 3bytes format

*** 24 bits

SND_PCM_FORMAT_S24 Signed 24 bit CPU endian
SND_PCM_FORMAT_U24 Unsigned 24 bit CPU endian
SND_PCM_FORMAT_S24_LE Signed 24 bit Little Endian
SND_PCM_FORMAT_S24_BE Signed 24 bit Big Endian
SND_PCM_FORMAT_U24_LE Unsigned 24 bit Little Endian
SND_PCM_FORMAT_U24_BE Unsigned 24 bit Big Endian
SND_PCM_FORMAT_S24_3LE Signed 24bit Little Endian in 3bytes format
SND_PCM_FORMAT_S24_3BE Signed 24bit Big Endian in 3bytes format
SND_PCM_FORMAT_U24_3LE Unsigned 24bit Little Endian in 3bytes format
SND_PCM_FORMAT_U24_3BE Unsigned 24bit Big Endian in 3bytes format

*** 32 bits

SND_PCM_FORMAT_S32_LE Signed 32 bit Little Endian
SND_PCM_FORMAT_S32_BE Signed 32 bit Big Endian
SND_PCM_FORMAT_U32_LE Unsigned 32 bit Little Endian
SND_PCM_FORMAT_U32_BE Unsigned 32 bit Big Endian
SND_PCM_FORMAT_FLOAT_LE Float 32 bit Little Endian, Range -1.0 to 1.0
SND_PCM_FORMAT_FLOAT_BE Float 32 bit Big Endian, Range -1.0 to 1.0
SND_PCM_FORMAT_S32 Signed 32 bit CPU endian
SND_PCM_FORMAT_U32 Unsigned 32 bit CPU endian
SND_PCM_FORMAT_FLOAT Float 32 bit CPU endian

*** 64 bits
SND_PCM_FORMAT_FLOAT64 Float 64 bit CPU endian
SND_PCM_FORMAT_FLOAT64_LE Float 64 bit Little Endian, Range -1.0 to 1.0
SND_PCM_FORMAT_FLOAT64_BE Float 64 bit Big Endian, Range -1.0 to 1.0

*/
