/**************************************************************************************

	MUSTUXLIB - THE COMMON LIBRARY FOR ALL MUSTUX APPLICATIONS
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/


#include <stdio.h>
#include <unistd.h>
#include <qstring.h>
#include <sys/time.h>
#include "MustuxAudioBus.hh"
#include "MustuxDebugger.hh"
#include "MustuxAudioDeviceMapper.hh"


Bus::Bus(int pParentCardId, int pId, int pStream, QString desc)
	{
	PENTERCONS4;
	parentCardId= pParentCardId;
	id = pId;
	stream=pStream;
	descriptor=desc;
	status=CLOSED;
	stereoBus=false;
	transferBuffer=(char*) 0;
	levels = (float*) 0;
	transferSize=-1;
	fragmentsPerSecond = 0;
	transferResult = new MustuxAudioBusTransferResult();

	busMode[0 ] = new BusMode(22050, MONO,   16);
	busMode[1 ] = new BusMode(32000, MONO,   16);
	busMode[2 ] = new BusMode(44100, MONO,   16);
	busMode[3 ] = new BusMode(44100, MONO,   20);
	busMode[4 ] = new BusMode(44100, MONO,   24);
	busMode[5 ] = new BusMode(44100, MONO,   32);
	busMode[6 ] = new BusMode(44100, MONO,   64);
	busMode[7 ] = new BusMode(48000, MONO,   16);
	busMode[8 ] = new BusMode(48000, MONO,   20);
	busMode[9 ] = new BusMode(48000, MONO,   24);
	busMode[10] = new BusMode(48000, MONO,   32);
	busMode[11] = new BusMode(48000, MONO,   64);
	busMode[12] = new BusMode(96000, MONO,   16);
	busMode[13] = new BusMode(96000, MONO,   20);
	busMode[14] = new BusMode(96000, MONO,   24);
	busMode[15] = new BusMode(96000, MONO,   32);
	busMode[16] = new BusMode(96000, MONO,   64);
	busMode[17] = new BusMode(22050, STEREO, 16);
	busMode[18] = new BusMode(32000, STEREO, 16);
	busMode[19] = new BusMode(44100, STEREO, 16);
	busMode[20] = new BusMode(44100, STEREO, 20);
	busMode[21] = new BusMode(44100, STEREO, 24);
	busMode[22] = new BusMode(44100, STEREO, 32);
	busMode[23] = new BusMode(44100, STEREO, 64);
	busMode[24] = new BusMode(48000, STEREO, 16);
	busMode[25] = new BusMode(48000, STEREO, 20);
	busMode[26] = new BusMode(48000, STEREO, 24);
	busMode[27] = new BusMode(48000, STEREO, 32);
	busMode[28] = new BusMode(48000, STEREO, 64);
	busMode[29] = new BusMode(96000, STEREO, 16);
	busMode[30] = new BusMode(96000, STEREO, 20);
	busMode[31] = new BusMode(96000, STEREO, 24);
	busMode[32] = new BusMode(96000, STEREO, 32);
	busMode[33] = new BusMode(96000, STEREO, 64);
	PMESG2("%s Bus %d [ card %d ] : \"%s\" created",
		stream==MustuxAudioDeviceMapper::CAPTURE?"CAPTURE ":"PLAYBACK",
		pId,parentCardId,
		(const char*) get_full_name().ascii());
	PEXITCONS4;
	}


Bus::~Bus()
	{
	delete transferResult;
	}


void Bus::probe_valid_modes()
	{
	PENTER3;
	stereoBus=false;
	PMESG2("Probing valid modes for bus %s ...",(get_real_descriptor()).ascii());
	for (int j=0; j<MAX_SCB_MODES; j++)
		{
		PMESG2_START("  Trying %s : ",(busMode[j]->get_name()).ascii());
		int r = supports(busMode[j]->rate,busMode[j]->bitDepht,busMode[j]->channels,stream);
		if (r<0)
			{
			QString em = MustuxAudioDeviceMapper::get_error_mesg(r);
			PMESG2_END(" FAILED : %s",em.ascii());
			busMode[j]->valid=false;
			}
		else
			{
			PMESG2_END(" OK");
			busMode[j]->valid=true;
			if (busMode[j]->channels==STEREO)
				stereoBus=true;
			}
		}
	PEXIT3;
	}



// THIS IS JUST NOT WORKING ! FIX ME...
int Bus::supports(int pRate, int pBitDepht, int pChannels, int pStream)
	{
	PENTER4;
	if (pBitDepht==8)
		{
		PEXIT4;
		return MustuxAudioDeviceMapper::ERROR_8_BITS_BANNED_IN_MUSTUX_APPLICATIONS;
		}
	format = MustuxAudioDeviceMapper::get_format(pBitDepht);
	if (format == SND_PCM_FORMAT_UNKNOWN)
		{
		PEXIT4;
		return MustuxAudioDeviceMapper::ERROR_CANNOT_DEFINE_FORMAT;
		}
	snd_pcm_hw_params_t* tempHwParams;
	if (pStream == MustuxAudioDeviceMapper::CAPTURE)
		{
		if (snd_pcm_open(&pcmHandle, ((const char*) get_real_descriptor().ascii()), SND_PCM_STREAM_CAPTURE, 0) < 0)
			{
			PERROR("Cannot open %s for probing MustuxAudioDeviceMapper::CAPTURE capabilities",((const char*) get_real_descriptor().ascii()));
			PEXIT4;
			return MustuxAudioDeviceMapper::ERROR_CANNOT_OPEN_BUS;
			}
		}
	else
		{
		if (snd_pcm_open(&pcmHandle, ((const char*) get_real_descriptor().ascii()), SND_PCM_STREAM_PLAYBACK, 0) < 0)
			{
			PERROR("Cannot open %s for probing PLAYBACK capabilities",((const char*) get_real_descriptor().ascii()));
			PEXIT4;
			return MustuxAudioDeviceMapper::ERROR_CANNOT_OPEN_BUS;
			}
		}
	snd_pcm_hw_params_alloca(&tempHwParams);
	if (snd_pcm_hw_params_any(pcmHandle, tempHwParams) < 0)
		{
		snd_pcm_close(pcmHandle);
		PEXIT4;
		return MustuxAudioDeviceMapper::ERROR_CANNOT_SET_PARAMS_ANY;
		}
	if (snd_pcm_hw_params_test_rate     (pcmHandle, tempHwParams, pRate, 0)!=0)
		{
		snd_pcm_close(pcmHandle);
		PEXIT4;
		return MustuxAudioDeviceMapper::ERROR_BUS_DOES_NOT_SUPPORT_THIS_SAMPLE_RATE;
		}
	if (snd_pcm_hw_params_test_format   (pcmHandle, tempHwParams, format)!=0)
		{
		snd_pcm_close(pcmHandle);
		PEXIT4;
		return MustuxAudioDeviceMapper::ERROR_BUS_DOES_NOT_SUPPORT_THIS_BIT_DEPTH;
		}
	if (snd_pcm_hw_params_test_channels (pcmHandle, tempHwParams, pChannels)!=0)
		{
		snd_pcm_close(pcmHandle);
		PEXIT4;
		return MustuxAudioDeviceMapper::ERROR_BUS_DOES_NOT_SUPPORT_THIS_CHANNELS;
		}
	snd_pcm_close(pcmHandle);
	PEXIT4;
	return 1;
	}




bool Bus::valid_mode(int pRate, int pBitDepht, int pChannels)
	{
	PENTER3;
	for (int m=0; m<MAX_SCB_MODES; m++)
		{
		int ra = busMode[m]->rate;
		int bd = busMode[m]->bitDepht;
		int ch = busMode[m]->channels;
		if ( (ra==pRate) &&
		(bd==pBitDepht) &&
		(ch==pChannels)
		)
			{
			PEXIT3;
			return busMode[m]->valid;
			}
		}
	PEXIT3;
	return false;
	}



int Bus::open(int pRate, int pBitDepht, int whichChannels, bool block, float pSuggestedFragmentsPerSecond)
	{
	PENTER2;
	fragmentsPerSecond = pSuggestedFragmentsPerSecond;
	int channels; // number of channels effectively open ( not the access code MONO,STEREO,ONLY_LEFT,ONLY_RIGHT, AND SO ON...)
	int r;
	QString busFullReference = get_short_name()+" ("+get_real_descriptor()+")";
	const char* cBusFullReference = (const char*) busFullReference.ascii();
	PMESG2("Opening %s bus : %s ...",stream==MustuxAudioDeviceMapper::PLAYBACK?"PLAYBACK":"CAPTURE",cBusFullReference);
	if (stream==MustuxAudioDeviceMapper::CAPTURE)
		{
		if (allocated_for_capture())
			{
			PMESG2("Bus seems to be allocated for capture. No need to re-open it.");
			PEXIT2;
			return 1;
			}
		if (stereoBus)
			{
			if ((whichChannels==MustuxAudioDeviceMapper::ONLY_LEFT_CHANNEL) && (status==CAPTURING_FROM_LEFT_CHANNEL))
				{
				PERROR("One or more channels of bus %s is already in use.",cBusFullReference);
				PEXIT2;
				return MustuxAudioDeviceMapper::ERROR_BUSY_BUS;
				}
			if ((whichChannels==MustuxAudioDeviceMapper::ONLY_RIGHT_CHANNEL) && (status==CAPTURING_FROM_RIGHT_CHANNEL))
				{
				PERROR("One or more channels of bus %s is already in use.", cBusFullReference);
				PEXIT2;
				return MustuxAudioDeviceMapper::ERROR_BUSY_BUS;
				}
			if ((whichChannels==MustuxAudioDeviceMapper::STEREO) && (status==CAPTURING_FROM_LEFT_CHANNEL || status==CAPTURING_FROM_RIGHT_CHANNEL))
				{
				PERROR("One or more channels of bus %s is already in use.",cBusFullReference);
				PEXIT2;
				return MustuxAudioDeviceMapper::ERROR_BUSY_BUS;
				}
			channels=2;
			}
		else
			{
			if (whichChannels==MustuxAudioDeviceMapper::STEREO)
				{
				PERROR("Bus %s does not support STEREO stuff",cBusFullReference);
				PEXIT2;
				return MustuxAudioDeviceMapper::ERROR_MONO_BUS;
				}
			channels=1;
			}
		}
	else
		{
		if (allocated_for_playback())
			{
			if ((whichChannels!=MustuxAudioDeviceMapper::MONO) && (status!=PLAYING_TO_MONO))
				{
				PMESG2("Bus seems to be allocated for playback. No need to re-open it.");
				PEXIT2;
				return 0;
				}
			PERROR("Bus %s is busy. ",cBusFullReference);
			PEXIT2;
			return -1;
			}
		else
			{
			if (whichChannels==MustuxAudioDeviceMapper::MONO)
				channels=1;
			else // STEREO, ONLY_LEFT or ONLY_RIGHT
				channels=2;
			}
		}

	// not open yet ? so let's try and open it ! :-)
	if (!valid_mode(pRate, pBitDepht, channels))
		{
		PERROR("Invalid mode for bus %s : %d Hz, %d bits, %s",cBusFullReference,pRate,pBitDepht, (channels==1?"MONO":"STEREO"));
		PEXIT2;
		return MustuxAudioDeviceMapper::ERROR_INVALID_BUS;
		}
	if (stream==MustuxAudioDeviceMapper::PLAYBACK)
		{
		r = snd_pcm_open(&pcmHandle, ((const char*) get_real_descriptor().ascii()), SND_PCM_STREAM_PLAYBACK, block ? 0 : SND_PCM_NONBLOCK);
		if ( r < 0)
			{
			PERROR("Cannot open bus %s for PLAYBACK . ALSA ERROR : %s",cBusFullReference,snd_strerror(r));
			PEXIT2;
			return MustuxAudioDeviceMapper::ERROR_CANNOT_OPEN_PLAYBACK_BUS;
			}
		}
	else if (stream==MustuxAudioDeviceMapper::CAPTURE)
		{
		r = snd_pcm_open(&pcmHandle, ((const char*) get_real_descriptor().ascii()), SND_PCM_STREAM_CAPTURE, 0);
		if (r < 0)
			{
			PERROR("Cannot open %s for MustuxAudioDeviceMapper::CAPTURE . ALSA ERROR : %s",cBusFullReference,snd_strerror(r));
			PEXIT2;
			return MustuxAudioDeviceMapper::ERROR_CANNOT_OPEN_CAPTURE_BUS;
			}
		}
	// some parameters declaration/initialization
	int dir;
	int triedPeriodSize;
	int numPeriods = 2; // IMPROVE ME (?)
	blockSize = channels * (pBitDepht/8);
	int chunckSize = pRate*blockSize;
	int fragmentSize = (int) ( chunckSize / fragmentsPerSecond);
	snd_pcm_uframes_t minPeriodSize, maxPeriodSize;

	// starting the ALSA settings
	snd_pcm_hw_params_alloca(&hwparams);
	r = snd_pcm_hw_params_any(pcmHandle, hwparams);
	if (r < 0)
		{
		PERROR("Cannot configure bus %s. ALSA ERROR : %s ",cBusFullReference,snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	// set the interleaved read/write format
	r = snd_pcm_hw_params_set_access(pcmHandle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED);
	if (r < 0)
		{
		PERROR("Invalid access format to bus %s: %s",cBusFullReference,snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	// set the sample format
	format = MustuxAudioDeviceMapper::get_format(pBitDepht);
	if (format == SND_PCM_FORMAT_UNKNOWN)
		{
		PERROR("Cannot define pcm format to %d bits in bus %s",pBitDepht, cBusFullReference);
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	r = snd_pcm_hw_params_set_format(pcmHandle, hwparams, format);
	if (r < 0)
		{
		PERROR("Invalid sample format to bus %s : %s",cBusFullReference,snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	// set the count of channels
	r = snd_pcm_hw_params_set_channels(pcmHandle, hwparams, channels);
	if (r < 0)
		{
		PERROR("Invalid number of channels (%d) to bus %s : %s",cBusFullReference,  channels, snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	// set the stream rate
	r = snd_pcm_hw_params_set_rate(pcmHandle, hwparams, pRate, 0);
	if (r < 0)
		{
		PERROR("Invalid sample rate (%d Hz) to bus %s . ALSA ERROR : %s", pRate, cBusFullReference, snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	// the period/buffer setup process
	r = snd_pcm_hw_params_set_periods(pcmHandle, hwparams, numPeriods, 0) ;
	if (r < 0)
		{
		PERROR("Cannot set periods count to %d on bus %s. ALSA ERROR : %s", numPeriods, cBusFullReference, snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	r = snd_pcm_hw_params_get_period_size_min (hwparams, &minPeriodSize, &dir );
	if (r<0)
		{
		PERROR("Cannot get minimum period size on bus %s . ALSA ERROR : %s", cBusFullReference, snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}
	r = snd_pcm_hw_params_get_period_size_max (hwparams, &maxPeriodSize, &dir );
	if (r<0)
		{
		PERROR("Cannot get maximum period size on bus %s . ALSA ERROR : %s", cBusFullReference, snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	PMESG2("Detected min/max period size : %d - %d",minPeriodSize,maxPeriodSize);

	periodSize = fragmentSize / blockSize;
	// TODO : check if periodSize is allow (is within mix/max)
	triedPeriodSize = (int) periodSize;

	if (stream==MustuxAudioDeviceMapper::PLAYBACK)
		{
		PMESG2("Trying OPEN PLAYBACK BUS with these values: fps=%2.3f  fragmentSize=%d  periods=%d  periodSize=%d",
			fragmentsPerSecond, fragmentSize, numPeriods, periodSize);                int rp2 = snd_pcm_hw_params_set_period_size (pcmHandle, hwparams, periodSize, 0);
		r = snd_pcm_hw_params_set_period_size (pcmHandle, hwparams, periodSize, 0);
		if (r < 0)
			{
			PERROR("setting period size to %d on bus %s. ALSA ERROR : %s", periodSize,cBusFullReference,snd_strerror(r));
			goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
			}
		}
	else
		{
		PMESG2("Trying OPEN CAPTURE BUS with these values: fps=%2.3f  fragmentSize=%d  periods=%d  periodSize=%d",
			fragmentsPerSecond, fragmentSize, numPeriods, periodSize);
		r = snd_pcm_hw_params_set_period_size_near (pcmHandle, hwparams, &periodSize, &dir);
		if (r < 0)
			{
			PERROR("setting period size to %d on bus %s. ALSA ERROR : %s", periodSize,cBusFullReference, snd_strerror(r));
			goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
			}
		}

	r = snd_pcm_hw_params_get_period_size (hwparams, &periodSize, &dir);
	if (r < 0)
		{
		PERROR("Could not confirm actual periodSize value after setting bus %s . ALSA ERROR : ",cBusFullReference, snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	if (triedPeriodSize!=periodSize)
		{
		PMESG2("Could not set periodSize to %d . ALSA let me use only : %d . Continuing anyway...",triedPeriodSize, periodSize);
		}


	// write the parameters to descriptor
	r = snd_pcm_hw_params(pcmHandle, hwparams);
	if (r < 0)
		{
		PERROR("Unable to set hw params : %s",snd_strerror(r));
		goto ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION;
		}

	transferSize = periodSize * blockSize;
	transferBuffer=new char[transferSize];
	bzero(transferBuffer, transferSize);

	// set the actual values for fragmentSize and fps
	if (fragmentSize != transferSize)
		{
		fragmentSize = transferSize;
		fragmentsPerSecond = (float) chunckSize / fragmentSize;
		PMESG2("Fixing fragmentSize and fps to match transferSize : fragmentSize=%d fps=%2.3f",fragmentSize,fragmentsPerSecond);
		}

	if (stream==MustuxAudioDeviceMapper::PLAYBACK)
		{
		if (channels==1)
			status=PLAYING_TO_MONO;
		else
			status=PLAYING_TO_STEREO; // shouldnt it have status for "playing to left only/ playing to right only" situatiosn ???
		}
	else
		{
		if (whichChannels==MustuxAudioDeviceMapper::ONLY_LEFT_CHANNEL)
			status=CAPTURING_FROM_LEFT_CHANNEL;
		else if (whichChannels==MustuxAudioDeviceMapper::ONLY_RIGHT_CHANNEL)
			status=CAPTURING_FROM_RIGHT_CHANNEL;
		else if (whichChannels==MustuxAudioDeviceMapper::STEREO)
			status=CAPTURING_FROM_STEREO;
		else if (whichChannels==MustuxAudioDeviceMapper::MONO)
			status=CAPTURING_FROM_MONO;
		}

	if (status!=CLOSED)
		{
		PMESG2("BUS %s %s OPEN : %d Hz, %s, %d bits periodSize=%d transferSize=%d",
			stream==MustuxAudioDeviceMapper::CAPTURE?"IN":"OUT",
			cBusFullReference,
			pRate ,
			channels==1?"MONO":"STEREO",
			pBitDepht,
			(int) periodSize,
			(int) transferSize);
		}
	else
		PERROR("Status is still closed, at this point ??!?!?!");

	levels = new float[channels];

	PEXIT2;
	return 1;

ALSA_ERROR_AFTER_OPEN_AND_HWPARAMS_ALLOCATION:
	if (snd_pcm_close(pcmHandle)<0)
		{
		PERROR ("Could not close pcmHandle after previous ALSA error");
		}
	// SHOULDNT I CALL snd_pcm_hw_params_free(hwparams) now ?
	// If I try I get a segfault.. but why.. ?
	PEXIT2;
	return r;
	}




MustuxAudioBusTransferResult* Bus::transfer(int bytesToTransfer)
	{
	PENTER3;
	transferResult->clean();
	int periodsToTransfer;
	if ( (bytesToTransfer==-1) || (bytesToTransfer > periodSize) ) // -1 means ALL bytes
		periodsToTransfer = periodSize;
	else
		periodsToTransfer = bytesToTransfer / blockSize;
	int periodsTransferred;
	char* ptr = transferBuffer;
	while (periodsToTransfer > 0)
		{
		if (stream==MustuxAudioDeviceMapper::CAPTURE)
			periodsTransferred = snd_pcm_readi(pcmHandle, ptr, periodsToTransfer);
		else
			periodsTransferred = snd_pcm_writei(pcmHandle, ptr, periodsToTransfer);
		if ( periodsTransferred >= 0 )
			{
			ptr += periodsTransferred * blockSize;
			transferResult->bytesTransferred += periodsTransferred * blockSize;
			periodsToTransfer -= periodsTransferred;
			}
		else
			{
			int err;
			if ( periodsTransferred == -EAGAIN ) // overrun
				{
				PWARN2("Buffer OVERRUN, retrying ...");
				snd_pcm_wait(pcmHandle, 1000);
				continue;
				}
			else if (periodsTransferred == -EPIPE) // underrun
				{
				PWARN2("Buffer UNDERRUN, recovering ...");
				snd_pcm_status_t *bus_status;
				snd_pcm_status_alloca(&bus_status);
				snd_pcm_status(pcmHandle, bus_status);
				if (snd_pcm_status_get_state(bus_status) == SND_PCM_STATE_XRUN)
					{
					struct timeval now, diff, tstamp;
					gettimeofday(&now, 0);
					snd_pcm_status_get_trigger_tstamp(bus_status, &tstamp);
					timersub(&now, &tstamp, &diff);
					PWARN("Buffer UNDERRUN of at least %.3f msecs, recovering ...", diff.tv_sec * 1000 + diff.tv_usec / 1000.0);
					}
				err = snd_pcm_prepare(pcmHandle);
				if ( err < 0 )
					{
					PERROR("Can't recover : %s",snd_strerror(periodsTransferred));
					transferResult->error = MustuxAudioDeviceMapper::ERROR_UNDERRUN;
					transferResult->xrun=-1;
					}
				}
			else // unknown error. try to recover anyway
				{
				PWARN("Unknown problem in transfer. Trying to recover anyway...");
				err = snd_pcm_prepare(pcmHandle);
				if ( err < 0)
					PERROR("Can't recover : %s\n", snd_strerror(err));
				}
			break;
			}
		}
	transferResult->message=MustuxAudioDeviceMapper::get_error_mesg(transferResult->error);
	PEXIT3;
	return transferResult;
	}



void Bus::clean_buffer()
	{
	PENTER3;
	bzero(transferBuffer, transferSize);
	PEXIT3;
	}



int Bus::close()
	{
	PENTER2;
	if (status==CLOSED)
		{
		PMESG2("Bus %s Seems to be already closed. Ignoring close call...",(const char*) get_real_descriptor().ascii());
		PEXIT2;
		return 1;
		}
	if (transferBuffer)
		delete transferBuffer;
	transferBuffer=(char*) 0;
	if ( snd_pcm_close(pcmHandle) <0 )
		{
		PERROR("Cannot close Bus %s", (const char*) get_full_name() );
		PEXIT2;
		return -1;
		}
	status=CLOSED;
	PMESG2(" BUS %s CLOSED", (const char*) get_full_name() );
	// SHOULDNT I CALL THIS ? (it crashes !) snd_pcm_hw_params_free(hwparams);
	PEXIT2;
	return 1;
	}


int Bus::get_transfer_size()
	{
	return transferSize;
	}


char* Bus::get_transfer_buffer(int whichChannels)
	{
	return transferBuffer; // IMPROVE ME.. Mustux consider whichChannels some way...
	}


bool Bus::is_stereo()
	{
	return stereoBus;
	}


QString Bus::get_short_name()
	{
	QString spId; spId.setNum(id+1);
	QString s;
	if (id==0)
		s="st";
	else if (id==1)
		s="nd";
	else if (id==2)
		s="rd";
	else s="th";
	QString br = spId+s+" "+(stream==MustuxAudioDeviceMapper::CAPTURE?"IN":"OUT");
	return MustuxAudioDeviceMapper::get_card_name(parentCardId)+" - "+br;
	}

QString Bus::get_full_name()
	{
	QString spId; spId.setNum(id+1);
	QString s;
	if (id==0)
		s="st";
	else if (id==1)
		s="nd";
	else if (id==2)
		s="rd";
	else s="th";
	QString br = spId+s+" "+(stream==MustuxAudioDeviceMapper::CAPTURE?"IN":"OUT");
	return MustuxAudioDeviceMapper::get_card_name(parentCardId)+" - "+br+" ("+get_real_descriptor()+")";
	}

QString Bus::get_descriptor()
	{
	return get_real_descriptor();
	}


float Bus::get_fps()
	{
	return fragmentsPerSecond;
	}

bool Bus::allocated_for_playback()
	{
	bool b = ( (status==PLAYING_TO_STEREO) || (status==PLAYING_TO_LEFT_CHANNEL) || (status==PLAYING_TO_RIGHT_CHANNEL));
	return b;
	}

bool Bus::allocated_for_capture()
	{
	return ( (status==CAPTURING_FROM_RIGHT_CHANNEL) || (status==CAPTURING_FROM_LEFT_CHANNEL) || (status==CAPTURING_FROM_STEREO));
	}

QString Bus::get_real_descriptor()
	{
	if (MustuxAudioDeviceMapper::get_plughw_use())
		return "plug"+descriptor;
	else
		return descriptor;
	}


float* Bus::get_levels()
	{
	int channels = is_stereo() ? 2 : 1;

	if((transferSize < 0) || (!transferBuffer))	//This can happen at the time the buffers are cleared! of course, it's a change of 1:100000 or something but it happens!!
		return levels;				//LG: but the buffers are only bzero'd, not destroyed. They are always valid (as long the bus is open, ofcourse). They are invalid only when bus is closed, but nobody is supposed to call geT_levels from a closed bus... (RFC)
	char* p  = transferBuffer;
	float* max = new float[channels];
	for (int k=0; k<channels; k++) max[k]=0.0f;
	unsigned short sho;
	int totalsamples = transferSize/(channels*sizeof(short));
	for (int i=0; i <totalsamples; i++)
		{
		for (int k=0; k<channels; k++)
			{
			sho = (unsigned short) abs(*(short*)p);
			if (sho > max[k]) max[k] = (float) sho;
			p+=sizeof(short);
			}
		}
	for (int k=0; k<channels; k++)
		levels[k] = max[k];
	delete max;
	return levels;
	}



// eof


