/*
 * libproccpuinfo.c - platform independent C API for reading /proc/cpuinfo
 *
 * Copyright (C) 2006  Thomas Cort <linuxgeek@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "proccpuinfo.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

char *proccpuinfo_filename = NULL;
char *proccpuinfo_format = NULL;

enum { architecture = 0, hardware_platform = 1, frequency = 2, cache = 3, cpus = 4, bogomips = 5};

enum { 
	proccpuinfo_i386 = 0,
	proccpuinfo_x86_64 = 1,
	proccpuinfo_ia64 = 2,
	proccpuinfo_alpha = 3,
	proccpuinfo_sparc = 4,
	proccpuinfo_hppa = 5,
	proccpuinfo_mips = 6,
	proccpuinfo_powerpc = 7,
	proccpuinfo_powerpc64 = 8,
	proccpuinfo_arm = 9,
	proccpuinfo_s390 = 10,
	proccpuinfo_s390x = 11,
	proccpuinfo_sh = 12,
	proccpuinfo_m68k = 13,
	proccpuinfo_cris = 14,
	proccpuinfo_frv = 15,
	proccpuinfo_vax = 16,
	proccpuinfo_m32r = 17,
	proccpuinfo_unknown = 18
};

struct proccpuinfo_arch_list_t {
        const char *name;
	unsigned char architecture[3];
	unsigned char hardware_platform[3];
	char freq_delim;
	unsigned char frequency[3];
	char cache_delim;
	unsigned char cache_size[4];
	unsigned char count_cpus;
	unsigned char cpus;
	unsigned char bogomips[4];
} proccpuinfo_archlist[] = {
	{ "i386",		{31, 0, 0}, {41, 0, 0}, '\0', {16, 0, 0},  ' ', {13, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "x86_64",		{31, 0, 0}, {41, 0, 0}, '\0', {16, 0, 0},  ' ', {13, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "ia64",		{23, 0, 0}, {40, 0, 0}, '\0', {16, 0, 0}, '\0', { 0, 0, 0, 0}, 1, 34, { 1, 0, 0, 0} },
	{ "alpha",		{19, 0, 0}, {36, 0, 0},  ' ', {22, 0, 0},  'K', {25, 0, 0, 0}, 0, 21, { 1, 0, 0, 0} },
	{ "sparc",		{38, 0, 0}, {15, 0, 0}, '\0', { 8,12, 0}, '\0', { 0, 0, 0, 0}, 0, 32, {11, 2, 0, 0} },
	{ "hppa",		{15, 0, 0}, {30, 0, 0}, '\0', {16, 0, 0}, '\0', { 0, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "mips",		{19, 0, 0}, {36, 0, 0}, '\0', { 0, 0, 0}, '\0', { 0, 0, 0, 0}, 1, 34, { 1, 0, 0, 0} },
	{ "powerpc",		{15, 0, 0}, {29, 0, 0},  'M', {14, 0, 0},  'K', {25,26,42, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "powerpc64",		{15, 0, 0}, {29, 0, 0},  'M', {14, 0, 0},  'K', {25,26,42, 0}, 1, 34, { 0, 0, 0, 0} },
	{ "arm",		{33,15, 0}, {29,24, 0},  'M', {14, 0, 0}, '\0', { 4, 0, 0, 0}, 1,  0, { 1, 2, 3, 0} },
	{ "s390",		{37, 0, 0}, {41, 0, 0}, '\0', { 0, 0, 0}, '\0', { 0, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "s390x",		{37, 0, 0}, {41, 0, 0}, '\0', { 0, 0, 0}, '\0', { 0, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "sh",			{18, 0, 0}, {27,29, 0},  'M', {17, 7, 0},  'K', {13, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "m68k",		{ 5, 0, 0}, {28, 0, 0},  'M', {10, 0, 0}, '\0', { 0, 0, 0, 0}, 1,  0, { 2, 0, 0, 0} },
	{ "cris",		{15, 0, 0}, {19, 0, 0}, '\0', { 0, 0, 0},  ' ', {13, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "frv",		{ 6, 0, 0}, {35, 0, 0},  ' ', { 9, 0, 0}, '\0', { 0, 0, 0, 0}, 1,  0, { 2, 0, 0, 0} },
	{ "vax",		{15, 0, 0}, {20, 0, 0}, '\0', { 0, 0, 0}, '\0', { 0, 0, 0, 0}, 1,  0, { 1, 0, 0, 0} },
	{ "m32r",		{18, 0, 0}, {27, 0, 0},  'M', { 7, 0, 0}, '\0', { 0, 0, 0, 0}, 1, 34, { 3, 0, 0, 0} },
	{ "unknown",		{ 0, 0, 0}, { 0, 0, 0}, '\0', { 0, 0, 0}, '\0', { 0, 0, 0, 0}, 1,  0, { 0, 0, 0, 0} }
};

#define NUM_ARCHES (19)

char *__proccpuinfo_key[] = {
	"unknown",			/*  0 */
	"BogoMIPS",			/*  1 */
	"BogoMips",			/*  2 */
	"bogomips",			/*  3 */
	"Cache size",			/*  4 */
	"CPU",				/*  5 */
	"CPU-Core",			/*  6 */
	"CPU clock",			/*  7 */
	"CPU0ClkTck",			/*  8 */
	"Clock-Core",			/*  9 */
	"Clocking",			/* 10 */
	"Cpu0Bogo",			/* 11 */
	"Cpu0ClkTck",			/* 12 */
	"cache size",			/* 13 */
	"clock",			/* 14 */
	"cpu",				/* 15 */
	"cpu MHz",			/* 16 */
	"cpu clock",			/* 17 */
	"cpu family",			/* 18 */
	"cpu model",			/* 19 */
	"cpu type",			/* 20 */
	"cpus detected",		/* 21 */
	"cycle frequency [Hz]",		/* 22 */
	"family",			/* 23 */
	"Hardware",			/* 24 */
	"L2 cache",			/* 25 */
	"l2 cache",			/* 26 */
	"Machine",			/* 27 */
	"MMU",				/* 28 */
	"machine",			/* 29 */
	"model",			/* 30 */
	"model name",			/* 31 */
	"ncpus probed",			/* 32 */
	"Processor",			/* 33 */
	"processor",			/* 34 */
	"System",			/* 35 */
	"system type",			/* 36 */
	"Type",				/* 37 */
	"type",				/* 38 */
	"unknown",			/* 39 */
	"vendor",			/* 40 */
	"vendor_id",			/* 41 */
	"board l2"			/* 42 */
};

#if defined(__i386__)
	int proccpuinfo_arch = proccpuinfo_i386;
#elif defined(__x86_64__)
	int proccpuinfo_arch = proccpuinfo_x86_64;
#elif defined(__ia64__)
	int proccpuinfo_arch = proccpuinfo_ia64;
#elif defined(__alpha__)
	int proccpuinfo_arch = proccpuinfo_alpha;
#elif defined(__sparc__) || defined(sparc)
	int proccpuinfo_arch = proccpuinfo_sparc;
#elif defined(__hppa__)
	int proccpuinfo_arch = proccpuinfo_hppa;
#elif defined(__mips__)
	int proccpuinfo_arch = proccpuinfo_mips;
#elif defined(__powerpc__)
	int proccpuinfo_arch = proccpuinfo_powerpc;
#elif defined(__powerpc64__)
	int proccpuinfo_arch = proccpuinfo_powerpc64;
#elif defined(__arm__)
	int proccpuinfo_arch = proccpuinfo_arm;
#elif defined(__s390__)
	int proccpuinfo_arch = proccpuinfo_s390;
#elif defined(__s390x__)
	int proccpuinfo_arch = proccpuinfo_s390x;
#elif defined(__sh__)
	int proccpuinfo_arch = proccpuinfo_sh;
#elif defined(__m68k__)
	int proccpuinfo_arch = proccpuinfo_m68k;
#elif defined(__cris__)
	int proccpuinfo_arch = proccpuinfo_cris;
#elif defined(__frv__)
	int proccpuinfo_arch = proccpuinfo_frv;
#elif defined(__vax__) || defined(__VAX__)
	int proccpuinfo_arch = proccpuinfo_vax;
#elif defined(__m32r__)
	int proccpuinfo_arch = proccpuinfo_m32r;
#else
	 int proccpuinfo_arch = proccpuinfo_unknown;
#endif

int   proccpuinfo_get_arch()     { return proccpuinfo_arch;}
char *proccpuinfo_get_filename() { return proccpuinfo_filename; }
char *proccpuinfo_get_format()   { return proccpuinfo_format; }

void proccpuinfo_set_arch(int x)       { proccpuinfo_arch     = x; }
void proccpuinfo_set_filename(char *f) { proccpuinfo_filename = f; }
void proccpuinfo_set_format(char *f)   { proccpuinfo_format   = f; }


int proccpuinfo_decode_arch(char *s) {
	int i;

	for (i = 0; i < NUM_ARCHES; i++)
		if (!strcmp(proccpuinfo_archlist[i].name,s))
			return i;

	return proccpuinfo_unknown;
}

void __proccpuinfo_remove_whitespace(char *buf) {
	/* first eat trailing space */
	char *tmp = buf + strlen(buf) - 1;
	while (tmp > buf && isspace(*tmp))
		*tmp-- = '\0';
	/* then eat leading space */
	tmp = buf;
	while (*tmp && isspace(*tmp))
		*tmp++;
	if (tmp != buf)
		memmove(buf, tmp, strlen(tmp)+1);
}

char *__proccpuinfo_get_value(int target_key) {
	FILE *fp;
	char key[64], value[257], *target_value;

	if ((fp = fopen(proccpuinfo_get_filename(), "r")) != NULL) {
		while (fscanf(fp, proccpuinfo_get_format(), key, value) != EOF) {
			if (ferror(fp)) {
				fclose(fp);
				return NULL;
			}

			__proccpuinfo_remove_whitespace(key);
			if (strlen(key) == strlen(__proccpuinfo_key[target_key]) && !strcmp(key, __proccpuinfo_key[target_key])) {
				__proccpuinfo_remove_whitespace(value);

				if (!(target_value = (char *) malloc(strlen(value)+1))) {
					fclose(fp);
					return NULL;
				}

				strncpy(target_value,value,strlen(value)+1);
				fclose(fp);
				return target_value;
			}
		}
		fclose(fp);
	}

	return NULL;
}

int __proccpuinfo_get_count(int target_key) {
	FILE *fp;
	char key[64], value[257], *target_value;
	unsigned int count = 0;

	if ((fp = fopen(proccpuinfo_get_filename(), "r")) != NULL) {
		while (fscanf(fp, proccpuinfo_get_format(), key, value) != EOF) {
			if (ferror(fp)) {
				fclose(fp);
				return count;
			}

			__proccpuinfo_remove_whitespace(key);
			if (!strcmp(key, __proccpuinfo_key[target_key])) {
				count++;
			}
		}
		fclose(fp);
	}

	return count;
}

proccpuinfo *proccpuinfo_read() {
	proccpuinfo *info;
	char *freq, *_cache, *_cpus, *_bogomips;
	int i;

	if (!(info = (proccpuinfo *) malloc(sizeof(proccpuinfo))))
		return NULL;

	memset(info,0,sizeof(proccpuinfo));

	if (!proccpuinfo_get_filename())
			proccpuinfo_set_filename("/proc/cpuinfo");

	if (!proccpuinfo_get_format()) {
		if (proccpuinfo_get_arch() == proccpuinfo_s390 || proccpuinfo_get_arch() == proccpuinfo_s390x)
			proccpuinfo_set_format("%[^\t :]%*[ :]%[^\n]\n");
		else
			proccpuinfo_set_format("%[^\t:]\t:%[^\n]\n");
	}


	for (i = 0; proccpuinfo_archlist[proccpuinfo_get_arch()].architecture[i] && i < 3; i++)
		if ((info->architecture = __proccpuinfo_get_value(proccpuinfo_archlist[proccpuinfo_get_arch()].architecture[i])))
			break;

	for (i = 0; proccpuinfo_archlist[proccpuinfo_get_arch()].hardware_platform[i] && i < 3; i++)
		if ((info->hardware_platform = __proccpuinfo_get_value(proccpuinfo_archlist[proccpuinfo_get_arch()].hardware_platform[i])))
			break;

	for (i = 0; proccpuinfo_archlist[proccpuinfo_get_arch()].frequency[i] && i < 3; i++) {
		if ((freq = __proccpuinfo_get_value(proccpuinfo_archlist[proccpuinfo_get_arch()].frequency[i]))) {
			if (proccpuinfo_archlist[proccpuinfo_get_arch()].freq_delim && !(proccpuinfo_get_arch() == proccpuinfo_alpha && freq[strlen(freq)-1] != '.')) {
				for (i = 0; i < strlen(freq); i++) {
					if (freq[i] == proccpuinfo_archlist[proccpuinfo_get_arch()].freq_delim) {
						freq[i] = '\0';
						info->frequency = strtod(freq,NULL);
						break;
					}
				}
			} else {
				if (proccpuinfo_get_arch() == proccpuinfo_sparc && freq[0] == '0')
					freq[1] = 'x';
				info->frequency = strtod(freq,NULL);
			}
			if (proccpuinfo_get_arch() == proccpuinfo_alpha || (proccpuinfo_get_arch() == proccpuinfo_sparc && proccpuinfo_archlist[proccpuinfo_get_arch()].frequency[i]  == 12))
				info->frequency = info->frequency/1000000;
			free(freq);
			break;
		}
	}

	for (i = 0; proccpuinfo_archlist[proccpuinfo_get_arch()].cache_size[i] && i < 4; i++) {
		if ((_cache = __proccpuinfo_get_value(proccpuinfo_archlist[proccpuinfo_get_arch()].cache_size[i]))) {
			if (proccpuinfo_archlist[proccpuinfo_get_arch()].cache_delim) {
				for (i = 0; i < strlen(_cache); i++) {
					if (_cache[i] == proccpuinfo_archlist[proccpuinfo_get_arch()].cache_delim) {
						_cache[i] = '\0';
						info->cache = atoi(_cache);
						break;
					}
				}
			} else {
				info->cache = strtod(_cache,NULL);
			}
			free(_cache);
			break;
		}
	}


	if (proccpuinfo_archlist[proccpuinfo_get_arch()].count_cpus) {
		info->cpus = __proccpuinfo_get_count(proccpuinfo_archlist[proccpuinfo_get_arch()].cpus);
	} else {
		if ((_cpus = __proccpuinfo_get_value(proccpuinfo_archlist[proccpuinfo_get_arch()].cpus))) {
			if (_cpus) {
				info->cpus = atoi(_cpus);
				free(_cpus);
			}
		}
	}

	if (info->cpus < 1)
		info->cpus = 1;

	for (i = 0; proccpuinfo_archlist[proccpuinfo_get_arch()].bogomips[i] && i < 4; i++) {
		if ((_bogomips = __proccpuinfo_get_value(proccpuinfo_archlist[proccpuinfo_get_arch()].bogomips[i]))) {
			if (proccpuinfo_get_arch() == proccpuinfo_s390 || proccpuinfo_get_arch() == proccpuinfo_s390x) {
				char *p = _bogomips;
				for (i = 0; i < strlen(_bogomips) && p[i] != ':'; i++);
				info->bogomips = strtod(p+i+1,NULL);
			} else {
				info->bogomips = strtod(_bogomips,NULL);
			}
			free(_bogomips);
			break;
		}
	}

	return info;
}

void proccpuinfo_free(proccpuinfo *info) {
	if (!info)
		return;

	free(info->architecture);
	free(info->hardware_platform);

	free(info);
}
