#include <Python.h>
#include <gmp.h>

/*
    This file is part of primerange.

    primerange is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    primerange is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with primerange.  If not, see <http://www.gnu.org/licenses/>.

    (c) 2011 Gary Wright
    http://wrightsolutions.co.uk/contact
    http://identi.ca/gnubyexample
    http://gnumbers.blogspot.com/

*/

	const char  outfstem [25] = "/tmp/setz_exp_esl";
	unsigned int	debug_flag = 0;
	char      *fexpk;
	char      *fout36;
	char      *fplanned;
	char      *fnotes;
	char      *frinprogress;
	FILE      *outfexpk;
	FILE      *outfout36;
	FILE      *outfplanned;
	FILE      *outfnotes;
	FILE      *outfrinprogress;
	unsigned long multiple_of_1365 = 294076965;
   unsigned int slice_size_max_2exp = 3;
   unsigned long long exp_shift_start = 0;
   int process_expk_optimisation = -999;
	mpz_t		ONE;
	mpz_t		THIRTEENSIXTYFIVE;
	mpz_t		FOURZERONINEFIVE;
	mpz_t		FIVEFOURSIXTY;
	mpz_t		MULTIPLE_OF_1365; 
	mpz_t		N;
	mpz_t		R_IN_PROGRESS_LIMITED_INPUT;
	mpz_t		R_IN_PROGRESS_LIMITED;
   mpz_t		R1, R2, R; 
   mpz_t		A;



/* no static prefix as may be called by other functions defined in this .c file */
unsigned long size2gmp(unsigned long num_long) {
  mpz_t NUM_LONG;
  unsigned long sibt = 0;
  mpz_init_set_ui(NUM_LONG, num_long);
  sibt = (long)mpz_sizeinbase(NUM_LONG,2);
  mpz_clear(NUM_LONG);
  return sibt;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(size2gmp__doc__,
"Calculate (and return) number of portions.");

/* Wrapper around the underlying C function */
static PyObject *
py_size2gmp(PyObject *self, PyObject *args)
{
  /* The function size2gmp() really only supports calls from
	  within this .c file itself, however a part functioning wrapper
	  is provided anyway.
  */
   unsigned long s2gmpresult;
	unsigned long num_l;
	/* "args" signature is long */
	/* ':size2gmp' for error messages */
	if (!PyArg_ParseTuple(args, "l:size2gmp", &num_l))
		return NULL;

	if (num_l < 0 || num_l > ULONG_MAX) {
	  return NULL;
	}

	s2gmpresult = size2gmp(num_l);

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)s2gmpresult);
}

/* no static prefix as may be called by other functions defined in this .c file */
unsigned long long size2longlong(unsigned long long num_ll) {
  mpz_t NUM_LL;
  unsigned long sibt = 0;
  mpz_init_set_ui(NUM_LL, num_ll);
  sibt = (long long)mpz_sizeinbase(NUM_LL,2);
  mpz_clear(NUM_LL);
  return sibt;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(size2longlong__doc__,
"Calculate (and return) number of portions - ll variant.");

/* Wrapper around the underlying C function */
static PyObject *
py_size2longlong(PyObject *self, PyObject *args)
{
  /* The function size2longlong() really only supports calls from
	  within this .c file itself, however a part functioning wrapper
	  is provided anyway.
  */
   unsigned long long s2result;
	unsigned long num_l;
	/* "args" signature is long */
	/* ':size2longlong' for error messages */
	if (!PyArg_ParseTuple(args, "l:size2longlong", &num_l))
		return NULL;

	if (num_l < 0 || num_l > ULONG_MAX) {
	  return NULL;
	}

	s2result = size2longlong((unsigned long long)num_l);

	if (s2result > ULONG_MAX) {
	  /* Decision to return (long) only - no long long */
	  return NULL;
	}

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)s2result);
}

/* no static prefix as may be called by other functions defined in this .c file */
unsigned int calculate_split(unsigned int sibt, unsigned int tstbit) {
  /* sibt is short for size in base 2 */
  /* tstbit is the result of examining the (sibt-1)th bit */
  unsigned int ports = 1048576;
  unsigned int ports_2exp = 1;
  /* We will calculate ports, but worth remembering that
	  for gigantic exponents, portions = 8192 is not enough */
  extern unsigned int slice_size_max_2exp;
  if (sibt > 3) {
	 if (sibt > slice_size_max_2exp) {
		ports_2exp = sibt - slice_size_max_2exp;
	 } else {
		ports_2exp = sibt - 2;
	 }
  }
  /* There will be more splits if the (sibt-1)th bit is set */
  ports_2exp += tstbit;
  ports = pow(2,ports_2exp);
  if (debug_flag > 0) {
	 printf("calculate_split should return a split that means individual portions"
			  " are <= 2^%d in size\n",slice_size_max_2exp);
	 printf("calculate_split is running with sibt==%d and tstbit=%d\n",sibt,tstbit);
	 printf("returning ports=%d\n",ports);
  }
  return ports;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(calculate_split__doc__,
"Calculate (and return) number of portions.");

/* Wrapper around the underlying C function */
static PyObject *
py_calculate_split(PyObject *self, PyObject *args)
{
   unsigned int splitresult;
	unsigned int sibt_given;
	unsigned int tstbit_given;
	/* "args" signature is integer,integer */
	/* ':calculate_split' for error messages */
	if (!PyArg_ParseTuple(args, "ii:calculate_split", &sibt_given, &tstbit_given))
		return NULL;

  /* sibt_given is short for size in base 2 */
  /* tstbit_given is the result of examining the (sibt-1)th bit */
	if (tstbit_given < 0 || tstbit_given > 1) {
	  return NULL;
	}

	splitresult = calculate_split(sibt_given, tstbit_given);

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)splitresult);
}

/* no static prefix as may be called by other functions defined in this .c file */
unsigned int calculate_split_wrapper_int(unsigned int exp) {
  unsigned long size2exp = 0;
  unsigned long tstbit_working = 0;
  unsigned long tstbit_result = 0;
  unsigned int portions_int = 0;
  size2exp = size2gmp(exp);
  tstbit_working = exp >> (size2exp - 2);
  tstbit_result = tstbit_working & 1;
  portions_int = calculate_split(size2exp,tstbit_result);
  return portions_int;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(calculate_split_wrapper_int__doc__,
"Derive 2nd of argument of 2 for calculate_split() from just single argument input.");

/* Wrapper around the underlying C function */
static PyObject *
py_calculate_split_wrapper_int(PyObject *self, PyObject *args)
{
   unsigned long cswrapperresult;
	unsigned int exp_given;
	/* "args" signature is long */
	/* ':calculate_split_wrapper_int' for error messages */
	if (!PyArg_ParseTuple(args, "i:calculate_split_wrapper_int", &exp_given))
		return NULL;

	if (exp_given < 2) {
	  return NULL;
	}

	cswrapperresult = calculate_split_wrapper_int(exp_given);

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)cswrapperresult);
}

/* no static prefix as may be called by other functions defined in this .c file */
unsigned long long calculate_split_wrapper_longlong(unsigned int exp) {
  unsigned long long size2exp = 0;
  unsigned long tstbit_working = 0;
  unsigned long tstbit_result = 0;
  unsigned int portions_int = 0;
  printf("size2longlong() is next call.\n");
  size2exp = size2longlong(exp);
  printf("size2longlong() returned result, now continuing.\n");
  tstbit_working = exp >> (size2exp - 2);
  tstbit_result = tstbit_working & 1;
  portions_int = calculate_split((int)size2exp,tstbit_result);
  return (long long)portions_int;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(calculate_split_wrapper_longlong__doc__,
"Derive 2nd of argument of 2 for calculate_split() from just single argument input.");

/* Wrapper around the underlying C function */
static PyObject *
py_calculate_split_wrapper_longlong(PyObject *self, PyObject *args)
{
  /* The function calculate_split_wrapper_longlong really only supports
	  calls from within this .c file itself, however
	  a part functioning wrapper is provided anyway.
  */
   unsigned long cswrapperresult;
	unsigned long exp_l;
	/* "args" signature is long */
	/* ':calculate_split_wrapper_longlong' for error messages */
	if (!PyArg_ParseTuple(args, "l:calculate_split_wrapper_longlong", &exp_l))
		return NULL;

	if (exp_l < 2 || exp_l > ULONG_MAX) {
	  return NULL;
	}

	cswrapperresult = \
	  calculate_split_wrapper_longlong((unsigned long long)exp_l);

	if (cswrapperresult > ULONG_MAX) {
	  /* Decision to return (long) only - no long long */
	  return NULL;
	}

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)cswrapperresult);
}

/* no static prefix as may be called by other functions defined in this .c file */
unsigned long mod_where_multiple_of_1365_in_exponent_limited(
				 unsigned int portion_count_limit, mpz_t EXP, unsigned int exp_un2k) {

	 mpz_t		SL2K, EX2K;
	 extern mpz_t		R_IN_PROGRESS_LIMITED_INPUT;
	 extern mpz_t		R_IN_PROGRESS_LIMITED;
	 mpz_t		R_IN_PROGRESS_LTD; /* Used only internally within function */

	unsigned int portion_count_exp = 0;
	unsigned int portions_exp = 128;
	unsigned int slice = 0;
	unsigned int extra = 0;

	unsigned long size10long = 0;

	 long clock_mod_where_limited_start = 0;
	 long clocks_mod_where_limited = 0;
	 long elapsed_secs_limited = 0;

	extern unsigned int	debug_flag;

	if (debug_flag > 0) {
		printf("mod_where_multiple_of_1365_limited() processing exp_un2k=%d begins...\n",
				 exp_un2k);
		size10long = mpz_sizeinbase(N,10);
		printf("N has size %ld (approx)\n",size10long);
		size10long = mpz_sizeinbase(R_IN_PROGRESS_LIMITED_INPUT,10);
		printf("before processing (R_IN_PROGRESS_LIMITED_INPUT)"
				 " we were given a %ld digit (approx) starting point\n",
				 size10long);
		size10long = mpz_sizeinbase(EXP,10);
		if (size10long < 81) {
		  gmp_printf("EXP=%Zd\n",EXP);
		}
	}

	clock_mod_where_limited_start = clock();

	if (portion_count_limit > exp_un2k) {
	  if (debug_flag > 0) {
		 printf("mod_where_multiple_of_1365_limited() %d > exp_un2k ( %d > %d).\n",
				  portion_count_limit,portion_count_limit,exp_un2k);
		 printf("calculate_split call not required therefore.\n");
		 printf("mod_where_multiple_of_1365_limited() jumped straight to mpz_powm().\n");
	  }
	  mpz_powm(R_IN_PROGRESS_LIMITED, R_IN_PROGRESS_LIMITED_INPUT, EXP, N);
	  clocks_mod_where_limited = (clock() - clock_mod_where_limited_start);
	  elapsed_secs_limited = clocks_mod_where_limited / CLOCKS_PER_SEC;
	  if (debug_flag > 0) {

		 printf("mod_where_multiple_of_1365_limited(%d,,%d)"
				 " elapsed %ld seconds.\n",portion_count_limit,exp_un2k,
				 elapsed_secs_limited);

		 size10long = mpz_sizeinbase(R_IN_PROGRESS_LIMITED,10);
		 printf("after processing (R_IN_PROGRESS_LIMITED)"
				 " we have now a %ld digit (approx) result saved\n",
				 size10long);
		 if (size10long < 81) {
			gmp_printf("jump straight to mpz_powm for mod(blah^%Zd,N) resulted in"
						  " R_IN_PROGRESS_LIMITED=%Zd\n",EXP,R_IN_PROGRESS_LIMITED);
		 }
	  }
	  fprintf(outfnotes, "mod_where_multiple_of_1365_limited(%d,,%d)"
				 " elapsed %ld seconds.\n",portion_count_limit,exp_un2k,
				 elapsed_secs_limited);
	  fflush(outfnotes);
	  return 0;
	}

	portions_exp = calculate_split_wrapper_int(exp_un2k);
	if (portions_exp < 1) {
	  mpz_set(R_IN_PROGRESS_LIMITED, R_IN_PROGRESS_LIMITED_INPUT);
	  return 101;
	}

	 slice = exp_un2k / portions_exp;
	 extra = exp_un2k - ((portions_exp - 1) * slice);

	 if (debug_flag > 0) {
		printf("slice now populated.\n");
		printf("slice=%d now populated.\n",slice);
		printf("...limited() slice and extra now populated.\n");
		printf("slice=%d and extra=%d now populated.\n",
				 slice,extra);
	 }

	 if (pow(2,slice) > portion_count_limit) {
		/* slice = 9 making 2^9 > 256 would return 151 error here if
			you provided 256 as portion_count_limit to indicate that
			exponentiation should never be attempted for any exponent
			larger than 256. */
		return 151;
	 }

	 mpz_init(SL2K);
	 mpz_ui_pow_ui(SL2K,2,slice);
	 if (mpz_cmp_ui(SL2K,portion_count_limit) > 0) {
		return 151;
	 }

	 mpz_init(EX2K);
	 mpz_ui_pow_ui(EX2K,2,extra);

	 mpz_init(R_IN_PROGRESS_LTD);
	 mpz_powm(R_IN_PROGRESS_LTD, R_IN_PROGRESS_LIMITED_INPUT, EX2K, N);

	 for (portion_count_exp = 1; portion_count_exp < portions_exp; portion_count_exp++) {
		if (debug_flag > 0) {
		  printf("Iteration having portion_count=%d ( <= %d ) has started.\n",
				 portion_count_exp,portion_count_limit);
		}
		mpz_powm(R_IN_PROGRESS_LTD, R_IN_PROGRESS_LTD, SL2K, N); 
	 }

	 clocks_mod_where_limited = (clock() - clock_mod_where_limited_start);
	 elapsed_secs_limited = clocks_mod_where_limited / CLOCKS_PER_SEC;

	 mpz_set(R_IN_PROGRESS_LIMITED, R_IN_PROGRESS_LTD);
	 mpz_clear(R_IN_PROGRESS_LTD);

	 if (debug_flag > 0) {

		printf("mod_where_multiple_of_1365_limited(%d,,%d)"
				 " elapsed %ld seconds.\n",portion_count_limit,exp_un2k,
				 elapsed_secs_limited);

		size10long = mpz_sizeinbase(R_IN_PROGRESS_LIMITED,10);
		printf("after processing (R_IN_PROGRESS_LIMITED)"
				 " we have now a %ld digit (approx) result saved\n",
				 size10long);
		if (size10long < 81) {
		  gmp_printf("R_IN_PROGRESS_LIMITED=%Zd\n",
						  R_IN_PROGRESS_LIMITED);
		}
	 }

	 fprintf(outfnotes, "mod_where_multiple_of_1365_limited(%d,,%d)"
				 " elapsed %ld seconds.\n",portion_count_limit,exp_un2k,
				 elapsed_secs_limited);
	 fflush(outfnotes);

	 return 0;
 }

/* Short description of the function (one liner). */
PyDoc_STRVAR(mod_where_multiple_of_1365_in_exponent_limited__doc__,
"modulo of small to large exponent where exponent is 2 to some power utilising slice + extra");

/* Wrapper around the underlying C function */
static PyObject *
py_mod_where_multiple_of_1365_in_exponent_limited(PyObject *self, PyObject *args)
{
	unsigned int port_count_limit;
	unsigned int e_un2k;
	long mwresult;
	mpz_t E;
	/* "args" signature is integer, integer */
	/* ':mod_where' for error messages */
	if (!PyArg_ParseTuple(args, "ii:mod_where", &port_count_limit, &e_un2k))
		return NULL;

	/* parameters that are invalid might be corrected here */
	if (e_un2k < 1) e_un2k = 1;

	/* assumption here is that internal calls to the C function might
		already have initialised and set R_IN_PROGRESS_LIMITED_INPUT, however
		calls from Python (testing etc)
		will not have set R_IN_PROGRESS_LIMITED_INPUT.
	*/
	mpz_init_set_ui(R_IN_PROGRESS_LIMITED_INPUT,6);
	mpz_init(N);
	mpz_mul_2exp(N,FOURZERONINEFIVE,e_un2k);
	mpz_add(N,N,ONE);

	mpz_init(E);
	mpz_mul_2exp(E,ONE,e_un2k);
	
	mwresult = mod_where_multiple_of_1365_in_exponent_limited(
		port_count_limit,E,e_un2k);
	
	mpz_clear(E);

	/* Convert from a C type to a Python type */
	return PyInt_FromLong((long)mwresult);
}

/* no static prefix as may be called by other functions defined in this .c file */
unsigned int nonresidue_value(unsigned int nrv_limit) {
  unsigned int nrv_candidate = 0;
  int nrv_result = -999;
  for(nrv_candidate = 2; nrv_candidate < nrv_limit ; nrv_candidate++) {
	 nrv_result = mpz_ui_kronecker(nrv_candidate, N);
	 if (nrv_result < 1)
		break;
  }
  /* function needs enhancing as does not deal properly 
	  with case where nrv_limit is set low
	  and/or for loop hits nrv_limit.
	  Suggest supplying nrv_limit as 1000 until enhanced. */
  return nrv_candidate;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(nonresidue_value__doc__,
"Non Residue Value. Supports single argument so process does not search forever.");

/* Wrapper around the underlying C function */
static PyObject *
py_nonresidue_value(PyObject *self, PyObject *args)
{
   unsigned long nrvresult;
	unsigned int nrv_lim;
	/* "args" signature is long */
	/* ':nonresidue_value' for error messages */
	if (!PyArg_ParseTuple(args, "i:nonresidue_value", &nrv_lim))
		return NULL;

	if (nrv_lim < 2) {
	  return NULL;
	}

	if (nrv_lim < 20) {
	  /* temporary lower limit until nonresidue_value() is enhanced */
	  return NULL;
	}

	nrvresult = nonresidue_value(nrv_lim);

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)nrvresult);
}

unsigned long mod_where_multiple_of_1365_in_exponent_longway(unsigned long long exp) {
  /* ft is short for 512th - bit of a stretch but go with it */
  /* gt is the final portion k_minus_one - (511 * ft) */
  /* ft and gt where named before variable 'portions' was introduced
	  and I retain these names */
  mpz_t		FT2K, GT2K;
  mpz_t		FT2K_BOOSTED;
  extern mpz_t		A;
  extern mpz_t		N;
  extern mpz_t		R1, R2, R; 
  extern mpz_t		R_IN_PROGRESS_LIMITED_INPUT;
  extern	mpz_t		R_IN_PROGRESS_LIMITED;
  mpz_t		R_IN_PROGRESS;
  unsigned int multiple_of_1365_multiplier = 0;
  unsigned long long k_minus_one = 1;
  unsigned long long exp_done = 0;
  unsigned long long ft = 0;
  unsigned long long ft2k = 0;
  unsigned long long gt = 0;
  unsigned long long gt2k = 0;
  int ft2k_plus_gt2k_sign = 0;
  unsigned int portions_boost_marker = 0;
  /* gt2k as int and gt as long long seems like a mismatch - depends on usage */
  unsigned int portion_count = 0;
  unsigned int portions = 1048576;

  unsigned long size10long = 0;

  unsigned long long outR1len = 0;
  char      *outR1;
  char      *outR1tail;
  unsigned long long outR2len = 0;
  char      *outR2;
  char      *outR2tail;

  long clock_mod_where_start = 0;
  long clocks_mod_where = 0;
  long elapsed_secs = 0;

  k_minus_one = exp - 1;
  multiple_of_1365_multiplier = multiple_of_1365 / 1365;

  mpz_powm_ui(R1, A, multiple_of_1365_multiplier, N); /* modulo and 'powers of powers' */
  if (debug_flag > 0) {
	 printf("...longway() multiple_of_1365_multiplier=%d.\n",
			  multiple_of_1365_multiplier);
	 printf("...longway() R1 initialised and populated using (mod N).\n");

	 outR1 = mpz_get_str(NULL,10,R1);
	 outR1len = strlen(outR1);

	 if (outR1len < 81) {
		printf("R1 has %lld decimal digits and" 
				  " in full R1=%s\n",outR1len,outR1);
		fprintf(outfnotes,"exp=%lld gives R1 having %lld decimal digits and"
				  " in full R1=%s\n",exp,outR1len,outR1);
		fflush(outfnotes);
	 } else {
		printf("R1 has %lld decimal digits and ends %s\n",
				 outR1len,&outR1[(outR1len-10)]);

		asprintf(&outR1tail,"%s",&outR1[(outR1len-10)]);
		fprintf(outfnotes,"exp=%lld gives R1 having %lld decimal digits and"
				  " ending %s\n",exp,outR1len,outR1tail);
		fflush(outfnotes);
	 }
  }

  mpz_powm_ui(R2, R1, 1365, N); /* R2 has acounted for original multiple_of_1365 so far */
  if (debug_flag > 0) {
	 printf("...longway() R2 initialised and populated using (mod N).\n");

	 outR2 = mpz_get_str(NULL,10,R2);
	 outR2len = strlen(outR2);

	 if (outR2len < 81) {
		printf("R2 has %lld decimal digits and" 
				  " in full R2=%s\n",outR2len,outR2);
		fprintf(outfnotes,"exp=%lld gives R2 having %lld decimal digits and"
				  " in full R2=%s\n",exp,outR2len,outR2);
		fflush(outfnotes);
	 } else {
		printf("R2 has %lld decimal digits and ends %s\n",
				 outR2len,&outR2[(outR2len-10)]);

		asprintf(&outR2tail,"%s",&outR2[(outR2len-10)]);
		fprintf(outfnotes,"exp=%lld gives R2 having %lld decimal digits and"
				  " ending %s\n",exp,outR2len,outR2tail);
		fflush(outfnotes);
	 }

  }

  if (debug_flag > 0) {
	 printf("k_minus_one has binary size %lld\n",size2longlong(k_minus_one));
  }
  portions = calculate_split_wrapper_longlong(k_minus_one);
  if (debug_flag > 0) {
	 printf("portions now populated and equal to %d.\n",portions);
	 printf("k_minus_one has binary size %lld\n",size2longlong(k_minus_one));
  }
  ft = k_minus_one / portions;
  if (debug_flag > 0) {
	 printf("ft=%lld and known that %lld*%d = %lld.\n",
			  ft,ft,(portions-1),(ft*(portions-1)));
  }
  gt = k_minus_one - ((portions - 1) * ft);
  if (debug_flag > 0) { printf("gt=%lld now (provisionally) populated.\n",gt); }

  portions_boost_marker = portions;
  if ((2*gt) > (3*ft)) {
	 portions_boost_marker = (ft * (gt/ft));
	 if (debug_flag > 0) { 
		printf("portions_boost_marker=%d now populated.\n",
				 portions_boost_marker);
	 }
	 gt = gt - portions_boost_marker;
	 if (debug_flag > 0) {
		printf("gt has been lowered using portions_boost_marker=%d.\n",
			  portions_boost_marker);
	 }
	 portions_boost_marker = portions - portions_boost_marker - 1;
  }
  if (debug_flag > 0) {
	 printf("gt=%lld now populated.\n",gt);
	 printf("portions_boost_marker=%d ... think 'portions complement' - 1\n",
			  portions_boost_marker);
  }

  if (debug_flag > 0 && gt < 1) {
	 printf("gt=%lld meaning adjustment process malfunction!"
			  " Ignore results.\n",gt);
  }

  mpz_init(FT2K);
  mpz_ui_pow_ui(FT2K,2,ft);
  if (mpz_cmp_ui(FT2K,ULLONG_MAX) < 1) {
	 ft2k = pow(2,ft);
  }

  if (debug_flag > 0) {
	 size10long = mpz_sizeinbase(FT2K,10);
	 if (size10long < 81) {
		gmp_printf("FT2K has size %d and equals %Zd\n",size10long,FT2K);
	 } else {
		printf("FT2K has size %ld\n",size10long);
	 }
  }

  mpz_init(FT2K_BOOSTED);
  mpz_ui_pow_ui(FT2K_BOOSTED,2,(ft+1));

  mpz_init(GT2K);
  mpz_ui_pow_ui(GT2K,2,gt);
  if (mpz_cmp_ui(GT2K,ULLONG_MAX) < 1) {
	 gt2k = pow(2,gt);
  }

  if (debug_flag > 0) {
	 size10long = mpz_sizeinbase(FT2K,10);
	 if (size10long < 81) {
		gmp_printf("FT2K has size %d and equals %Zd\n",size10long,FT2K);
	 } else {
		printf("FT2K has size %ld\n",size10long);
	 }
  }

  if (ft2k > 0 && gt2k > 0) { ft2k_plus_gt2k_sign = 1; }

  size10long = mpz_sizeinbase(GT2K,10);
  if (debug_flag > 0) { printf("GT2K has size %ld\n",size10long); }

  mpz_init_set(R_IN_PROGRESS,R2);
  gmp_fprintf(outfrinprogress,"R_IN_PROGRESS=R2=%Zd\n",R_IN_PROGRESS);
  fflush(outfrinprogress);

  mpz_init_set(R_IN_PROGRESS_LIMITED_INPUT,R2);
  if (debug_flag > 0) { printf("R_IN_PROGRESS_LIMITED_INPUT=R2\n"); }

  /* because of division and rounding gt may well be significantly
	  larger than ft, so raise to gt should be attempted early */
  /* ensure R_IN_PROGRESS is set before making next function call. */
  mod_where_multiple_of_1365_in_exponent_limited(256,GT2K,gt);
  /* mpz_powm(R_IN_PROGRESS, R_IN_PROGRESS, GT2K, N); */

  if (debug_flag > 0) {
	 printf("...longway() continues now after gt2k's processed"
			  " via ...limited()\n");
  }

  mpz_set(R_IN_PROGRESS,R_IN_PROGRESS_LIMITED);
  if (debug_flag > 0) { printf("R_IN_PROGRESS=R_IN_PROGRESS_LIMITED\n"); }

  clock_mod_where_start = clock();
  /* clocks_mod_where = (clock() - clock_mod_where_start); */

  for (portion_count = 1; portion_count <= portions_boost_marker; portion_count++) {
	 clocks_mod_where = (clock() - clock_mod_where_start);
	 elapsed_secs = clocks_mod_where / CLOCKS_PER_SEC;
	 if (debug_flag > 0) {
		printf("%ld secs - Loop iter with portion_count=%d started.\n",
				 elapsed_secs,portion_count);
	 }
	 mpz_powm(R_IN_PROGRESS, R_IN_PROGRESS, FT2K, N); 
	 if (ft2k_plus_gt2k_sign > 0) { exp_done += ft2k; }
	 /*
		 exponent_size = mpz_sizeinbase(HALF_OF_TWO_TO_K,10);
		  gmp_printf("EXPONENT (reduction 1) has size %d\n",exponent_size);
	 */
  }
  
  if (debug_flag > 0) {
	 printf("...longway() exp_done=%lld at point where switch to boosted\n",exp_done);
  }

  for (portion_count = portions_boost_marker + 1; portion_count < portions; portion_count++) {
	 clocks_mod_where = (clock() - clock_mod_where_start);
	 elapsed_secs = clocks_mod_where / CLOCKS_PER_SEC;
	 if (debug_flag > 0) {
		printf("%ld secs - Loop iter (boosted) with portion_count=%d started.\n",
				 elapsed_secs,portion_count);
	 }
	 mpz_powm(R_IN_PROGRESS, R_IN_PROGRESS, FT2K_BOOSTED, N); 
	 if (ft2k_plus_gt2k_sign > 0) { exp_done = exp_done + 2*ft2k; }
  }

  if (debug_flag > 0) {
	 printf("...longway() exp_done=%lld after ft2k's processed\n",exp_done);
  }

  mpz_set(R, R_IN_PROGRESS);

  if (debug_flag > 0) {
	 size10long = mpz_sizeinbase(R,10);
	 printf("...longway() R has size %ld\n",size10long);

  }

  mpz_clear(FT2K);
  mpz_clear(GT2K);
  mpz_clear(FT2K_BOOSTED);
  mpz_clear(R_IN_PROGRESS);
  /* return value for convenience only, see mpz R usually */
  return 0;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(mod_where_multiple_of_1365_in_exponent_longway__doc__,
"mod_where_multiple_of_1365_in_exponent with no optimisations");

/* Wrapper around the underlying C function */
static PyObject *
py_mod_where_multiple_of_1365_in_exponent_longway(PyObject *self, PyObject *args)
{
   unsigned long mwresult;
	unsigned long exp_l;
	/* "args" signature is long */
	/* ':mod_where_multiple_of_1365_in_exponent_longway' for error messages */
	if (!PyArg_ParseTuple(args, "l:mod_where_multiple_of_1365_in_exponent_longway", &exp_l))
		return NULL;

	if (exp_l < 2 || exp_l > ULONG_MAX) {
	  return NULL;
	}

	mwresult = \
	  mod_where_multiple_of_1365_in_exponent_longway((unsigned long long)exp_l);

	/* Convert from a C type to a Python type */
	return PyInt_FromLong((long)mwresult);
}

unsigned long process_expk(unsigned int e_inc, unsigned int proc_expk_optimisation) {

  int mod_where_return = -1;
  unsigned long long expk = 1;
  unsigned long long rshift_size10 = 0;
  extern unsigned long long exp_shift_start;
  unsigned long long k_minus_one = 0;
  unsigned int       nonresidue_val;
  unsigned int exponent_size10 = 0;
  unsigned long size10long = 0;
  unsigned long long outD36len = 0;
  extern int process_expk_optimisation;

  mpz_t		K;
  mpz_t		TWO_TO_K, HALF_OF_TWO_TO_K;
  mpz_t		NMO, NMO_HALVED;
  extern mpz_t		A;
  extern mpz_t		N;
  extern mpz_t		R1, R2, R; 
  mpz_t		RSHIFT,RPO;
  mpz_t		D,DMO;

  char      *outD36;
  char      *outD36tail;

  long clock_nmohalved_start = 0;
  long clocks_nmohalved = 0;
  long elapsed_secs_nmohalved = 0;

  /* expk = k_minus_one + 1 + e_inc; */
  expk = multiple_of_1365 + exp_shift_start + e_inc;
  mpz_init_set_ui(K,expk);
  k_minus_one = expk - 1;
  if (debug_flag > -999) {
	 printf("e_inc+exp_shift_start+multiple_of_1365=%d+%lld+%ld=%lld.\n",
			  e_inc,exp_shift_start,multiple_of_1365,expk);
  }
  process_expk_optimisation = proc_expk_optimisation;
  mpz_init(HALF_OF_TWO_TO_K);
  mpz_init(TWO_TO_K);
  mpz_ui_pow_ui(HALF_OF_TWO_TO_K,2,k_minus_one);
  mpz_mul_ui(TWO_TO_K,HALF_OF_TWO_TO_K,2);
  mpz_init(NMO);
  mpz_init(N);
  mpz_mul(NMO,TWO_TO_K,MULTIPLE_OF_1365);
  if (debug_flag > 0) {
	 printf("NMO initialised and populated.\n");
	 size10long = mpz_sizeinbase(NMO,10);
	 printf("NMO has size %ld\n",size10long);
  }
  mpz_add_ui(N,NMO,1);
  if (debug_flag > 0) {
	 printf("N initialised and populated.\n");
	 /*  mpz_init_set_str(N, "41", 10);*/
	 size10long = mpz_sizeinbase(N,10);
	 printf("N has size10 of %ld\n",size10long);
  }

  mpz_init(A);  /* A is the nonresidue value which we will raise to a power using EXPONENT */
  nonresidue_val = nonresidue_value(1000);
  mpz_set_ui(A, nonresidue_val);
  /*  mpz_init_set_str(A, "2", 10); */
  if (debug_flag > 0) {
	 printf("A=%d is being used as nonresidue value.\n",nonresidue_val);
  }

  mpz_init(NMO_HALVED);
  if (debug_flag > 0) {
	 printf("NMO_HALVED will be formed next using mpz_mul_ui().\n");
  }
  clock_nmohalved_start = clock();
  mpz_mul_ui(NMO_HALVED,HALF_OF_TWO_TO_K,multiple_of_1365);
  clocks_nmohalved = (clock() - clock_nmohalved_start);
  elapsed_secs_nmohalved = clocks_nmohalved / CLOCKS_PER_SEC;

  fprintf(outfnotes,"process_expk() NMO_HALVED forming - elapsed"
			 " %ld seconds\n",elapsed_secs_nmohalved);
  fflush(outfnotes);

  exponent_size10 = mpz_sizeinbase(NMO_HALVED,10);

  if (debug_flag > 0) {
	 printf("process_expk() NMO_HALVED forming - elapsed %ld seconds\n",
			  elapsed_secs_nmohalved);
	 printf("EXPONENT (original)=NMO_HALVED has size %d\n",exponent_size10);
  }

  gmp_fprintf(outfplanned,
				    "expk=%d planned processing involves nonresidue A=%Zd\n"
				    "...which is then raised to exponent having %d decimal digits"
				  " labelled NMO_HALVED.\n",
				  expk,A,exponent_size10);
  /*
	   "...which is then raised to exponent %Zd.\n",
		  expk,A,NMO_HALVED);
  */
  fflush(outfplanned);

  if (debug_flag > 0) {
	 printf("EXPONENT (original)=NMO_HALVED summary in .planned file\n");
  }

  /* next few lines are a very loose attempt to bound things (based
	    on exponent size) so that memory requirements
		 are no more than 4 gig (4096 megabytes */
  if (exponent_size10 > 9999999) {
	 slice_size_max_2exp = 5; /* 6 just too much, 8 way too much */
  }

  mpz_init(R);  /* R is the remainder/residue of the congruence */
  mpz_init(R1);
  mpz_init(R2);

  if (proc_expk_optimisation < 10) {
	 mod_where_return = mod_where_multiple_of_1365_in_exponent_longway(expk);
	 if (debug_flag > 0) {
		printf("mod_where_multiple_of_1365_in_exponent_longway() completed.\n");
	 }
  } else if (proc_expk_optimisation < 20) {
	 /* optimisation 1 */
	 /* mod_where_return = mod_where_multiple_of_1365_in_exponent(); */
	 mod_where_return = mod_where_multiple_of_1365_in_exponent_longway(expk);
	 if (debug_flag > 0) {
		printf("mod_where_multiple_of_1365_in_exponent_longway is processed"
				 " using single optimisations.\n");
	 }
  } else {
	 /* optimisation 1 */
	 /* optimisation 2 */
	 /* mod_where_return = mod_where_multiple_of_1365_in_exponent(); */
	 mod_where_return = mod_where_multiple_of_1365_in_exponent_longway(expk);
	 if (debug_flag > 0) {
		printf("mod_where_multiple_of_1365_in_exponent_longway processed"
				 " using both optimisations.\n");
	 }
  }

  if (debug_flag > 0) {
	 printf("mod_where_multiple_of_1365_in_exponent_longway()"
			  " returned %d.\n",mod_where_return);
  }
  fprintf(outfnotes,"mod_where_multiple_of_1365_in_exponent_longway()"
			 " returned %d.\n",mod_where_return);
  fflush(outfnotes);

  if (mod_where_return != 0) { return mod_where_return; }

  mpz_clear(R1);
  mpz_clear(R2);
  /* mpz_powm(R, A, NMO_HALVED, N);
	    Above line fails ...as EXPONENT has size 873049404; which results in ...
		 GNU MP: Cannot allocate memory (size=185613271040) */
  if (debug_flag > 0) {
	 printf("RSHIFT calculation next now that R initialised and populated.\n");
  }

  mpz_init(RSHIFT);
  mpz_sub(RSHIFT,N,R);
  /* printf("RSHIFT initialised and populated.\n"); */
  if (mpz_cmp_ui(RSHIFT,1) == 0) {
	 printf("expk=%lld RSHIFT=1 checked using mpz_cmp_ui()"
			  " your lucky day Jim Dixon!\n",expk);
	 fprintf(outfnotes,"expk=%lld RSHIFT=1 checked using mpz_cmp_ui()"
			  " your lucky day Jim Dixon!\n",expk);
	 fflush(outfnotes);
  } else {
	 rshift_size10 = mpz_sizeinbase(RSHIFT,10);
	 if (rshift_size10 < 81) {
		gmp_printf("expk=%lld RSHIFT=%Zd\n",expk,RSHIFT);
		gmp_fprintf(outfnotes,"expk=%lld RSHIFT=%Zd\n",expk,RSHIFT);
		fflush(outfnotes);
		/*  mpz_get_str(outRSHIFT,10,RSHIFT); */
	 } else {
		printf("expk=%lld RSHIFT has %lld decimal digits (approx.)\n",
				 expk,rshift_size10);
		fprintf(outfnotes,"expk=%lld RSHIFT has %lld decimal"
				  " digits (approx.)\n",expk,rshift_size10);

		fflush(outfnotes);
	 }
  }
  gmp_fprintf(outfnotes,
				  "expk=%d workings A=%Zd explained ... %Zd(RSHIFT) + (R)%Zd gives"
				  " N=%Zd using",expk,A,RSHIFT,R,N);
  gmp_fprintf(outfnotes,
				  " nonresidue A=%Zd which is then"
				  " raised to exponent %Zd(NMO_HALVED).\n",A,NMO_HALVED);
  fflush(outfnotes);

  mpz_init(RPO);
  mpz_add_ui(RPO,R,1);

  if (mpz_cmp(RPO,N) == 0) {
	 fprintf(outfexpk,"%lld achieved test result -1"
				" so deterministic if N really is proth.\n",expk);
	 fflush(outfexpk);
	 if (debug_flag > -999) {
		/*
		  printf("Deterministic: If N=%Zd is truly a Proth Number, then N has been **proven Prime**\n", N);
		  printf("...using deterministic Proth Test that returned the required '-1'.\n");
		*/
		printf("%lld achieved test result -1 so deterministic if N really is proth.\n",
				 expk);
	 }
  }

  mpz_init(D);
  mpz_init(DMO);
  mpz_mul_2exp(DMO,MULTIPLE_OF_1365,expk);
  mpz_add_ui(D,DMO,1);
  mpz_clear(K);
  outD36 = mpz_get_str(NULL,36,D);
  outD36len = strlen(outD36);

  printf("D36 ends %s and that concludes this test. *** End of exponent test ***\n",
			&outD36[(outD36len-10)]);

  asprintf(&outD36tail,"%s",&outD36[(outD36len-10)]);
  fprintf(outfout36,"EXPK=%lld gets %s is TAIL of full base 36 representation D36=%s\n",
			 expk, outD36tail, outD36);
  fflush(outfout36);

  mpz_clear(TWO_TO_K);
  mpz_clear(HALF_OF_TWO_TO_K);
  mpz_clear(NMO);
  mpz_clear(NMO_HALVED);
  mpz_clear(N);
  mpz_clear(A);

  mpz_clear(RSHIFT);
  mpz_clear(R);
  mpz_clear(RPO);

  mpz_clear(D);
  mpz_clear(DMO);

  return 0;
}


/* Short description of the function (one liner). */
PyDoc_STRVAR(process_expk__doc__,"Process gigantic exponent.");

/* Wrapper around the underlying C function */
static PyObject *
py_process_expk(PyObject *self, PyObject *args)
{
   unsigned long procresult;
	unsigned int exp_inc;
	int p_expk_optimisation = -99;
	extern int process_expk_optimisation;

	/* "args" signature is integer,integer */
	/* ':process_expk' for error messages */
	if (!PyArg_ParseTuple(args, "i|i:process_expk", &exp_inc, &p_expk_optimisation))
		return NULL;

	if (p_expk_optimisation >= 0) {
	  process_expk_optimisation = p_expk_optimisation;
	} else {
	  process_expk_optimisation = 0;
	  /* zero, indicating 'no optimisation' should be set rather
		  than possible undefined value extern might hold currently */
	}

	if (exp_inc < 0) {
	  return NULL;
	}

	procresult = process_expk(exp_inc, p_expk_optimisation);

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long)procresult);
}

/* static prefix if you only need to access it directly from python */
static long setz_esl_init(unsigned int m1365,  unsigned long long e_shift_start) {
	extern FILE		*outfexpk;
	extern FILE		*outfout36;
	extern FILE		*outfplanned;
	extern FILE		*outfnotes;
	extern FILE		*outfrinprogress;
	extern unsigned int	debug_flag;
	extern unsigned long multiple_of_1365;
   extern unsigned long long exp_shift_start;
	extern mpz_t		ONE;
	extern mpz_t		THIRTEENSIXTYFIVE;
	extern mpz_t		FOURZERONINEFIVE;
	extern mpz_t		FIVEFOURSIXTY;
	extern mpz_t		MULTIPLE_OF_1365; 

	mpz_init_set_ui(ONE,1);
	mpz_init_set_ui(THIRTEENSIXTYFIVE,1365);
	mpz_init_set_ui(FOURZERONINEFIVE,4095);
	mpz_init_set_ui(FIVEFOURSIXTY,5460);

	multiple_of_1365 = m1365;
	if (debug_flag > 0) {
	  printf("esl_init() initialisation in progress using"
				" multiple_of_1365=%ld\n",multiple_of_1365);
	}

	exp_shift_start = e_shift_start;

	mpz_init_set_ui(MULTIPLE_OF_1365,multiple_of_1365);
	if (debug_flag > 0) {
	  gmp_printf("esl_init() initialisation in progress using"
					 " MULTIPLE_OF_1365=%Zd\n",MULTIPLE_OF_1365);
	}

  asprintf(&fexpk,"%s.expk",outfstem);
  outfexpk = fopen(fexpk,"w");

  asprintf(&fout36,"%s.out36",outfstem);
  outfout36 = fopen(fout36,"w");

  asprintf(&fnotes,"%s.notes",outfstem);
  outfnotes = fopen(fnotes,"w");

  fprintf(outfnotes,"esl_init() initialisation in progress"
			  " using multiple_of_1365=%ld\n",multiple_of_1365);
  fflush(outfnotes);

  asprintf(&fplanned,"%s.planned",outfstem);
  outfplanned = fopen(fplanned,"w");

  asprintf(&frinprogress,"%s.rinprogress",outfstem);
  outfrinprogress = fopen(frinprogress,"w");

	return 0;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(setz_esl_init__doc__,
"Initialisation. Here you would include opening any files that you expect to write to by one/several functions");

/* Wrapper around the underlying C function */
static PyObject *
py_setz_esl_init(PyObject *self, PyObject *args)
{
   long initresult;
	extern unsigned int	debug_flag;
	extern unsigned int	slice_size_max_2exp;
   /* extern unsigned long long exp_shift_start; */
	unsigned int mult1365;
   unsigned long long e_shift_start;
	unsigned int d_flag;
	unsigned int max2exp;
	unsigned int zerosappend = 0;
	/* "args" signature is integer,long then optional
		integer, integer, integer */
	/* ':setz_esl_init' for error messages */
	if (!PyArg_ParseTuple(args, "il|iii:setz_esl_init", &mult1365,
								 &e_shift_start, &d_flag, &max2exp, &zerosappend))
		return NULL;
	
	/* where e_shift_start <= 31000 take this to mean that e_shift_start
		is in fact gigantic, and argument is a shorthand for larger value */
	if (zerosappend > 0) {
	  /* append zeros to the number supplied as an argument */
	  e_shift_start = e_shift_start*10^zerosappend;
	  /* So 3028 as 2nd arg and 6 as 5th argument would see
		  shift_start as 3,028,000,000 which is gigantic.
		  So 30280 as 2nd arg and 5 as 5th argument achieves same  */
	}

	if (d_flag < 1000) {
	  debug_flag = d_flag; /* set extern debug_flag */
	}

	if (max2exp < 1 || max2exp > 10) {
	  return NULL;
	}
	if (max2exp > 0) {
	  slice_size_max_2exp = max2exp; /* set extern slice_size_max_2exp */
	  /* slice_size_max_2exp = 5; 6 just too much, 8 way too much */
	}

	initresult = setz_esl_init(mult1365, e_shift_start);

	/* Convert from a C type to a Python type */	
	return PyInt_FromLong((long) initresult);
}

/* static prefix if you only need to access it directly from python */
static long setz_esl_closedown(void) {
	extern FILE		*outfexpk;
	extern FILE		*outfout36;
	extern FILE		*outfplanned;
	extern FILE		*outfnotes;
	extern FILE		*outfrinprogress;
	extern unsigned long multiple_of_1365;
	extern mpz_t		ONE;

	fflush(outfexpk);
	fclose(outfexpk);
	fflush(outfout36);
	fclose(outfout36);
	fflush(outfplanned);
	fclose(outfplanned);
	mpz_clear(ONE);
	fprintf(outfnotes,"esl_closedown() complete for all processing that used multiple_of_1365=%ld\n",
			multiple_of_1365);
	fflush(outfnotes);
	fclose(outfnotes);
	fflush(outfrinprogress);
	fclose(outfrinprogress);
	return 0;
}

/* Short description of the function (one liner). */
PyDoc_STRVAR(setz_esl_closedown__doc__,
"You might consider this de-initialisation and here you would include closing any files opened during initialisation");

/* Wrapper around the underlying C function */
static PyObject *
py_setz_esl_closedown(PyObject *self, PyObject *args)
{
   long closeresult;
	/* "args" - there are none */
	
	closeresult = setz_esl_closedown();

	/* Convert from a C type to a Python type */
	return PyInt_FromLong((long) closeresult);
}

/* Short description of the extension module (one liner). */
PyDoc_STRVAR(setz_exp_small_to_the_gigantic_unopt__doc__,
"Utilities for setZ (long) to support A^K where A is small and K is gigantic");

/* List of methods defined in this module.
	2nd argument in each entry should begin py_ */
/* "METH_VARARGS" or "METH_NOARGS" for 3rd argument of each method listed.
	Last entry in the list should be a {NULL, NULL} entry */
static PyMethodDef setz_exp_small_to_the_gigantic_unopt_methods[] = {
	{"setz_esl_init",  py_setz_esl_init, METH_VARARGS, setz_esl_init__doc__},
	{"setz_esl_closedown",  py_setz_esl_closedown, METH_NOARGS, setz_esl_closedown__doc__},
	{"size2gmp",  py_size2gmp, METH_VARARGS, size2gmp__doc__},
	{"size2longlong",  py_size2longlong, METH_VARARGS, size2longlong__doc__},
	{"calculate_split",  py_calculate_split, METH_VARARGS, calculate_split__doc__},
	{"calculate_split_wrapper_int",  py_calculate_split_wrapper_int,
	 METH_VARARGS, calculate_split_wrapper_int__doc__},
	{"calculate_split_wrapper_longlong",  py_calculate_split_wrapper_longlong,
	 METH_VARARGS, calculate_split_wrapper_longlong__doc__},
	{"mod_where_multiple_of_1365_in_exponent_limited",
	 py_mod_where_multiple_of_1365_in_exponent_limited,
	 METH_VARARGS, mod_where_multiple_of_1365_in_exponent_limited__doc__},
	{"nonresidue_value", py_nonresidue_value, METH_VARARGS, nonresidue_value__doc__},
	{"mod_where_multiple_of_1365_in_exponent_longway",
	 py_mod_where_multiple_of_1365_in_exponent_longway,
	 METH_VARARGS, mod_where_multiple_of_1365_in_exponent_longway__doc__},
	{"process_expk", py_process_expk, METH_VARARGS, process_expk__doc__},
	{NULL, NULL}
};

/* For this module "setz_exp_small_to_the_gigantic_unopt" the api expects
	an initialization function named "initsetz_exp_small_to_the_gigantic_unopt". */
/* PyMODINIT_FUNC on next line might aid portability */
PyMODINIT_FUNC
initsetz_exp_small_to_the_gigantic_unopt(void)
{
	Py_InitModule3("setz_exp_small_to_the_gigantic_unopt", setz_exp_small_to_the_gigantic_unopt_methods,
                   setz_exp_small_to_the_gigantic_unopt__doc__);
}


