#include <stdio.h>
#include <math.h>
#include <gmp.h>
#include <string.h>
#include <stddef.h>
#include <sysexits.h>
#include <stdlib.h>

/*
    This file is part of primerange.

    primerange is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    primerange is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with primerange.  If not, see <http://www.gnu.org/licenses/>.

    (c) 2011 Gary Wright
    http://wrightsolutions.co.uk/contact
    http://identi.ca/gnubyexample
    http://gnumbers.blogspot.com/

*/

  mpz_t         D10;
  mpz_t         DM5;
  mpz_t         K, L, M, KBYL;
  mpz_t         C;
  mpz_t         BIVARIATE_CONSTANT;
  mpz_t         BSQRM4AC, BSQR, TWENTYFOURC;
  mpz_t         QUADFORM_NUMERATOR_FROM_DIFF;
  mpz_t         QUADFORM_NUMERATOR_FROM_SUM;
  mpz_t         QUADFORM_NUMERATOR_SHIFT;
  mpz_t         QUAD_RESULT_FROM_N_DIFF;
  mpz_t         QUAD_RESULT_FROM_N_SUM;
  mpz_t         R1, R2, ROOT_DIFF;
  mpz_t         FIRST, SECOND, FIRSTBYSECOND;
  mpz_t         SQRTD10;
  char          *outK;
  char          *outL;
  char          *outM;
  char          *outFIRST;
  char          *outSECOND;
  char          *outFIRSTBYSECOND;
  char          *outBSQRM4AC;
  char          *mode;
  const char  outfstem [15] = "/tmp/primerange";
  char        *finput = "/tmp/primerange.composite2largefac";
  const char  *delimiters = " \n";

  unsigned int  prime_rating = 0;
  unsigned int  nbr_of_types = 1;
  unsigned int  algorithms = 1;
  unsigned int  perfect_square_target = 1;
  unsigned int  perfect_square_count = 0;
  unsigned int  search_percent = 10;
  unsigned int  kl_flag = 0;
/* set debug_flag = 0 to be verbose only when indicated by 'mode' */
  unsigned int  debug_flag = 0;
/* 4 thousand million is actually 4294967296 but +/- is near enough */
  const unsigned int fourbillion = 4294967295;
  const int     twobillion = 2147483648;

  unsigned int  even_i;
  unsigned int  even_i_by_6;
  int           even_outer;
  /*;
  int           even_lower;
  int           even_upper;
  */
/* if you cannot fit even_diff_k_l into a 4 billion limit of storage
   then you probably should be worrying about runtime */
  int           even_diff_k_l;
  int           b;
  double        sqrt2test;

  char        *fkl;
  char        *fnotes;
  char        *f1;
  char        *f2;
  char        *fcomp;
  char buf [3000000];
  char *givenD10;
  FILE        *outfkl;
  FILE        *outfnotes;
  FILE        *outf1;
  FILE        *outf2;
  FILE        *outfcomp;

/* Typically this program will be run with arguments 1 1
which says there is just one type of number in the input file
'composite', and only 1 algorithm should be tried to identify
integer factors.

If the first argument is other than 1 then the program assumes
mixed input. That means that there might be some primes in there
and some composites also.

If the second argument is > 1 then several algorithms will be
attempted including trial division.

Note: Trial division will usually assume that the numbers have
already been tested for low valued prime factors by some
quarter trial limited trial division.

Second argument > 1 is useful for comparing the timings of the
different algorithms to verify which is best for your use.

The third argument says how many perfect square hits to achieve
before program stops processing.

If the third argument is 1 then do not seek multiple answers in
the search for perfect squares. If you know a number has two divisors
then you can be confident in giving 999 for third argument if you want.
Having a value greater than 1 for third argument is going to extend
the runtime certainly.

The fourth argument sets a percentage limit on the search range.
For large numbers it might be unwise to run with a fourth argument
greater than 25. Runtime will be long for large numbers if you fail
to limit the search range (avoid using 100 for fourth argument!).

Typical usage of this program:
  mpz_two_large_factors_of_known_form.o 1 1 minimal 1 1  
  mpz_two_large_factors_of_known_form.o 1 1 roots 1 5  
Testing usage of this program:
  mpz_two_large_factors_of_known_form.o 1 1 roots 4 100

*/

int main(int argc, char *argv[]) { 

  asprintf(&fkl,"%s.kl",outfstem);
  outfkl = fopen(fkl,"w");
  asprintf(&fnotes,"%s.notes",outfstem);
  outfnotes = fopen(fnotes,"w");
  asprintf(&f1,"%s.1",outfstem);
  outf1 = fopen(f1,"w");
  asprintf(&f2,"%s.2",outfstem);
  outf2 = fopen(f2,"w");
  asprintf(&fcomp,"%s.composite",outfstem);
  outfcomp = fopen(fcomp,"w");

  mode = "minimal";
  if (argc > 1) {
    nbr_of_types = atoi(argv[1]);
    if (argc > 2) {
      algorithms = atoi(argv[2]);
      printf("Evaluating using %d algorithm(s)\n",algorithms);
      if (argc > 3) {
	mode = argv[3];
	printf("Program called with %d arguments\n",argc-1);
	if (strcmp(mode,"roots") == 0) {
	  kl_flag = 1;
	} else if (strcmp(mode,"debug") == 0) {
	  debug_flag = 1;
	  /* the fact that you want verbose output suggests
	     that you would also want kl reporting.
	     WARNING: The 'debug' option results in a lot
	     of memory transfer between mpz and readable char
	     output. Some of this is uncleared and only designed
	     for use in a test environment with 4GB of memory
	     or more.
	  */
	  kl_flag = 1;
	}
	if (argc > 4) {
	  perfect_square_target = atoi(argv[4]);
	  if (argc > 5) {
	    search_percent = atoi(argv[5]);
	  }
	}
      }
    }
  }
  /* debug_flag is the only flag which cannot be switched 
     directly via the command line. However when mode is not 'roots'
     or 'debug' then the program switches debug_flag to zero
  */
  /*
  if (strcmp(mode,"roots") == 0) {
  */
    /* debug_flag = 1; 
       "roots" encountered so leave debug_flag as set in program itself 
    */
  /*
  }
  */
  if (kl_flag = 1) {
    /* debug_flag = 1; 
       "roots" or "debug" leave debug_flag as set in program itself.
    */
  } else {
    debug_flag = 0;
  }

  switch(nbr_of_types) {
  case 0:
    exit(EX_USAGE);
    break;
  case 1:
    mpz_init(K);
    mpz_init(L );
    mpz_init(M);
    mpz_init(C);
    /*
    mpz_init(BSQRM4AC);
    mpz_init(BSQR);
    */
    mpz_init(TWENTYFOURC);
    mpz_init(QUADFORM_NUMERATOR_FROM_DIFF);
    mpz_init(QUADFORM_NUMERATOR_FROM_SUM);
    mpz_init(QUADFORM_NUMERATOR_SHIFT);
    mpz_init(QUAD_RESULT_FROM_N_DIFF);
    mpz_init(QUAD_RESULT_FROM_N_SUM);
    /*
    mpz_init(R1);
    mpz_init(R2);
    */
    mpz_init(ROOT_DIFF);
    mpz_init(FIRST);
    mpz_init(SECOND);
    mpz_init(FIRSTBYSECOND);
    mpz_init(SQRTD10);
    process_file_composite2largefac();
    break;
  case 2:
    printf("Expecting nbr_of_types: %d\n",nbr_of_types);
    /* no break; thanks */
  default:
    finput = "/tmp/primerange.probprime";
    printf("Reading mixed input from .probprime file\n");
    process_file_mixed_input();
  } /* end case */

  fclose(outfkl);
  fclose(outfnotes);
  fclose(outf1);
  fclose(outf2);
  fclose(outfcomp);


}

int process_file_composite2largefac() {
  /*
  char *outK;
  char *outL;
  */
  FILE *inf = fopen(finput, "r");
  while (fgets(buf,3000000,inf) != NULL) {
    givenD10 = strtok(buf,delimiters);
    mpz_set_str(D10,givenD10,10);
    fprintf(outfkl, "%s",givenD10);
    printf("%s processing begun...\n",givenD10);
    mpz_sub_ui(DM5, D10, 5);
    if (mpz_tdiv_q_ui(M, DM5, 6) > 0) {
      printf("%s processing found unexpected input number!\n",givenD10);
    } else {
      outM = mpz_get_str(NULL,10,M);
      mpz_add_ui(BIVARIATE_CONSTANT,M,1);
      /* BIVARIATE_CONSTANT is always positive */
      if (mpz_odd_p(BIVARIATE_CONSTANT) > 0) {
	printf("%s processing hit an issue as"
	       " BIVARIATE_CONSTANT is odd rather than even!\n",givenD10);
      } else {
	if (debug_flag == 1) {
	  search_heavy_io(2);
	  if (perfect_square_count < perfect_square_target) {
	    search_heavy_io(-2);
	  }
	} else {
	  search(2);
	  if (perfect_square_count < perfect_square_target) {
	    search(-2);
	  }
	}
	/* following line just pushes a new line really */
	fprintf(outfkl, " \n",givenD10);
      } /* end else */
    } /* end else */
    perfect_square_count = 0;
  } /* end while */   
}

int search(int step) {
  /* 	even_outer = fourbillion;
	mpz_tdiv_q_ui(even_upper,BIVARIATE_CONSTANT,6); */
  int quot = 0;  /* quotient of 6/step */
  mpz_sqrt(SQRTD10,D10);
  even_outer = mpz_get_ui(SQRTD10);

  for (even_i = 0; even_i < even_outer;even_i += step) {
    /* if (factor_of_m_or_prob_prime(two_three_remove(even_i)) == 0) { */
    /*	  if (passes_b_criteria(two_three_remove(even_i)) == 0) { */
    /* continue;
    */
    if (step > 0) {
      even_diff_k_l = even_i;
    } else {
      even_diff_k_l = -1 * even_i;
    }
    /* even_i_by_6 = abs(even_diff_k_l) */
    even_i_by_6 = even_i * 6;
    mpz_init(BSQRM4AC);
    mpz_init(BSQR);
    mpz_ui_pow_ui(BSQR,even_i_by_6,2);

    /* C will have the wrong sign in the next line 
       which explains mpz_mul_si using -24 that follows. */
    if (step > 0) {
      mpz_sub_ui(C,BIVARIATE_CONSTANT,even_i);
    } else {
      mpz_add_ui(C,BIVARIATE_CONSTANT,even_i);
    }
    /* above is coded as an if else as mpz_sub_si does
       not exist in the version of gmp I am using */

    if (mpz_mod_ui(BSQRM4AC,C,6) == 0) {
      /* only interested in C that is divisible by 6 */
      process_c();
      if (quot == 0) {
	quot = 6 / step;
	step = abs(step) * quot;
	fprintf(outfnotes, "%s iterating until %d"
	  " using step of "
	  "%d\n",givenD10,even_outer,step);
	fflush(outfnotes);
      }

    }

    if (perfect_square_count >= perfect_square_target) {
      printf("%s process has achieved target ..."
	     " quitting now.\n",givenD10);
      break;
    }

    mpz_clear(BSQRM4AC);
    mpz_clear(BSQR);
  }   /* end for loop */

}

int process_c() {
    mpz_mul_si(TWENTYFOURC,C,-24);
    mpz_sub(BSQRM4AC,BSQR,TWENTYFOURC);
    if (mpz_mod_ui(QUADFORM_NUMERATOR_SHIFT,BSQRM4AC,144) > 0) {
      return;
    }
    if (mpz_perfect_square_p(BSQRM4AC)) {
      outBSQRM4AC = mpz_get_str(NULL,10,BSQRM4AC);
      printf("%s found perfect"
	     " square=%s using even_diff_k_l="
	     "%d!\n",givenD10,outBSQRM4AC,even_diff_k_l);
      fprintf(outfkl, ",%s",outBSQRM4AC);
      fflush(outfkl);
      if (kl_flag == 1) {
	mpz_sqrt(QUADFORM_NUMERATOR_SHIFT,BSQRM4AC);
	/* Numerator form: -b + sqrt() */
	mpz_sub_ui(QUADFORM_NUMERATOR_FROM_DIFF,QUADFORM_NUMERATOR_SHIFT,even_i_by_6);
	if (mpz_tdiv_q_ui(QUAD_RESULT_FROM_N_DIFF,QUADFORM_NUMERATOR_FROM_DIFF,12) == 0) {
	  if (even_diff_k_l > 0) {
	    /* mpz_neg(R2,QUAD_RESULT_FROM_DIFF); */
	    mpz_set(K,QUAD_RESULT_FROM_N_DIFF);
	  } else {
	    /* mpz_neg(R1,QUAD_RESULT_FROM_DIFF); */
	    mpz_set(L,QUAD_RESULT_FROM_N_DIFF);
	  }
	}
	/* Numerator form: -b - sqrt() */
	mpz_add_ui(QUADFORM_NUMERATOR_FROM_SUM,QUADFORM_NUMERATOR_SHIFT,even_i_by_6);
	if (mpz_tdiv_q_ui(QUAD_RESULT_FROM_N_SUM,QUADFORM_NUMERATOR_FROM_SUM,12) == 0) {
	  if (even_diff_k_l > 0) {
	    /* mpz_neg(R2,QUAD_RESULT_FROM_SUM); */
	    mpz_set(L,QUAD_RESULT_FROM_N_SUM);
	  } else {
	    /* mpz_neg(R1,QUAD_RESULT_FROM_SUM); */
	    mpz_set(K,QUAD_RESULT_FROM_N_SUM);
	  }
	}
	outK = mpz_get_str(NULL,10,K);  
	outL = mpz_get_str(NULL,10,L);  
	mpz_mul(KBYL,K,L);
	if (mpz_sgn(KBYL) > 0) {
	  fprintf(outfkl, ",%s",outK);
	  fprintf(outfkl, ",%s",outL);
	  /*
	    mpz_sub(ROOT_DIFF,R2,R1); 
	    if (mpz_cmp(ROOT_DIFF,even_diff_k_l) == 0) {
	    fprintf(outfkl, ",%s",outK);
	    }
	  */
	  mpz_set_ui(FIRST,1);
	  mpz_addmul_ui(FIRST,K,6);
	  outFIRST = mpz_get_str(NULL,10,FIRST);  
	  mpz_set_si(SECOND,-1);
	  mpz_addmul_ui(SECOND,L,6);
	  outSECOND = mpz_get_str(NULL,10,SECOND);  
	  mpz_mul(FIRSTBYSECOND,FIRST,SECOND);
	  outFIRSTBYSECOND = mpz_get_str(NULL,10,FIRSTBYSECOND);  
	  fprintf(outfkl, ",%s factors as"
		  " %s x %s",outFIRSTBYSECOND,outFIRST,outSECOND);
	  fprintf(outfkl, " using even_diff_k_l="
		  "%d (< %d)",even_diff_k_l,even_outer);
	} /* end if mpz_sgn() */
      } /* end kl processing */
      perfect_square_count += 1;
    } else {
      if (debug_flag == 1) {
	printf("%s -v- process "
	       " even_diff_k_l=%d\n",givenD10,even_diff_k_l);
      }
    }
}


int process_file_mixed_input() {
  FILE *inf = fopen(finput, "r");
  while (fgets(buf,3000000,inf) != NULL) {
    /* printf("%s\n",buf); */
    strtok(buf,"=");
    givenD10 = strtok(NULL," ");
    mpz_set_str(D10,givenD10,10);
    prime_rating = mpz_probab_prime_p(D10,10);
    printf("2 is better than 1 and we get: %d\n",prime_rating);
    switch (prime_rating) {
    case 1:
      fprintf(outf1, "%s\n",givenD10);
      break;
    case 2:
      fprintf(outf2, "%s\n",givenD10);
      break;
    default:
      fprintf(outfcomp, "%s\n",givenD10);
      break;
    } /* end switch */
  } /* end while */    
}


int search_heavy_io(int step) {
	/* 	even_outer = fourbillion;
		mpz_tdiv_q_ui(even_upper,BIVARIATE_CONSTANT,6); */
	mpz_sqrt(SQRTD10,D10);
	even_outer = mpz_get_ui(SQRTD10);

	fprintf(outfnotes, "%s iterating until %d"
		" using step of "
		"%d\n",givenD10,even_outer,step);
	fflush(outfnotes);

	for (even_i = 0; even_i < even_outer;even_i += 2) {
	  /* even_diff_k_l = two_three_remove(even_i); */
	  /*
	  printf("%s using even_i=%d after"
		 " two_three_remove()\n",givenD10,even_diff_k_l);
	  */
	  /* if (factor_of_m_or_prob_prime(two_three_remove(even_i)) == 0) { */
	  /*
	  if (passes_b_criteria(two_three_remove(even_i)) == 0) {
	    continue;
	  }
	  */
	  /*
	  even_diff_k_l = two_three_remove(even_i);
	  sqrt2test = sqrt(even_diff_k_l);
	  if (sqrt2test != (int)sqrt2test) {
	    printf("skipping even_i=%d after looking"
		   " at sqrt(two_three_remove\n",even_diff_k_l);
	    continue;
	  }
	  */
	  if (step > 0) {
	    even_diff_k_l = even_i;
	  } else {
	    even_diff_k_l = -1 * even_i;
	  }
	  /* even_i_by_6 = abs(even_diff_k_l) */
	  even_i_by_6 = even_i * 6;
	  mpz_init(BSQRM4AC);
	  mpz_init(BSQR);
	  mpz_ui_pow_ui(BSQR,even_i_by_6,2);
	  outBSQRM4AC = mpz_get_str(NULL,10,BSQR);
	    printf("%s -v- process "
		   " BSQR=%s\n",givenD10,outBSQRM4AC);

	  /* C will have the wrong sign in the next line 
	     which explains mpz_mul_si using -24 that follows. */
	  if (step > 0) {
	    mpz_sub_ui(C,BIVARIATE_CONSTANT,even_i);
	  } else {
	    mpz_add_ui(C,BIVARIATE_CONSTANT,even_i);
	  }
	  /* above is coded as an if else as mpz_sub_si does
	     not exist in the version of gmp I am using */

	  if (mpz_mod_ui(BSQRM4AC,C,6) == 0) {
	    /* only interested in C that is divisible by 6 */
	    process_c_heavy_io();
	  }

	  if (perfect_square_count >= perfect_square_target) {
	    printf("%s process has achieved target ..."
		   " quitting now.\n",givenD10);
	    break;
	  }

	  mpz_clear(BSQRM4AC);
	  mpz_clear(BSQR);
	}   /* end for loop */
}


int process_c_heavy_io() {
	  outBSQRM4AC = mpz_get_str(NULL,10,C);
	    printf("%s -v- process (note the minus preceding"
		   " C) -C=%s\n",givenD10,outBSQRM4AC);

	  mpz_mul_si(TWENTYFOURC,C,-24);
	  mpz_sub(BSQRM4AC,BSQR,TWENTYFOURC);
	  if (mpz_mod_ui(QUADFORM_NUMERATOR_SHIFT,BSQRM4AC,144) > 0) {
	    return;
	  }
	  outBSQRM4AC = mpz_get_str(NULL,10,BSQRM4AC);
	  if (mpz_perfect_square_p(BSQRM4AC)) {
	    printf("%s found perfect"
		   " square=%s using even_diff_k_l="
		   "%d!\n",givenD10,outBSQRM4AC,even_diff_k_l);
	    fprintf(outfkl, ",%s",outBSQRM4AC);
	    fflush(outfkl);
	    if (kl_flag == 1) {
	      printf("%s kl processing\n",givenD10);
	      mpz_sqrt(QUADFORM_NUMERATOR_SHIFT,BSQRM4AC);
	      /* Numerator form: -b + sqrt() */
	      mpz_sub_ui(QUADFORM_NUMERATOR_FROM_DIFF,QUADFORM_NUMERATOR_SHIFT,even_i_by_6);
	      if (mpz_tdiv_q_ui(QUAD_RESULT_FROM_N_DIFF,QUADFORM_NUMERATOR_FROM_DIFF,12) == 0) {
		if (even_diff_k_l > 0) {
		  /* mpz_neg(R2,QUAD_RESULT_FROM_DIFF); */
		  mpz_set(K,QUAD_RESULT_FROM_N_DIFF);
		} else {
		  /* mpz_neg(R1,QUAD_RESULT_FROM_DIFF); */
		  mpz_set(L,QUAD_RESULT_FROM_N_DIFF);
		}
	      }
	      /* Numerator form: -b - sqrt() */
	      mpz_add_ui(QUADFORM_NUMERATOR_FROM_SUM,QUADFORM_NUMERATOR_SHIFT,even_i_by_6);
	      if (mpz_tdiv_q_ui(QUAD_RESULT_FROM_N_SUM,QUADFORM_NUMERATOR_FROM_SUM,12) == 0) {
		if (even_diff_k_l > 0) {
		  /* mpz_neg(R2,QUAD_RESULT_FROM_SUM); */
		  mpz_set(L,QUAD_RESULT_FROM_N_SUM);
		} else {
		  /* mpz_neg(R1,QUAD_RESULT_FROM_SUM); */
		  mpz_set(K,QUAD_RESULT_FROM_N_SUM);
		}
	      }
	      outK = mpz_get_str(NULL,10,K);  
	      outL = mpz_get_str(NULL,10,L);  
	      mpz_mul(KBYL,K,L);
 	      if (mpz_sgn(KBYL) > 0) {
		fprintf(outfkl, ",%s",outK);
		fprintf(outfkl, ",%s",outL);
		/*
		  mpz_sub(ROOT_DIFF,R2,R1); 
		  if (mpz_cmp(ROOT_DIFF,even_diff_k_l) == 0) {
		  fprintf(outfkl, ",%s",outK);
		  }
		*/
		mpz_set_ui(FIRST,1);
		mpz_addmul_ui(FIRST,K,6);
		outFIRST = mpz_get_str(NULL,10,FIRST);  
		mpz_set_si(SECOND,-1);
		mpz_addmul_ui(SECOND,L,6);
		outSECOND = mpz_get_str(NULL,10,SECOND);  
		mpz_mul(FIRSTBYSECOND,FIRST,SECOND);
		outFIRSTBYSECOND = mpz_get_str(NULL,10,FIRSTBYSECOND);  
		fprintf(outfkl, ",%s factors as"
			" %s x %s",outFIRSTBYSECOND,outFIRST,outSECOND);
		fprintf(outfkl, " using even_diff_k_l="
			"%d (< %d)",even_diff_k_l,even_outer);
	      } /* end if mpz_sgn() */
	    } /* end kl processing */
	    perfect_square_count += 1;
	    printf("%s incremented count so now perfect_square_count="
		   "%d\n",givenD10,perfect_square_count);

	  } else {
	    printf("%s -v- process "
		   " even_diff_k_l=%d\n",givenD10,even_diff_k_l);

	    outBSQRM4AC = mpz_get_str(NULL,10,TWENTYFOURC);
	    printf("%s process "
		   " 24C=%s\n",givenD10,outBSQRM4AC);

	    outBSQRM4AC = mpz_get_str(NULL,10,BSQRM4AC);
	    printf("%s -^- process rejecting"
		   " BSQRM4AC=%s\n",givenD10,outBSQRM4AC);
	  }

}


int two_three_remove(int full) {
  double divresult;

  /* Not intended as an efficient production algorithm, more
     as a helper function for testing questions in your analysis.
     Bit testing and gmp functions
     mpz_scan1 and mpz_tdiv_q_2exp might well be employed if
     you want something fast.
  */

  /* return (int)(full / 2); */
  /* printf("full starts out as %d\n",full); */
  divresult = full / 2.0;
  while (divresult > 1 && divresult == (int)divresult) {
    full = divresult;
    divresult = full / 2.0;
  }
  /* printf("full stage1 completed with full=%d\n",full); */
  divresult = full / 3.0;
  while (divresult > 1 && divresult == (int)divresult) {
    full = divresult;
    divresult = full / 3.0;
  }
  /* printf("full stage2 completed with full=%d\n",full); */
  return (int)full;
}


int two_three_remove_prob_prime(int full) {
  mpz_t         FULL;
  double divresult;

  /* Not intended as an efficient production algorithm, more
     as a helper function for testing questions in your analysis.
     Bit testing and gmp functions
     mpz_scan1 and mpz_tdiv_q_2exp might well be employed if
     you want something fast.
  */

  divresult = full / 2.0;
  while (divresult > 1 && divresult == (int)divresult) {
    full = divresult;
    divresult = full / 2.0;
  }
  divresult = full / 3.0;
  while (divresult > 1 && divresult == (int)divresult) {
    full = divresult;
    divresult = full / 3.0;
  }
  mpz_init_set_ui(FULL,full);
  if(mpz_probab_prime_p(FULL,10) > 0) {
    return full;
  }
  return 0;
}


int factor_of_m_or_prob_prime(int full) {
  if (full < 2) {
    return 1;
    /* this function is primarily used as a filter
       to remove even_i values for which processing
       is seen as not worthwhile. zero is worthwhile
       and give return 1. Give return 1 for 1 also
       to shortcut 'factor of m' always true, although
       we do not really expect 1 as an input value.
    */
  }

  if (debug_flag == 1) {
    printf("%s full is even_i after two_three_remove and"
	   " equals %d\n",givenD10,full);
  }

  if (mpz_tdiv_ui(M, full) == 0) {
    return 1;
    /* return 1 to indicate full is a divisor of m */
  }

  mpz_t         FULL;
  mpz_init_set_ui(FULL,full);
  if(mpz_probab_prime_p(FULL,10) > 0) {
    return 1;
    /* return 1 to indicate full is a probable prime */
  }

  /* returning 0 says full is neither a divisor of m
     nor a probable prime */
  return 0;
}


int factor_of_m(int full) {
  if (full < 2) {
    return 1;
    /* this function is primarily used as a filter
       to remove even_i values for which processing
       is seen as not worthwhile. zero is worthwhile
       and give return 1. Give return 1 for 1 also
       to shortcut 'factor of m' always true, although
       we do not really expect 1 as an input value.
    */
  }

  if (debug_flag == 1) {
    printf("%s full is even_i after two_three_remove and"
	   " equals %d\n",givenD10,full);
  }

  if (mpz_tdiv_ui(M, full) == 0) {
    return 1;
    /* return 1 to indicate full is a divisor of m */
  }

  /* returning 0 says full is not a divisor of m */
  return 0;
}


int passes_b_criteria(int full) {
  /* passes_b_criteria is a future implementation, that
     includes the functionality of factor_of_m_or_prob_prime()
     but without the printf() statement that makes assumption
     about caller.
  */

  /* Number pattern experiments. Try what you like in here.
  if (full == 23851) {
    return 1;
  }
  */

  if (factor_of_m(full) > 0) {
    return 1;
  }

  mpz_t         FULL;
  mpz_init_set_ui(FULL,full);
  if(mpz_probab_prime_p(FULL,10) > 0) {
    return 1;
    /* return 1 to indicate full is a probable prime */
  }

  /* returning 0 says full is neither a divisor of m
     nor a probable prime */
  return 0;
}
