#!/usr/bin/env python
#-*- coding: latin-1 -*-

# mdt25.py 081204
# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with This program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Download mdt25 rasters from IDEE WCS.

Usage:

python mtd25.py x1 y1 x2 y2 dst
python mtd25.py x y -w width [-h height] dst
python mtd25.py x y -r radius dst

Several 25 km x 25 km files are created if necessary and lower left
coordinates added to the file names.

Format is AsciiGrid with float32 pixel values and EPSG:23030 srs but
projection information is not stored. If gdal python bindings are available
then projection information is added to the files and GeoTIFF files with int16
pixel values and a VRT mosaic of them are generated.

"""

import os
import sys
import urllib
import getopt
from helper import add_coords

try:
    from gdalhelper import translate, add_srs, gdal
    from gdal_vrtmerge import gdal_vrtmerge
    gdal_available = True
except ImportError, e:
    print e
    gdal_available = False

def usage():
    """Print usage advices."""
    print __doc__

def processargs(args):
    """Process command line arguments.

    Return a tuple (x1, y1, x2, y2, dstname).

    """

    opts, args = getopt.gnu_getopt(args, 'h:w:r:')
    optdict =dict(opts)

    if '-r' in optdict:
        radius = int(optdict['-r'])
        x = int(args[0])
        y = int(args[1])
        x1 = x - radius
        y1 = y - radius
        x2 = x + radius
        y2 = y + radius
        dstname = args[2]

    elif '-w' in optdict:
        width = int(optdict['-w'])
        x1 = int(args[0])
        y1 = int(args[1])
        x2 = x1 + width
        if '-h' in optdict:
            height = int(optdict['-h'])
        else:
            height = width
        y2 = y1 + height
        dstname = args[2]

    else:
        x1 = int(args[0])
        y1 = int(args[1])
        x2 = int(args[2])
        y2 = int(args[3])
        dstname = args[4]

    return x1, y1, x2, y2, dstname

def download_mdt25(x1, y1, x2, y2, name=None):
    """Download a mdt25 raster file from IDEE WCS.

    Format is AsciiGrid with float32 pixel values and EPSG:23030 srs but
    projection information is stored only if python gdal bindings are 
    available. If name is None or '' then a name is automagically generated
    from coordinates.

    """

    if name is None or name == '':
        name = '%d_%d_%d_%d.asc' % (x1, y1, x2, y2)

    url = 'http://www.idee.es/wcs/IDEE-WCS-UTM30N/wcsServlet?' + \
          'SERVICE=WCS&' + \
          'REQUEST=GetCoverage&' + \
          'VERSION=1.1.1c1&' + \
          'CRS=EPSG:23030&' + \
          'BBOX=%d,%d,%d,%d&' + \
          'COVERAGE=MDT25_peninsula_ZIP&' + \
          'RESX=25&RESY=25&' + \
          'FORMAT=AsciiGrid&' + \
          'EXCEPTIONS=XML'

    url = url % (x1, y1, x2, y2)
    urllib.urlretrieve(url, name)
    if gdal_available:
        add_srs(name, 'EPSG:23030')

def download_mdt25_mosaic(x1, y1, x2, y2, name, tilesize=25000):
    """Download a mdt25 mosaic from IDEE WCS.

    Returns a list of file names. See download_mdt25 for details.

    """

    names = []
    for xx1 in range(x1, x2, tilesize):
        xx2 = min(x2, xx1 + tilesize)
        for yy1 in range(y1, y2, tilesize):
           yy2 = min(y2, yy1 + tilesize)
           tilename = add_coords(name, xx1, yy1)
           download_mdt25(xx1, yy1, xx2, yy2, tilename)
           names.append(tilename)
    return names


if __name__ == '__main__':
    try:
        x1, y1, x2, y2, dstname = processargs(sys.argv[1:])
    except Exception, e:
        print e
        usage()
        sys.exit(1)

    dx = x2 - x1
    dy = y2 - y1
    if max(dx, dy) > 35000:
        names = download_mdt25_mosaic(x1, y1, x2, y2, dstname, tilesize=2500)
    else:
        download_mdt25(x1, y1, x2, y2, dstname)
        names = [dstname]

    if gdal_available:
        tifnames = []
        for name in names:
            tifname, ext = os.path.splitext(name)
            tifname = os.path.extsep.join((tifname, 'tif'))
            translate(name, tifname, ot=gdal.GDT_Int16)
            tifnames.append(tifname)
        if len(tifnames) > 1:
            vrtname, ext = os.path.splitext(dstname)
            vrtname = os.path.extsep.join((vrtname, 'vrt'))
            gdal_vrtmerge(tifnames, vrtname)

