!
!
!   This file is part of Posix90
!
!   This program is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Public License as published by
!   the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   This program is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU General Public License for more details.
!
!   You should have received a copy of the GNU General Public License
!   along with this program; if not, write to the Free Software
!   Foundation, Inc., 51 Franklin Street, Fifth Floor,
!   Boston, MA 02110-1301, USA. 
!
module f90_unix_io
  use f90_unix_errno
  use f90_unix_env, only : sizet_kind, NULL
  use f90_unix_tools, only : C0
  implicit none
  
  include 'f90_unix_io_const.inc'
  
  type FILE
     integer(file_kind) :: fp
  end type FILE
  
  interface fread
     module procedure fread_str, fread_str_array
  end interface
  
  interface fwrite
     module procedure fwrite_str, fwrite_str_array
  end interface
  
  interface associated
     module procedure associated_fp
  end interface
  
contains
  
  logical function associated_fp(fp)
    type(FILE), intent(in) :: fp
    associated_fp = (fp%fp /= NULL)
  end function associated_fp
  
  type(FILE) function fopen(path, mode, errno)
    character(len=*), intent(in) :: path, mode
    integer, intent(out), optional :: errno
    type(FILE) :: fp

    if(present(errno)) errno = 0
    call set_errno

    call c_fopen(fp%fp, path//C0, mode//C0)
    fopen = fp

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fopen('"//trim(path)//"','"//trim(mode)//"')")
          stop
       end if
    endif

  end function fopen

  integer function fclose(fp, errno)
    type(FILE), intent(inout) :: fp
    integer, intent(out), optional :: errno
    integer, external :: c_fclose

    if(present(errno)) errno = 0
    call set_errno

    fclose = c_fclose(fp%fp)

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror("f90_unix_io::fclose")
          stop
       end if
    endif

  end function fclose
  
  type(FILE) function popen(command, mode, errno)
    character(len=*), intent(in) :: command, mode
    integer, intent(out), optional :: errno
    type(FILE) :: fp

    if(present(errno)) errno = 0
    call set_errno

    call c_popen(fp%fp,command//C0, mode//C0)
    popen = fp

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::popen('"//trim(command)//"','"//trim(mode)//"')")
          stop
       end if
    endif
  end function popen

  integer function pclose(fp, errno)
    type(FILE), intent(inout) :: fp
    integer, intent(out), optional :: errno
    integer, external :: c_pclose

    if(present(errno)) errno = 0
    call set_errno

    pclose = c_pclose(fp%fp)

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror("f90_unix_io::pclose")
          stop
       end if
    endif

  end function pclose
  
  integer(sizet_kind) function fread_str(str, len, fp, errno)
    character(len=*), intent(in) :: str
    integer(sizet_kind), intent(in) :: len
    type(FILE), intent(inout) :: fp
    integer, intent(out), optional :: errno
    integer(sizet_kind), external :: c_fread_str

    if(present(errno)) errno = 0
    call set_errno

    fread_str = c_fread_str(str, len, fp%fp)

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fread_str")
          stop
       end if
    endif

  end function fread_str

  integer(sizet_kind) function fread_str_array(str, len, fp, errno)
    character(len=*), intent(in) :: str(:)
    integer(sizet_kind), intent(in) :: len
    type(FILE), intent(inout) :: fp
    integer, intent(out), optional :: errno
    integer, external :: c_fread_str_array

    if(present(errno)) errno = 0
    call set_errno

    fread_str_array = c_fread_str_array(str, len, size(str,1), fp%fp) 

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fread_str_array")
          stop
       end if
    endif
    
  end function fread_str_array
  
  integer(sizet_kind) function fwrite_str(str, len, fp, errno)
    character(len=*), intent(in) :: str
    integer(sizet_kind), intent(in) :: len
    type(FILE), intent(inout) :: fp
    integer, intent(out), optional :: errno
    integer, external :: c_fwrite_str
 
    if(present(errno)) errno = 0
    call set_errno

    fwrite_str = c_fwrite_str(str, len, fp%fp)

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fwrite_str")
          stop
       end if
    endif
    
  end function fwrite_str

  integer(sizet_kind) function fwrite_str_array(str, len, fp, errno)
    character(len=*), intent(in) :: str(:)
    integer(sizet_kind), intent(in) :: len
    type(FILE), intent(inout) :: fp
    integer, intent(out), optional :: errno
    integer, external :: c_fwrite_str_array

    if(present(errno)) errno = 0
    call set_errno

    fwrite_str_array = c_fwrite_str_array(str, len, size(str,1), fp%fp) 

    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fwrite_str_array")
          stop
       end if
    endif
    
  end function fwrite_str_array
  
  subroutine fgets(str, strlen, fp, errno)
    character(len=*), intent(inout) :: str
    integer, intent(out) :: strlen
    type(FILE), intent(in) :: fp
    integer, intent(out), optional :: errno

    if(present(errno)) errno = 0
    call set_errno

    call c_fgets(str,fp%fp)
    call fortranify_string(str, strlen)
    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fgets")
          stop
       end if
    endif
    
  end subroutine fgets
  
  subroutine fputs(str, fp, errno)
    character(len=*), intent(in) :: str
    type(FILE):: fp
    integer, intent(out), optional :: errno

    if(present(errno)) errno = 0
    call set_errno

    call c_fputs(str//C0, fp%fp)
    if(present(errno)) then
       errno = get_errno()
    else
       if(get_errno() /= 0) then
          call perror(&
               &"f90_unix_io::fputs")
          stop
       end if
    endif
  end subroutine fputs
  
  type(FILE) function stdin()
    call c_stdin(stdin%fp)
  end function stdin
  
  type(FILE) function stdout()
    call c_stdout(stdout%fp)
  end function stdout
  
  type(FILE) function stderr()
    call c_stderr(stderr%fp)
  end function stderr
  
end module f90_unix_io
