/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "player.h"
#include "game.h"

extern pGame Game;

CPlayer::CPlayer(void)
{
	mMaxPower			= 2000.f;
	mGravity			= 0.f;
	mBearing			= 0.f;
	mFriction			= 3.f;
	mRotateSpeed		= PI;
	mRadius				= 8.f;
	mAntiRotateCoeff	= 0.25f;
	mWeaponHeight		= 0.f;
	mStartLife			= 100.f;
	mWeapon2Nbr			= 3;

	mShoot		= 0;
	mShootId	= 0;

	mPower = 0.f;
	mCooldown	= 0.f;
	mLife		= -1.f;
	mStatus		= STATUS_KILLED;
	mWait		= -1.f;	// Comme a on part direct

	mAttacker = NULL;
	mDamage = 0.f;
	mGroundCollision = false;
	
	// Cration de la traine
	mTail = new CEmitter;
	mTail->setMaterial(new CMaterial(MediaManager->Get<CTexture>("fx/tail.png"),0.08f));
	mTail->setSpeed(CVector3(0.f,0.f,0.f),CVector3(50.f,50.f,50.f));
	mTail->setFriction(CVector3(0.f,100.f,0.f));
	mTail->setPartLife(1.f,0.5f);
	mTail->setPartSize(100.f,100.f);
	mTail->setPartRotation(PI/2.f,false);
	mTail->setFrequency(0.f);
	AddChild(mTail);
	
	// Cration de la fume
	mSmoke = new CEmitter;
	mSmoke->setMaterial(new CMaterial(MediaManager->Get<CTexture>("fx/smoke.png"),0.4f));
	mSmoke->setSpeed(CVector3(0.f,0.f,0.f),CVector3(10.f,10.f,10.f));
	mSmoke->setFriction(CVector3(0.f,100.f,0.f));
	mSmoke->setPartLife(3.0f,0.2f);
	mSmoke->setPartSize(60.f,50.f);
	mSmoke->setPartRotation(PI,true);
	mSmoke->setFrequency(0.f);
	AddChild(mSmoke);

	// Cration du feu
	mFire = new CEmitter;
	mFire->setMaterial(new CMaterial(MediaManager->Get<CTexture>("fx/fire.png"),0.4f));
	mFire->setSpeed(CVector3(0.f,0.f,0.f),CVector3(50.f,50.f,50.f));
	mFire->setFriction(CVector3(0.f,0.f,0.f));
	mFire->setPartLife(0.5f,0.25f);
	mFire->setPartSize(20.f,45.f);
	mFire->setPartRotation(PI*10.f,true);
	mFire->setFrequency(0.f);
	AddChild(mFire);

	// Cration des sons
	mGun = new CSound(MediaManager->Get<CSample>("sound/gun.ogg"));
	AddChild(mGun);

	// Cration du curseur
	mCursor = new CPlain(	2500.f,2500.f,
				new CMaterial(MediaManager->Get<CTexture>("empire.png"),0.9f));
	mCursor->Attach(Game->mMapCursors);

	mHeatOfBattle = 0.f;
}

CPlayer::~CPlayer(void)
{
	mCursor->Attach(NULL);
}

void CPlayer::setCursor(const std::string &texture)
{
	mCursor->setMaterial(new CMaterial(MediaManager->Get<CTexture>(texture),0.9f));
}

void CPlayer::Parameter(const std::string &param,const std::string &data)
{
	std::stringstream sdata(data);

	if(param == "power")				sdata>>mMaxPower;
	else if(param == "gravity")			sdata>>mGravity;
	else if(param == "bearing")			sdata>>mBearing;
	else if(param == "friction")		sdata>>mFriction;
	else if(param == "rotatespeed")		sdata>>mRotateSpeed;
	else if(param == "collisionradius")	sdata>>mRadius;
	else if(param == "antirotatecoeff")	sdata>>mAntiRotateCoeff;
	else if(param == "weaponheight")	sdata>>mWeaponHeight;
	else if(param == "startlife")		sdata>>mStartLife;
	else if(param == "weapon1")			mWeapon[1] = data;
	else if(param == "weapon2")			mWeapon[2] = data;
	else if(param == "weapon3")			mWeapon[3] = data;
	else if(param == "weapon2nbr")		sdata>>mWeapon2Nbr;
	else CEntity::Parameter(param,data);
}

void CPlayer::Input(int flags,buffer_t &data)
{
	CNetEntity::Input(flags,data);
	
	if(flags & DATA_VELOCITY)
	{
		mVelocity.x = data.readFloat32();
		mVelocity.y = data.readFloat32();
		mVelocity.z = data.readFloat32();
	}

	if(flags & DATA_ROTATION)
	{
		CVector3 position = getLocalMatrix().getTranslation();

		CVector3 axis;
		axis.x = data.readFloat32();
		axis.y = data.readFloat32();
		axis.z = data.readFloat32();
		float angle = data.readFloat32();
		CQuaternion rotation(axis,angle);	
		
		Identity();
		Translate(position);
		Rotate(rotation);
	}

	if(flags & DATA_STATUS)
	{
		unsigned status = data.readInt8();
		ChangeStatus(status);
		mLife = data.readFloat32();	// TODO: mettre life ailleurs !
	}
}

void CPlayer::OutputData(int flags,buffer_t &data)
{
	CNetEntity::OutputData(flags,data);
	
	if(flags & DATA_VELOCITY)
	{
		data.writeFloat32(mVelocity.x);
		data.writeFloat32(mVelocity.y);
		data.writeFloat32(mVelocity.z);
	}

	if(flags & DATA_ROTATION)
	{
		CQuaternion rotation = getLocalMatrix().getRotation();
		CVector3 axis = rotation.getAxis();
		data.writeFloat32(axis.x);
		data.writeFloat32(axis.y);
		data.writeFloat32(axis.z);
		data.writeFloat32(rotation.getAngle());
	}

	if(flags & DATA_STATUS)
	{
		data.writeInt8(mStatus);
		data.writeFloat32(mLife);
	}
}

bool CPlayer::Update(double time)
{
	if(!CNetEntity::Update(time)) return false;
	usercmd_t &cmd = mCommand;

	CVector3	position	= getLocalMatrix().getTranslation();
	CQuaternion	rotation	= getLocalMatrix().getRotation();
	CVector3	dirx	= getLocalMatrix().getAxisX().Normalize();
	CVector3	diry	= getLocalMatrix().getAxisY().Normalize();
	CVector3	dirz	= getLocalMatrix().getAxisZ().Normalize();
	
	if(mStatus & STATUS_KILLED) 
	{
		if(mStatus & STATUS_CRASHED)
		{
			cmd.stick.x=0.f;
			cmd.stick.y=0.f;
			cmd.stick.z=0.f;
			cmd.buttons=0x0;
		}
		else{
			cmd.stick.x=0.3f;
			cmd.stick.y=-0.66f;
			cmd.stick.z=0.33f;
			cmd.buttons=0x0;
		}
	}

	if(cmd.stick.x < -1.f) cmd.stick.x = -1.f;
	else if(cmd.stick.x > 1.f) cmd.stick.x = 1.f;
	if(cmd.stick.y < -1.f) cmd.stick.y = -1.f;
	else if(cmd.stick.y > 1.f) cmd.stick.y = 1.f;
	if(cmd.stick.z < 0.f) cmd.stick.z = 0.f;
	else if(cmd.stick.z > 1.f) cmd.stick.z = 1.f;

	mSmoke->setFrequency((1.f-mLife/mStartLife)*25.f);
	mTail->setFrequency(mVelocity.Norm()/20.f);
	
	pSound engine = Call("engine");
	if(engine != NULL)
	{
		engine->setPitch((mPower/mMaxPower)*(0.75f+0.5f*dirz.Dotpoint(CVector3(0.f,-1.f,0.f))));
		engine->setGain(mCommand.stick.z);
	}
	
	float yaw,pitch;
	yaw 	= -cmd.stick.x;
	pitch	= -cmd.stick.y;
	
	mPower+=(mMaxPower*cmd.stick.z-mPower)*(1.f-std::exp(-float(time)*0.3f));	
	
	// Application du principe fondamental de la dynamique
	CVector3 acceleration =	diry*mBearing*std::abs(mVelocity.Dotpoint(dirz))		// portance
							+ dirz*mPower											// propulsion
							- mVelocity*mFriction;									// friction
	acceleration.y+= -mGravity;														// poids
	if(mStatus & STATUS_KILLED) acceleration.y-=500.f;
	mVelocity += acceleration*float(time);
	CVector3 move = mVelocity*float(time);

	// Correctif de position
	move += getCorrective(time);

	// Gestion de la rotation
	CQuaternion rotvelocity = CQuaternion(CVector3(0.f,1.f,0.f),yaw)*CQuaternion(CVector3(1.f,0.f,0.f),pitch)*CQuaternion(CVector3(0.f,0.f,1.f),-yaw);
	rotvelocity *= (rotation/CQuaternion(dirz)).Power(0.35f);
	if(!(mStatus & STATUS_KILLED)) rotvelocity *= (rotation/CQuaternion(CVector3(dirz.x,0.f,dirz.z))).Power(mAntiRotateCoeff);
	CQuaternion rotmove = rotvelocity.Power(float(time)*mPower/mMaxPower);
	
	// Collision
	CVector3 newmove,intersect;
	if(Game->mTerrainObject->Collision(position,move,mRadius,&newmove,&intersect))
	{
		CVector3 normal = position - intersect;
		normal.Normalize();
		int response = CollisionResponse(intersect,normal);

		if(response & 0x02)
		{
			CVector3 axis = newmove.Crosspoint(move);
			rotation = CQuaternion(axis,newmove.Norm()/(10*mRadius))*rotation;
		}
		else {
			CVector3 dirz = getLocalMatrix().getAxisZ().Normalize();
			
			CQuaternion targetrot = CQuaternion(CVector3(1.f,0.f,0.f),std::atan(normal.z));
			targetrot*= CQuaternion(CVector3(0.f,0.f,1.f),-std::atan(normal.x));
			targetrot*= CQuaternion(CVector3(dirz.x,0.f,dirz.z));
			
			rotmove *= (rotation/targetrot).Power(float(time)*2.f);
		}

		if(response & 0x01)
		{
			move = newmove;
			mVelocity = mVelocity*(0.5f + 0.5f*mVelocity.Dotpoint(newmove/newmove.Norm())/mVelocity.Norm());
		}
	}
	
	position += move;
	rotation *= rotmove;
	
	float t = 1.f-std::exp(-float(time)*2.f);
	if(position.z<2000.f)		rotation = rotation.Lerp(CVector3(0.f,0.f,1.f),t);
	else if(position.z>48000.f)	rotation = rotation.Lerp(CVector3(0.f,0.f,-1.f),t);
	if(position.x<2000.f)		rotation = rotation.Lerp(CVector3(1.f,0.f,0.f),t);
	else if(position.x>48000.f)	rotation = rotation.Lerp(CVector3(-1.f,0.f,0.f),t);
	
	if(position.y<0.f)		position.y=0.f;
	if(position.y>10000.f)	position.y=10000.f;
	
	Identity();
	Translate(position);
	Rotate(rotation);

	setChanged(DATA_CLIENT);
	
	setAnimationSpeed(cmd.stick.z+0.1);
	
	// Curseur sur la carte
	mCursor->Identity();
	mCursor->Translate(CVector3(position.x-25000.f,25000.f-position.z,0.f));
	mCursor->Rotate(CQuaternion(CVector3(0.f,0.f,1.f),-std::atan2(dirz.x,-dirz.z)+PI));
	
	if(mWeapon[1].size()>0 && (cmd.buttons & 0x1)) 
		mShoot = 1;
	else if(mWeapon[2].size()>0 && (cmd.buttons & 0x8)) mShoot = 2;
	else if(mWeapon[2].size()>0 && (cmd.buttons & 0x10)) mShoot = 3;
	//else if(mWeapon[3].size()>0 && (cmd.buttons & 0x20)) mShoot = 4;
	else mShoot = 0;
	
	if(mCooldown < 0.f) mCooldown+=time;
	while(mShoot != 0 && mCooldown >= 0.f) 
	{
		++mShootId;
		event_t ev;
		ev.id = EVENT_SHOOT;
		ev.param = mShoot%0x10 + mShootId*0x10;
		ev.stamp = Engine->getTimestamp()-mCooldown;
		Event(ev);
	}

	if(mShoot == 1) {
		if(!mGun->isPlaying()) mGun->Play(true);
	}
	else mGun->Stop();
	
	// Calcule si on est au coeur d'une bataille
	float dist = std::numeric_limits<float>::infinity();
	for(CGame::NetEntitiesMap_t::iterator it=Game->mNetEntities.begin(); it!=Game->mNetEntities.end(); ++it)
	{
		if(it->first >= INDEX_NONPLAYER) break;
		if(it->second == this) continue;
		if(it->second->getGroup() != getGroup()) 
			dist = std::min(dist,position.Distance(it->second->getLocalMatrix().getTranslation()));
	}
	
	mHeatOfBattle+= (1.f-dist/3000.f)*0.25f*time;
	
	if(mHeatOfBattle < 0.f) mHeatOfBattle = 0.f;
	else if(mHeatOfBattle > 1.f) mHeatOfBattle = 1.f;
	return true;
}

bool CPlayer::Event(const event_t &ev) 
{
	if(CNetEntity::Event(ev)) return true;

	switch(ev.id)
	{
		// Gestion du tir
		case EVENT_SHOOT:
		{
			int weapon = ev.param % 0x10;
			int mShootId = ev.param / 0x10;
			switch(weapon)	
			{
				// Mitrailleuse
				case 1:
				{
					mCooldown -= 0.08f;
					mShootId %= 2;
					
					CVector3 pos(4.f-8.f*mShootId,mWeaponHeight,40.f);
					CVector3 speed = getLocalMatrix().getAxisZ().Normalize()*3000.f;

					pBullet bullet = new CBullet(this,getLatency(),pos*getLocalMatrix(),speed);
					bullet->Script(mWeapon[1]);
					bullet->Attach(getFather());
					bullet->Update(Engine->getTimestamp()-ev.stamp);
					break;
				}

				// Canons
				case 2:
				case 3:
				{
					mCooldown -= 1.f/mWeapon2Nbr;
					mShootId %= mWeapon2Nbr;
					
					CVector3 pos((5-weapon*2)*40.f,mWeaponHeight,40.f*(mWeapon2Nbr/2-mShootId));
					CVector3 speed = getLocalMatrix().getAxisX().Normalize()*1500.f;
					if(weapon==3) speed*=-1.f;
					
					pBullet bullet = new CBullet(this,getLatency(),pos*getLocalMatrix(),speed+mVelocity);
					bullet->Script(mWeapon[2]);
					bullet->Attach(getFather());
					bullet->Update(Engine->getTimestamp()-ev.stamp);
			
					pSound sound = new CSound(MediaManager->Get<CSample>("sound/canon.ogg"));
					sound->Translate(pos*getLocalMatrix());
					sound->Attach(getFather());
					sound->setLife(2.);
					sound->Play();
					
					// fume du canon
					pEmitter particles = new CEmitter;
					particles->setMaterial(new CMaterial(MediaManager->Get<CTexture>("fx/smoke.png"),0.6f));
					particles->setFrequency(100.f);
					particles->setSpeed(mVelocity+speed*0.1f,CCoord3(50.f,50.f,50.f),false);
					particles->setPartLife(1.f,0.1f);
					particles->setFriction(CCoord3(0.f,250.f,0.f)-speed*0.1f,false);
					particles->setPartSize(25.f,25.f);
					particles->setPartRotation(PI/2,true);
					particles->Translate(pos);
					particles->setLife(0.20f);
					AddChild(particles);

					/*pPointLight light=new CPointLight;
					light->setAmbient(CColor(150,150,80));
					light->setDiffuse(CColor(250,250,160));
					light->setSpecular(CColor(240,240,170));
					light->setAttenuation(1.0f,0.05f,0.f);
					light->Attach(bullet);
					light->setLife(0.3f);*/
					break;
				}

				default:
					return false;

			}
			break;
		}

		// case EVENT_...

		default:
			return false;
	}

	return true;
}


void CPlayer::ApplyCommand(const usercmd_t & cmd)
{
	// TODO: p des trucs  faire...
}

void CPlayer::ServerUpdate(double time)
{
	if(mWait>0.f) mWait-=time;
	else {
		if(mAttacker != NULL)
		{
			mLife-=mDamage;
			mDamage = 0.f;
			mAttacker = NULL;
		}
		
		if(mLife<0.f)
		{
			ChangeStatus(mStatus | STATUS_KILLED);
			setChanged(DATA_STATUS);
			mLife = 0.f;
		}
		else {
			if(ChangeStatus(mStatus & (~STATUS_KILLED)))
				setChanged(DATA_STATUS);

			for(CGame::NetEntitiesMap_t::iterator it=Game->mNetEntities.begin(); it!=Game->mNetEntities.end(); ++it)
			{
				if(it->first >= INDEX_NONPLAYER) break;
				if(it->second == this) continue;
				float dist = getLocalMatrix().getTranslation().Distance(it->second->getLocalMatrix().getTranslation());
				if(dist <= getRadius()+it->second->getRadius())
					mLife = -1.f;
			}

			if(mGroundCollision) mLife-=40.f;	// Pas de comptage par frame

			/*switch(mShoot)
			{
			case 4:
				while(mCooldown >= 0.f)
				{
					for(int i=0; i<2;++i)
					{
						pRocket rocket = new CRocket(mGroup);
						int object = std::max(INDEX_NONPLAYER,(--Game->mNetEntities.end())->first+1);
						Game->mNetEntities.insert(Game->mNetEntities.end(),std::make_pair(object,rocket));
						rocket->Script(mWeapon[3]);
						rocket->Transform(getGlobalMatrix());
						rocket->Translate(CVector3(4.f-8.f*i,mWeaponHeight,4.f));
						rocket->Attach(getFather());
						rocket->Update(mCooldown);
					}

					mCooldown -= 10.f;
				}
				break;
			}*/
		}
	}
}

bool CPlayer::ChangeStatus(unsigned status)
{
	if(status == mStatus) return false;
	
	int added = status & (~mStatus);

	if(added & STATUS_KILLED)
	{
		mWait = 10.;
		mLife = 0.f;
		mShoot = 0;
		mGun->Stop();
		mFire->setFrequency(50.f);

		// Explosion
		pEntity explosion = new CEntity("fx/big_exp.p3d");
		explosion->Translate(getLocalMatrix().getTranslation());
		explosion->Attach(getFather());
		
		// Dsactivation de la camera interne
		pEntity called = Call("camera");
		if(called != NULL) called->setName("camera_disabled");
	}

	if(added & STATUS_CRASHED)
	{
		pSound engine = Call("engine");
		if(engine != NULL) engine->Stop();
	}
	
	int removed = (~status) & mStatus;

	if(removed & STATUS_KILLED)
	{
		status&=(~STATUS_CRASHED);
		
		mWait = 0.;
		mLife = mStartLife;
		mFire->setFrequency(0.f);
		resetCommand();

		CCoord3 position;
		CQuaternion rotation;
		
		if(getGroup()=="pirate")
		{
				position.set(10000.f+500*getIdentifier(),4000.f,10000.f);
				rotation.FromAxis(CVector3(0.f,1.f,0.f),0.f);
		}
		else {
				position.set(40000.f-500*getIdentifier(),4000.f,40000.f);
				rotation.FromAxis(CVector3(0.f,1.f,0.f),PI);
		}

		Identity();
		Translate(position);
		Rotate(rotation);

		// Ractivation de la camera interne
		pEntity called = Call("camera_disabled");
		if(called != NULL) called->setName("camera");

		pSound engine = Call("engine");
		if(engine != NULL) engine->Play(true);
	}

	if(mStatus != status) setChanged(mStatus);
	mStatus = status;
	return true;
}

int CPlayer::CollisionResponse(const CCoord3 &intersect,const CVector3 &normal)
{
	mGroundCollision = false;
	
	CVector3	diry = getLocalMatrix().getAxisY().Normalize();
	float factor = normal.Dotpoint(diry);

	if(factor < 0.9f && !(mStatus & STATUS_CRASHED))
	{
		pEntity impact = new CEntity("fx/impact.p3d");
		impact->Attach(getFather());
		impact->Translate(intersect);
	
		if(factor < 0.7f)
		{
			if(mStatus & STATUS_KILLED)
			{
				ChangeStatus(mStatus | STATUS_CRASHED);
			}
			else {
				pEntity explosion = new CEntity("fx/big_impact.p3d");
				explosion->Attach(getFather());
				explosion->Translate(intersect);
				explosion->Rotate(normal);
				
				mGroundCollision = true;
			}
		}
	}
	
	if(mStatus & STATUS_CRASHED) return 0xFF;
	else return 0x01;
}

void CPlayer::BulletResponse(pNetEntity owner,float damage,const CCoord3 &intersect,const CVector3 &direction)
{
	pSound sound = new CSound(MediaManager->Get<CSample>("sound/hit.ogg"));
	sound->Attach(getFather());
	sound->Translate(intersect);
	sound->Play();
	sound->setLife(0.3f);

	pEntity explosion = new CEntity("fx/small_exp.p3d");
	explosion->Translate(intersect);
	explosion->Attach(getFather());

	mAttacker = owner;
	mDamage+=damage;

	mHeatOfBattle+=0.5f;
}
