/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "netentity.h"
#include "game.h"

extern pGame Game;

CNetEntity::CNetEntity(void)
{
	setChanged(0xFF);
	mLatency = 0.;
	mRadius = 1.f;
	
	mIdentifier = -1;
	
	// Cration du texte
	pMaterial font = new CMaterial(MediaManager->Get<CTexture>("font.png"));
	font->setAmbient(CColor(0,0,0,250));
	font->setDiffuse(CColor(0,0,0,250));
	font->setSpecular(CColor(0,0,0,250));
	mNameText = new CText;
	mNameText->setFont(font,10.f,20.f);
	mNameText->setCentered(true);
	mNameText->LockDirection(Game->mCamera);
	mNameText->setEnv(GL_CULL_FACE,false);
	AddChild(mNameText);

	mCommand.buttons = 0x0;
	resetCommand();
	resetGroup();
}

CNetEntity::~CNetEntity(void)
{

}

void CNetEntity::setIdentifier(int identifier)
{
	CGame::NetEntitiesMap_t::iterator it = Game->mNetEntities.find(mIdentifier);
	if(it!=Game->mNetEntities.end()) Game->mNetEntities.erase(it);
	Game->mNetEntities[identifier] = this;
	mIdentifier = identifier;
}

int CNetEntity::getIdentifier(void) const
{
	return mIdentifier;
}

void CNetEntity::setGroup(const std::string &group)
{
	mGroup = group;
}

void CNetEntity::resetGroup(void)
{
	mGroup.clear();
}

void CNetEntity::setName(const std::string &name)
{
	mName = name;
	mNameText->setText(name);
}

const std::string &CNetEntity::getGroup(void) const
{
	return mGroup;
}

const std::string &CNetEntity::getName(void) const
{
	return mName;
}

void CNetEntity::setNameVisibility(bool visible)
{
	if(visible) mNameText->Attach(this);
	else mNameText->Detach();
}

float CNetEntity::getRadius(void) const
{
	return mRadius;
}

void CNetEntity::setCommand(const usercmd_t & cmd)
{
	if((mCommand.stick-cmd.stick).abs()<CVector3(1.f,1.f,1.f)*EPSILON)
		setChanged(DATA_COMMAND);

	unsigned changed = ((cmd.buttons & ~mCommand.buttons) | (mCommand.buttons & ~cmd.buttons));
	mCommand = cmd;
	mCommand.changed = changed;
}

void CNetEntity::resetCommand(void)
{
	mCommand.stick.setNull();
	mCommand.stickAux.setNull();
	mCommand.changed = mCommand.buttons;
	mCommand.buttons = 0x0;
	setChanged(DATA_COMMAND);
}

const usercmd_t &CNetEntity::getCommand(void) const
{
	return mCommand;
}

void CNetEntity::Input(double latency,int flags,buffer_t &data)
{
	// REM: La latence n'est pas  prendre en compte grce  la compensation sur les projectiles
	//double stamp = Engine->getTimestamp();//-mLatency;

	/*int nbr = data.readInt8();
	while(nbr-- > 0)
	{
		event_t ev;
		ev.id = data.readInt8();
		ev.param = data.readInt8();
		ev.stamp = stamp-data.readInt16()*0.001;	// bug !
		
		Event(ev);
	}*/
	
	mLatency = latency;
	if(flags & DATA_LATENCY)
		mLatency += data.readInt16()*0.001;
	Log<<latency<<std::endl;

	Input(flags,data);
	setChanged(flags);
}

// TODO: compression des donnes
void CNetEntity::Input(int flags,buffer_t &data)
{
	if(flags & DATA_CREATE)
	{
		mGroup = data.readString();
		mName = data.readString();
		
		std::string script = data.readString();
		// TODO
		if(getScript().empty()) 
			Script(script);	// charge le modle

		mNameText->Identity();
		mNameText->Translate(CVector3(0.f,mRadius*2,0.f));
		mNameText->Scale(CVector3(2.f,2.f,2.f));
	}
	
	if(flags & DATA_COMMAND)
	{	
		mCommand.stick.x = data.readFloat32();
		mCommand.stick.y = data.readFloat32();
		mCommand.stick.z = data.readFloat32();
		mCommand.buttons = data.readInt8();
	}
	
	if(flags & DATA_POSITION)
	{
		CVector3 position;
		position.x = data.readFloat32();
		position.y = data.readFloat32();
		position.z = data.readFloat32();
		mCorrective = position - getLocalMatrix().getTranslation();
	}
}

void CNetEntity::OutputData(int flags,buffer_t &data)
{
	if(flags & DATA_LATENCY)
		data.writeInt16(mLatency*1000.);
	
	if(flags & DATA_CREATE)
	{
		data.writeString(mGroup);
		data.writeString(mName);
		data.writeString(getScript());
	}
	
	if(flags & DATA_COMMAND)
	{
		data.writeFloat32(mCommand.stick.x);
		data.writeFloat32(mCommand.stick.y);
		data.writeFloat32(mCommand.stick.z);
		data.writeInt8(mCommand.buttons);
	}

	if(flags & DATA_POSITION)
	{
		CVector3 position = getLocalMatrix().getTranslation()+mCorrective;
		data.writeFloat32(position.x);
		data.writeFloat32(position.y);
		data.writeFloat32(position.z);
	}
}

/*void CNetEntity::OutputEvents(double stamp,buffer_t &data)
{
	EventsMap_t::iterator it=mEvents.upper_bound(stamp);
	int nbr = std::distance(it,mEvents.end());
	data.writeInt8(nbr);
	while(it!=mEvents.end())
	{
		data.writeInt8(it->second.id);						// id
		data.writeInt8(it->second.param);					// param
		data.writeInt16((stamp - it->second.stamp)*1000.);	// decal
		++it;
	}
}*/

double CNetEntity::getLatency(void) const
{
	return mLatency;
}

CVector3 CNetEntity::getCorrective(double time)
{
	float t = 1.f - std::exp(-float(time)*5.f);
	CVector3 move = mCorrective*t;
	mCorrective-=move;
	return move;
}

bool CNetEntity::Update(double time)
{
	if(!isAnimationPlaying())
	{
		setAnimationSpeed(1.f);
		PlayAnimation();
	}

	ApplyCommand(mCommand);

	/*EventsMap_t::iterator it = mEvents.upper_bound(Engine->getTime()-EVENTS_BACKLOG);
	mEvents.erase(mEvents.begin(),it);*/

	mNameText->Identity();
	mNameText->Translate(CVector3(0.f,mRadius*1.5f,0.f));
	
	return CEntity::Update(time);
}

bool CNetEntity::Event(const event_t &ev) 
{
	// Sauvegarde l'vnement
	//mEvents[ev.stamp] = ev;

	return false;	// pas de traitement ici
}

void CNetEntity::ApplyCommand(const usercmd_t & cmd)
{
	// Dummy
}

void CNetEntity::ServerUpdate(double time)
{
	// Dummy
}

void CNetEntity::BulletResponse(pNetEntity owner,float damage,const CCoord3 &intersect,const CVector3 &direction)
{
	// Dummy
}

void CNetEntity::setChanged(int flags)
{
	double stamp = Engine->getTime();
	
	int i = 0;
	while(flags)
	{
		if(flags & 0x1) 
			mLastChange[i] = stamp;

		flags = flags >> 1;
		++i;
	}
}

int CNetEntity::getChanged(double stamp) const
{
	int flags = 0x0;
	int bit = 0x1;
	for(int i=0; i<MAX_DATA_FLAGS; ++i)
	{
		if(mLastChange[i] > stamp) flags |= bit;
		bit = bit << 1;
	}
	return flags;
}

double CNetEntity::getTimeShift(const CCoord3 position)
{
	double sum = 0.;
	double sum_coeffs = 0.;
	for(CGame::NetEntitiesMap_t::iterator it=Game->mNetEntities.begin(); it!=Game->mNetEntities.end(); ++it)
	{
		double inv_dist = 1./position.Distance(it->second->getLocalMatrix().getTranslation());
		sum += it->second->mLatency * inv_dist;
		sum_coeffs += inv_dist;
	}
	return sum / sum_coeffs;
}
