/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "client.h"

CClient::CClient(const std::string &host)
{
	mSockStream = INVALID_SOCKET;
	mSockDgram = INVALID_SOCKET;
	
	addrinfo aiHints;
	memset(&aiHints, 0, sizeof(aiHints));
	addrinfo *aiList = NULL;
	
	try {
		
		aiHints.ai_family = AF_UNSPEC;
		aiHints.ai_socktype = 0;
		aiHints.ai_protocol = 0;
		aiHints.ai_flags = 0;
		
		// Boucle de recherche d'un couple de ports identiques pour les sockets de datagrammes et de flux
		int random = rand();
		int i;
		for(i=0; i<16384; ++i)
		{
			std::stringstream ssport;
			ssport << 49152 + (random + i) % 16384;

			// Obtention de l'adresse locale
			if(getaddrinfo(NULL, ssport.str().c_str(), &aiHints, &aiList) != 0)
				throw CException("Impossible d'obtenir l'adresse locale");
			
			// Cration de la socket de flux
			mSockStream = socket(aiList->ai_family,SOCK_STREAM,0);
			
			// Attache
			if(bind(mSockStream,(sockaddr*)aiList->ai_addr,aiList->ai_addrlen) != 0)
			{
				freeaddrinfo(aiList);
				close(mSockStream);
				if(sockerrno == EADDRINUSE) continue;
				throw CException("Impossible d'attacher  l'adresse locale");
			}
			
			// Cration de la socket de datagramme
			mSockDgram = socket(aiList->ai_family,SOCK_DGRAM,0);
			
			// Attache
			if(bind(mSockDgram,(sockaddr*)aiList->ai_addr,aiList->ai_addrlen) != 0)
			{
				freeaddrinfo(aiList);
				close(mSockStream);
				close(mSockDgram);
				if(sockerrno == EADDRINUSE) continue;
				throw CException("Impossible d'attacher  l'adresse locale");
			}

			break;
		}
		if(i == 16384) throw CException("Impossible de trouver des ports libres");
		
		// Obtention de l'adresse distante
		if(getaddrinfo(host.c_str(), NET_PORT, &aiHints, &aiList) != 0)
			throw CException("Hote inconnu");
		
		// Connexions
		if(connect(mSockStream,(sockaddr*)aiList->ai_addr,aiList->ai_addrlen) != 0) 
			throw CException("Connexion impossible");
		if(connect(mSockDgram,(sockaddr*)aiList->ai_addr,aiList->ai_addrlen) != 0) 
			throw CException("Connexion impossible");
		
		freeaddrinfo(aiList);
		aiList = NULL;

		ctl_t b = 1;
		if(ioctl(mSockStream,FIONBIO,&b)<0 || ioctl(mSockDgram,FIONBIO,&b)<0)
			throw CException("Impossible d'utiliser des sockets non bloquantes");

	}
	catch(...)
	{
		if(aiList != NULL) freeaddrinfo(aiList);
		if(mSockDgram != INVALID_SOCKET) close(mSockDgram);
		if(mSockStream != INVALID_SOCKET) close(mSockStream);
		throw;
	}

	Log<<"Connecte a "<<host<<std::endl;
	
	mCurrentStamp = 0;
	mLastRecvStamp = 0;
	mLastAckStamp = 0;
	mLastRecvLocalStamp = 0.;
	mLatency = 0.;
	mSendCooldown = 0.;

	mLocalEntity = -1;
}

CClient::~CClient(void)
{
	close(mSockDgram);
	close(mSockStream);
}


bool CClient::Update(double time)
{
	mCurrentStamp = Engine->getTime();
	
	// Boucle de rception du socket de flux
	char chr;
	while(recv(mSockStream,&chr,1,0) == 1)
	{
		if(chr!='\r' && chr!='\n')
		{
			if(chr==8) {
				if(!mRecvLine.empty())
					mRecvLine.erase(--mRecvLine.end());
			}
			else mRecvLine+= chr;
		}
		if(chr=='\n') {
			size_t p = mRecvLine.find_first_of(' ');
			mRecvLine.erase(p,1);
			CGame::Process(mRecvLine.substr(0,p),mRecvLine.substr(p,std::string::npos));	// TODO
			mRecvLine.clear();
		}

	}
	if(sockerrno != EWOULDBLOCK) 
		Log<<"Erreur: retour recv sur stream = "<<sockerrno<<std::endl;

	// Boucle de rception du socket de datagrammes
	char buffer[NET_BUFFERSIZE];
	int size;
	while((size = recv(mSockDgram,buffer,NET_BUFFERSIZE,0)) > 0)
	{
		buffer_t data(buffer,size);

		// Lit l'en-tte
		double stamp	 = data.readTime();		// stamp courant
		double ack		 = data.readTime();		// dernier stamp reu par le serveur
		double decal	 = data.readTime();		// temps d'envoi du dernier, permet de calculer la latence
		mLocalEntity	 = data.readInt8();		// entite controlle
		
		if(stamp <= mLastRecvStamp) continue;	// droppe si trop vieux ou dj reu
		mLastRecvStamp = stamp;
		mLastAckStamp = ack;
		mLastRecvLocalStamp = mCurrentStamp;
		
		double latency = (mCurrentStamp - (ack+decal))/2;		// calcul de la latence
		mLatency = (mLatency*9 + latency)/10;					// moyennage

		for(NetEntitiesMap_t::iterator it=mNetEntities.begin(); data.left(); ++it)
		{
			int object	= data.readInt8();
			int flags	= data.readInt8();
			
			// Attention: valuation optimise
			if(it==mNetEntities.end() || it->first > object)
			{
				if(!(flags & DATA_CREATE)) break;	// Le serveur fait n'importe quoi !

				pNetEntity entity;
				if(object < INDEX_NONPLAYER) entity = new CPlayer;
				else entity = new CNetEntity;
				entity->Attach(mScene->getRootEntity());
				entity->setIdentifier(object);

				entity->Input(mLatency,flags,data);
				
				if(object < INDEX_NONPLAYER) 
					pPlayer(entity)->setCursor(entity->getGroup()+".png");
				--it;
			}
			else if(it->first == object)
			{
				/*if(flags & DATA_CREATE)
				{
					it->second->Attach(NULL);
					if(object < INDEX_NONPLAYER) it->second = new CPlayer;
					else it->second = new CNetEntity;
					it->second->Attach(mScene->getRootEntity());
				}*/
				it->second->Input(mLatency,flags,data);
			}
			else if(it->first < object)
			{
				it->second->Attach(NULL);
				mNetEntities.erase(it--);
			}
		}
	}
	if(sockerrno != EWOULDBLOCK) 
		Log<<"Erreur: retour recv sur dgram = "<<sockerrno<<std::endl;
	
	// Rcupration et excution de l'entre joueur
	pPlayer player = getNetEntity(mLocalEntity);
	if(player != NULL) 
	{
		usercmd_t cmd = player->getCommand();
		SampleInput(cmd,time);
		player->setCommand(cmd);
		
		if(mTargetCamera != player) 
		{
			if(mTargetCamera != NULL) 
			{
				mTargetCamera->setCursor(mTargetCamera->getGroup()+".png");
				mTargetCamera->setNameVisibility(true);
			}
			player->setCursor("player.png");
			player->setNameVisibility(false);
			mTargetCamera = pNetEntity(player);
		}
	}

	// Mise  jour du jeu
	if(!CGame::Update(time)) return false;
	if(player != NULL) 
		UpdateCamera(player->getCommand(),time);
	
	// Envoi
	mSendCooldown-=time;
	if(mSendCooldown <= 0.)
	{
		mSendCooldown+= 1./NET_UPDATERATE;

		buffer_t data;
		data.writeTime(mCurrentStamp);							// stamp
		data.writeTime(mLastRecvStamp);							// ack
		data.writeTime(mCurrentStamp - mLastRecvLocalStamp);	// decal	
		
		if(player != NULL)
		{
			int flags = player->getChanged(mLastAckStamp);
			flags &= DATA_CLIENT;

			data.writeInt8(mLocalEntity);		// object
			data.writeInt8(flags);				// flags
					
			//player->OutputEvents(mLastAckStamp,data);	//events
			player->OutputData(flags,data);				// data
		}

		if(send(mSockDgram,data.ptr(),data.size(),0) != data.size())
				std::cout<<"Erreur: retour sendto sur dgram = "<<sockerrno<<std::endl;
	}

	return true;
}


bool CClient::Process(const std::string &command,const std::string &data)
{
	std::string message = command+' '+data+'\n';
	send(mSockStream,message.data(),message.size(),0);
	return true;
}
