/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "sound.h"
#include "mediamanager.h"
#include "sample.h"
#include "music.h"

CCoord3 CSound::ListenerOldPos;
CVector3 CSound::ListenerVelocity;

void CSound::Init(void)
{
	alGetError();
	ALCdevice *device=alcOpenDevice("arts");
	if(!device) device=alcOpenDevice(NULL);
	if (device)
	{
		int attrlist[]={ALC_SYNC,AL_FALSE,ALC_REFRESH,10,ALC_FREQUENCY,44100,ALC_INVALID};
		ALCcontext *context=alcCreateContext(device,attrlist);
		alcMakeContextCurrent(context);
		alcProcessContext(context);
	}
	else std::cout<<"Attention: Impossible d'ouvrir le perirpherique de son"<<std::endl;

	// Paramtres gnraux OpenAL
	alDistanceModel(AL_INVERSE_DISTANCE_CLAMPED);
	
	alDopplerFactor(20.f);		// TODO: fonctions de rglage
	alDopplerVelocity(343.3f);
	alSpeedOfSound(343.3f);
}

void CSound::Cleanup(void)
{
	ALCcontext *context=alcGetCurrentContext();
	ALCdevice *device=alcGetContextsDevice(context);
	alcMakeContextCurrent(NULL);
	alcDestroyContext(context);
	alcCloseDevice(device);
}

void CSound::Process(void)
{
	alcProcessContext(alcGetCurrentContext());
}

void CSound::setGlobalGain(float volume)
{
	alListenerf(AL_GAIN,volume);
}

void CSound::Listener(pEntity entity,double time)
{
	CMatrix4 global = entity->getGlobalMatrix();
	CCoord3 pos = global.getTranslation();
	alListenerfv(AL_POSITION,pos);

	float orientation[6];
	*reinterpret_cast<CVector3*>(orientation) = -global.getAxisZ();
	*reinterpret_cast<CVector3*>(orientation+3) = global.getAxisY();
	alListenerfv(AL_ORIENTATION,orientation);

	if(time>EPSILON)
	{
		float t = 1.f-std::exp(-float(time)*20.f);
		ListenerVelocity += ((pos-ListenerOldPos)/time - ListenerVelocity)*t;
		alListenerfv(AL_VELOCITY,ListenerVelocity);
		ListenerOldPos=pos;
	}
}

CSound::CSound(pSample sample)
{
	alGenSources(1,&mSource);
	
	// Paramtres de la source
	alSourcef(mSource, AL_REFERENCE_DISTANCE, 100.f);
	alSourcef(mSource, AL_ROLLOFF_FACTOR, 1.f);

	mSample=sample;
}

CSound::~CSound(void)
{
	alSourceStop(mSource);
	alDeleteSources(1,&mSource);
}

void CSound::setSample(pSample sample)
{
	Stop();
	mSample = sample;
}

pSample CSound::getSample(void) const
{
	return mSample;
}

void CSound::Play(bool loop)
{
	if(mSample!=NULL) mSample->Play(mSource,loop);
	mOldPos=getGlobalMatrix().getTranslation();
}

void CSound::Stop(void)
{
	if(mSample!=NULL) mSample->Stop(mSource);
}

float CSound::getGain(void)
{
	float value;
	alGetSourcef(mSource,AL_GAIN,&value);
	return value;
}

float CSound::getPitch(void)
{
	float value;
	alGetSourcef(mSource,AL_PITCH,&value);
	return value;
}

void CSound::setGain(float volume)
{
	alSourcef(mSource,AL_GAIN,volume);
}

void CSound::setPitch(float pitch)
{
	alSourcef(mSource,AL_PITCH,pitch);
}

bool CSound::isPlaying(void)
{
	ALenum state;
	alGetSourcei(mSource,AL_SOURCE_STATE,&state);
	return (state==AL_PLAYING);
}

void CSound::Parameter(const std::string &param,const std::string &data)
{
	if(param == "file" || param == "sample")	setSample(MediaManager->Get<CSample>(data));
	else if(param == "music")					setSample(MediaManager->Get<CMusic>(data));
	else if(param == "play")					Play(data=="true");
	else CEntity::Parameter(param,data);
}

bool CSound::Update(double time)
{
	if(mSample!=NULL)
	{
		CCoord3 pos=getGlobalMatrix().getTranslation();
		alSourcefv(mSource,AL_POSITION,pos);
		
		if(time>EPSILON)
		{
			if(isHierarchyChanged()) mOldPos = pos;
			float t = 1.f-std::exp(-float(time)*20.f);
			mVelocity += ((pos-mOldPos)/time - mVelocity)*t;
			alSourcefv(mSource,AL_VELOCITY,mVelocity); // bug sous Debian GNU/Linux lenny, OpenAL 0.0.8-6 ?
			mOldPos=pos;
		}
		
		mSample->Update(mSource);
	}

	return CEntity::Update(time);
}




