/***************************************************************************
 *   Copyright (C) 2006 by Paul-Louis Ageneau                              *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
// Constructeur par dfaut
inline CQuaternion::CQuaternion(void) : x(0.f),y(0.f),z(0.f),w(1.f)
{

}

// Constructeur surcharg
inline CQuaternion::CQuaternion(float _x,float _y,float _z,float _w) : x(_x),y(_y),z(_z),w(_w)
{

}

// Constructeur  partir d'un axe et d'un angle
inline CQuaternion::CQuaternion(const CVector3 &axe,float angle)
{
	FromAxis(axe,angle);
}

// Constructeur  partir d'un matrice
inline CQuaternion::CQuaternion(const CMatrix4 &matrix)
{
	FromMatrix(matrix);
}

// Constructeur  partir des angles d'Euler
inline CQuaternion::CQuaternion(float x,float y,float z)
{
	FromEuler(x,y,z);
}

// Constructeur  partir d'une direction
inline CQuaternion::CQuaternion(const CVector3 &direction)
{
	FromDirection(direction);
}

// Destructeur
inline CQuaternion::~CQuaternion(void)
{

}
	
inline void CQuaternion::Identity(void)
{
	x=y=z=0.f;
	w=1.f;
}

inline float CQuaternion::Norm(void) const
{
	return std::sqrt(x*x + y*y + z*z + w*w);
}

inline CQuaternion &CQuaternion::Normalize(void)
{
	float norm = Norm();
	
	x /= norm;
	y /= norm;
	z /= norm;
	w /= norm;

	return (*this);
}
	
inline void CQuaternion::FromAxis(const CVector3 &axis,float angle)
{
	float sin = std::sin(angle/2);
	x = axis.x * sin;
	y = axis.y * sin;
	z = axis.z * sin;
	w = std::cos(angle/2);

	Normalize();
}

inline void CQuaternion::FromEuler(float ax,float ay,float az)
{
	CQuaternion qx(CVector3(1.f, 0.f, 0.f),ax);
	CQuaternion qy(CVector3(0.f, 1.f, 0.f),ay);
	CQuaternion qz(CVector3(0.f, 0.f, 1.f),az);

	*this = qx * qy * qz;
}

inline void CQuaternion::FromMatrix(CMatrix4 matrix)
{
	matrix.Normalize();
	float Trace = matrix(0, 0) + matrix(1, 1) + matrix(2, 2) + matrix(3, 3);

    if (Trace > 0.f)
    {
        float s = std::sqrt(Trace) * 2.0f;
        x = (matrix(2, 1) - matrix(1, 2)) / s;
        y = (matrix(0, 2) - matrix(2, 0)) / s;
        z = (matrix(1, 0) - matrix(0, 1)) / s;
        w = 0.25f * s;
    }
    else
    {
        if ((matrix(0, 0) > matrix(1, 1)) && (matrix(0, 0) > matrix(2, 2)))
        {
            float s = std::sqrt(1.f + matrix(0, 0) - matrix(1, 1) - matrix(2, 2)) * 2.f;
            x = 0.25f * s;
            y = (matrix(0, 1) + matrix(1, 0)) / s;
            z = (matrix(0, 2) + matrix(2, 0)) / s;
            w = (matrix(1, 2) + matrix(2, 1)) / s;
        }
        else if (matrix(1, 1) > matrix(2, 2))
        {
            float s = std::sqrt(1.f - matrix(0, 0) + matrix(1, 1) - matrix(2, 2)) * 2.f;
            x = (matrix(0, 1) + matrix(1, 0)) / s;
            y = 0.25f * s;
            z = (matrix(1, 2) + matrix(2, 1)) / s;
            w = (matrix(0, 2) + matrix(2, 0)) / s;
        }
        else
        {
            float s = std::sqrt(1.f - matrix(0, 0) - matrix(1, 1) + matrix(2, 2)) * 2.f;
            x = (matrix(0, 2) + matrix(2, 0)) / s;
            y = (matrix(1, 2) + matrix(2, 1)) / s;
            z = 0.25f * s;
            w = (matrix(0, 1) + matrix(1, 0)) / s;
        }
    }

	Normalize();
}

inline void CQuaternion::FromDirection(const CVector3 &direction)
{
	// OPTI
	FromAxis(CVector3(0.f,1.f,0.f),std::atan2(direction.x,direction.z));
	(*this)*=CQuaternion(CVector3(1.f,0.f,0.f),-std::atan2(direction.y,std::sqrt(direction.x*direction.x+direction.z*direction.z)));

}

inline CQuaternion CQuaternion::operator*(const CQuaternion &q) const
{
	 return CQuaternion(w * q.x + x * q.w + y * q.z - z * q.y,
                       w * q.y + y * q.w + z * q.x - x * q.z,
                       w * q.z + z * q.w + x * q.y - y * q.x,
                       w * q.w - x * q.x - y * q.y - z * q.z).Normalize();
}

inline CQuaternion CQuaternion::operator/(CQuaternion q) const
{
	// Chemin le plus court parmis les 2 possibles
	float dot = x*q.x+y*q.y+z*q.z+w*q.w;
	if (dot < 0.f)
	{
		q.x=-q.x;
		q.y=-q.y;
		q.z=-q.z;
		q.w=-q.w;
	}

	CQuaternion q0=(*this);
	q0.Normalize();
	q.Normalize();
	return q0.Conjugate()*q;
}

inline CQuaternion &CQuaternion::operator*=(const CQuaternion &q)
{
   (*this)=(*this)*q;
   return (*this);
}

inline float CQuaternion::getAngle(void) const
{
	return std::acos(w)*2.0f;
}

inline CVector3 CQuaternion::getAxis(void) const
{
	float sinus = std::sqrt(1 - w*w);
	if(sinus <= std::numeric_limits<float>::epsilon())
		sinus = 1.f;
	return CVector3(x,y,z)/sinus;
}


inline CQuaternion CQuaternion::Power(float t) const
{
	float angle = getAngle()/2;

	CVector3 u(x,y,z);
	u.Normalize();
	u*=std::sin(t*angle);
	return CQuaternion(u.x,u.y,u.z,std::cos(t*angle)).Normalize();
}

inline CQuaternion CQuaternion::Conjugate(void) const
{
	return CQuaternion(-x, -y, -z, w);
} 

inline CQuaternion CQuaternion::Lerp(CQuaternion q,float t) const
{ 
	return (*this)*((*this)/q).Power(t);
}

inline CMatrix4 CQuaternion::ToMatrix(void) const
{
	return CMatrix4(	1.f - 2.f*y*y - 2.f*z*z,	2.f*x*y - 2.f*z*w,			2.f*x*z + 2.f*y*w,			0.f,
						2.f*x*y + 2.f*z*w,			1.f - 2.f*x*x - 2.f*z*z,	2.f*z*y - 2.f*x*w,			0.f,
						2.f*x*z - 2.f*y*w,			2.f*z*y + 2.f*x*w,			1.f - 2.f*x*x - 2.f*y*y,	0.f,
						0.f,						0.f,						0.f,						1.f);
}

inline float CQuaternion::getEulerX(void) const
{
	return std::atan(2.f*(x*y+z*w)/(1.f-2.f*(y*y+z*z)));
}

inline float CQuaternion::getEulerY(void) const
{
	return - std::asin(2.f*(x*z-w*y));
}

inline float CQuaternion::getEulerZ(void) const
{
	return std::atan(2.f*(x*w+y*z)/(1.f-2.f*(z*z+w*w)));
}
