/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "octree.h"
#include "bufferarray.h"
#include "bufferobject.h"



COctree::COctree(	const float *vertices, 
					index_t nvertices,
					index_t *indices, 
					index_t nindices,
					const float **texcoords,
					int ntexcoords) : CMesh(vertices,nvertices,indices,nindices,texcoords,ntexcoords)
{
	Build();
}

COctree::COctree(pMesh mesh)
{
	Copy(mesh);
	Build();
}

COctree::~COctree(void)
{

}

void COctree::Build(void)
{
	if(mNbrIndices == 0) return;

	const float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_READ_ONLY);
	
	// Contruction de l'arbre
	mRootNode=new CNode;
	
	index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_WRITE);
	
	for(MaterialsMap_t::const_iterator it=mMaterials.begin(); it!=mMaterials.end(); ++it)
	{
		index_t begin,end;
		begin=it->first;

		MaterialsMap_t::const_iterator next_it=it;
		++next_it;
		if(next_it==mMaterials.end()) end=mIndexBuffer->getCount();
		else end=next_it->first;

		mRootNode->Build(begin,end-begin,indices,vertices,it->second);
	}

	mIndexBuffer->Unlock();
	mVertexBuffer->Unlock();
}

void COctree::Update(void)
{
	const float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_READ_ONLY);
	const index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_ONLY);
	mRootNode->Update(indices,vertices);
	mVertexBuffer->Unlock();
	mIndexBuffer->Unlock();
}

int COctree::Draw(int pass,double frame,pMaterial force)
{
	if(pass != 1) return 0;

	int count=0;
	
	pFrustum frustum=new CFrustum;
	frustum->Extract();
	
	EnableBuffers();
	count=mRootNode->Draw(frustum,mIndexBuffer,force);
	DisableBuffers();

	return count;
}

float COctree::Intersect(double frame,const CCoord3 &pos,const CVector3 &move,float radius,CCoord3 *intersection)
{
	const float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_READ_ONLY);
	const index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_ONLY);
	float t=mRootNode->Intersect(pos,move,radius,intersection,indices,vertices);
	mVertexBuffer->Unlock();
	mIndexBuffer->Unlock();
	return t;
}

COctree::CNode::CNode(void)
{

}

COctree::CNode::~CNode(void)
{

}

int COctree::CNode::Draw(pFrustum frustum, pIndexBuffer indexbuffer,pMaterial force)
{
	if(frustum->BoxInFrustum(mMin,mMax))
	{
		int count=0;
		
		if(force != NULL)
		{
			if(!mSegments.empty() && force->Bind())
			{
				index_t nbr=(--mSegments.end())->second.second - mSegments.begin()->second.first;
				glDrawElements(GL_TRIANGLES,nbr,GL_UNSIGNED_INT,indexbuffer->Offset(mSegments.begin()->second.first));
				count+=nbr/3;
				force->Unbind();
			}
		}
		else for(SegmentsMap_t::const_iterator it=mSegments.begin(); it!=mSegments.end(); ++it)
		{
			if(it->first->Bind())	// Attache du matriau
			{
				const Segment_t &segment=it->second;
			
				index_t nbr=segment.second-segment.first;
				glDrawElements(GL_TRIANGLES,nbr,GL_UNSIGNED_INT,indexbuffer->Offset(segment.first));
				count+=nbr/3;

				it->first->Unbind();
			}
		}
		
		if(mChilds[0]!=NULL)
			for(int c=0; c<8; ++c)
				count+=mChilds[c]->Draw(frustum,indexbuffer,force);

		return count;
	}
	else return 0;
}

int COctree::CNode::getSubNode(const CCoord3 &pos)
{
	int subnode=0;
	CCoord3 centre=(mMin+mMax)/2.f;
	if(pos.x>centre.x) subnode+=1;
	if(pos.y>centre.y) subnode+=2;
	if(pos.z>centre.z) subnode+=4;
	return subnode;
}

void COctree::CNode::FitBox(const CCoord3 &pos)
{
	mMin.x=std::min(mMin.x,pos.x);
	mMin.y=std::min(mMin.y,pos.y);
	mMin.z=std::min(mMin.z,pos.z);

	mMax.x=std::max(mMax.x,pos.x);
	mMax.y=std::max(mMax.y,pos.y);
	mMax.z=std::max(mMax.z,pos.z);
}

void COctree::CNode::Build(index_t pos,index_t nbr,index_t *indices,const float *vertices,pMaterial material)
{
	if(!nbr) return;
	
	if(mChilds[0]==NULL && mSegments.empty()) 
		mMin=mMax=CCoord3(vertices + indices[pos]*3);
	
	for(index_t i=pos; i<pos+nbr; i+=3)
		for(int j=0; j<3; ++j)
			FitBox(CCoord3(vertices + indices[i+j]*3));
	
	if(nbr<=OCTREE_MAXFACES)			// si le nombre de faces n'est pas trop lev
	{
		mSegments[material]=Segment_t(pos,pos+nbr);

	} else {				// sinon on le subdivise
		
		// cration des botes des noeuds enfants
		if(mChilds[0]==NULL)
			for(int c=0; c<8; ++c)
				mChilds[c]=new CNode;

		index_t *buffer=new index_t[nbr];
		index_t end=pos;
		index_t begin;
			
		for(int c=0; c<8; ++c)
		{		
			begin=end;

			// Pour chaque face
			for(index_t i=pos; i<pos+nbr; i+=3)
			{
				// Position des 3 vertices de la face
				const float *v1 = vertices + indices[i]*3;
				const float *v2 = vertices + indices[i+1]*3;
				const float *v3 = vertices + indices[i+2]*3;
				CCoord3 centre = (CCoord3(v1)+CCoord3(v2)+CCoord3(v3))/3.f;

				if(getSubNode(centre)==c)	// OPTI
				{
					std::copy(indices+i,indices+i+3,buffer+end-pos);
					end+=3;
				};
			}

			mChilds[c]->Build(begin,end-begin,buffer-pos,vertices,material);
		}

		std::copy(buffer,buffer+nbr,indices+pos);
	}
}

void COctree::CNode::Update(const index_t *indices,const float *vertices)
{
	if(!mSegments.empty())
	{
		mMin=mMax=CCoord3(vertices + indices[mSegments.begin()->second.first]*3);
		for(SegmentsMap_t::const_iterator it=mSegments.begin(); it!=mSegments.end(); ++it)
		{
			const Segment_t &segment=it->second;
			for(index_t i=segment.first; i<segment.second; i+=3)
				for(int j=0; j<3; ++j)
					FitBox(CCoord3(vertices + indices[i+j]*3));
		}
	}
	else if(mChilds[0]!=NULL) {
		mMin=mMax=mChilds[0]->mMin;
	} else return;
		
	if(mChilds[0]!=NULL)
		for(int c=0; c<8; ++c)
		{
			mChilds[c]->Update(indices,vertices);
			FitBox(mChilds[c]->mMin);
			FitBox(mChilds[c]->mMax);
		}
}

float COctree::CNode::Intersect(const CCoord3 &pos,const CVector3 &move,float radius,CCoord3 *intersection,const index_t *indices,const float *vertices)
{
	if(mSegments.empty() && mChilds[0]==NULL) return std::numeric_limits<float>::infinity();

	// Si le point n'est pas directement  l'intrieur de la boite, test d'intersection
	if(!pos.isInBox(mMin,mMax))
	{
		// Approximation par sphere (beaucoup plus rapide)
		float t;
		CCoord3 centre=(mMin+mMax)/2.f;
		float totalradius=centre.Distance(mMin)+radius;
		if(centre.Distance(pos) > totalradius) {
			t=pos.IntersectSphere(move,centre,totalradius);
			if(t>1.f) return std::numeric_limits<float>::infinity();
		}
	}
	
	// cherche la premire face entrant en collision
	float nearestDistance = std::numeric_limits<float>::infinity();
	CCoord3 nearestIntersection;
	
	for(SegmentsMap_t::const_iterator it=mSegments.begin(); it!=mSegments.end(); ++it)
	{
		const Segment_t &segment=it->second;
		
		for(index_t i=segment.first; i<segment.second; ++i)
		{
			CCoord3 v1(vertices + indices[i]*3);
			CCoord3 v2(vertices + indices[++i]*3);
			CCoord3 v3(vertices + indices[++i]*3);
			
			CCoord3 intersect;
			float t=pos.Intersect(move,radius,v1,v2,v3,&intersect);
			if(t<nearestDistance)
			{
				nearestDistance=t;
				nearestIntersection=intersect;
			}
		}
	}

	if(mChilds[0]!=NULL)
		for(int i=0; i<8; ++i)
		{
			CCoord3 intersect;
			float t=mChilds[i]->Intersect(pos,move,radius,&intersect,indices,vertices);
				if(t<nearestDistance)
				{
					nearestDistance=t;
					nearestIntersection=intersect;
				}
		}

	if(intersection) *intersection=nearestIntersection;
	return nearestDistance;
}
