/***************************************************************************
 *   Copyright (C) 2006 by Paul-Louis Ageneau                              *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

// Constructeur par dfaut
inline CMatrix4::CMatrix4(float m11, float m12, float m13, float m14,
                          float m21, float m22, float m23, float m24,
                          float m31, float m32, float m33, float m34,
                          float m41, float m42, float m43, float m44) :
a11(m11), a12(m12), a13(m13), a14(m14),
a21(m21), a22(m22), a23(m23), a24(m24),
a31(m31), a32(m32), a33(m33), a34(m34),
a41(m41), a42(m42), a43(m43), a44(m44)
{

}

// Identite
inline void CMatrix4::Identity()
{
    a11 = 1.0f; a12 = 0.0f; a13 = 0.0f; a14 = 0.0f;
    a21 = 0.0f; a22 = 1.0f; a23 = 0.0f; a24 = 0.0f;
    a31 = 0.0f; a32 = 0.0f; a33 = 1.0f; a34 = 0.0f;
    a41 = 0.0f; a42 = 0.0f; a43 = 0.0f; a44 = 1.0f;
}

// Dterminant
inline float CMatrix4::Determinant() const
{
    float A = a22 * (a33 * a44 - a43 * a34) - a32 * (a23 * a44 - a43 * a24) + a42 * (a23 * a34 - a33 * a24);
    float B = a12 * (a33 * a44 - a43 * a34) - a32 * (a13 * a44 - a43 * a14) + a42 * (a13 * a34 - a33 * a14);
    float C = a12 * (a23 * a44 - a43 * a24) - a22 * (a13 * a44 - a43 * a14) + a42 * (a13 * a24 - a23 * a14);
    float D = a12 * (a23 * a34 - a33 * a24) - a22 * (a13 * a34 - a33 * a14) + a32 * (a13 * a24 - a23 * a14);

    return a11 * A - a21 * B + a31 * C - a41 * D;
}

// Tranpose
inline CMatrix4 CMatrix4::Transpose() const
{
    return CMatrix4(a11, a21, a31, a41,
                    a12, a22, a32, a42,
                    a13, a23, a33, a43,
                    a14, a24, a34, a44);
}

// Inverse
inline CMatrix4 CMatrix4::Inverse() const
{
    CMatrix4 result;
    float det = Determinant();
	AssertZero(det);
   
	result.a11 =  (a22 * (a33 * a44 - a34 * a43) - a32 * (a23 * a44 - a43 * a24) + a42 * (a23 * a34 - a33 *  a24)) / det;
    result.a12 = -(a12 * (a33 * a44 - a43 * a34) - a32 * (a13 * a44 - a43 * a14) + a42 * (a13 * a34 - a33 *  a14)) / det;
    result.a13 =  (a12 * (a23 * a44 - a43 * a24) - a22 * (a13 * a44 - a43 * a14) + a42 * (a13 * a24 - a23 *  a14)) / det;
    result.a14 = -(a12 * (a23 * a34 - a33 * a24) - a22 * (a13 * a34 - a33 * a14) + a32 * (a13 * a24 - a23 *  a14)) / det;

    result.a21 = -(a21 * (a33 * a44 - a34 * a43) - a23 * (a31 * a44 - a34 * a41) + a24 * (a31 * a43 - a33 *  a41)) / det;
    result.a22 =  (a11 * (a33 * a44 - a34 * a43) - a13 * (a31 * a44 - a34 * a41) + a14 * (a31 * a43 - a33 *  a41)) / det;
    result.a23 = -(a11 * (a23 * a44 - a24 * a43) - a13 * (a21 * a44 - a24 * a41) + a14 * (a21 * a43 - a23 *  a41)) / det;
    result.a24 =  (a11 * (a23 * a34 - a24 * a33) - a13 * (a21 * a34 - a24 * a31) + a14 * (a21 * a33 - a23 *  a31)) / det;

    result.a31 =  (a21 * (a32 * a44 - a34 * a42) - a22 * (a31 * a44 - a34 * a41) + a24 * (a31 * a42 - a32 *  a41)) / det;
    result.a32 = -(a11 * (a32 * a44 - a34 * a42) - a12 * (a31 * a44 - a34 * a41) + a14 * (a31 * a42 - a32 *  a41)) / det;
    result.a33 =  (a11 * (a22 * a44 - a24 * a42) - a12 * (a21 * a44 - a24 * a41) + a14 * (a21 * a42 - a22 *  a41)) / det;
    result.a34 = -(a11 * (a22 * a34 - a24 * a32) - a12 * (a21 * a34 - a24 * a31) + a14 * (a21 * a32 - a22 *  a31)) / det;

    result.a41 = -(a21 * (a32 * a43 - a33 * a42) - a22 * (a31 * a43 - a33 * a41) + a23 * (a31 * a42 - a32 *  a41)) / det;
    result.a42 =  (a11 * (a32 * a43 - a33 * a42) - a12 * (a31 * a43 - a33 * a41) + a13 * (a31 * a42 - a32 *  a41)) / det;
    result.a43 = -(a11 * (a22 * a43 - a23 * a42) - a12 * (a21 * a43 - a23 * a41) + a13 * (a21 * a42 - a22 *  a41)) / det;
    result.a44 =  (a11 * (a22 * a33 - a23 * a32) - a12 * (a21 * a33 - a23 * a31) + a13 * (a21 * a32 - a22 *  a31)) / det;

    return result;
}

inline CMatrix4 CMatrix4::NoTranslation(void) const
{
	 return CMatrix4(a11, a12, a13, 0.f,
                    a21, a22, a23, 0.f,
                    a31, a32, a33, 0.f,
                    a41, a42, a43, 1.f);
}

inline void CMatrix4::setTranslation(const CVector3 &v)
{
    a14 = v.x;
    a24 = v.y;
    a34 = v.z;
}

inline void CMatrix4::setScale(const CVector3 &s)
{
    CVector3 scale = s/getScale();
	*reinterpret_cast<CVector3*>(&a11)*=scale.x;
	*reinterpret_cast<CVector3*>(&a12)*=scale.y;
	*reinterpret_cast<CVector3*>(&a13)*=scale.z;
}

inline void CMatrix4::setRotation(float x,float y,float z)
{	
	CVector3 scale = getScale();
	
	float cosx = std::cos(x);
	float cosy = std::cos(y);
	float cosz = std::cos(z);

	float sinx = std::sin(x);
	float siny = std::sin(y);
	float sinz = std::sin(z);
	
	a11 = cosz*cosy*scale.x;
	a21 = sinz*cosy*scale.x;
	a31 = -siny*scale.x;

	a12 = (cosz*siny*sinx - sinz*cosx)*scale.y;
	a22 = (sinz*siny*sinx + cosx*cosz)*scale.y;
	a32 = sinx*cosy*scale.y;

	a13 = (cosz*siny*cosx + sinz*sinx)*scale.z;
	a23 = (sinz*siny*cosx - cosz*sinx)*scale.z;
	a33 = cosx*cosy*scale.z;
}

inline void CMatrix4::setRotation(const CQuaternion &q)
{
	CMatrix4 mat = q.ToMatrix();
	CVector3 scale = getScale();

	a11 = mat.a11*scale.x;
	a21 = mat.a21*scale.x;
	a31 = mat.a31*scale.x;

	a12 = mat.a12*scale.y;
	a22 = mat.a22*scale.y;
	a32 = mat.a32*scale.y;

	a13 = mat.a13*scale.z;
	a23 = mat.a23*scale.z;
	a33 = mat.a33*scale.z;
}

// TODO
/*
inline void CMatrix4::LookAt(const CVector3 &from, const CVector3 &to, const CVector3 &up)
{
    CVector3 zaxis = to - from;
    zaxis.Normalize();
    CVector3 xaxis = up.Crosspoint(zaxis);
    xaxis.Normalize();
    CVector3 yaxis = zaxis.Crosspoint(xaxis);

    a11 = xaxis.x; a12 = xaxis.y; a13 = xaxis.z; a14 = -xaxis.Crosspoint(from);
    a21 = yaxis.x; a22 = yaxis.y; a23 = yaxis.z; a24 = -yaxis.Crosspoint(from));
    a31 = zaxis.x; a32 = zaxis.y; a33 = zaxis.z; a34 = -zaxis.Crosspoint(from));
    a41 = 0.0f;    a42 = 0.0f;    a43 = 0.0f;    a44 = 1.0f;
}
*/

inline void CMatrix4::Normalize(void)
{
    CVector3 scale = getScale();
	*reinterpret_cast<CVector3*>(&a11)/=scale.x;
	*reinterpret_cast<CVector3*>(&a12)/=scale.y;
	*reinterpret_cast<CVector3*>(&a13)/=scale.z;
}

inline void CMatrix4::Translate(const CVector3 &v)
{
	CMatrix4 mat;
	mat.setTranslation(v);
	(*this)*=mat;
}

inline void CMatrix4::Rotate(float x,float y,float z)
{
	CMatrix4 mat;
	mat.setRotation(x,y,z);
	(*this)*=mat;
}

inline void CMatrix4::Rotate(const CQuaternion &q)
{
	(*this)*=q.ToMatrix();
}

inline void CMatrix4::Scale(const CVector3 &s)
{
	CMatrix4 mat;
	mat.a11 = s.x;
	mat.a22 = s.y;
	mat.a33 = s.z;
	(*this)*=mat;
}

inline CVector3 CMatrix4::getTranslation(void) const
{
	return CVector3(a14,a24,a34);
}

inline CQuaternion CMatrix4::getRotation(void) const
{
	return CQuaternion(*this);
}

inline CVector3 CMatrix4::getAxisX(void) const
{
	return CVector3(a11,a21,a31);
}

inline CVector3 CMatrix4::getAxisY(void) const
{
	return CVector3(a12,a22,a32);
}

inline CVector3 CMatrix4::getAxisZ(void) const
{
	return CVector3(a13,a23,a33);
}

inline CVector3 CMatrix4::getScale(void) const
{
	return CVector3(CVector3(a11,a21,a31).Norm(),
					CVector3(a12,a22,a32).Norm(),
					CVector3(a13,a23,a33).Norm());
}

inline CVector3 CMatrix4::Transform(const CVector3& v, float w) const
{
    return CVector3(v.x * a11 + v.y * a12 + v.z * a13 + w * a14,
                   v.x * a21 + v.y * a22 + v.z * a23 + w * a24,
                   v.x * a31 + v.y * a32 + v.z * a33 + w * a34);
}

inline void CMatrix4::Apply(void) const
{
	glMultMatrixf(operator const float*());
}

inline void CMatrix4::Load(void) const
{
	glLoadMatrixf(operator const float*());
}

inline void CMatrix4::Get(GLenum name)
{
	glGetFloatv(name,(operator float*()));
}

// multiplication de 2 matrices
inline CMatrix4 CMatrix4::operator *(const CMatrix4& m) const
{
    return CMatrix4(m.a11 * a11 + m.a21 * a12 + m.a31 * a13 + m.a41 * a14,
					m.a12 * a11 + m.a22 * a12 + m.a32 * a13 + m.a42 * a14,
                    m.a13 * a11 + m.a23 * a12 + m.a33 * a13 + m.a43 * a14,
                    m.a14 * a11 + m.a24 * a12 + m.a34 * a13 + m.a44 * a14,

                    m.a11 * a21 + m.a21 * a22 + m.a31 * a23 + m.a41 * a24,
                    m.a12 * a21 + m.a22 * a22 + m.a32 * a23 + m.a42 * a24,
                    m.a13 * a21 + m.a23 * a22 + m.a33 * a23 + m.a43 * a24,
                    m.a14 * a21 + m.a24 * a22 + m.a34 * a23 + m.a44 * a24,

                    m.a11 * a31 + m.a21 * a32 + m.a31 * a33 + m.a41 * a34,
                    m.a12 * a31 + m.a22 * a32 + m.a32 * a33 + m.a42 * a34,
                    m.a13 * a31 + m.a23 * a32 + m.a33 * a33 + m.a43 * a34,
                    m.a14 * a31 + m.a24 * a32 + m.a34 * a33 + m.a44 * a34,

                    m.a11 * a41 + m.a21 * a42 + m.a31 * a43 + m.a41 * a44,
                    m.a12 * a41 + m.a22 * a42 + m.a32 * a43 + m.a42 * a44,
                    m.a13 * a41 + m.a23 * a42 + m.a33 * a43 + m.a43 * a44,
                    m.a14 * a41 + m.a24 * a42 + m.a34 * a43 + m.a44 * a44);
}

// multiplication *=
inline const CMatrix4& CMatrix4::operator *=(const CMatrix4& m)
{
    *this = *this * m;

    return *this;
}


// accs aux lments
inline float& CMatrix4::operator ()(std::size_t i, std::size_t j)
{
    return operator float*()[i + 4 * j];
}

// accs constant aux lments
inline const float& CMatrix4::operator ()(std::size_t i, std::size_t j) const
{
   return operator const float*()[i + 4 * j];
}

// cast en const float*
inline CMatrix4::operator const float*() const
{
    return &a11;
}

// cast en float*
inline CMatrix4::operator float*()
{
    return &a11;
}
