/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "frustum.h"

CFrustum::CFrustum(void)
{

}

CFrustum::~CFrustum(void)
{

}

// extrait les plans du frustum
void CFrustum::Extract(void)
{
	CMatrix4 modl,proj;
	modl.Get(GL_MODELVIEW_MATRIX);   // Rcupere la matrice de modelisation
	proj.Get(GL_PROJECTION_MATRIX);  // Rcupere la matrice de projection

	// On multiplie la matrice de projection par celle de modelisation
	proj*=modl;
	
	// Extrait le plan de DROITE
	mPlane[0].a = proj(3,0) - proj(0,0);
	mPlane[0].b = proj(3,1) - proj(0,1);
	mPlane[0].c = proj(3,2) - proj(0,2);
	mPlane[0].d = proj(3,3) - proj(0,3);
	mPlane[0].Normalize();

	// Extrait le plan de GAUCHE
	mPlane[1].a = proj(3,0) + proj(0,0);
	mPlane[1].b = proj(3,1) + proj(0,1);
	mPlane[1].c = proj(3,2) + proj(0,2);
	mPlane[1].d = proj(3,3) + proj(0,3);
	mPlane[1].Normalize();

	// Extrait le plan du BAS
	mPlane[2].a = proj(3,0) + proj(1,0);
	mPlane[2].b = proj(3,1) + proj(1,1);
	mPlane[2].c = proj(3,2) + proj(1,2);
	mPlane[2].d = proj(3,3) + proj(1,3);
	mPlane[2].Normalize();

	// Extrait le plan du HAUT
	mPlane[3].a = proj(3,0) - proj(1,0);
	mPlane[3].b = proj(3,1) - proj(1,1);
	mPlane[3].c = proj(3,2) - proj(1,2);
	mPlane[3].d = proj(3,3) - proj(1,3);
	mPlane[3].Normalize();

	// Extrait le plan ELOIGNE
	mPlane[4].a = proj(3,0) - proj(2,0);
	mPlane[4].b = proj(3,1) - proj(2,1);
	mPlane[4].c = proj(3,2) - proj(2,2);
	mPlane[4].d = proj(3,3) - proj(2,3);
	mPlane[4].Normalize();

	// Extrait le plan PROCHE
	mPlane[5].a = proj(3,0) + proj(2,0);
	mPlane[5].b = proj(3,1) + proj(2,1);
	mPlane[5].c = proj(3,2) + proj(2,2);
	mPlane[5].d = proj(3,3) + proj(2,3);
	mPlane[5].Normalize();
}

/*
Calcule si un point est sur un des 6 plans du frustum

  formule A * X + B * Y + C * Z + D
	A, B, C, et D = 4 nombres qui definissent le plan et
	X, Y, et Z = coordonnes du point
*/
bool CFrustum::PointInFrustum(const CCoord3 &p) const
{
	for(int i=0; i<6; ++i)	// pour chaque plan
	{
		// si le point est en dehors d'un des plans
		float d=mPlane[i].a*p.x + mPlane[i].b*p.y + mPlane[i].c*p.z + mPlane[i].d;
		if(d <= 0.f) return false;
	}
	return true;
}

//Calcule si une sphere est dans le frustum
bool CFrustum::SphereInFrustum(const CCoord3 &centre, float radius) const
{
	for(int i=0; i<6; ++i)	// pour chaque plan
	{
		float d=mPlane[i].a*centre.x + mPlane[i].b*centre.y + mPlane[i].c*centre.z + mPlane[i].d;
		if(d <= -radius) return false;	// si on est en dehors d'un des plans
	}
	return true;
}

//Calcule si une sphere est dans le frustum avec centre=(0,0,0)
bool CFrustum::SphereInFrustum(float radius) const
{
	for(int i=0; i<6; ++i)	// pour chaque plan
		if(mPlane[i].d <= -radius) return false;	// si on est en dehors d'un des plans
	return true;
}

//Calcule pour une box
bool CFrustum::BoxInFrustum(const CCoord3 &p1,const CCoord3 &p2) const
{
	for(int p = 0; p < 6; p++ )
	{
		if( mPlane[p].a*p2.x + mPlane[p].b*p2.y + mPlane[p].c*p2.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p1.x + mPlane[p].b*p2.y + mPlane[p].c*p2.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p2.x + mPlane[p].b*p1.y + mPlane[p].c*p2.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p1.x + mPlane[p].b*p1.y + mPlane[p].c*p2.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p2.x + mPlane[p].b*p2.y + mPlane[p].c*p1.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p1.x + mPlane[p].b*p2.y + mPlane[p].c*p1.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p2.x + mPlane[p].b*p1.y + mPlane[p].c*p1.z + mPlane[p].d > 0 )
			continue;
		if( mPlane[p].a*p1.x + mPlane[p].b*p1.y + mPlane[p].c*p1.z + mPlane[p].d > 0 )
			continue;
		return false;
	}
	return true;
}
