/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "entity.h"
#include "paulo3d.h"

// Constructeur
CEntity::CEntity(const std::string &filename) :		// Constructeur depuis fichier
	mFather(NULL),
	mInternalRadius(-1.f),
	mExternalRadius(-1.f),
	mAnimPlaying(false),
	mAnimSpeed(1.),
	mCurrentFrame(0.),
	mAnimStart(0.),
	mAnimEnd(-1.),
	mLife(-1.f),
	mHierarchyChanged(true)
{
	if(!filename.empty()) Script(filename);
}

// Destructeur
CEntity::~CEntity(void)
{
	pEntity child = getChild();
	while(child != NULL)
	{
		child->mFather = NULL;
		child = child->getBrother();
	}

	if(mBrother != NULL) mBrother->mBigBrother = mBigBrother;
}

pScriptable CEntity::Tag(const std::string &tag)
{
	pEntity child;
	if(tag == "entity")					child = new CEntity;
	else if(tag == "object")			child = new CObject;
	else if(tag == "camera")			child = new CCamera;
	else if(tag == "pointlight")		child = new CPointLight;
	else if(tag == "directionallight")	child = new CDirectionalLight;
	else if(tag == "emitter")			child = new CEmitter;
	else if(tag == "sound")				child = new CSound;
	else if(tag == "text")				child = new CText;
	else if(tag == "plain")				child = new CPlain;

	if(child != NULL) {
		AddChild(child);
		return child;
	}

	if(tag == "material")
	{
		// TODO: Problme de rechargement des matriaux
		pMaterial mat = new CMaterial;
		ResourceManager->Add("temp_script_material",mat);
		Parameter("material","temp_script_material");
		ResourceManager->Remove("temp_script_material");
		return mat;
	}
	else if(tag == "animation")
	{
		// TODO: Problme de rechargement
		pAnimation anim = new CAnimation;
		setAnimation(anim);
		PlayAnimation();
		return anim;
	}
	
	return NULL;
}

void CEntity::Parameter(const std::string &param,const std::string &data)
{
	std::stringstream sdata(data);
	
	if(param == "name")					mName = data;
	else if(param == "translate" 
			|| param =="position")		Translate(CVector3(data));
	else if(param == "rotate")			Rotate(CQuaternion(data));
	else if(param == "scale")			Scale(CVector3(data));
	else if(param == "life")			sdata>>mLife;
	else if(param == "file")			Script(data);
	else if(param == "animspeed")		sdata>>mAnimSpeed;
	else if(param == "flags")
	{
		std::string::size_type p = data.find("light");
		if(p!=std::string::npos) setEnv(GL_LIGHTING,p==0 || !(data[--p]=='-'));
		p = data.find("fog");
		if(p!=std::string::npos) setEnv(GL_FOG,p==0 || !(data[--p]=='-'));
		p = data.find("cull");
		if(p!=std::string::npos) setEnv(GL_CULL_FACE,p==0 || !(data[--p]=='-'));
	}
}

void CEntity::Attach(CEntity *parent)
{
	if(parent == getFather()) return;
	Detach();
	if(parent) parent->AddChild(this);
}

void CEntity::Detach(void)
{
	if(mFather) 
	{
		pEntity lock = this;	// Empche la destruction

		if(mFather->mChild == this) 
		{
			mFather->mChild = mBrother;
			if(mBrother != NULL) mBrother->mBigBrother = NULL;
		}
		else {
			mBigBrother->mBrother = mBrother;
			if(mBrother != NULL) mBrother->mBigBrother = mBigBrother;
		}
		
		mFather = NULL;
		mBrother = NULL;
		mBigBrother = NULL;
		mHierarchyChanged = true;
	}
}

bool CEntity::RemoveBrother(pEntity ptr)
{
	if(ptr==NULL) return false;
	if(mBrother != NULL) 
	{
		if(mBrother == ptr) 
		{
			pEntity brother = mBrother;
			mBrother = brother->mBrother;
			brother->mBrother = NULL;
			brother->mFather = NULL;
			brother->mBigBrother = NULL;
			brother->mHierarchyChanged = true;
			return true;
		}
		else return mBrother->RemoveBrother(ptr);
	}
	else return false;
}

void CEntity::AddChild(pEntity ptr)
{
	if(ptr==NULL) return;
	ptr->Detach();
	ptr->mBrother = mChild;
	if(mChild != NULL) mChild->mBigBrother = ptr;
	ptr->mFather = this;
	ptr->mBigBrother = NULL;
	mChild = ptr;
	mChild->mHierarchyChanged = true;
}

bool CEntity::RemoveChild(pEntity ptr)
{
	if(ptr==NULL) return false;
	if(mChild != NULL) 
	{
		if(mChild == ptr) 
		{
			pEntity child = mChild;
			mChild = child->mBrother;
			child->mBrother = NULL;
			child->mFather = NULL;
			child->mBigBrother = NULL;
			child->mHierarchyChanged = true;
			return true;
		}
		else return mChild->RemoveBrother(ptr);
	}
	else return false;
}

pEntity CEntity::getFather(void)
{
	return mFather;
}

pEntity CEntity::getBrother(int num)
{
	if(num == 0) return mBrother;
	else if(mBrother != NULL) return mBrother->getBrother(--num);
	else return NULL;
}

int CEntity::getBrothersCount(void)
{
	if(mBrother != NULL) return mBrother->getBrothersCount()+1;
	else return 0;
}

pEntity CEntity::getChild(int num)
{
	if(num == 0) return mChild;
	else if(mChild != NULL) return mChild->getBrother(--num);
	else return NULL;
}

int CEntity::getChildsCount(void)
{
	if(mChild != NULL) return mChild->getBrothersCount()+1;
	else return 0;
}

pEntity CEntity::Call(const std::string &name)
{
	if(name == mName) return this;
	
	pEntity child = getChild();
	while(child != NULL)
	{
		pEntity called = child->Call(name);
		if(called != NULL) return called;
		child = child->getBrother();
	}

	return NULL;
}

void CEntity::Translate(const CVector3 &v)
{
	mLocalMatrix.Translate(v);
}

void CEntity::Rotate(float x,float y,float z)
{
	mLocalMatrix.Rotate(x,y,z);
}

void CEntity::Rotate(const CQuaternion &q)
{
	mLocalMatrix.Rotate(q);
}

void CEntity::Scale(const CVector3 &s)
{
	mLocalMatrix.Scale(s);
}

CMatrix4 &CEntity::getLocalMatrix(void)
{
	return mLocalMatrix;
}

CMatrix4 CEntity::getGlobalMatrix(void) const
{
	if(mFather) return mFather->getGlobalMatrix()*mLocalMatrix;
	else return mLocalMatrix;
}
void CEntity::setTransformation(const CMatrix4 &matrix)
{
	mLocalMatrix=matrix;
}

void CEntity::Transform(const CMatrix4 &matrix)
{
	mLocalMatrix*=matrix;
}

void CEntity::AbsoluteTransform(const CMatrix4 &matrix)
{
	mLocalMatrix=matrix*mLocalMatrix;
}

void CEntity::Identity(void)
{
	mLocalMatrix.Identity();
}

const CMatrix4 *CEntity::getTextureMatrix(int unit) const
{
	TextureMatrixMap_t::const_iterator it=mTextureMatrixes.find(unit);
	if(it!=mTextureMatrixes.end()) return &it->second;
	else return NULL;
}

void CEntity::setTextureMatrix(const CMatrix4 *matrix,int unit)
{
	if(matrix!=NULL) mTextureMatrixes[unit]=*matrix;
	else {
		TextureMatrixMap_t::iterator it=mTextureMatrixes.find(unit);
		if(it!=mTextureMatrixes.end()) mTextureMatrixes.erase(it);
	}
}

void CEntity::setEnv(GLenum name,bool state)
{
	mEnv[name] = state;
}

void CEntity::setLife(float life)
{
	mLife=life;
}

float CEntity::getLife(void)
{
	return mLife;
}

void CEntity::setAnimation(pAnimation animation)
{
	mAnimation = animation;
}

pAnimation CEntity::getAnimation(void) const
{
	return mAnimation;
}

void CEntity::PlayAnimation(void)
{
	mAnimPlaying = true;

	pEntity child = getChild();
	while(child != NULL)
	{
		child->PlayAnimation();
		child = child->getBrother();
	}
}

void CEntity::StopAnimation(void)
{
	mAnimPlaying = false;
	pEntity child = getChild();
	while(child != NULL)
	{
		child->StopAnimation();
		child = child->getBrother();
	}	
}

bool CEntity::isAnimationPlaying(void)
{
	return mAnimPlaying;
}

void CEntity::setAnimationFrame(double start,double end)
{
	mAnimStart = start;
	mAnimEnd = end;
	mCurrentFrame = start;
	
	if(mAnimation != NULL)
	{
		Identity();
		if(mAnimation->getPositionCount() > 0) Translate(mAnimation->getPosition(start));
		if(mAnimation->getRotationCount() > 0) Rotate(mAnimation->getRotation(start));
		if(mAnimation->getScaleCount() > 0)	Scale(mAnimation->getScale(start));
	}

	pEntity child = getChild();
	while(child != NULL)
	{
		child->setAnimationFrame(start,end);
		child = child->getBrother();
	}
}

void CEntity::setAnimationSpeed(double speed)
{
	mAnimSpeed = speed;
}

float CEntity::getAbsoluteAnimationSpeed(void) const
{
	if(mFather) return mAnimSpeed*mFather->getAbsoluteAnimationSpeed();
	else return mAnimSpeed;
}

bool CEntity::Update(double time)
{
	if(mAnimPlaying)
	{
		mCurrentFrame+=time*getAbsoluteAnimationSpeed();
		if(mAnimEnd >= 0.)
		{
			if(mCurrentFrame>=mAnimEnd) mCurrentFrame+=mAnimStart-mAnimEnd;
			else if(mCurrentFrame<mAnimStart) mCurrentFrame+=mAnimEnd-mAnimStart;
		}
		
		if(mAnimation != NULL && mAnimation->getCount()>0)
		{
			Identity();
			if(mAnimation->getPositionCount() > 0)	Translate(mAnimation->getPosition(mCurrentFrame));
			if(mAnimation->getRotationCount() > 0)	Rotate(mAnimation->getRotation(mCurrentFrame));
			if(mAnimation->getScaleCount() > 0)		Scale(mAnimation->getScale(mCurrentFrame));
		}
	}

	if(mLife>0.f)
	{
		mLife-=time;
		if(mLife<=0.f) return false;
	}

	if(mLockDirection != NULL)
	{
		CCoord3 pos =  getGlobalMatrix().getTranslation();
		CCoord3 scale = getGlobalMatrix().getScale();
		CVector3 direction = mLockDirection->getGlobalMatrix().getTranslation()-pos;
		
		pos =  getLocalMatrix().getTranslation();
		scale = getLocalMatrix().getScale();
		
		Identity();
		Translate(pos);
		Scale(scale);		
		
		if(mFather != NULL)
		{
			CQuaternion rot = mFather->getGlobalMatrix().getRotation();
			Rotate(rot.Conjugate());
		}

		Rotate(direction);
	}
	
	pEntity child = getChild();
	while(child != NULL)
	{
		pEntity next = child->getBrother();
		if(!child->Update(time)) child->Detach();
		child = next;
	}

	mHierarchyChanged = false;	// reset le flag de changement de hirarchie
	return true;
}

int CEntity::Display(int pass)
{
	int count = 0;
	
	if(pass <= 0)	// Pas d'environnement pour la pass 0
	{
		// Dessin
		count+=Draw(pass);

		// Passage aux enfants et calcul du rayon externe
		mExternalRadius = mInternalRadius;
		pEntity child = getChild();
		while(child != NULL)
		{
			count+=child->Display(pass);
			
			if(child->mExternalRadius >= 0.f)
			{
				float d=child->mLocalMatrix.getTranslation().Norm();
				d+=child->mExternalRadius*child->mLocalMatrix.getScale().abs().max();
				mExternalRadius=std::max(mExternalRadius,d);
			}
			
			child = child->getBrother();
		}
	}
	else {	// pass > 0
		if(pass == 1)
		{
			// Test de visibilit pendant la pass 1
			mExternalVisible = false;
			mInternalVisible = false;
			if(mExternalRadius >= 0.f)
			{
				// Application de la matrice locale
				glMatrixMode(GL_MODELVIEW);			
				glPushMatrix();
				mLocalMatrix.Apply();
				
				// Extrait le frustum
				pFrustum frustum = new CFrustum;
				frustum->Extract();					
				
				// Enlve la matrice locale
				glPopMatrix();						
				
				if(frustum->SphereInFrustum(mExternalRadius))		// sphre externe
				{
					mExternalVisible=true;
					if(mInternalRadius >= 0.f)
					{
						if(frustum->SphereInFrustum(mInternalRadius))	// sphre interne
							mInternalVisible=true;
					}	
				}
			}
		}

		if(mExternalVisible)
		{
			// Application de la matrice locale
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			mLocalMatrix.Apply();

			// Application des matrices de texture
			glMatrixMode(GL_TEXTURE);
			if(GLEW_ARB_multitexture)	// avec multitexturing
			{
				int max_textures;
				glGetIntegerv(GL_MAX_TEXTURE_UNITS,&max_textures);

				for(int unit=0; unit<max_textures; ++unit)
				{
					glActiveTextureARB(GL_TEXTURE0+unit);
					glPushMatrix();
					TextureMatrixMap_t::const_iterator it=mTextureMatrixes.find(unit);
					if(it!=mTextureMatrixes.end())
						it->second.Apply();
				}
			}
			else {	//sans multitexturing
				glPushMatrix();
				TextureMatrixMap_t::const_iterator it=mTextureMatrixes.find(0);
				if(it!=mTextureMatrixes.end()) it->second.Apply();
			}
			
			// Application de l'environnement de dessin
			glPushAttrib(GL_ENABLE_BIT);
			glEnable(GL_NORMALIZE);	// TODO: pas optimis
			for(EnvMap_t::iterator it = mEnv.begin(); it!= mEnv.end(); ++it)
			{
				if(it->second) glEnable(it->first);
				else glDisable(it->first);
			}
			
			// Inverse le sens des faces si la base est indirecte
			CVector3 scale = mLocalMatrix.getScale();
			bool direct=(scale.x*scale.y*scale.z >= 0.f);
			if(!direct)	// base indirecte
			{
				GLint front;
				glGetIntegerv(GL_FRONT_FACE,&front);
				if(front == GL_CCW) glFrontFace(GL_CW);
				else glFrontFace(GL_CCW);
			}
			
			if(mInternalVisible) count+=Draw(pass);
			
			// Passage aux enfants
			pEntity child = getChild();
			while(child != NULL)
			{
				count+=child->Display(pass);
				child = child->getBrother();
			}
					
			if(!direct)	// base indirecte
			{
				GLint front;
				glGetIntegerv(GL_FRONT_FACE,&front);
				if(front == GL_CCW) glFrontFace(GL_CW);
				else glFrontFace(GL_CCW);
			}
			
			// Enlve la matrice locale
			glMatrixMode(GL_MODELVIEW);
			glPopMatrix();

			// Enlve les matrices de textures
			glMatrixMode(GL_TEXTURE);
			if(GLEW_ARB_multitexture)	// avec multitexturing
			{
				int max_textures;
				glGetIntegerv(GL_MAX_TEXTURE_UNITS,&max_textures);

				for(int unit=0; unit<max_textures; ++unit)
				{
					glActiveTextureARB(GL_TEXTURE0+unit);
					glPopMatrix();
				}
			}
			else glPopMatrix(); //sans multitexturing
				
			// Dsactivation de l'environnement de dessin
			glPopAttrib();
		}
	}

	return count;
}

int CEntity::Draw( int pass)
{
	// Dummy
	return 0;
}

float CEntity::Intersect(const CCoord3 &pos,const CVector3 &move,float radius,CCoord3 *intersection)
{
	// Dummy
	return std::numeric_limits<float>::infinity();
}

float CEntity::Intersection(CCoord3 pos,CVector3 move,float radius,CCoord3 *intersection)
{
	// Pour dsactiver toutes les collisions (debug only)
	//return std::numeric_limits<float>::infinity();
	
	// Passe les vecteurs dans le repre local
	CMatrix4 inverse=getLocalMatrix().Inverse();
	pos=pos*inverse;
	inverse(0,3)=inverse(1,3)=inverse(2,3)=0.f;
	move=move*inverse;
	radius*=inverse.getScale().Norm()/SQRT3;	// Rem: Mauvais fonctionnement en cas de scale non uniforme !
	
	// OPTI
	float dist=pos.Norm();
	
	bool pass = true;
	if(mExternalRadius>=0.f)
		if(dist>mExternalRadius+radius)
		{
			float t=pos.IntersectSphere(move,CCoord3(0.f,0.f,0.f),mExternalRadius+radius);
			if(t<0.f || t>1.f) pass=false;
		}

	if(pass)
	{
		float t;
		bool intersect = true;
		if(mInternalRadius>=0.f)
			if(dist > mInternalRadius+radius)
			{
				t=pos.IntersectSphere(move,CCoord3(0.f,0.f,0.f),mInternalRadius+radius);
				intersect=(t>=0.f && t<=1.f); 
			}

		if(intersect) t=Intersect(pos,move,radius,intersection);
		else t=std::numeric_limits<float>::infinity();

		pEntity child = getChild();
		while(child != NULL)
		{
			CCoord3 childintersection;
			float tchild=child->Intersection(pos,move,radius,&childintersection);
				if(tchild<t)
				{
					t=tchild;
					if(intersection) *intersection=childintersection;
				}
			
			child = child->getBrother();
		}

		if(intersection) (*intersection)=(*intersection)*getLocalMatrix();
		return t;
	}
	else return std::numeric_limits<float>::infinity();
}

bool CEntity::Collision(const CCoord3 &pos,const CVector3 &move,float radius,CVector3 *result,CVector3 *intersect)
{
	CCoord3 intersection;
	float t1=Intersection(pos,move,radius,&intersection);

	// si il y a collision
	if(t1<=1.f)
	{
		if(intersect) *intersect = intersection;
		if(!result) return true;	// pas besoin de calculer la collision dans ce cas

		CVector3 v=move*(t1-EPSILON);

		// plan de sliding
		CCoord3 slideOrigin=intersection;
		CVector3 slideNormal=intersection-pos;

		// projette la destination sur le plan de sliding
		CCoord3 destination=pos+move;
		float t2=destination.IntersectPlane(slideNormal,slideOrigin,slideNormal);
		CCoord3 newdestination=destination+slideNormal*(t2-EPSILON);

		CVector3 newmove=newdestination-intersection;
		CCoord3 newpos=pos+v;

		CVector3 r;
		Collision(newpos,newmove,radius,&r,NULL);	// collisions suivantes
		*result=v+r;
		return true;
	}
	else 
	{
		if(result) *result=move;
		return false;
	}
}

void CEntity::setRadius(float radius)
{
	mInternalRadius=radius;
}



