/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "emitter.h"
#include "mediamanager.h"

// Constructeur
CEmitter::CEmitter(void) :
	mFrequency(10.f),
	mStartSpeed(0.f,10.f,0.f),
	mRandomSpeed(0.f,0.f,0.f),
	mIsRelativeSpeed(true),
	mStartLife(1.f),
	mRandomLife(0.f),
	mFriction(0.f,-1.f,0.f),
	mIsRelativeFriction(false),
	mSizex(1.f),
	mSizey(1.f),
	mRotSpeed(0.f),
	mIsRotRandom(false),
	mCooldown(0.f)
{

}


// Destructeur
CEmitter::~CEmitter(void)
{

}

void CEmitter::setMaterial(pMaterial material)
{
	mMaterial=material;
}
	
void CEmitter::setFrequency(float frequency)
{
	mFrequency=std::fabs(frequency);
	mCooldown=std::min(mCooldown,1.f/mFrequency);
}

void CEmitter::setSpeed(const CVector3 &speed,const CVector3 &random,bool relative)
{
	mStartSpeed=speed;
	mRandomSpeed=random.abs();
	mIsRelativeSpeed=relative;
}

void CEmitter::setPartLife(float life,float random)
{
	mStartLife=std::fabs(life);
	mRandomLife=std::fabs(random);
}

void CEmitter::setFriction(const CVector3 &friction,bool relative)
{
	mFriction=friction;
	mIsRelativeFriction=relative;
}

void CEmitter::setPartSize(float x,float y)
{
	mSizex=std::fabs(x);
	mSizey=std::fabs(y);
}

void CEmitter::setPartRotation(float speed,bool random)
{
	mRotSpeed=speed;
	mIsRotRandom=random;
}

void CEmitter::Parameter(const std::string &param,const std::string &data)
{
	std::stringstream sdata(data);

	if(param == "material")					setMaterial(MediaManager->Get<CMaterial>(data));
	else if(param == "frequency")			{sdata>>mFrequency; mCooldown=std::min(mCooldown,1.f/mFrequency);}
	else if(param == "speed")				sdata>>mStartSpeed;
	else if(param == "randomspeed")			sdata>>mRandomSpeed;
	else if(param == "speedrelative")		mIsRelativeSpeed=(data=="true");
	else if(param == "friction")			sdata>>mFriction;
	else if(param == "frictionrelative")	mIsRelativeFriction=(data=="true");
	else if(param == "partlife")			sdata>>mStartLife;
	else if(param == "partrandomlife")		sdata>>mRandomLife;
	else if(param == "size")				{sdata>>mSizex; sdata>>mSizey;}
	else if(param == "rotspeed")			mRotSpeed = getangle(sdata);
	else if(param == "rotrandom")			mIsRotRandom=(data=="true");
	else CEntity::Parameter(param,data);
}

bool CEmitter::Update(double time)
{
	mCooldown+=time;
	
	float period=1.f/mFrequency;
	if(mCooldown>=period)
	{
		CMatrix4 matrix(getGlobalMatrix());
		CVector3 pos = matrix.getTranslation();
		CMatrix4 rotation = matrix.NoTranslation();

		if(isHierarchyChanged()) mOldPos = pos;
		
		// les nouvelles particules
		float startCooldown = mCooldown;
		do {
			mCooldown-=period;
			pParticle p = CreateParticle();

			p->mMaterial =	mMaterial;
			p->mPos =		p->mPos*matrix + (mOldPos-pos)*mCooldown/startCooldown;
			p->mStartLife =	mStartLife;
			if(mRandomLife>0.f) p->mStartLife+=(float(std::rand()*2.f/RAND_MAX-1.f))*mRandomLife;
			p->mLife = p->mStartLife;

			p->mSpeed =		mStartSpeed + CVector3(	(float(std::rand())*2.f/RAND_MAX-1.f)*mRandomSpeed.x,
													(float(std::rand())*2.f/RAND_MAX-1.f)*mRandomSpeed.y,
													(float(std::rand())*2.f/RAND_MAX-1.f)*mRandomSpeed.z);
			if(mIsRelativeSpeed) p->mSpeed = p->mSpeed*rotation;
			
			p->mSizex =	mSizex;
			p->mSizey =	mSizey;
			p->mAnglez = float(std::rand())*2*PI/RAND_MAX;

			CScene::Current->AddParticle(p);

		} while(mCooldown>=period);

		mOldPos = pos;
	}
	
	if(mIsRelativeFriction) mGlobalFriction = mFriction*getGlobalMatrix().NoTranslation();
	else mGlobalFriction = mFriction;
	return CEntity::Update(time);
}

pParticle CEmitter::CreateParticle(void)
{
	return new CParticle(this);
}

/*
CBoxEmitter::CBoxEmitter(const CBox &box) : mBox(box)
{

}

CBoxEmitter::~CBoxEmitter(void)
{

}

void CBoxEmitter::setBox(const CBox &box)
{
	mBox=box;
}

pParticle CBoxEmitter::CreateParticle(void)
{
	pParticle p=new Particle;
	p->Pos.x=rand(_Box.p1.x,_Box.p2.x);
	p->Pos.y=rand(_Box.p1.y,_Box.p2.y);
	p->Pos.z=rand(_Box.p1.z,_Box.p2.z);
	return p;
}
*/
