/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#ifndef ANIMATION_H
#define ANIMATION_H

#include "vector3.h"
#include "quaternion.h"
#include "smartptr.h"
#include "include.h"

// Classe CLerp, stockant une animation linaire par keyframes
template<class T> class CLerp
{
public:
	void AddKey(double frame,const T &key);
	void DeleteKey(double frame);
	void DeleteInterval(double frame1,double frame2);
	void DeleteBefore(double frame);
	void DeleteAfter(double frame);

	// TODO: l'extrapolation est automatique
	T getFrame(double frame);
	size_t getCount(void);

private:
	typedef std::map<double,T> KeyFramesMap_t;
	KeyFramesMap_t mKeyFrames;
};

template<class T> void CLerp<T>::AddKey(double frame,const T &key)
{
	mKeyFrames[frame] = key;
}

template<class T> void CLerp<T>::DeleteKey(double frame)
{
	mKeyFrames.erase(frame);
}

template<class T> void CLerp<T>::DeleteInterval(double frame1,double frame2)
{
	typename KeyFramesMap_t::iterator it_begin = mKeyFrames.upper_bound(frame1);
	typename KeyFramesMap_t::iterator it_end = mKeyFrames.lower_bound(frame2);
	mKeyFrames.erase(it_begin,it_end);
}

template<class T> void CLerp<T>::DeleteBefore(double frame)
{
	typename KeyFramesMap_t::iterator it = mKeyFrames.lower_bound(frame);
	mKeyFrames.erase(mKeyFrames.begin(),it);
}

template<class T> void CLerp<T>::DeleteAfter(double frame)
{
	typename KeyFramesMap_t::iterator it = mKeyFrames.upper_bound(frame);
	mKeyFrames.erase(it,mKeyFrames.end());
}

template<class T> T CLerp<T>::getFrame(double frame)
{
	if(mKeyFrames.empty()) throw CException("Aucune clef d'animation");
	if(mKeyFrames.size() == 1) return mKeyFrames.begin()->second;
	typename KeyFramesMap_t::iterator it_end = mKeyFrames.lower_bound(frame);
	
	if(it_end == mKeyFrames.end()) --it_end;		// extrapolation  la fin
	if(it_end == mKeyFrames.begin()) ++it_end;		// extrapolation au dbut
	
	typename KeyFramesMap_t::iterator it_begin = it_end;
	--it_begin;

	float t = (frame - it_begin->first)/(it_end->first - it_begin->first);

	return it_begin->second.Lerp(it_end->second,t);
}

template<class T> size_t CLerp<T>::getCount(void)
{
	return mKeyFrames.size();
}

// Classe CAnimation, stockant une animation complte (position+rotation+scale)
class CAnimation : public IScriptable
{
private:
	CLerp<CVector3> 	mPosition;
	CLerp<CQuaternion> 	mRotation;
	CLerp<CVector3> 	mScale;
	double mLoop;

public:
	CAnimation(void)
	{
		mLoop = std::numeric_limits<double>::max();
	}

	~CAnimation(void) {}

	inline void setLoop(double frame)	{mLoop = frame;}
	
	inline void AddPositionKey(double frame,const CVector3 &key) 	{ mPosition.AddKey(frame,key); }
	inline void AddRotationKey(double frame,const CQuaternion &key)	{ mRotation.AddKey(frame,key); }
	inline void AddScaleKey(double frame,const CVector3 &key)		{ mScale.AddKey(frame,key); }

	inline size_t getCount(void)					{ return mPosition.getCount()+mRotation.getCount()+mScale.getCount(); }
	inline size_t getPositionCount(void)			{ return mPosition.getCount(); }
	inline size_t getRotationCount(void)			{ return mRotation.getCount(); }
	inline size_t getScaleCount(void)				{ return mScale.getCount(); }
	inline CVector3 getPosition(double frame)		{ return mPosition.getFrame(std::fmod(frame,mLoop)); }
	inline CQuaternion getRotation(double frame)	{ return mRotation.getFrame(std::fmod(frame,mLoop)); }
	inline CVector3 getScale(double frame)			{ return mScale.getFrame(std::fmod(frame,mLoop)); }
	
	inline void DeleteKey(double frame)
	{
		mPosition.DeleteKey(frame);
		mRotation.DeleteKey(frame);
		mScale.DeleteKey(frame);
	}
	
	inline void DeleteInterval(double frame1,double frame2)
	{
		mPosition.DeleteInterval(frame1,frame2);
		mRotation.DeleteInterval(frame1,frame2);
		mScale.DeleteInterval(frame1,frame2);
	}

	inline void DeleteBefore(double frame)
	{
		mPosition.DeleteBefore(frame);
		mRotation.DeleteBefore(frame);
		mScale.DeleteBefore(frame);
	}

	inline void DeleteAfter(double frame)
	{
		mPosition.DeleteAfter(frame);
		mRotation.DeleteAfter(frame);
		mScale.DeleteAfter(frame);
	}

	inline pScriptable Tag(const std::string &tag)
	{
		return NULL;
	}
	
	inline void Parameter(const std::string &param,const std::string &data)
	{
		std::stringstream sdata(data);
		double frame;
		sdata>>frame;
		
		if(param == "position")			{ CVector3 pos; sdata>>pos; AddPositionKey(frame,pos); }
		else if(param == "rotation")	{ CQuaternion rot; sdata>>rot; AddRotationKey(frame,rot); }
		else if(param == "scale")		{ CVector3 scale; sdata>>scale; AddScaleKey(frame,scale); }
		else if(param == "loop")		mLoop = frame;
	}
};

typedef CSmartPtr<CAnimation> pAnimation;

#endif	// ANIMATION_H
