/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "3dsloader.h"
#include "mediamanager.h"
#include "resourcemanager.h"
#include "material.h"
#include "texture.h"

C3dsLoader::C3dsLoader(void)
{

}

C3dsLoader::~C3dsLoader(void)
{

}

CMesh* C3dsLoader::Load(const std::string &filename)
{
	// Ouverture du fichier
	std::ifstream file;
	file.open(filename.c_str(),std::ios::in|std::ios::binary);
	if(!file.is_open()) throw CLoadingFailed(filename, "Ouverture impossible");
	
	uint16_t	chunk_id;		// ID du noeud
	uint32_t	chunk_lenght;	// longueur du noeud
	uint16_t	quantity;
	
	char buffer[256];

	enum {AMBIENT,DIFFUSE,SPECULAR}	colorType;
	enum {SHININESS,TRANSPARENCY}	percentType;

	std::stack<pMaterial> materials;
	materials.push(new CMaterial);

	std::vector<float32_t> Vertices;
	std::vector<float32_t> TexCoords;

	typedef CMesh::index_t index_t;
	std::vector<index_t> TempIndices;
	typedef std::map<pMaterial,std::vector<index_t> > IndicesMap_t;
	IndicesMap_t Indices;

	index_t	BaseVertices=0;
	index_t	BaseTexCoords=0;
	
	// Boucle de lecture
	while (file.good())						
	{	
		file.read((char*)&chunk_id,2);
		if (file.gcount()<2) break;		// fin du fichier
		file.read((char*)&chunk_lenght,4);	
		if(chunk_lenght<6) throw CLoadingFailed(filename, "Format invalide ou fichier corrompu");
		
		switch (chunk_id)
        {	
			// Noeud principal
			case 0x4d4d: 
				break;    

			// Noeud diteur
			case 0x3d3d:
				break;

			// Noeud des matriaux
			case 0xAFFF:
				break;

			// Noeud d'un nouveau materiau
			case 0xA000:
				{
					file.getline(buffer,256,0);		// lecture du nom
					std::string name(filename+'|'+buffer);	

					pMaterial mat=CResourceManager::Instance()->Get<CMaterial>(name);
					if(mat!=NULL)	// si dj charg
					{
						file.seekg(chunk_lenght-7-strlen(buffer),std::ios::cur);
						materials.push(mat);
					}
					else {
						materials.push(new CMaterial);
						CResourceManager::Instance()->Add(name,materials.top());
					}
				}
				break;
			
			// Noeud de couleur ambiante d'un materiau
			case 0xA010:
				colorType=AMBIENT;
				break;
			
			// Noeud de couleur diffuse d'un materiau
			case 0xA020:
				colorType=DIFFUSE;
				break;
			
			// Noeud de couleur speculaire d'un materiau
			case 0xA030:
				colorType=SPECULAR;
				break;
		
			// Noeud de l'clat d'un materiau
			case 0xA040:
				percentType=SHININESS;
				break;
			
			// Noeud de la transparence d'un materiau
			case 0xA050:
				percentType=TRANSPARENCY;
				break;
			
			// Noeud de la texture d'un materiau
			case 0xA200:
				break;

			// Noeud du fichier de texture d'un materiau
			case 0xA300:
				{				
					file.getline(buffer,256,0);
					pTexture texture=CResourceManager::Instance()->Get<CTexture>(buffer);
					
					if(texture==NULL)
					{
						texture=CMediaManager::Instance()->Get<CTexture>(buffer);
						if(texture!=NULL) CResourceManager::Instance()->Add(buffer,texture);
					}

					materials.top()->setTexture(texture);
				}
				break;

			// Noeud de couleur (flottants)
			case 0x0010:
				{
					float32_t r,g,b;
					file.read ((char*)&r,4);
					file.read ((char*)&g,4);
					file.read ((char*)&b,4);

					switch (colorType)
					{
						case AMBIENT: 
							materials.top()->setAmbient(CColor(r,g,b)*.2f); 
							break;
						case DIFFUSE: 
							materials.top()->setDiffuse(CColor(r,g,b)); 
							break;
						case SPECULAR: 
							materials.top()->setSpecular(CColor(r,g,b));
							break;
					}
					break;
				}

			
			// Noeud de couleur avec correction gamma (flottants)
			case 0x0013:
				{
					float32_t r,g,b;
					file.read ((char*)&r,4);
					file.read ((char*)&g,4);
					file.read ((char*)&b,4);
						
					switch (colorType)
					{
						case AMBIENT:
							materials.top()->setAmbient(CColor(r,g,b));
							break;
						case DIFFUSE:
							materials.top()->setDiffuse(CColor(r,g,b));
							break;
						case SPECULAR: 
							materials.top()->setSpecular(CColor(r,g,b));
							break;
					}
					break;
				}
			
			// Noeud de couleur (bytes)
				case 0x0011:
				{
					uint8_t r,g,b;
					file.read ((char*)&r,1);
					file.read ((char*)&g,1);
					file.read ((char*)&b,1);
						
					switch (colorType)
					{
						case AMBIENT: 
							materials.top()->setAmbient(CColor(r,g,b)*.2f); 
							break;
						case DIFFUSE: 
							materials.top()->setDiffuse(CColor(r,g,b)); 
							break;
						case SPECULAR: 
							materials.top()->setSpecular(CColor(r,g,b));
							break;
					}
					break;
				}
			
			// Noeud de couleur avec correction gamma (bytes)
			case 0x0012:
				{
					uint8_t r,g,b;
					file.read ((char*)&r,1);
					file.read ((char*)&g,1);
					file.read ((char*)&b,1);
						
					switch (colorType)
					{
						case AMBIENT: 
							materials.top()->setAmbient(CColor(r,g,b)); 
							break;
						case DIFFUSE:
							materials.top()->setDiffuse(CColor(r,g,b));
							break;
						case SPECULAR: 
							materials.top()->setSpecular(CColor(r,g,b));
							break;
					}
					break;
				}
			
			
			// Noeud de pourcentage	(word)
			case 0x0030:
				{
					sint16_t value;
					file.read ((char*)&value,2);
					switch (percentType)
					{
						case SHININESS: 
							materials.top()->setShininess(int((100-value)*1.28f));
							break;
						case TRANSPARENCY: 
							materials.top()->setAlpha(float(100-value)/100.f);
							break;
					}
					break;
				}
			
			
			// Noeud de pourcentage	(flottant)
			case 0x0031:
				{
					float value;
					file.read((char*)&value,4);
					switch (percentType)
					{
						case SHININESS: materials.top()->setShininess(int((100.f-value)*1.28f)); break;
						case TRANSPARENCY: materials.top()->setAlpha(1.f-value/100.f); break;
					}
					break;
				}
			
				
			// Nouvel object
			case 0x4000:
				file.getline(buffer,256,0);
				break;

			// Nouveau mesh
			case 0x4100:
				break;
			
			// Liste des vertex
			case 0x4110:
				{
					file.read((char*)&quantity,2);
					BaseVertices=Vertices.size()/3;
					Vertices.resize((BaseVertices+quantity)*3);
					BaseTexCoords=TexCoords.size()/2;
					TexCoords.resize((BaseTexCoords+quantity)*2);
					
					for(unsigned int i=BaseVertices*3; i<Vertices.size(); i+=3)
					{
						file.read((char*)&Vertices[i],4);
						file.read((char*)&Vertices[i+2],4);
						file.read((char*)&Vertices[i+1],4);
						Vertices[i+2]=-Vertices[i+2];
					}
				}
				break;

			// Liste des faces
			case 0x4120:
				{
					file.read((char*)&quantity,2);
					TempIndices.resize(quantity*3);
					
					for(unsigned int i=0; i<TempIndices.size(); i+=3)
					{
						uint16_t nbr;
						// lecture des indices des vertices
						file.read((char*)&nbr,2);
						TempIndices[i]=nbr+BaseVertices;
						file.read((char*)&nbr,2);
						TempIndices[i+1]=nbr+BaseVertices;
						file.read((char*)&nbr,2);
						TempIndices[i+2]=nbr+BaseVertices;
						
						// lecture des flags de la face (ne servent  rien)
						uint16_t face_flags;
						file.read((char*)&face_flags,2);
					}
				}
				break;
				
			//Bloc matriau
			case 0x4130:
				{
					file.getline(buffer,256,0);
					std::string name(filename+'|'+buffer);

					pMaterial mat=CResourceManager::Instance()->Get<CMaterial>(name);
					if (mat==NULL) mat=CMaterial::Default;
					
					file.read((char*)&quantity,2);
					for(uint16_t i=0; i<quantity; ++i)
					{
						uint16_t nbr;
						file.read((char*)&nbr,2);
						
						if(i<TempIndices.size()*3)
						{
							Indices[mat].push_back(TempIndices[nbr*3]);
							Indices[mat].push_back(TempIndices[nbr*3+1]);
							Indices[mat].push_back(TempIndices[nbr*3+2]);
						}
					}
				}
				break;
			
			// Coordonnes de mapping
			case 0x4140:
				{
					file.read((char*)&quantity,2);

					for(unsigned int i=BaseTexCoords*2; i<TexCoords.size(); i+=2)
					{
						file.read((char*)&TexCoords[i],4);
						file.read((char*)&TexCoords[i+1],4);
					}
				}
                break;
			
			// Chunk inconnu
			default:
				file.seekg(chunk_lenght-6,std::ios::cur);	// 6 = taille du header du chunk
		} 
	}

	file.close();

	TempIndices.clear();

	for(IndicesMap_t::const_iterator it=Indices.begin(); it!=Indices.end(); ++it)
		TempIndices.insert(TempIndices.end(), it->second.begin(), it->second.end());

	const float *texcoords=&TexCoords[0];
	CMesh *mesh=new CMesh(&Vertices[0],Vertices.size()/3,&TempIndices[0],TempIndices.size(),&texcoords,1);

	index_t count=0;
	for(IndicesMap_t::const_iterator it=Indices.begin(); it!=Indices.end(); ++it)
	{
		mesh->setMaterial(it->first,count);
		count+=it->second.size();
	}

	return mesh;
}
