/*
 * pdsninput.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsninput.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsNInput                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsNInput *pds_ninput_new(void)
 *  \brief Crea una neurona de tipo PdsNInput e inicia con cero todos los elementos.
 *  \return Un puntero al vector de tipo PdsNInput.
 *  \ingroup PdsNInputGroup
 */
PdsNInput *pds_ninput_new(void)
{
	PdsNInput *NInput=NULL;
	
	NInput=(PdsNInput *)calloc(1,sizeof(PdsNInput));
	if(NInput==NULL)	return NULL;

	NInput->Y[0]=0.0;
	NInput->Y[1]=0.0;

	return NInput;
}

/** \fn int pds_ninput_get_input(const PdsNInput *NInput,PdsSnReal *m)
 *  \brief Devuelve el valor de la entrada  de la neurona NInput.
 *  \param[in] NInput La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la entrada.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NInput==NULL). 
 *  \ingroup PdsNInputGroup
 */
int pds_ninput_get_input(const PdsNInput *NInput,PdsSnReal *m)
{
	*m=0;
	if(NInput==NULL)		return FALSE;

	*m=*(NInput->Y);

	return TRUE;
}


/** \fn int pds_ninput_get_output(const PdsNInput *NInput,PdsSnReal *m)
 *  \brief Devuelve el valor de la salida Y de la neurona NInput.
 *  \param[in] NInput La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NInput==NULL). 
 *  \ingroup PdsNInputGroup
 */
int pds_ninput_get_output(const PdsNInput *NInput,PdsSnReal *m)
{
	*m=0;
	if(NInput==NULL)		return FALSE;

	*m=*(NInput->Y);

	return TRUE;
}


/** \fn int pds_ninput_set_input(PdsNInput *NInput,PdsSnReal m)
 *  \brief Escribe el valor en la entrada de la neurona NInput.
 *  \param[in,out] NInput La neurona en consulta.
 *  \param[in] m El valor que se cargará en la entrada.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NInput==NULL). 
 *  \ingroup PdsNInputGroup
 */
int pds_ninput_set_input(PdsNInput *NInput,PdsSnReal m)
{
	if(NInput==NULL)		return FALSE;

	*(NInput->Y)=m;

	return TRUE;
}

/** \fn void pds_ninput_free(PdsNInput *NInput)
 *  \brief Libera una neurona de tipo puntero PdsNInput.
 *  \param[in,out] NInput La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNInputGroup
 */
void pds_ninput_free(PdsNInput *NInput)
{
	if(NInput!=NULL)
	{
		free(NInput);
	}
}


/** \fn void pds_ninput_destroy(PdsNInput **NInput)
 *  \brief Libera una neurona de tipo puntero PdsNInput, y limpia el puntero con NULL.
 *  \param[in,out] NInput La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNInputGroup
 */
void pds_ninput_destroy(PdsNInput **NInput)
{
	if((*NInput)!=NULL)
	{
		free(*NInput);
		(*NInput)=NULL;
	}
}

