/*
 * pdsregionrect.h
 * 
 * Copyright 2018 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsregionrect.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 * \brief Archivo de definición de variable.
 *
 */
#ifndef __PDSREGIONRECT_H__
#define __PDSREGIONRECT_H__


#include <pds/pdsraglobal.h>


#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsRegionRectGroup Módulo PdsRegionRect.
 *  \brief Funciones que trabajan con Regiones.
 * @{
 */

#ifndef PDS_OK
/*!
  \def PDS_OK
  Valor retornado en caso de exito, este es un valor diferente que cero.
*/
	#define PDS_OK 1
#endif

#ifndef PDS_WRONG
/*!
  \def PDS_WRONG
  Valor retornado en caso de fracaso, este valor es igual a cero.
*/
	#define PDS_WRONG 0
#endif


/*! \struct PdsRegionRect
 *  \brief La estructura tipo  PdsRegionRect .
 *  Esta estructura genera unnumero complejo.
 *  Para usar incluir pdsregionrect/pdsregionrect.h.
 *  \ingroup PdsRegionRectGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Linea inicial */
	PdsRaInteger L0;
	/*! Columna inicial */
	PdsRaInteger C0;
	/*! Numero de Lineas */
	PdsRaNatural Nlin;
	/*! Numero de Columnas */
	PdsRaNatural Ncol;

}PdsRegionRect;

/** @name pds_region_rect
 *  Creando una nueva estructura
 * @{
 */

/** \fn PdsRegionRect pds_region_rect(PdsRaInteger L0,PdsRaInteger C0,PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea la variable PdsRegionRect.
 *  \param[in] L0 Linea inicial.
 *  \param[in] C0 Columna inicial.
 *  \param[in] Nlin Numero de lineas.
 *  \param[in] Ncol Numero de columnas.
 *  \return La variable PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
PdsRegionRect pds_region_rect(PdsRaInteger L0,PdsRaInteger C0,PdsRaNatural Nlin,PdsRaNatural Ncol);

//@}

/** @name pds_region_rect_<moving>
 *  Moviendo de posicion la region
 * @{
 */


/** \fn PdsRegionRect pds_region_rect_transfer(PdsRegionRect R,PdsRaInteger L,PdsRaInteger C)
 *  \brief Crea una variable PdsRegionRect a partir de suma R+(L,C).
 *  \param[in] R La region a transferir.
 *  \param[in] L Numero de lineas a sumar.
 *  \param[in] C Numero de columnas a sumar.
 *  \return La variable PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
PdsRegionRect pds_region_rect_transfer(PdsRegionRect R,PdsRaInteger L,PdsRaInteger C);


/** \fn PdsRegionRect pds_region_rect_move(PdsRegionRect R,PdsRaInteger L0,PdsRaInteger C0)
 *  \brief Crea una variable PdsRegionRect, siendo esta una version de R movimentada a (L0,C0).
 *  \param[in] R La region a mover.
 *  \param[in] L0 Nueva linea de origen.
 *  \param[in] C0 Nueva columna de origen
 *  \return La variable PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
PdsRegionRect pds_region_rect_move(PdsRegionRect R,PdsRaInteger L0,PdsRaInteger C0);


//@}

/** @name pds_region_rect_<set_operations>
 *  Operaciones de conjunto con las regiones
 * @{
 */

/** \fn int pds_region_rect_intersection(PdsRegionRect A,PdsRegionRect B,PdsRegionRect *R)
 *  \brief Intersecta dos regiones.
 *  \param[in] A Primera region.
 *  \param[in] B Segunda region.
 *  \param[out] R Region de interesecion si existe, o ninguna mudanza si  no.
 *  \return PDS_OK si existe la intersecion, o PDS_WRONG si no existe.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_intersection(PdsRegionRect A,PdsRegionRect B,PdsRegionRect *R);


/** \fn int int pds_region_rect_is_inside(PdsRegionRect R,PdsRegionRect Rin)
 *  \brief Indica si la region Rin está dentro de la region R.
 *  \param[in] Rin Suúesta region interna de R.
 *  \param[in] R Region madre o contenedora.
 *  \return PDS_OK si Rin está adentro de R, o PDS_WRONG si no.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_is_inside(PdsRegionRect Rin,PdsRegionRect R);


//@}

/** @name pds_region_rect_<printf>
 *  Imprimir datos de la region
 * @{
 */

/** \fn int pds_region_rect_printf(PdsRegionRect R)
 *  \brief Imprime la variable PdsRegionRect.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_printf(PdsRegionRect R);

/** \fn int pds_region_rect_stylized_printf(PdsRegionRect R)
 *  \brief Imprime la variable PdsRegionRect.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_stylized_printf(PdsRegionRect R);

/** \fn int pds_region_rect_fprintf(FILE *fd, PdsRegionRect R)
 *  \brief Imprime en un FILE la variable PdsRegionRect.
 *  \param[in] fd File descriptor.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_fprintf(FILE *fd, PdsRegionRect R);

/** \fn int pds_region_rect_stylized_fprintf(FILE *fd, PdsRegionRect R)
 *  \brief Imprime en un FILE la variable PdsRegionRect.
 *  \param[in] fd File descriptor.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_stylized_fprintf(FILE *fd, PdsRegionRect R);


//@}
 


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSREGIONRECT_H__ */
