/*
 * pdsmatrix.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsrautils.h>
#include <pds/pdsmatrix.h>


////////////////////////////////////////////////////////////////////////////////
////                        Funciones Extras                                ////
////////////////////////////////////////////////////////////////////////////////

int pds_matrix_find_row_in_col(const PdsMatrix *Matrix, PdsRaNatural *row, PdsRaNatural *E, PdsRaNatural col, PdsRaNatural r1);

////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsMatrix                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsMatrix *pds_matrix_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea una matriz de tipo PdsMatrix.
 *  \param[in] Nlin Es el número de lineas de la matriz.
 *  \param[in] Ncol Es el número de columnas de la matriz.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
{
	PdsMatrix *Matrix=NULL;
	PdsRaNatural i;

	Matrix=(PdsMatrix *)calloc(1,sizeof(PdsMatrix));
	if(Matrix==NULL) {return NULL;}

	Matrix->Nlin=Nlin;
	Matrix->Ncol=Ncol;
	Matrix->M=(PdsRaReal **)calloc(Matrix->Nlin,sizeof(PdsRaReal *));
	if(Matrix->M==NULL) 
	{
		free(Matrix);
		return NULL;
	}
	
	for(i=0;i<Matrix->Nlin;i++)
	{
		Matrix->M[i]=(PdsRaReal *)calloc(Matrix->Ncol,sizeof(PdsRaReal));
		if(Matrix->M[i]==NULL) 
		{
			for(i--;i>=0;i--)	free(Matrix->M[i]);
			free(Matrix);
			return NULL;
		}
	}

	return Matrix;
}


/** \fn PdsMatrix *pds_matrix_new_matrix(const PdsMatrix *MatSrc)
 *  \brief Crea una matriz de tipo PdsMatrix a partir de la matriz MatSrc.
 *  \param[in] MatSrc Matriz que se usará como imagen como fuente.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_matrix(const PdsMatrix *MatSrc)
{
	PdsMatrix *Matrix=NULL;
	PdsRaNatural i,j;

	if(MatSrc==NULL)	return NULL;

	Matrix=(PdsMatrix *)calloc(1,sizeof(PdsMatrix));
	if(Matrix==NULL) {return NULL;}

	Matrix->Nlin=MatSrc->Nlin;
	Matrix->Ncol=MatSrc->Ncol;
	Matrix->M=(PdsRaReal **)calloc(Matrix->Nlin,sizeof(PdsRaReal *));
	if(Matrix->M==NULL) 
	{
		free(Matrix);
		return NULL;
	}
	
	for(i=0;i<Matrix->Nlin;i++)
	{
		Matrix->M[i]=(PdsRaReal *)calloc(Matrix->Ncol,sizeof(PdsRaReal));
		if(Matrix->M[i]==NULL) 
		{
			for(i--;i>=0;i--)	free(Matrix->M[i]);
			free(Matrix);
			return NULL;
		}
		else
		{
			for(j=0;j<Matrix->Ncol;j++)	Matrix->M[i][j]=MatSrc->M[i][j];
		}
	}

	return Matrix;
}


/** \fn PdsMatrix *pds_matrix_new_load_data(const char* datafile)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos de la linea
 *  line del archivo datafile.
 *  Usa TAB como delimitador de elemento y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la funcion retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o line inexistente)
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_load_data(const char* datafile)
{
	PdsRaNatural i,j,nc,nl;
	int id;
	int tmp;
	FILE *fd=NULL;
	char x[__MAX_CHAR_DATA_SIZE__];
	PdsMatrix *Matrix=NULL;
	
	fd=fopen(datafile,"r");
	if(fd==NULL)	return NULL;

	id=pds_load_number_of_columns_and_lines(fd,&nc,&nl);
	if(id==FALSE)	
	{
		fclose(fd);
		return NULL;
	}


	Matrix=pds_matrix_new(nl,nc);
	if(Matrix==NULL)	
	{
		fclose(fd);
		return NULL;
	}

	for(j=0;j<nl;j++)
	for(i=0;i<nc;i++)
	{
		x[0]=0;
		tmp=fscanf(fd,"%s",x);
		if(strlen(x)==0)	break;

		Matrix->M[j][i]=atof(x);
	}
	
	fclose(fd);
	return Matrix;
}


/** \fn int pds_matrix_copy_matrix(PdsMatrix *Matrix,PdsRaNatural X1,PdsRaNatural Y1,const PdsMatrix *MatSrc,PdsRaNatural X2,PdsRaNatural Y2)
 *  \brief Copia en la matriz Matrix en la posicion (X1,Y1) la matriz MatSrc 
 *  desde la posicion (X2,Y2).
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(X1,Y1)=MatSrc(X2,Y2)</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] X1 Linea en la que se copiará.
 *  \param[in] Y1 Columna en la que se copiará.
 *  \param[in] MatSrc Matriz que se usara como fuente.
 *  \param[in] X2 Linea que se copiará.
 *  \param[in] Y2 Columna que se copiará.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_matrix(PdsMatrix *Matrix,PdsRaNatural X1,PdsRaNatural Y1,const PdsMatrix *MatSrc,PdsRaNatural X2,PdsRaNatural Y2)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return FALSE;
	if(MatSrc==NULL)	return FALSE;

	if((Matrix->Nlin-X1) < (MatSrc->Nlin-X2))	Nlin=Matrix->Nlin-X1;
	else						Nlin=MatSrc->Nlin-X2;

	if((Matrix->Ncol-Y1) < (MatSrc->Ncol-Y2))	Ncol=Matrix->Ncol-Y1;
	else						Ncol=MatSrc->Ncol-Y2;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i+X1][j+Y1]=MatSrc->M[i+X2][j+Y2];
	}

	return TRUE;
}


/** \fn int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
 *  \brief Copia vector VecSrc en la columna col de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] col Columna a escribir.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
{
	PdsRaNatural i,j;
	PdsRaNatural N;

	if(Matrix==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;
	if(col>=Matrix->Ncol)	return FALSE;

	if( Matrix->Nlin < VecSrc->Nel )	N=Matrix->Nlin;
	else					N=VecSrc->Nel;

	for(i=0;i<N;i++)	Matrix->M[i][col]=VecSrc->V[i];

	return TRUE;
}


/** \fn int pds_matrix_copy_vector_lin(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural lin)
 *  \brief Copia vector VecSrc en la linea lin de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] lin Linea a escribir.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_vector_lin(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural lin)
{
	PdsRaNatural i,j;
	PdsRaNatural N;

	if(Matrix==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;
	if(lin>=Matrix->Nlin)	return FALSE;

	if( Matrix->Ncol < VecSrc->Nel )	N=Matrix->Ncol;
	else					N=VecSrc->Nel;

	for(i=0;i<N;i++)	Matrix->M[lin][i]=VecSrc->V[i];

	return TRUE;
}

/** \fn int pds_matrix_copy_identity(PdsMatrix *Matrix,PdsRaNatural X,PdsRaNatural Y,PdsRaReal Val)
 *  \brief Copia en la matriz Matrix una matriz identidad(Val) en la posicion (X,Y).
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(NLin,NCol)=MatSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] X Linea en la que se copiará.
 *  \param[in] Y Columna en la que se copiará.
 *  \param[in] Val El valor que será escrito en la diagonal.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_identity(PdsMatrix *Matrix,PdsRaNatural X,PdsRaNatural Y,PdsRaReal Val)
{
	PdsRaNatural i,j;
	PdsRaNatural N;

	if(Matrix==NULL)	return FALSE;

	if((Matrix->Nlin-X) < (Matrix->Ncol-Y))	N=(Matrix->Nlin-X);
	else					N=(Matrix->Ncol-Y);

	for(i=0;i<N;i++)
	{
		for(j=0;j<N;j++)
			if(i==j)	Matrix->M[i+X][j+Y]=Val;
			else		Matrix->M[i+X][j+Y]=0;
	}

	return TRUE;
}


/** \fn int pds_matrix_init_identity(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con una diagonal de valor 
 *  Val y llena el resto de ceros. <br><b>Matrix=Val*I</b>.
 *  \param[in,out] Matrix La matriz a iniciar.
 *  \param[in] Val El valor que será escrito en la diagonal.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_identity(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			if(i==j)	Matrix->M[i][j]=Val;
			else		Matrix->M[i][j]=0;
		}
	}

	return TRUE;
}


/** \fn int pds_matrix_init_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con una matriz.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix=MatSrc</b>.
 *  \param[in,out] Matrix La matriz a iniciar.
 *  \param[in] MatSrc Matriz que se usara como fuente.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return FALSE;
	if(MatSrc==NULL)	return FALSE;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=MatSrc->M[i][j];
	}

	return TRUE;
}


/** \fn int pds_matrix_init_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con un valor. <br><b>Matrix=Val</b>.
 *  \param[in,out] Matrix La matriz a iniciar con un valor.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_value(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		Matrix->M[i][j]=Val;
	}

	return TRUE;
}


/** \fn int pds_matrix_add_identity(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Suma a la matriz de tipo puntero PdsMatrix con una diagonal de valor 
 *  Val y suma cero al resto. <br><b>Matrix=Matrix+Val*I</b>.
 *  \param[in,out] Matrix La matriz destino y operando de la suma.
 *  \param[in] Val El valor que será sumado a la diagonal.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_identity(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			if(i==j)	Matrix->M[i][j]=Matrix->M[i][j]+Val;
		}
	}

	return TRUE;
}



/** \fn int pds_matrix_add_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Suma la matriz de tipo puntero PdsMatrix con una matriz MatSrc y lo 
 *  carga en Matrix. Si los tamaños son diferentes intersecta los tamaños y hace 
 *  la copia en la  intersección solamente. <br><b>Matrix=Matrix+MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se sumará a Matrix.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return FALSE;
	if(MatSrc==NULL)	return FALSE;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]+MatSrc->M[i][j];
	}

	return TRUE;
}


/** \fn int pds_matrix_add_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Suma a la matriz de tipo puntero PdsMatrix un valor. <br><b>Matrix=Matrix+Val</b>.
 *  \param[in,out] Matrix La matriz a la cual se le sumará un valor.
 *  \param[in] Val Es el valor que se sumará a los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_value(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]+Val;
	}

	return TRUE;
}


/** \fn int pds_matrix_sub_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Resta la matriz de tipo puntero PdsMatrix con una matriz MatSrc y lo 
 *  carga en Matrix. Si los tamaños son diferentes intersecta los tamaños y hace 
 *  la copia en la  intersección solamente. <br><b>Matrix=Matrix-MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se sumará a Matrix.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_sub_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return FALSE;
	if(MatSrc==NULL)	return FALSE;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]-MatSrc->M[i][j];
	}

	return TRUE;
}


/** \fn int pds_matrix_mul_matrix_elements(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Multiplica la matriz de tipo puntero PdsMatrix con una matriz MatSrc 
 *  elemento a elemento y lo carga en Matrix. Si los tamaños son diferentes 
 *  intersecta los tamaños y hace la copia en la  intersección solamente. <br><b>Matrix=Matrix.MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se multiplicará a Matrix.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_matrix_elements(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return FALSE;
	if(MatSrc==NULL)	return FALSE;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]*MatSrc->M[i][j];
	}

	return TRUE;
}


/** \fn int pds_matrix_mul_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
 *  \brief Multiplica la matriz MatSrc1 (Nlin1xNcol1) con la matriz MatSrc2 (Nlin2xNcol2)
 *  y lo carga en Matrix (Nlin1xNcol2). <br><b>Matrix=MatSrc1*MatSrc2</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino, esta matriz ya debe 
 *  estar creada, todo su contenido será sobrescrito.
 *  \param[in] MatSrc1 Matriz que se multiplicará a MatSrc2.
 *  \param[in] MatSrc2 Matriz que se multiplicará a MatSrc1.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  Puede retornar FALSE si alguna de las matrices es NULL, o si el número de 
 *  lineas (Nlin1) de Matrix no corresponde al número de lineas (Nlin1) de MatSrc1, 
 *  o si el número de columnas (Ncol2) de Matrix no corresponde al número de 
 *  columnas (Ncol2) de MatSrc2, o si Ncol1!=Nlin2, e cualquiera de esos errores
 *  la matriz Matrix no es alterada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
{
	PdsRaNatural i,j,k;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;
	PdsRaNatural N;
	PdsRaReal S;

	if(Matrix==NULL)	return FALSE;
	if(MatSrc1==NULL)	return FALSE;
	if(MatSrc2==NULL)	return FALSE;

	if(Matrix->Nlin  != MatSrc1->Nlin)	return FALSE;
	if(Matrix->Ncol  != MatSrc2->Ncol)	return FALSE;

	if(MatSrc1->Ncol != MatSrc2->Nlin)	return FALSE;

	Nlin=Matrix->Nlin;
	Ncol=Matrix->Ncol;

	N=MatSrc1->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		{
			S=0;
			for(k=0;k<N;k++)
			{
				S=S+MatSrc1->M[i][k]*MatSrc2->M[k][j];
			}
			Matrix->M[i][j]=S;
		}
	}

	return TRUE;
}


/** \fn PdsMatrix *pds_matrix_mul_matrix_new(const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
 *  \brief Multiplica la matriz MatSrc1 (Nlin1xNcol1) con la matriz MatSrc2 (Nlin2xNcol2)
 *  y devuelve una matriz Matrix (Nlin1xNcol2). <br><b>Matrix=MatSrc1*MatSrc2</b>.
 *  \param[in] MatSrc1 Matriz que se multiplicará a MatSrc2.
 *  \param[in] MatSrc2 Matriz que se multiplicará a MatSrc1.
 *  \return La multiplicación de las matrices si todo fue bien o NULL si no.
 *  Puede retornar NULL si alguna de las matrices es NULL, o si Ncol1!=Nlin2.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mul_matrix_new(const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
{
	PdsRaNatural i,j,k;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;
	PdsRaNatural N;
	PdsRaReal S;

	PdsMatrix *Matrix=NULL;

	if(MatSrc1==NULL)	return NULL;
	if(MatSrc2==NULL)	return NULL;

	if(MatSrc1->Ncol != MatSrc2->Nlin)	return NULL;

	Nlin=MatSrc1->Nlin;
	Ncol=MatSrc2->Ncol;

	N   =MatSrc1->Ncol;

	Matrix=pds_matrix_new(Nlin,Ncol);
	if(Matrix==NULL)	return NULL;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		{
			S=0;
			for(k=0;k<N;k++)
			{
				S=S+MatSrc1->M[i][k]*MatSrc2->M[k][j];
			}
			Matrix->M[i][j]=S;
		}
	}

	return Matrix;
}


/** \fn int pds_matrix_mul_vector(PdsVector *A,const PdsMatrix *M,const PdsVector *V)
 *  \brief Multiplica la matriz M con el vector V y devuelve un vector A. 
 *  <br><b>A=M*V</b>.
 *  \param[out] A Vector donde se cargará la multiplicación.
 *  \param[in] M Matriz que se multiplicará.
 *  \param[in] V Vector que se multiplicará.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  Puede retornar FALSE si la matriz no tiene el mismo número de columnas que elementos
 *  tiene V. O si el numero de elementos de A es distinto al número de lineas de M.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_vector(PdsVector *A,const PdsMatrix *M,const PdsVector *V)
{
	PdsRaNatural i,j;

	if(M==NULL)	return FALSE;
	if(V==NULL)	return FALSE;
	if(A==NULL)	return FALSE;

	if(M->Ncol!=V->Nel)	return FALSE;
	if(M->Nlin!=A->Nel)	return FALSE;

	for(i=0;i<M->Nlin;i++)
	{
		A->V[i]=0;
		for(j=0;j<M->Ncol;j++)
		{
			A->V[i]=A->V[i]+M->M[i][j]*V->V[j];
		}
	}
	return TRUE;
}


/** \fn PdsVector *pds_matrix_mul_vector_new(const PdsMatrix *M,const PdsVector *V)
 *  \brief Multiplica la matriz M con el vector V y devuelve un vector A. 
 *  <br><b>A=M*V</b>.
 *  \param[in] M Matriz que se multiplicará.
 *  \param[in] V Vector que se multiplicará.
 *  \return La multiplicación de la matriz por el vector si todo fue bien o NULL si no.
 *  Puede retornar NULL si la matriz no tiene el mismo número de columnas que elementos
 *  tiene V. O si el numero de elementos de A es distinto al número de lineas de M.
 *  \ingroup PdsMatrixGroup
 */
PdsVector *pds_matrix_mul_vector_new(const PdsMatrix *M,const PdsVector *V)
{
	PdsRaNatural i,j;
	PdsVector *A=NULL;

	if(M==NULL)	return NULL;
	if(V==NULL)	return NULL;

	if(M->Ncol!=V->Nel)	return NULL;

	A=pds_vector_new(M->Nlin);
	if(A==NULL)	return NULL;

	for(i=0;i<M->Nlin;i++)
	{
		A->V[i]=0;
		for(j=0;j<M->Ncol;j++)
		{
			A->V[i]=A->V[i]+M->M[i][j]*V->V[j];
		}
	}
	return A;
}


/** \fn int pds_matrix_mul_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Multiplica a la matriz de tipo puntero PdsMatrix por un valor.  <br><b>Matrix=Val*Matrix</b>.
 *  \param[in,out] Matrix La matriz a la cual se le multiplicará un valor.
 *  \param[in] Val Es el valor que se sumará a los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_value(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]*Val;
	}

	return TRUE;
}


/** \fn PdsMatrix *pds_matrix_transpose_new(const PdsMatrix *M)
 *  \brief Genera la transpuesta de la matriz.  <br><b>Mt=M^T</b>.
 *  \param[in] M La matriz a la cual se le aplicará la transpuesta.
 *  \return La transpuesta de la matriz o NULL en caso de error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_transpose_new(const PdsMatrix *M)
{
	PdsRaNatural i,j;
	PdsMatrix *Mt=NULL;

	if(M==NULL)	return NULL;

	Mt=pds_matrix_new(M->Ncol,M->Nlin);
	if(Mt==NULL)	return NULL;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<M->Nlin;j++)
		{
			Mt->M[i][j]=M->M[j][i];
		}
	}
	return Mt;
}


/** \fn int pds_matrix_transpose(PdsMatrix *Mt,const PdsMatrix *M)
 *  \brief Genera la transpuesta de la matriz.  <br><b>Mt=M^T</b>.
 *  \param[in] Mt La transpuesta de la matriz M.
 *  \param[in] M La matriz a la cual se le aplicará la transpuesta.
 *  \return RUE si todo fue bien o FALSE si no.(ej. M==NULL Mt==NULL).
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_transpose(PdsMatrix *Mt,const PdsMatrix *M)
{
	PdsRaNatural i,j;

	if(M==NULL)	return FALSE;
	if(Mt==NULL)	return FALSE;

	if(M->Nlin!=Mt->Ncol)	return FALSE;
	if(M->Ncol!=Mt->Nlin)	return FALSE;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<M->Nlin;j++)
		{
			Mt->M[i][j]=M->M[j][i];
		}
	}
	return TRUE;
}


/** \fn PdsMatrix *pds_matrix_mtm_new(const PdsMatrix *M)
 *  \brief Genera  <br><b>M^T * M</b>.
 *  \param[in] M La matriz a la cual se le aplicará la operación.
 *  \return La <br><b>M^T * M</b> de la matriz o NULL en caso de error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mtm_new(const PdsMatrix *M)
{
	PdsRaNatural i,j,k;
	PdsMatrix *MtM=NULL;
	PdsRaReal S;

	if(M==NULL)	return NULL;

	MtM=pds_matrix_new(M->Ncol,M->Ncol);
	if(MtM==NULL)	return NULL;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<=i;j++)
		{
			S=0;
			for(k=0;k<M->Nlin;k++)
			{
				S=S+M->M[k][i]*M->M[k][j];
			}
			MtM->M[i][j]=S;
		}
	}

	for(i=0;i<M->Ncol;i++)
	{
		for(j=i+1;j<M->Ncol;j++)
		{
			MtM->M[i][j]=MtM->M[j][i];
		}
	}

	return MtM;
}


/** \fn PdsMatrix *pds_matrix_mtm_ai_new(const PdsMatrix *M,PdsRaReal Alpha)
 *  \brief Genera  <br><b>M^T * M + Alpha * I</b>.
 *  \param[in] M La matriz a la cual se le aplicará la operación.
 *  \param[in] Alpha Factor que se multiplicará a la identidad.
 *  \return La <br><b>M^T * M + Alpha * I</b> de la matriz o NULL en caso de 
 *  error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mtm_ai_new(const PdsMatrix *M,PdsRaReal Alpha)
{
	PdsRaNatural i,j,k;
	PdsMatrix *MtMaI=NULL;
	PdsRaReal S;

	if(M==NULL)	return NULL;

	MtMaI=pds_matrix_new(M->Ncol,M->Ncol);
	if(MtMaI==NULL)	return NULL;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<=i;j++)
		{
			S=0;
			for(k=0;k<M->Nlin;k++)
			{
				S=S+M->M[k][i]*M->M[k][j];
			}
			MtMaI->M[i][j]=S;
		}
	}

	for(i=0;i<M->Ncol;i++)
	{
		for(j=i+1;j<M->Ncol;j++)
		{
			MtMaI->M[i][j]=MtMaI->M[j][i];
		}
	}

	for(i=0;i<M->Ncol;i++)
	{
			MtMaI->M[i][i]=MtMaI->M[i][i]+Alpha;
	}


	return MtMaI;
}


/** \fn int pds_matrix_linear_least_square(const PdsVector *U,const PdsMatrix *H,PdsVector *Z)
 *  \brief Encuentra Z  <br><b>U=HZ</b><br><b>Z=(H'H)^(-1)H'U</b>.
 *
 *  \f[ e^2=||U - H Z||^2 \f]
 *  \f[ \frac{\partial e^2}{\partial Z}=0 \f]
 *  \f[ Z=( H^T H )^{-1} H^T U \f]
 *  \param[in] U Valor de salida del sistema de ecuaciones <b>U=HZ</b>.
 *  \param[in] H Matriz del sistema.
 *  \param[out] Z Incognita del sistema, aquí es cargado el resultado.
 *  \return TRUE si todo fue bien o FALSE si no. (ej U==NULL,H==NULL o Z==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_linear_least_square(const PdsVector *U,const PdsMatrix *H,PdsVector *Z)
{
	PdsRaNatural i,j;
	PdsMatrix *HtH=NULL;
	PdsMatrix *HtH1=NULL;
	PdsVector *HtU=NULL;

	if(U->Nel!=H->Nlin)	return FALSE;
	if(Z->Nel!=H->Ncol)	return FALSE;

	HtH=pds_matrix_mtm_new(H);
	if(HtH==NULL)	return FALSE;

	HtH1=pds_matrix_inverse_matrix_new(HtH);
	if(HtH1==NULL)	
	{
		pds_matrix_free(HtH);
		return FALSE;
	}

	pds_matrix_free(HtH);

	HtU=pds_vector_new (H->Ncol);
	if(HtU==NULL)	
	{
		pds_matrix_free(HtH);
		pds_matrix_free(HtH1);
		return FALSE;
	}

	for(i=0;i<H->Ncol;i++)
	{
		HtU->V[i]=0;
		for(j=0;j<H->Nlin;j++)
		HtU->V[i]=HtU->V[i]+H->M[j][i]*U->V[j];
	}

	pds_matrix_mul_vector(Z,HtH1,HtU);

	pds_matrix_free(HtH1);
	pds_vector_free(HtU);

	return TRUE;
}


/** \fn int pds_matrix_tikhonov_nolinear_least_square(const PdsVector *U,const PdsMatrix *J,const PdsVector *F,PdsRaReal Alpha,PdsVector *Z)
 *  \brief Encuentra Z  una iteración de <br><b>U=F(Z)
 *  </b><br><b>Z=Z+(J'J + Alpha I)^(-1)(J'(U-F(Z))-Alpha Z)</b>.
 *
 *  \f[ e^2={||U-F(Z)||}^2+\alpha {||Z||}^2 \f]
 *  \f[ \frac{\partial e^2}{\partial Z}=0 \f]
 *  \f[ Z=Z+ {[ {J(Z)}^T {J(Z)} + \alpha I]}^{-1} [{J(Z)}^T (U-F(Z))-\alpha Z] \f]
 *  \param[in] U Valor de salida del sistema de ecuaciones <b>U=HZ</b>.
 *  \param[in] J matriz jacobiano de F con respecto a Z.
 *  \param[in] F Vector que depende de Z.
 *  \param[in] Alpha Factor de regularización.
 *  \param[out] Z Incognita del sistema, aquí es cargado el resultado.
 *  \return TRUE si todo fue bien o FALSE si no. (ej U==NULL,H==NULL o Z==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_tikhonov_nolinear_least_square(const PdsVector *U,const PdsMatrix *J,const PdsVector *F,PdsRaReal Alpha,PdsVector *Z)
{
	PdsRaNatural i,j;

	PdsMatrix *JtJaI=NULL;
	PdsMatrix *JtJaI1=NULL;
	////////////////////////////////////////////////////////////////////////////
	PdsVector *JtUFaZ=NULL;
	////////////////////////////////////////////////////////////////////////////
	PdsVector *V=NULL;
	PdsRaReal S;
	
	if(U==NULL)	return FALSE;
	if(J==NULL)	return FALSE;
	if(F==NULL)	return FALSE;
	if(Z==NULL)	return FALSE;

	if(U->Nel!=F->Nel)	return FALSE;
	if(U->Nel!=J->Nlin)	return FALSE;
	if(Z->Nel!=J->Ncol)	return FALSE;

	JtJaI=pds_matrix_mtm_ai_new(J,Alpha);
	if(JtJaI==NULL)	return FALSE;

	JtJaI1=pds_matrix_inverse_matrix_new(JtJaI);
	pds_matrix_free(JtJaI);
	if(JtJaI1==NULL)	return FALSE;
	
	////////////////////////////////////////////////////////////////////////////

	JtUFaZ=pds_vector_new(J->Ncol);
	if(JtUFaZ==NULL)	
	{
		pds_matrix_free(JtJaI1);            
		return FALSE;
	}

	for(i=0;i<J->Ncol;i++)
	{
		S=-Alpha*Z->V[i];
		for(j=0;j<J->Nlin;j++)
		{
			S = S + J->M[j][i] * (U->V[j]-F->V[j]);
		}

		JtUFaZ->V[i]=S;
	}
	
	////////////////////////////////////////////////////////////////////////////
    	
	V=pds_matrix_mul_vector_new(JtJaI1,JtUFaZ);
	pds_matrix_free(JtJaI1);	
	pds_vector_free(JtUFaZ);
	if(V==NULL)	return FALSE;
	
	pds_vector_add_vector(Z,V);  

	return TRUE;
}


/** \fn int pds_matrix_print(const PdsMatrix *Matrix)
 *  \brief Imprime en pantalla una matriz de tipo puntero PdsMatrix.
 *  \param[in] Matrix La matriz a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_printf(const PdsMatrix *Matrix)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;
	//printf("\n");
	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			printf("%e",Matrix->M[i][j]);
			
			if(j==(Matrix->Ncol-1))	printf("\n");
			else			printf("\t");
		}
	}
	return TRUE;
}


/** \fn int pds_matrix_fprintf(const PdsMatrix *Matrix,FILE *fd)
 *  \brief Imprime en el archivo que apunta fd una matriz de tipo puntero PdsMatrix.
 *  \param[in] Matrix La matriz a imprimir en fd.
 *  \param[in,out] fd El puntero al archivo.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fprintf(const PdsMatrix *Matrix,FILE *fd)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	//fprintf(fd,"\n");
	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			fprintf(fd,"%e",Matrix->M[i][j]);
			
			if(j==(Matrix->Ncol-1))	fprintf(fd,"\n");
			else			fprintf(fd,"\t");
		}
	}
	return TRUE;
}


/** \fn int pds_matrix_fscanf(PdsMatrix *Matrix, FILE *fd)
 *  \brief Inicializa una matriz con los datos del archivo apuntado por fd.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] Matrix Matriz en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo de donde se cargará los datos iniciales
 *  de la matriz.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fscanf(PdsMatrix *Matrix, FILE *fd)
{
	PdsRaNatural i,j;
	int id;
	int tmp;
	char x[__MAX_CHAR_DATA_SIZE__];
	
	if(Matrix==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	id=TRUE;

	for(j=0;(j<Matrix->Nlin)&&(feof(fd)==0);j++)
	{
		for(i=0;(i<Matrix->Ncol)&&(feof(fd)==0);i++)
		{
			do{
				x[0]=0;
				tmp=fscanf(fd,"%s",x);
			}while( (strlen(x)==0) && (feof(fd)==0) );

			if(feof(fd)!=0)	{id=FALSE;	break;}

			Matrix->M[j][i]=atof(x);
		}

		if(feof(fd)!=0)	{id=FALSE;	break;}
	}

	for(;j<Matrix->Nlin;j++)
	{
		for(;i<Matrix->Ncol;i++)
		{
			Matrix->M[j][i]=0.0;
		}
	}

	return id;
}


/** \fn int pds_matrix_fwrite(PdsMatrix *Matrix, FILE *fd)
 *  \brief Escribe los datos de un vector en el archivo binario apuntado por fd.
 *  \param[in] Matrix Matrix de donde se leeran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se escribiran los datos 
 *  del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fwrite(PdsMatrix *Matrix, FILE *fd)
{
	size_t n;
	PdsRaNatural i;
	
	if(Matrix==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<Matrix->Nlin;i++)
	{
		n=fwrite(Matrix->M[i],sizeof(Matrix->M[i][0]),Matrix->Ncol,fd);
	
		if(n!=Matrix->Ncol)	return FALSE;
	}
	return TRUE;
}


/** \fn int pds_matrix_fread(PdsMatrix *Matrix, FILE *fd)
 *  \brief Inicializa un vector con los datos del archivo binario apuntado por fd.
 *  \param[in] Matrix Matrix en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se cargará los datos 
 *  iniciales del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fread(PdsMatrix *Matrix, FILE *fd)
{
	PdsRaNatural i,j;
	size_t n;
	
	if(Matrix==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(j=0;(j<Matrix->Nlin)&&(feof(fd)==0);j++)
	{
		n=0;
		for(i=0;(i<Matrix->Ncol)&&(feof(fd)==0);i++)
		{
			n=fread(&(Matrix->M[j][i]),sizeof(Matrix->M[j][0]),1,fd);
			if(n!=1)	break;
		}
		if(n!=1)	break;
	}

	for(;j<Matrix->Nlin;j++)
	{
		for(;i<Matrix->Ncol;i++)
		{
			Matrix->M[j][i]=0.0;
			
		}
	}

	return TRUE;
}


/** \fn int pds_matrix_get_value(const PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y, PdsRaReal *m)
 *  \brief Devuelve el valor en la posición (x,y) de la matriz  Matrix.
 *  (x,y) inicia con (0,0).
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[in] x Linea de posición x, el primer valor es cero.
 *  \param[in] y Columna de posición y, el primer valor es cero.
 *  \param[out] m El valor en la posición (x,y), en caso de error por fuera de rango 
 *  (x,y) entonces carga 0, en caso de error de matriz nula carga cero.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_value(const PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y, PdsRaReal *m)
{
	*m=0;

	if(Matrix==NULL)	return FALSE;

	if((x>=0)&&(y>=0)&&(x<Matrix->Nlin)&&(y<Matrix->Ncol))	*m=Matrix->M[x][y];
	else							*m=0;

	return TRUE;
}


/** \fn int pds_matrix_set_value(PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y,PdsRaReal m)
 *  \brief Escribe el valor en la posición (x,y) de la matriz  Matrix.
 *  En caso de que (x,y) estuvieran fuera de rango, no se considera como error
 *  simplemente no se escribe nada. (x,y) inicia con (0,0).
 *  \param[in,out] Matrix La matriz a modificar.
 *  \param[in] x Linea de posición x, el primer valor es cero.
 *  \param[in] y Columna de posición y, el primer valor es cero.
 *  \param[in] m Valor real que se colocara en (x,y).
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_set_value(PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y,PdsRaReal m)
{
	PdsRaReal z=0;

	if(Matrix==NULL)	return FALSE;

	if((x>=0)&&(y>=0)&&(x<Matrix->Nlin)&&(y<Matrix->Ncol))	Matrix->M[x][y]=m;

	return TRUE;
}


/** \fn int pds_matrix_get_nlines(const PdsMatrix *Matrix, PdsRaNatural *Nlines)
 *  \brief Devuelve el número de lineas de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] Nlines En donde se guardará el número de lineas de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_nlines(const PdsMatrix *Matrix, PdsRaNatural *Nlines)
{
	*Nlines=0;

	if(Matrix==NULL)	return FALSE;

	*Nlines=Matrix->Nlin;

	return TRUE;
}


/** \fn int pds_matrix_get_ncolumns(const PdsMatrix *Matrix, PdsRaNatural *Ncolumns)
 *  \brief Devuelve el número de columnas de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] Ncolumns En donde se guardará el número de columnas de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_ncolumns(const PdsMatrix *Matrix, PdsRaNatural *Ncolumns)
{
	*Ncolumns=0;

	if(Matrix==NULL)	return FALSE;

	*Ncolumns=Matrix->Ncol;

	return TRUE;
}


/** \fn int pds_matrix_get_min_value(const PdsMatrix *Matrix, PdsRaReal *m)
 *  \brief Devuelve el valor mínimo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor mínimo de los elementos de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_min_value(const PdsMatrix *Matrix, PdsRaReal *m)
{
	PdsRaNatural i,j;
	*m=0;

	if(Matrix==NULL)	return FALSE;

	*m=Matrix->M[0][0];
	for(i=0;i<Matrix->Nlin;i++)
		for(j=0;j<Matrix->Ncol;j++)
			if(*m > Matrix->M[i][j])	*m=Matrix->M[i][j];

	return TRUE;
}


/** \fn int pds_matrix_get_max_value(const PdsMatrix *Matrix, PdsRaReal *m)
 *  \brief Devuelve el valor máximo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor máximo de los elementos de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_max_value(const PdsMatrix *Matrix, PdsRaReal *m)
{
	PdsRaNatural i,j;
	*m=0;

	if(Matrix==NULL)	return FALSE;

	*m=Matrix->M[0][0];
	for(i=0;i<Matrix->Nlin;i++)
		for(j=0;j<Matrix->Ncol;j++)
			if(*m < Matrix->M[i][j])	*m=Matrix->M[i][j];

	return TRUE;
}


/** \fn int pds_matrix_swap_rows(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2)
 *  \brief Intercambia dos lineas de la matriz  Matrix.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea a intercambiar con r2.
 *  \param[in] r2 La linea a intercambiar con r1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin, r2>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_swap_rows(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2)
{
	PdsRaNatural j;
	PdsRaReal tmp;

	if(Matrix==NULL)	return FALSE;
	if(r1>=Matrix->Nlin)	return FALSE;
	if(r2>=Matrix->Nlin)	return FALSE;

	for(j=0;j<Matrix->Ncol;j++)
	{
		tmp=Matrix->M[r1][j];
		Matrix->M[r1][j]=Matrix->M[r2][j];
		Matrix->M[r2][j]=tmp;
	}

	return TRUE;
}


/** \fn int pds_matrix_row_add_row_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2, PdsRaReal factor)
 *  \brief Multiplica la linea r2 por un factor y el resulltado lo resta a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}+factor*Matrix{r2}</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] r2 La linea r2.
 *  \param[in] factor Factor que multiplicará a la linea r2.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin, r2>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_add_row_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2, PdsRaReal factor)
{
	PdsRaNatural j;

	if(Matrix==NULL)	return FALSE;
	if(r1>=Matrix->Nlin)	return FALSE;
	if(r2>=Matrix->Nlin)	return FALSE;

	for(j=0;j<Matrix->Ncol;j++)
	{
		Matrix->M[r1][j]=Matrix->M[r1][j]+factor*Matrix->M[r2][j];
	}

	return TRUE;
}


/** \fn int pds_matrix_row_mul_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
 *  \brief Multiplica la linea r1 por un factor y el resulltado lo carga a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}*factor</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] factor Factor que multiplicará a la linea r1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_mul_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
{
	PdsRaNatural j;

	if(Matrix==NULL)	return FALSE;
	if(r1>=Matrix->Nlin)	return FALSE;

	for(j=0;j<Matrix->Ncol;j++)
	{
		Matrix->M[r1][j]=Matrix->M[r1][j]*factor;
	}

	return TRUE;
}


/** \fn int pds_matrix_row_div_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
 *  \brief Divide la linea r1 por un factor y el resulltado lo carga a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}/factor</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] factor Factor que dividirá a la linea r1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_div_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
{
	PdsRaNatural j;

	if(Matrix==NULL)	return FALSE;
	if(r1>=Matrix->Nlin)	return FALSE;

	for(j=0;j<Matrix->Ncol;j++)
	{
		Matrix->M[r1][j]=Matrix->M[r1][j]/factor;
	}

	return TRUE;
}


/** \fn int pds_matrix_reduced_matrix(const PdsMatrix *Matrix, PdsMatrix *MatReduced)
 *  \brief Reduce la matriz Matrix y la guarda en MatReduced. 
 *  <br><b>MatReduced=Reduced{Matrix}</b>.
 *
 *  Una matriz es reducida cuando:
 *  Es una matriz escalonada.
 *  Sus pivotes son todos iguales a 1
 *  En cada fila el pivote es el único elemento no nulo de su columna
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] MatReduced Matriz reducida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de MatReduced tenga distinto tamaño que Matrix, 
 *  MatReduced==NULL o Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_reduced_matrix(const PdsMatrix *Matrix, PdsMatrix *MatReduced)
{
	PdsRaNatural i,j,k;
	PdsRaNatural r,E,N;
	PdsRaReal factor;
	int id;

	if(Matrix==NULL)	return FALSE;
	if(MatReduced==NULL)	return FALSE;

	if(MatReduced->Nlin!=Matrix->Nlin)	return FALSE;
	if(MatReduced->Ncol!=Matrix->Ncol)	return FALSE;

	if(MatReduced->Ncol<MatReduced->Nlin)	N=MatReduced->Ncol;
	else					N=MatReduced->Nlin;

	id=pds_matrix_init_matrix(MatReduced,Matrix);
	if(id==FALSE)	return FALSE;

	for(i=0,j=0;(i<MatReduced->Nlin)&&(j<MatReduced->Ncol);j++)
	{
		r=0;E=1;
		// Como el primer elemento de la linea es cero busco otra que no 
		// lo sea, si no existe paso a la siguiente columna.
		if(MatReduced->M[i][j]==0)
		{
			// Esta función carga con E=1 encontró una linea para 
			// intercambiar y carga en r la linea.
			// Devuelve E=0 si no encontró ninguna linea
			id=pds_matrix_find_row_in_col(MatReduced,&r,&E,j,i+1);
			if(id==FALSE)	return FALSE;

			// Intercambia la linea con r.
			if(E==1) 
			{
				id=pds_matrix_swap_rows(MatReduced,r,i);
				if(id==FALSE)	return FALSE;
			}
		}
		// E==1 Indica que M[i][j]!=0, da igual si fue con ayuda de swap.
		// E==0 Indica que M[k][j]==0     i<= k <Nlin.
		if(E==1)
		{
			// Divido la linea en estudio para que inicie con 1.
			id=pds_matrix_row_div_factor(MatReduced,i,MatReduced->M[i][j]);
			if(id==FALSE)	return FALSE;

			for(k=i+1;k<MatReduced->Nlin;k++)
			{
				if(MatReduced->M[k][j]!=0)
				{
					id=pds_matrix_row_add_row_factor(MatReduced,k,i,-MatReduced->M[k][j]);
					if(id==FALSE)	return FALSE;
					//MatReduced->M[k][j]=0;//Esto porque no tiene buena precisión.
				}
			}
			i++;
		}
	}

	// Verifica si el ultimo elemento de la diagonal es distinto de cero
	// sino no tiene sentido seguir.
	if(MatReduced->M[N-1][N-1]==0)	return FALSE;

	// Comienza a restar desde la última fila, de manera que quede una diagonal
	// iniciando con 1 y seguido de ceros, hasta que termine la mayor matriz 
	// cuadrada que puede ser contenida por MatReduced.
	for(i=0;i<N;i++)
	{
		// Solo por si acaso verifico si cada elemento de la diagonal es
		// distinto de cero.
		if(MatReduced->M[(N-1)-i][(N-1)-i]!=0)
		{
			// Opero con las lineas para generar ceros en todas
			// las lineas arriba de la linea i.
			for(k=0;k<(N-1-i);k++)
			{
				id=pds_matrix_row_add_row_factor(MatReduced,k,(N-1)-i,-MatReduced->M[k][(N-1)-i]);
				if(id==FALSE)	return FALSE;
			}
		}
		else return FALSE;
	}

	return TRUE;
}


/** \fn int pds_matrix_inverse_matrix(const PdsMatrix *Matrix, PdsMatrix *MatInv)
 *  \brief Invierte la matriz Matrix y la guarda en MatInv. Ambas matrices deben
 *  existir y tener el mismo tamanho. <br><b>MatInv=Matrix^{-1}</b>.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] MatInv Matriz inversa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de MatInv tenga distinto tamaño que Matrix, 
 *  MatInv==NULL o Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_inverse_matrix(const PdsMatrix *Matrix, PdsMatrix *MatInv)
{
	PdsRaNatural i,j,k;
	PdsRaNatural r,E,N;
	PdsRaReal factor;
	PdsMatrix *Mat=NULL;
	int id;

	if(Matrix==NULL)	return FALSE;
	if(MatInv==NULL)	return FALSE;

	if(MatInv->Nlin!=Matrix->Nlin)	return FALSE;
	if(MatInv->Nlin!=Matrix->Ncol)	return FALSE;
	if(MatInv->Ncol!=Matrix->Nlin)	return FALSE;
	if(MatInv->Ncol!=Matrix->Ncol)	return FALSE;

	N=MatInv->Nlin;

	Mat=pds_matrix_new_matrix(Matrix);
	if(Mat==NULL)	{return FALSE;}

	id=pds_matrix_init_identity(MatInv,1.0);
	if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}

	for(i=0,j=0;(i<N)&&(j<N);j++)
	{
		r=0;E=1;
		// Como el primer elemento de la linea es cero busco otra que no 
		// lo sea, si no existe paso a la siguiente columna.
		if(Mat->M[i][j]==0)
		{
			// Esta función carga con E=1 encontró una linea para 
			// intercambiar y carga en r la linea.
			// Devuelve E=0 si no encontró ninguna linea
			id=pds_matrix_find_row_in_col(Mat,&r,&E,j,i+1);
			if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}

			// Intercambia la linea con r.
			if(E==1) 
			{
				id=pds_matrix_swap_rows(Mat,r,i);
				if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}

				id=pds_matrix_swap_rows(MatInv,r,i);
				if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}
			}
		}
		// E==1 Indica que M[i][j]!=0, da igual si fue con ayuda de swap.
		// E==0 Indica que M[k][j]==0     i<= k <Nlin.
		if(E==1)
		{

			factor=Mat->M[i][j];
			// Divido la linea en estudio para que inicie con 1.
			id=pds_matrix_row_div_factor(Mat,i,factor);
			if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}

			id=pds_matrix_row_div_factor(MatInv,i,factor);
			if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}

			for(k=i+1;k<Mat->Nlin;k++)
			{
				if(Mat->M[k][j]!=0)
				{
					factor=-Mat->M[k][j];
					id=pds_matrix_row_add_row_factor(Mat,k,i,factor);
					if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}
					//MatInv->M[k][j]=0;//Esto porque no tiene buena precisión.

					id=pds_matrix_row_add_row_factor(MatInv,k,i,factor);
					if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}
				}
			}
			i++;
		}
	}


	// Verifica si el ultimo elemento de la diagonal es distinto de cero
	// sino no tiene sentido seguir.
	if(Mat->M[N-1][N-1]==0)	{pds_matrix_free(Mat); return FALSE;}

	// Comienza a restar desde la última fila, de manera que quede una diagonal
	// iniciando con 1 y seguido de ceros, hasta que termine la mayor matriz 
	// cuadrada que puede ser contenida por MatInv.
	for(i=0;i<N;i++)
	{
		// Solo por si acaso verifico si cada elemento de la diagonal es
		// distinto de cero.
		if(Mat->M[(N-1)-i][(N-1)-i]!=0)
		{
			// Opero con las lineas para generar ceros en todas
			// las lineas arriba de la linea i.
			for(k=0;k<(N-1-i);k++)
			{
				factor=-Mat->M[k][(N-1)-i];
				id=pds_matrix_row_add_row_factor(Mat,k,(N-1)-i,factor);
				if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}

				id=pds_matrix_row_add_row_factor(MatInv,k,(N-1)-i,factor);
				if(id==FALSE)	{pds_matrix_free(Mat); return FALSE;}
			}
		}
		else return FALSE;
	}

	pds_matrix_free(Mat);
	return TRUE;
}


/** \fn PdsMatrix *pds_matrix_inverse_matrix_new(const PdsMatrix *Matrix)
 *  \brief Invierte la matriz Matrix y la guarda en MatInv. Ambas matrices deben
 *  existir y tener el mismo tamanho. <br><b>MatInv=Matrix^{-1}</b>.
 *  \param[in] Matrix La matriz de en consulta.
 *  \return Un puntero que apunta a una nueva estructura con la multiplicación.
 *  Retorna NULL si fallo en la inversión, o si Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_inverse_matrix_new(const PdsMatrix *Matrix)
{
	PdsRaNatural i,j,k;
	PdsRaNatural r,E,N;
	PdsRaReal factor;
	PdsMatrix *Mat=NULL;
	PdsMatrix *MatInv=NULL;
	int id;

	if(Matrix==NULL)	return NULL;

	if(Matrix->Ncol!=Matrix->Nlin)	return NULL;

	N=Matrix->Nlin;

	MatInv=pds_matrix_new(N,N);
	if(MatInv==NULL)	{return NULL;}

	Mat=pds_matrix_new_matrix(Matrix);
	if(Mat==NULL)	{pds_matrix_free(MatInv); return NULL;}

	id=pds_matrix_init_identity(MatInv,1.0);
	if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

	for(i=0,j=0;(i<N)&&(j<N);j++)
	{
		r=0;E=1;
		// Como el primer elemento de la linea es cero busco otra que no 
		// lo sea, si no existe paso a la siguiente columna.
		if(Mat->M[i][j]==0)
		{
			// Esta función carga con E=1 encontró una linea para 
			// intercambiar y carga en r la linea.
			// Devuelve E=0 si no encontró ninguna linea
			id=pds_matrix_find_row_in_col(Mat,&r,&E,j,i+1);
			if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

			// Intercambia la linea con r.
			if(E==1) 
			{
				id=pds_matrix_swap_rows(Mat,r,i);
				if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

				id=pds_matrix_swap_rows(MatInv,r,i);
				if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
			}
		}
		// E==1 Indica que M[i][j]!=0, da igual si fue con ayuda de swap.
		// E==0 Indica que M[k][j]==0     i<= k <Nlin.
		if(E==1)
		{

			factor=Mat->M[i][j];
			// Divido la linea en estudio para que inicie con 1.
			id=pds_matrix_row_div_factor(Mat,i,factor);
			if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

			id=pds_matrix_row_div_factor(MatInv,i,factor);
			if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

			for(k=i+1;k<Mat->Nlin;k++)
			{
				if(Mat->M[k][j]!=0)
				{
					factor=-Mat->M[k][j];
					id=pds_matrix_row_add_row_factor(Mat,k,i,factor);
					if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
					//MatInv->M[k][j]=0;//Esto porque no tiene buena precisión.

					id=pds_matrix_row_add_row_factor(MatInv,k,i,factor);
					if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
				}
			}
			i++;
		}
	}


	// Verifica si el ultimo elemento de la diagonal es distinto de cero
	// sino no tiene sentido seguir.
	if(Mat->M[N-1][N-1]==0)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

	// Comienza a restar desde la última fila, de manera que quede una diagonal
	// iniciando con 1 y seguido de ceros, hasta que termine la mayor matriz 
	// cuadrada que puede ser contenida por MatInv.
	for(i=0;i<N;i++)
	{
		// Solo por si acaso verifico si cada elemento de la diagonal es
		// distinto de cero.
		if(Mat->M[(N-1)-i][(N-1)-i]!=0)
		{
			// Opero con las lineas para generar ceros en todas
			// las lineas arriba de la linea i.
			for(k=0;k<(N-1-i);k++)
			{
				factor=-Mat->M[k][(N-1)-i];
				id=pds_matrix_row_add_row_factor(Mat,k,(N-1)-i,factor);
				if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

				id=pds_matrix_row_add_row_factor(MatInv,k,(N-1)-i,factor);
				if(id==FALSE)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
			}
		}
		else {pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
	}

	pds_matrix_free(Mat);
	return MatInv;
}


/** \fn int pds_matrix_is_equal (PdsMatrix *M1,PdsMatrix *M2)
 *  \brief Compara dos matrices.
 *  \param[in] M1 La matriz a comparar com M2.
 *  \param[in] M2 La matriz a comparar com M1.
 *  \return Retorna TRUE si las matrices son iguales, caso contrario retorna FALSE. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_equal (PdsMatrix *M1,PdsMatrix *M2)
{
	int i,j;
	if ((M1==NULL)|| (M2==NULL))	return FALSE;
	if ((M1->Nlin!=M2->Nlin)|| (M1->Ncol!=M2->Ncol))	return FALSE;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if(M1->M[i][j]!=M2->M[i][j])	return FALSE;
	}
	
	return TRUE;
}

/** \fn int pds_matrix_is_quasiequal (PdsMatrix *M1,PdsMatrix *M2,double u)
 *  \brief Compara dos matrices, y verifica si la diferencia absoluta de cada 
 *  elemento es menor que un umbral u.
 *  \param[in] M1 La matriz a comparar com M2.
 *  \param[in] M2 La matriz a comparar com M1.
 *  \param[in] u Es el umbrarl a testar.
 *  \return Retorna TRUE si las matrices son quasi iguales 
 *  (diferencia de elementos menoro igual a u), caso contrario retorna FALSE. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_quasiequal (PdsMatrix *M1,PdsMatrix *M2,double u)
{
	int i,j;
	if ((M1==NULL)|| (M2==NULL))	return FALSE;
	if ((M1->Nlin!=M2->Nlin)|| (M1->Ncol!=M2->Ncol))	return FALSE;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if (fabs(M1->M[i][j]-M2->M[i][j])>u)	return FALSE;
	}
	
	return TRUE;
}

/** \fn int pds_matrix_is_diagonal (PdsMatrix *M1)
 *  \brief Verifica si una matriz es diagonal.
 *
 *  Una matriz es diagonal cuando sus elementos exteriores a la diagonal 
 *  principal son nulos. Una matriz nula (llena de ceros) tambien es diagonal.
 *
 *  \param[in] M1 La matriz a verificar su diagonalidad.
 *  \return Retorna TRUE si las matriz es diagonal, caso contrario retorna FALSE. 
 *  Tambien retorna FALSE si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_diagonal (PdsMatrix *M1)
{
	int i,j;
	if (M1==NULL)	return FALSE;
	if (M1->Nlin!=M1->Ncol)	return FALSE;	
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if(i!=j)
		if(M1->M[i][j]!=0)	return FALSE;
	}
	
	return TRUE;
}

/** \fn int pds_matrix_is_quasidiagonal (PdsMatrix *M1,double u)
 *  \brief Verifica si una matriz es casi diagonal. Usa un umbral u, qualquier 
 *  valor menor o igual que u es considerado cero.
 *
 *  Una matriz es diagonal cuando sus elementos exteriores a la diagonal 
 *  principal son nulos. Una matriz nula (llena de ceros) tambien es diagonal.
 *
 *  \param[in] M1 La matriz a verificar su diagonalidad.
 *  \param[in] u Es el umbral bajo el cual todo lo demas se considera cero.
 *  \return Retorna TRUE si las matriz es diagonal, caso contrario retorna FALSE.
 *  Tambien retorna FALSE si la matriz no es cuadrada. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_quasidiagonal (PdsMatrix *M1,double u)
{
	int i,j;
	if (M1==NULL)	return FALSE;
	if (M1->Nlin!=M1->Ncol)	return FALSE;	
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if(i!=j)
		if(M1->M[i][j]>u)	return FALSE;
	}
	
	return TRUE;
}


/** \fn int pds_matrix_is_symetric (PdsMatrix *M1)
 *  \brief Verifica si una matriz es simétrica.
 *
 *  Una matriz es simétrica si coincide con su transpuesta.
 *
 *  \param[in] M1 La matriz a verificar su simetria.
 *  \return Retorna TRUE si las matriz es simetrica, caso contrario retorna FALSE. 
 *  Tambien retorna FALSE si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_symetric (PdsMatrix *M1)
{
	int i,j;
	if (M1==NULL)	return FALSE;
	if (M1->Nlin!=M1->Ncol)	return FALSE;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=i+1;j<M1->Ncol;j++)
	{
		if(M1->M[i][j]!=M1->M[j][i])	return FALSE;
	}
	
	return TRUE;
}


/** \fn int pds_matrix_is_antisymetric (PdsMatrix *M1)
 *  \brief Verifica si una matriz es anti simétrica.
 *
 *  Una matriz es anti simétrica cuando el opuesto de esta  coincide con su transpuesta.
 *  De esto se deduce que los elementos de la diagonal son nulos.
 *  <br><b>M1^t=-M1</b>.
 *
 *  \param[in] M1 La matriz a verificar su anti simetria.
 *  \return Retorna TRUE si las matriz es anti simetrica, caso contrario retorna FALSE. 
 *  Tambien retorna FALSE si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_antisymetric (PdsMatrix *M1)
{
	int i,j;
	if (M1==NULL)	return FALSE;
	if (M1->Nlin!=M1->Ncol)	return FALSE;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=i;j<M1->Ncol;j++)
	{
		if(M1->M[i][j]!=-M1->M[j][i])	return FALSE;
	}
	
	return TRUE;
}

/** \fn void pds_matrix_free(PdsMatrix *Matrix)
 *  \brief Libera una matriz de tipo puntero PdsMatrix.
 *  \param[in,out] Matrix la matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsMatrixGroup
 */
void pds_matrix_free(PdsMatrix *Matrix)
{
	PdsRaNatural i;

	if(Matrix!=NULL)
	{
		for(i=0;i<Matrix->Nlin;i++)
		{
			free(Matrix->M[i]);
		}
		free(Matrix);
	}
}


/** \fn void pds_matrix_destroy(PdsMatrix **Matrix)
 *  \brief Libera una matriz de tipo puntero PdsMatrix, y limpia el puntero con NULL.
 *  \param[in,out] Matrix la matriz a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsMatrixGroup
 */
void pds_matrix_destroy(PdsMatrix **Matrix)
{
	PdsRaNatural i;

	if((*Matrix)!=NULL)
	{
		for(i=0;i<(*Matrix)->Nlin;i++)
		{
			free((*Matrix)->M[i]);
		}
		free(*Matrix);
		*Matrix=NULL;
	}
}

////////////////////////////////////////////////////////////////////////////////
////                        Funciones Extras                                ////
////////////////////////////////////////////////////////////////////////////////

/** \fn int pds_matrix_find_row_in_col(const PdsMatrix *Matrix, PdsRaNatural *row, PdsRaNatural *E, PdsRaNatural col, PdsRaNatural r1)
 *  \brief Encuentra la primera linea row que sea mayor o igual a r1 y menor que 
 *  Nlin (r1<=row<Nlin), de tal forma que en la columna col  el elemento sea 
 *  distinto de cero.
 *  Osea busca en la columna "col" el primer elemento distinto de cero, pero lo
 *  busca en el intervalo de lineas r1<=row<Nlin.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] row La linea con el elemento distinto de cero entre [r1,Nlin).
 *  \param[out] E Si se encontró un elemento distinto de cero en la columna col, E=1,
 *  caso contrario E=1.
 *  \param[in] col Columna en la que buscaremos.
 *  \param[in] r1 Límite inferior de búsqueda de linea.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin, col>=Ncol o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_find_row_in_col(const PdsMatrix *Matrix, PdsRaNatural *row, PdsRaNatural *E, PdsRaNatural col, PdsRaNatural r1)
{
	PdsRaNatural i;

	if(Matrix==NULL)	return FALSE;
	if(r1 >=Matrix->Nlin)	return FALSE;
	if(col>=Matrix->Ncol)	return FALSE;

	for(i=r1;i<Matrix->Nlin;i++)
	{
		if(Matrix->M[i][col]!=0.0)
		{
			*E=1;
			*row=i;
			return TRUE;
		}
	}

	*E=0;
	*row=0;
	return TRUE;
}


