/*
 * pdsfifo.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsrautils.h>
#include <pds/pdsvector.h>
#include <pds/pdsfifo.h>

////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsFifo                                                ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsFifo *pds_fifo_new(PdsRaNatural N,PdsRaNatural M)
 *  \brief Crea una fifo de tipo PdsFifo, la fifo estará vacía y aceptará como
 *  máximo N vectores en la cola de tamaño M. La fifo está preparada para atender
 *  dos procesos que intenten simultáneamente leer y escribir en ella.
 *  \param[in] N Es el número máximo de elementos en la cola.
 *  \param[in] M Es el número de elementos de cada vector de la fifo.
 *  \return Un puntero a la Fifo de tipo PdsFifo.
 *  \ingroup PdsFifoGroup
 */
PdsFifo *pds_fifo_new(PdsRaNatural N,PdsRaNatural M)
{
	PdsFifo *FIFO=NULL;
	PdsRaNatural i,j;
	
	FIFO=(PdsFifo *)calloc(1,sizeof(PdsFifo));
	if(FIFO==NULL)	return NULL;

	FIFO->N=N;
	FIFO->M=M;
	FIFO->ReadID=0;
	FIFO->WriteID=0;
	FIFO->Nnodes=0;

	FIFO->V=(PdsVector **)calloc(N,sizeof(PdsVector *));
	if(FIFO->V==NULL)
	{
		free(FIFO);
		return NULL;
	}
	
	for(i=0;i<N;i++)
	{
		FIFO->V[i]=pds_vector_new(M);
		if(FIFO->V[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_vector_free(FIFO->V[j]);
			free(FIFO->V);
			free(FIFO);
			return NULL;
		}
	}
	
	FIFO->E=(PdsRaNatural *)calloc(N,sizeof(PdsRaNatural));
	if(FIFO->E==NULL)
	{
		for(j=0;j<N;j++)	pds_vector_free(FIFO->V[j]);
		free(FIFO->V);		
		free(FIFO);
		return NULL;
	}

	return FIFO;
}

/** \fn int pds_fifo_write(PdsFifo *FIFO,const PdsVector *V)
 *  \brief Escribe en la fifo FIFO el vector V. Si el tamaño del vector V y el 
 *  ancho de la fifo son diferentes se intersectan los tamaños y se hace la copia 
 *  en la intersección solamente. 
 *  \param[in,out] FIFO La fifo a escribir.
 *  \param[in] V El vector con los datos a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. FIFO==NULL, V==NULL o Número 
 *  máximo de elementos )
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_write(PdsFifo *FIFO,const PdsVector *V)
{
	int id;
	
	if(FIFO==NULL)			return FALSE;
	if(V==NULL)			return FALSE;
	if(FIFO->E[FIFO->WriteID]==1)	return FALSE;
	
	if(FIFO->Nnodes > FIFO->N)	return FALSE;

	id=pds_vector_init_vector(FIFO->V[FIFO->WriteID],V);
	if(id==FALSE)	return FALSE;

	id=FIFO->WriteID;

	FIFO->WriteID=(FIFO->WriteID+1)%FIFO->N;

	FIFO->Nnodes =FIFO->Nnodes+1;

	FIFO->E[id]=1;

	return TRUE;
}


/** \fn int pds_fifo_queue(const PdsFifo *FIFO,PdsRaNatural n)
 *  \brief Indica si existen como mínimo n vectores en el buffer. 
 *  \param[in] FIFO La fifo a leer.
 *  \param[in] n El número de vectores a leer.
 *  \return TRUE si todo fue bien y existen como mínimo n elementos o
 *  FALSE si no. (ej. FIFO==NULL, n==0, o fifo vacía )
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_queue(const PdsFifo *FIFO,PdsRaNatural n)
{
	PdsRaNatural i,id;
	
	if(FIFO==NULL)	return FALSE;
	if(n==0)	return FALSE;

	for(i=0,id=0;i<FIFO->N;i++)	id=id+FIFO->E[i];

	if(id>=n)	return TRUE;
	else 		return FALSE;
}


/** \fn int pds_fifo_read(PdsFifo *FIFO,PdsVector *V)
 *  \brief Lee en la fifo FIFO el vector V. Si el tamaño del vector V y el 
 *  ancho de la fifo son diferentes se intersectan los tamaños y se hace la copia 
 *  en la intersección solamente. 
 *  \param[in,out] FIFO La fifo a leer.
 *  \param[in] V El vector donde se copiará los datos.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. FIFO==NULL, V==NULL )
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_read(PdsFifo *FIFO,PdsVector *V)
{
	int id;

	if(FIFO==NULL)			return FALSE;
	if(V==NULL)			return FALSE;
	if(FIFO->E[FIFO->ReadID]==0)	return FALSE;
	
	id=pds_vector_init_vector(V,FIFO->V[FIFO->ReadID]);
	if(id==FALSE)	return FALSE;
	
	id=FIFO->ReadID;

	FIFO->ReadID=(FIFO->ReadID+1)%FIFO->N;

	if(FIFO->Nnodes==0)	
	{
		FIFO->ReadID=id;
		return FALSE;
	}
	else
	{
		FIFO->Nnodes =FIFO->Nnodes-1;
		FIFO->E[id]=0;

		return TRUE;
	}

}


/** \fn int pds_fifo_printf(const PdsFifo *FIFO)
 *  \brief Imprime en pantalla los vectores de la fifo FIFO, desde el primero en
 *  ser atendido en la cola hasta el último.
 *  \param[in] FIFO La fifo a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem FIFO==NULL)
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_printf(const PdsFifo *FIFO)
{
	PdsRaNatural id;

	if(FIFO==NULL)			return FALSE;
	if(FIFO->E[FIFO->ReadID]==0)	return FALSE;

	id=FIFO->ReadID;

	do
	{
		pds_vector_printf(FIFO->V[id]);
		id=(id+1)%FIFO->N;
	}while(FIFO->E[id]==1);
	
	return TRUE;
}



/** \fn int pds_fifo_isfull(const PdsFifo *FIFO)
 *  \brief Indica si la fifo esta llena. 
 *  \param[in] FIFO La fifo a leer.
 *  \return TRUE si todo fue bien y esta llena o
 *  FALSE si no. (ej. FIFO==NULL)
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_isfull(const PdsFifo *FIFO)
{
	if(FIFO==NULL)	return FALSE;

	if(FIFO->Nnodes==FIFO->N)	return TRUE;
	else 				return FALSE;
}


/** \fn int pds_fifo_isempty(const PdsFifo *FIFO)
 *  \brief Indica si la fifo esta vacia. 
 *  \param[in] FIFO La fifo a leer.
 *  \return TRUE si todo fue bien y esta vacia o
 *  FALSE si no. (ej. FIFO==NULL)
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_isempty(const PdsFifo *FIFO)
{
	if(FIFO==NULL)	return FALSE;

	if(FIFO->Nnodes==0)	return TRUE;
	else 			return FALSE;
}


/** \fn void pds_fifo_free(PdsFifo *FIFO)
 *  \brief Libera una fifo de tipo puntero PdsFifo.
 *  \param[in,out] FIFO La fifo a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFifoGroup
 */
void pds_fifo_free(PdsFifo *FIFO)
{
	PdsRaNatural i;
	if(FIFO!=NULL)
	{
		for(i=0;i<FIFO->N;i++)
		{
			pds_vector_free(FIFO->V[i]);
		}
		free(FIFO->V);
		free(FIFO->E);
		free(FIFO);
	}
}


/** \fn void pds_fifo_destroy(PdsFifo **FIFO)
 *  \brief Libera una fifo de tipo puntero PdsFifo, y limpia el puntero con NULL.
 *  \param[in,out] FIFO La fifo a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsFifoGroup
 */
void pds_fifo_destroy(PdsFifo **FIFO)
{
	PdsRaNatural i;
	if((*FIFO)!=NULL)
	{
		for(i=0;i<(*FIFO)->N;i++)
		{
			pds_vector_free((*FIFO)->V[i]);
		}
		free((*FIFO)->V);
		free((*FIFO)->E);
		free((*FIFO));
		(*FIFO)=NULL;
	}
}

