/*
 * pdsoctave.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsoctave.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-12-2011
 *  \brief Funciones para imprimir archivos png con OCTAVE.
 *  
 */

#ifndef __PDSOCTAVE_H__
#define __PDSOCTAVE_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <pds/pdsra.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsOctaveGroup Funciones para acceder a comandos de octave.
 *
 *  <br>
 *  Funciones.
 * @{
 */


/** \fn pds_octave_surf_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando surf.
 *
 *  <center>
 *  \image html pds_octave_surf_matrix.png "Usando la función surf() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_surf_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *imgfile);


/** \fn int pds_octave_pcolor_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando pcolor.
 *
 *  <center>
 *  \image html pds_octave_pcolor_matrix.png "Usando la función pcolor() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_pcolor_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *octfile,const char *imgfile);


/** \fn int pds_octave_imwrite_matrix(const PdsMatrix *M,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando imwrite en escala de grises.
 *
 *  <center>
 *  \image html pds_octave_imwrite_matrix.png "Usando la función imwrite() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir. Esta matriz debe tener elements entre 0 y 1.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_imwrite_matrix(const PdsMatrix *M,const char *octfile,const char *imgfile);


/** \fn int pds_octave_plot_vector(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime un vector V en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_vector(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *imgfile);


/** \fn int pds_octave_plot_vectors(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime un vector Y vs X en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] X Eje X del vector a imprimir.
 *  Si X==NULL entonces se mostrara la gráfica usando el numero de muestras
 *  como eje X.
 *  \param[in] Y Eje Y del vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_vectors(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *imgfile);


/** \fn int pds_octave_plot_compare_vectors(const PdsVector *X,const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *legendu,const char *legendv,const char *octfile,const char *imgfile)
 *  \brief Imprime un vector U y V en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_compare_vectors.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] X Vector correspondiente a el eje X de los vectores U y V.
 *  Si X==NULL entonces se mostrara la comparación usando el numero de muestras
 *  como eje X.
 *  \param[in] U Vector a imprimir.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] legendu Legenda de U.
 *  \param[in] legendv Legenda de V.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_compare_vectors(const PdsVector *X,const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *legendu,const char *legendv,const char *octfile,const char *imgfile);


/** \fn int pds_octave_plot_file_col(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime una columna de un archivo a un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] datafile Archivo de datos.
 *  \param[in] col Columna a imprimir, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_file_col(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *imgfile);



/** \fn int pds_octave_plot_file_cols(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime Y vs X de un par de columnas de un archivo a un archivo de imagen 
 *  usando como intermedio un archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] datafile Archivo de datos.
 *  \param[in] colx Columna a imprimir como eje X, inicia en 1.
 *  \param[in] coly Columna a imprimir como eje Y, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_file_cols(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *imgfile);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSOCTAVE_H__ */ 

