/*
 * pdsoctplotpng.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsoctplotpng.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-12-2011
 *  \brief Funciones para imprimir archivos png con OCTAVE.
 *  
 */

#ifndef __PDSOCTPLOTPNG_H__
#define __PDSOCTPLOTPNG_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <pds/pdsra.h>
#include <pds/pdsca.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsOctPlotPngGroup Funciones del módulo PdsOctPlotPng.
 *
 *  <br>
 *  Funciones.
 * @{
 */

/** \fn int pds_octplot_surf_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando surf.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_surf_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_pcolor_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando pcolor.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_pcolor_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_image_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando image.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_image_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz, const char *octfile,const char *pngfile);


/** \fn int pds_octplot_vector_in_png(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_vector_in_png(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile);

/** \fn int pds_octplot_vectors_in_png(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector Y vs X en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] X Eje X del vector a imprimir.
 *  \param[in] Y Eje Y del vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_vectors_in_png(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_cvector_in_png(const PdsCVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector complejo V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] V Vector complejo a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_cvector_in_png(const PdsCVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_cvectors_in_png(const PdsVector *X,const PdsCVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector complejo Y vs el vector real X en un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  \param[in] X Eje X del vector real a imprimir.
 *  \param[in] Y Eje Y del vector complejo a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_cvectors_in_png(const PdsVector *X,const PdsCVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_vector_compare_in_png(const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector U y V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] U Vector a imprimir.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_vector_compare_in_png(const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_afft_vector_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime el valor absoluto de la FFT de un vector V en un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  Imprime el valor absoluto de la FFT de V con 'o'.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_afft_vector_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_afft_vector8_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime el valor absoluto de la FFT de un vector V en un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  Imprime el valor absoluto de la FFT de V con 'o' y además el valor absoluto 
 *  de la FFT de V8=[V,0*V,0*V,0*V,0*V,0*V,0*V,0*V] con '--'.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_afft_vector8_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile);


/** \fn int pds_octplot_file_col_in_png(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime una columna de un archivo a un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] datafile Archivo de datos.
 *  \param[in] col Columna a imprimir, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_file_col_in_png(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *pngfile);



/** \fn int pds_octplot_file_cols_in_png(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime Y vs X de un par de columnas de un archivo a un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  \param[in] datafile Archivo de datos.
 *  \param[in] colx Columna a imprimir como eje X, inicia en 1.
 *  \param[in] coly Columna a imprimir como eje Y, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_file_cols_in_png(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *pngfile);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSOCTPLOTPNG_H__ */ 

