/*
 * pdsneuronml.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsneuronml.h
 *  \author Fernando Pujaico Rivera
 *  \date 05-01-2012
 *  \brief Funciones que trabajan con redes neuronales multi capas.
 *  
 *  <br>Estas funciones trabajan con redes neuronales multi capas de la forma.<br>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 */

#ifndef __PDSNEURONML_H__
#define __PDSNEURONML_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsnnglobal.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>
#include <pds/pdsnvector.h>


#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsNeuronMLGroup Módulo PdsNeuronML.
 *  \brief Funciones que trabajan con redes neuronales multi capas.
 *  
 *  <br>Estas funciones trabajan con redes neuronales multi capas de la forma.<br>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 * @{
 */


/*! \struct PdsNeuronML
 *  \brief La estructura tipo  PdsNeuronML .
 *  Esta estructura genera una red neuronal multi capa.
 *  Para usar incluir pds/pdsnn.h.
 *
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 *  \ingroup PdsNeuronMLGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un vector que indica el número de neuronas por capa. */
	PdsNnNatural *L;
	/*! Número de capas. */
	PdsNnNatural N;

	/*! Vector de neuronas de entrada Nel elementos. */
	PdsNIVector *LayerInput;
	/*! Arreglo de Nel elementos, cada elemento es un arreglo de neuronas. */
	PdsNVector **Layer;

	/*! Valor de error, un valor menor a este detiene el aprendizaje. */
	PdsNnReal MaxError;
	/*! Factor de aprendizaje. */
	PdsNnReal Alpha;
	/*! Número máximo de iteraciones. */
	PdsNnNatural MaxIter;
	/*! Número de iteraciones del ultimo entrenamiento. */
	PdsNnNatural Iter;
	/*! Habilitador de printf de avance. 1 habilitado 0 deshabilitado*/
	PdsNnBool ShowPrintf;

}PdsNeuronML;

/** @name pds_neuronml_<new>
 *  Creando nuevas neuronas
 *
 *  <center>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 *  </center>
 *  <br>
 * @{
 */

/** \fn PdsNeuronML *pds_neuronml_new(const PdsVector *L)
 *  \brief Crea una estructura de tipo PdsNeuronML.
 *
 *  <ul>
 *  <li>Inicia con cero todos los Pesos. </li>
 *  <li>El máximo margen de error RMS aceptado en la salida de la estructura 
    PdsNeuronML es PDS_DEFAULT_MAX_RMS_ERROR.</li>
 *  <li>El valor de de Alpha se inicia con PDS_DEFAULT_ALPHA.</li>
 *  <li>El valor U se inicia con 1.0.</li>
 *  <li>El valor Sigma se inicia con 1.0.</li>
 *  <li>El valor func(x) se inicia con tanh(x/2).</li>
 *  </ul> 
 *  
 *  \param[in] L Es un vector con el numero de elementos por capas.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new(const PdsVector *L);


/** \fn PdsNeuronML *pds_neuronml_new_load_data(const char *mlayer_filename)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia todos los elementos.
 *  desde un archivo de texto.
 *
 *  Los datos cargados son :
 *  <ul>
 *  <li>N, el numero de capas. </li>
 *  <li>MaxError, El máximo margen de error RMS aceptado en la salida.</li>
 *  <li>Alpha, El factor de aprendizaje.</li>
 *  <li>MaxIter, El máximo número de iteraciones.</li>
 *  <li>Iter, El número de iteraciones del último entrenamiento</li>.
 *  <li>ShowPrintf, Si la estructura muestra información adicional.</li>
 *  <li>L, Vector de N+1 elementos con el número de neuronas por capa.</li>
 *  <li>Vector de pesos de neurona.</li>
 *  <li>Valor de U y Valor de Sigma</li>
 *  <li>...</li>
 *  </ul> 
 *  
 *  \param[in] mlayer_filename Es el archivo de texto de donde se leen los datos.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_load_data(const char *mlayer_filename);


/** \fn int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayer_filename)
 *  \brief Salva una estructura de tipo PdsNeuronML en un archivo de texto.
 *
 *  Los datos salvados son :
 *  <ul>
 *  <li>N, el numero de capas. </li>
 *  <li>MaxError, El máximo margen de error RMS aceptado en la salida.</li>
 *  <li>Alpha, El factor de aprendizaje.</li>
 *  <li>MaxIter, El máximo número de iteraciones.</li>
 *  <li>Iter, El número de iteraciones del último entrenamiento</li>.
 *  <li>ShowPrintf, Si la estructura muestra información adicional.</li>
 *  <li>L, Vector de N+1 elementos con el número de neuronas por capa.</li>
 *  <li>Vector de pesos de neurona.</li>
 *  <li>Valor de U y Valor de Sigma</li>
 *  <li>...</li>
 *  </ul> 
 *  
 *  \param[in] NeuronML Es la red neuronal multicapa a salvar.
 *  \param[in] mlayer_filename Es el archivo de texto donde se guardará la red.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem: NeuronML==NULL o mlayer_filename=NULL).
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayer_filename);


//@}

/** @name pds_neuronml_<behavior_methods>
 *  Métodos que alteran o describen el comportamiento de la estructura.
 *
 * @{
 */


/** \fn int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal MaxError)
 *  \brief Coloca el maximo margen error RMS aceptado en la salida de la estructura
 *  PdsNeuronML.
 *
 *  Cualquier valor de error que es menor
 *  provoca que se detenga el algoritmo de aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxError Máximo error aceptado.
 *  El valor cero no es aceptado, en su defecto es escrito PDS_DEFAULT_MAX_RMS_ERROR.
 *  Valores negativos no son aceptados, en su defecto es escrito |MaxError|.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal MaxError);


/** \fn int pds_neuronml_set_max_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
 *  \brief Coloca el máximo numero de iteraciones aceptado, cualquier valor mayor
 *  provoca que se detenga el algoritmo de aprendizaje pds_neuronml_training(). si este número es cero
 *  entonces el algoritmo continua hasta que el error en la salida sea menor que
 *  NeuronML->Error.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxIter Número máximo de iteraciones.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter);


/** \fn int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
 *  \brief Devuelve el número de iteraciones del último entrenamiento hecho con
 *  la función pds_neuronml_training().
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Iter Número de iteraciones del último entrenamiento.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter);


/** \fn int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
 *  \brief Habilita la muestra en pantalla del avance del aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_enable_printf(PdsNeuronML *NeuronML);


//@}

/** @name pds_neuronml_<variable_methods>
 *  Métodos que alteran las variables de la estructura.
 *
 * @{
 */


/** \fn int pds_neuronml_init_weight_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los pesos usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_weight_uniform(PdsNeuronML *NeuronML,PdsUniform *RV);


/** \fn int pds_neuronml_init_u_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los umbrales U usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_u_uniform(PdsNeuronML *NeuronML,PdsUniform *RV);


/** \fn int pds_neuronml_set_sigma(PdsNeuronML *NeuronML,PdsNnReal Sigma)
 *  \brief Coloca el valor de Sigma en todas las neuronas
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Sigma Valor de sigma en la función de activación de las neuronas.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_sigma(PdsNeuronML *NeuronML,PdsNnReal Sigma);


//@}

/** @name pds_neuronml_<iterate_methods>
 *  Métodos sobre iteración de la estructura.
 *
 * @{
 */


/** \fn int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
 *  \brief Itera la red neuronal multicapa NeuronML.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \f[ y_j \leftarrow func\left(\frac{\theta_j}{\sigma_j}\right) \f]
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Input Datos de entrada de la red neuronal multicapa.
 *  \param[out] Output Datos de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output);


/** \fn int pds_neuronml_get_hard_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación con desición abrupta de los valores de salida
 *  de la red neuronal multicapa.
 *
 *  Se considera como un voto positivo "+1" cualquier valor mayor o igual a "0". 
 *  Se considera como un voto negativo "-1" cualquier valor menor a "0". 
 *  Luego se realiza la suma con signo de las votaciones.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación abrupta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_hard_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote);


/** \fn int pds_neuronml_get_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación con desición suave de los valores de salida
 *  de la red neuronal multicapa. 
 *
 *  Se realiza una suma simple de todos los valores de salida.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote);

//@}

/** @name pds_neuronml_<training_methods>
 *  Métodos sobre entrenamiento de la estructura.
 *
 * @{
 */

/** \fn int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
 *  \brief Entrena la red neuronal multicapa NeuronML. Usa el valor de Output como 
 *  entrenamiento, para finalizar carga el nuevo valor de salida en Output.
 *
 *  Bucle de retropropagación e iteración :
 *  <ul>
 *  <li>pds_nvector_evaluate_error()</li>
 *  <li>pds_nvector_backpropagate_error() : \f$ e_k \leftarrow e_k + \frac{func'\left(\frac{\theta_j}{\sigma_j}\right)}{\sigma_j} W_{kj} e_j \f$ </li>
 *  <li>pds_nvector_update_weight() : \f$ W_{kj} = W_{kj} + \alpha e_j \frac{func'\left(\frac{\theta_j}{\sigma_j}\right)}{\sigma_j} \frac{X_{k}}{X^TX} \f$</li>
 *  <li>pds_neuronml_iterate()</li>
 *  </ul>
 *  El bucle es repetido hasta que el error cuadratico medio en la salida sea menor
 *  NeuronML->MaxError o que el numero de iteraciones sea mayor que NeuronML->MaxIter
 *  (Solo si NeuronML->MaxIter es diferente de cero).
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Input Datos de entrada de la red neuronal multicapa.
 *  \param[in,out] Output Datos esperados de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output);


/** \fn int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
 *  \brief Coloca el valor del factor de aprendizaje Alpha.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Alpha Factor de aprendizaje.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha);


//@}

/** @name pds_neuronml_<fprintf>
 *  Funciones para leer y escribir datos de {Pesos, U, Sigma} de todas las neuronas.
 *
 * @{
 */


/** \fn int pds_neuronml_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i], el valor de U y Sigma.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  (neurona por neurona)
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fprintf(const PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_fscanf(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i], el valor de U  y Sigma.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  (neurona por neurona)
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fscanf(PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U y Sigma.
 *  (neurona por neurona)
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fwrite(const PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_fread(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U  y Sigma.
 *  (neurona por neurona)
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fread(PdsNeuronML *NeuronML, FILE *fd);

//@}

/** @name pds_neuronml_<free>
 *  Liberar memoria.
 *
 * @{
 */

/** \fn void pds_neuronml_free(PdsNeuronML *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_free(PdsNeuronML *NeuronML);


/** \fn void pds_neuronml_destroy(PdsNeuronML **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_destroy(PdsNeuronML **NeuronML);

//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


