/*
 * pdsneuronml.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsneuronml.h
 *  \author Fernando Pujaico Rivera
 *  \date 05-01-2012
 *  \brief Funciones que trabajan con vectores.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas"
 *  \b Nel es el número de elementos.
 */

#ifndef __PDSNEURONML_H__
#define __PDSNEURONML_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsnnglobal.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>
#include <pds/pdsnvector.h>


/** \defgroup PdsNeuronMLGroup Funciones del módulo PdsNeuronML.
 *  \brief Funciones que trabajan con redes neuronales multi capas.
 *  
 *  <br>Estas funciones trabajan con redes neuronales multi capas de la forma.<br>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 * @{
 */


#ifndef TRUE
/*! \def TRUE
 *  Define la palabra TRUE equivalente con 1.
 *  \ingroup PdsNeuronMLGroup
*/
	#define TRUE 1
#endif

#ifndef FALSE
/*! \def FALSE
 *  Define la palabra FALSE equivalente con 0.
 *  \ingroup PdsNeuronMLGroup
*/
	#define FALSE 0
#endif


/*! \struct PdsNeuronML
 *  \brief La estructura tipo  PdsNeuronML .
 *  Esta estructura genera una red neuronal multi capa.
 *  Para usar incluir pds/pdsnn.h.
 *  \ingroup PdsNeuronMLGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un vector que indica el número de neuronas por capa. */
	PdsNnNatural *L;
	/*! Número de capas. */
	PdsNnNatural N;

	/*! Vector de neuronas de entrada Nel elementos. */
	PdsNIVector *LayerInput;
	/*! Arreglo de Nel elementos, cada elemento es un arreglo de neuronas. */
	PdsNVector **Layer;

	/*! Valor de error. */
	PdsNnReal Error;
	/*! Factor de aprendizaje. */
	PdsNnReal Alpha;
	/*! Número máximo de iteraciones. */
	PdsNnNatural MaxIter;
	/*! Número de iteraciones del ultimo entrenamiento. */
	PdsNnNatural Iter;
	/*! Habilitador de printf de avance. 1 habilitado 0 deshabilitado*/
	PdsNnBool ShowPrintf;

}PdsNeuronML;


/** \fn PdsNeuronML *pds_neuronml_new(const PdsVector *L)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia con cero todos los 
 *  elementos. Pero con un error mínimo de 1/sqrt(2), que indica una incertidumbre 
 *  en la mitad de los datos de salida y los demás datos son aciertos. El valor de
 *  de Alpha se inicia con 0.5.
 *  \param[in] L Es un vector con el numero de elementos por capas.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new(const PdsVector *L);


/** \fn PdsNeuronML *pds_neuronml_new_load_data(const char *mlayerfile)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia todos los elementos.
 *  desde un archivo de texto.
 *  \param[in] mlayerfile Es el archivo de texto de donde se leen los datos.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_load_data(const char *mlayerfile);


/** \fn int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayerfile)
 *  \brief Salva una estructura de tipo PdsNeuronML en un archivo de texto.
 *  \param[in] NeuronML Es la red neuronal multicapa a salvar.
 *  \param[in] mlayerfile Es el archivo de texto donde se guardará la red.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem: NeuronML==NULL o mlayerfile=NULL).
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayerfile);


/** \fn int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal Error)
 *  \brief Coloca el maximo error aceptado, cualquier valor de error menor
 *  provoca que se detenga el algoritmo de aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Error Máximo error aceptado.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal Error);


/** \fn int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
 *  \brief Coloca el valor del factor de aprendizaje Alpha.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Alpha Factor de aprendizaje.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha);


/** \fn int pds_neuronml_set_max_number_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
 *  \brief Coloca el máximo numero de iteraciones aceptado, cualquier valor mayor
 *  provoca que se detenga el algoritmo de aprendizaje. si este número es cero
 *  entonces el algoritmo continua hasta que el error en la salida sea menor que
 *  NeuronML->Error.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxIter Número máximo de iteraciones.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_number_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter);


/** \fn int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
 *  \brief Devuelve el número de iteraciones del ultimo entrenamiento hecho con
 *  la función pds_neuronml_training.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Iter Número de iteraciones del último entrenamiento.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter);


/** \fn int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
 *  \brief Habilita la muestra en pantalla del avance del aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_enable_printf(PdsNeuronML *NeuronML);


/** \fn int pds_neuronml_init_uniform_synaptic_weight(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los pesos usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_uniform_synaptic_weight(PdsNeuronML *NeuronML,PdsUniform *RV);


/** \fn int pds_neuronml_init_uniform_u(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los umbrales U usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_uniform_u(PdsNeuronML *NeuronML,PdsUniform *RV);


/** \fn int pds_neuronml_set_sigma(PdsNeuronML *NeuronML,PdsNnReal Sigma)
 *  \brief Coloca el valor de Sigma en todas las neuronas
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Sigma Valor de sigma en la función de activación de las neuronas.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_sigma(PdsNeuronML *NeuronML,PdsNnReal Sigma);


/** \fn int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y)
 *  \brief Itera la red neuronal multicapa NeuronML.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] X Datos de entrada de la red neuronal multicapa.
 *  \param[out] Y Datos de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y);


/** \fn int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y)
 *  \brief Entrena la red neuronal multicapa NeuronML. Usa el valor de Y como 
 *  entrenamiento, para finalizar carga el nuevo valor de Y.
 *  \f[ e^{inputs}_i = W[i] e \f].
 *  \f[ W[i] = W[i] + \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] X Datos de entrada de la red neuronal multicapa.
 *  \param[in] Y Datos esperados de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y);


/** \fn int pds_neuronml_get_hard_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas de salida de la red neuronal multicapa, se considera como un voto 
 *  positivo "+1" cualquier valor mayor o igual a "0". Se considera como un voto 
 *  negativo "-1" cualquier valor menor a "0". Luego se realiza la suma con signo 
 *  de las votaciones.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación abrupta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_hard_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote);


/** \fn int pds_neuronml_get_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas del vector de salida de la red neuronal multicapa, se realiza una 
 *  suma simple de todos los valores de salida.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote);


/** \fn int pds_neuronml_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i], el valor de U y Sigma.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fprintf(const PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_fscanf(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i], el valor de U  y Sigma.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fscanf(PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U y Sigma.
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fwrite(const PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_fread(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U  y Sigma.
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fread(PdsNeuronML *NeuronML, FILE *fd);


/** \fn void pds_neuronml_free(PdsNeuronML *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_free(PdsNeuronML *NeuronML);


/** \fn void pds_neuronml_destroy(PdsNeuronML **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_destroy(PdsNeuronML **NeuronML);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


