/*
 * pdsmath.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsmath.h
 *  \brief Archivo de definición de funciones matemáticas
 *
 *  \author Fernando Pujaico Rivera
 */
#ifndef __PDSMATH_H__
#define __PDSMATH_H__


#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsMathGroup funciónes del módulo PdsMath.
 * @{
 */

/*! \def PDS_PHI
 *  \brief Es equivalente a <b>(1+sqrt(5))/2</b>. El llamado número aureo.
 *
 *  \f[ PDS\_PHI=\frac{1+\sqrt{5}}{2}\f]
 *  \ingroup PdsMathGroup
*/
#define PDS_PHI 1.618033988749894


/*! \def PDS_GOLDEN_RATIO
 *  \brief Es equivalente a <b>PDS_PHI = (1+sqrt(5))/2</b>. El llamado número aureo.
 *
 *  \f[ PDS\_GOLDEN\_RATIO=\frac{1+\sqrt{5}}{2}\f]
 *  \ingroup PdsMathGroup
*/
#define PDS_GOLDEN_RATIO PDS_PHI

/*! \def PDS_1_OVER_SQRT_2PI
 *  \brief Es equivalente a <b>1/sqrt(2*M_PI)</b>.
 *
 *  \f[ PDS\_1\_OVER\_SQRT\_2PI=\frac{1}{\sqrt{2 \pi}}\f]
 *  \ingroup PdsMathGroup
*/
#define PDS_1_OVER_SQRT_2PI 0.398942280401433

/*! \def PDS_2_OVER_SQRT_PI
 *  \brief Es equivalente a <b>2/sqrt(M_PI)</b>.
 *
 *  \f[ PDS\_2\_OVER\_SQRT\_PI=\frac{2}{\sqrt{\pi}}\f]
 *  \ingroup PdsMathGroup
*/
#define PDS_2_OVER_SQRT_PI  1.12837916709551

/*! \def PDS_LN2
 *  \brief Es equivalente a <b>ln(2)</b>.
 *
 *  \f[ PDS\_LN2=ln(2)\f]
 *  \ingroup PdsMathGroup
*/
#define PDS_LN2 6.93147180559945e-01

/*! \var PDS_MPB10
 *  \brief Es un arreglo de variables enteras con números primos
 *  de modo que <b>PDS_MPB10[d]</b> contiene el máximo número primo en base 10 
 *  con "d" decimales. El máximo valor de "d" es PDS_LENGTH_MPB10-1;
 *
 *  \ingroup PdsMathGroup
*/
unsigned long int PDS_MPB10[]={0,7,97,997,9973,99991,999983,9999991};

/*! \var PDS_LENGTH_MPB10
 *  \brief Es el número de elementos del arreglo PDS_MPB10 .
 *
 *  \ingroup PdsMathGroup
*/
unsigned long int PDS_LENGTH_MPB10=8;


/** \fn double pds_qfunc(double x)
 *  \brief Evalúa la función Q(x) 
 *  
 *  \f[ y=Q(x)=\frac{1}{\sqrt{2\pi}}\int_{x}^{\infty}exp({-\frac{u^2}{2}})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de Q(x).
 *  \ingroup PdsMathGroup
 */
double pds_qfunc(double x);


/** \fn double pds_qfuncinv(double x)
 *  \brief Evalúa la función Q^{-1}(x) , función Q inversa.
 *  
 *  \f[ y=Q^{-1}(x) \f]
 *  \f[ x=Q(y)=\frac{1}{\sqrt{2\pi}}\int_{y}^{\infty}exp({-\frac{u^2}{2}})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de Q^{-1}(x).
 *  \ingroup PdsMathGroup
 */
double pds_qfuncinv(double x);


/** \fn double pds_erf(double x)
 *  \brief Evalúa la función erf(x) 
 *  
 *  \f[ y=erf(x)=\frac{2}{\sqrt{\pi}}\int_{0}^{x}exp({-u^2})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de erf(x).
 *  \ingroup PdsMathGroup
 */
double pds_erf(double x);


/** \fn double pds_erfc(double x)
 *  \brief Evalúa la función erfc(x) 
 *  
 *  \f[ y=erfc(x)=1-\frac{2}{\sqrt{\pi}}\int_{0}^{x}exp({-u^2})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de erfc(x).
 *  \ingroup PdsMathGroup
 */
double pds_erfc(double x);


/** \fn double pds_sgn(double x)
 *  \brief Evalúa la función signo sgn(x) 
 *  
 *  \f[y=sgn(x)=\f] \f{eqnarray*}{
        1 &si& x>0 \\ 
        0 &si& x=0 \\ 
       -1 &si& x<0
   \f}
 *  \param[in] x Valor de entrada.
 *  \return El valor de sgn(x).
 *  \ingroup PdsMathGroup
 */
double pds_sgn(double x);


/** \fn float pds_fsgn(float x)
 *  \brief Evalúa la función signo sgn(x) 
 *  
 *  \f[y=sgn(x)=\f] \f{eqnarray*}{
        1 &si& x>0 \\ 
        0 &si& x=0 \\ 
       -1 &si& x<0
   \f}
 *  \param[in] x Valor de entrada.
 *  \return El valor de sgn(x).
 *  \ingroup PdsMathGroup
 */
float pds_fsgn(float x);

/** \fn int pds_isgn(int x)
 *  \brief Evalúa la función signo sgn(x) 
 *  
 *  \f[y=sgn(x)=\f] \f{eqnarray*}{
        1 &si& x>0 \\ 
        0 &si& x=0 \\ 
       -1 &si& x<0
   \f}
 *  \param[in] x Valor de entrada.
 *  \return El valor de sgn(x).
 *  \ingroup PdsMathGroup
 */
int pds_isgn(int x);

/** \fn double pds_sinc(double x)
 *  \brief Evalúa la función sinc(x)=sin(x)/x 
 *  
 *  \f[y=sinc(x)=\frac{sin(x)}{x}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de sinc(x).
 *  \ingroup PdsMathGroup
 */
double pds_sinc(double x);


/** \fn double pds_sigmoid(double x)
 *  \brief Evalúa la función sigmoid(x)=1/(1+e^{-x}) 
 *  
 *  \f[y=sigmoid(x)=\frac{1}{1+e^{-x}}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de sigmoid(x).
 *  \ingroup PdsMathGroup
 */
double pds_sigmoid(double x);


/** \fn double pds_gauss(double x,double U,double Sigma)
 *  \brief Evalúa la función gaussiana, o distribución gaussiana f(x)=N(U,Sigma^2).
 *  
 *  \f[y=\frac{1}{\sqrt{2\pi \sigma^2}}e^{-\frac{(x-U)^2}{2\sigma^2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \param[in] U El valor medio de x.
 *  \param[in] Sigma Es desvío padrón de x.
 *  \return El valor de f(x)=N(U,Sigma^2).
 *  \ingroup PdsMathGroup
 */
double pds_gauss(double x,double U,double Sigma);


/** \fn double pds_gauss2(double x,double U,double Sigma2)
 *  \brief Evalúa la función gaussiana, o distribución gaussiana f(x)=N(U,Sigma2).
 *  
 *  \f[y=\frac{1}{\sqrt{2\pi \sigma^2}}e^{-\frac{(x-U)^2}{2\sigma^2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \param[in] U El valor medio de x.
 *  \param[in] Sigma2 Es la varianza de x, Sigma^2.
 *  \return El valor de f(x)=N(U,Sigma^2).
 *  \ingroup PdsMathGroup
 */
double pds_gauss2(double x,double U,double Sigma2);


/** \fn double pds_gnorm(double x)
 *  \brief Evalúa la función gaussiana normalizada, o distribución gaussiana f(x)=N(0,1.0).
 *  
 *  \f[y=\frac{1}{\sqrt{2\pi}}e^{-\frac{x^2}{2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x)=N(0,1.0).
 *  \ingroup PdsMathGroup
 */
double pds_gnorm(double x);


/** \fn double pds_exp22(double x)
 *  \brief Evalúa la función f(x)=exp(-x^2/2).
 *  
 *  \f[y=e^{-\frac{x^2}{2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x).
 *  \ingroup PdsMathGroup
 */
double pds_exp22(double x);


/** \fn double pds_exp2(double x)
 *  \brief Evalúa la función f(x)=exp(-x^2).
 *  
 *  \f[y=e^{-x^2}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x).
 *  \ingroup PdsMathGroup
 */
double pds_exp2(double x);


/** \fn double pds_exp1(double x)
 *  \brief Evalúa la función f(x)=exp(-x).
 *  
 *  \f[y=e^{-x}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x).
 *  \ingroup PdsMathGroup
 */
double pds_exp1(double x);

/** \fn double pds_hb(double x)
 *  \brief Evalúa la función de entropía binaria Hb(x)=-x*log2(x)-(1-x)*log2(1-x).
 *  
 *  \f[y=H_b(x)=-x log_2(x)-(1-x) log_2(1-x)\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de Hb(x).
 *  \ingroup PdsMathGroup
 */
double pds_hb(double x);

/** \fn double pds_hbinv(double h)
 *  \brief Retorna el valor de x de la función de entropía binaria para un
 *  valor de h dado h=Hb(x)=-x*log2(x)-(1-x)*log2(1-x).
 *  
 *  \f[h=H_b(x)=-x log_2(x)-(1-x) log_2(1-x)\f] 
 *  \param[in] h Valor de entrada.
 *  \return El valor de x en h=Hb(x).
 *  \ingroup PdsMathGroup
 */
double pds_hbinv(double h);

/** \fn double pds_nchoosek(unsigned int n,unsigned int k)
 *  \brief Retorna el combinatorio (n,k)
 *  
 *  \f[ {n \choose k}=\frac{n!}{k!(n-k)!} \f] 
 *  \param[in] n Valor superior del combinatorio.
 *  \param[in] k Valor inferior del combinatorio.
 *  \return El valor del combinatorio (n,k).
 *  \ingroup PdsMathGroup
 */
double pds_nchoosek(unsigned int n,unsigned int k);


/** \fn double pds_binomial(unsigned int n,unsigned int k, double p)
 *  \brief Retorna la distribucion binomial(n,k,p)
 *  
 *  \f[ f(n,k,p)={n \choose k} p^k (1-p)^{n-k}\f] 
 *
 *  \param[in] n Numero total de ensayos
 *  \param[in] k Numero de eventos encontrdos.
 *  \param[in] p probbilidad de contecer un evento en un ensayo.
 *  \return El valor de f(n,k,p).
 *  \ingroup PdsMathGroup
 */
double pds_binomial(unsigned int n,unsigned int k, double p);


/** \fn double pds_integration(double (*f)(double), double a,double b,unsigned int n)
 *  \brief Evalúa la integral de a-->b de la función f(x), aplicando la regla de 
 *  Simpson con n divisiones, si n no es par internamente la función hace n=n+1.
 *  
 *  \f[S_n=\int_{a}^{b}f(x)dx\f] 
 *  \f[h=\frac{b-a}{n}\f] 
 *  \f[x_i=a+h~i\f] 
 *  \f[S_n=\frac{h}{3}(f(x_0)+f(x_n)+4\left [ f(x_1)+f(x_3)+\cdots +f(x_{n-1}) \right ]+2\left [ f(x_2)+f(x_4)+\cdots +f(x_{n-2}) \right ])\f] 
 *  \param[in] f La función a integrar.
 *  \param[in] a Límite inferior de la integral.
 *  \param[in] b Límite superior de la integral.
 *  \param[in] n Es el número de divisiones.
 *  \return El valor de la integral o cero si hubo un error, ejemplo b<a o n<=0.
 *  \ingroup PdsMathGroup
 */
double pds_integration(double (*f)(double), double a,double b,unsigned int n);


/** \fn double pds_integration2(double (*f)(double), double a,double b, double fa,double fb,unsigned int n)
 *  \brief Evalúa la integral de a-->b de la función f(x), aplicando la regla de 
 *  Simpson con n divisiones, si n no es par internamente la función hace n=n+1.
 *  
 *  \f[S_n=\int_{a}^{b}f(x)dx\f] 
 *  \f[h=\frac{b-a}{n}\f] 
 *  \f[x_i=a+h~i\f] 
 *  \f[S_n=\frac{h}{3}(f(a)+f(b)+4\left [ f(x_1)+f(x_3)+\cdots +f(x_{n-1}) \right ]+2\left [ f(x_2)+f(x_4)+\cdots +f(x_{n-2}) \right ])\f] 
 *  \param[in] f La función a integrar.
 *  \param[in] a Límite inferior de la integral.
 *  \param[in] b Límite superior de la integral.
 *  \param[in] fa El resultado de evaluar f(a).
 *  \param[in] fb El resultado de evaluar f(b).
 *  \param[in] n Es el número de divisiones.
 *  \return El valor de la integral o cero si hubo un error, ejemplo b<a o n<=0.
 *  \ingroup PdsMathGroup
 */
double pds_integration2(double (*f)(double), double a,double b, double fa,double fb,unsigned int n);


/** \fn double pds_integration_inf(double (*f)(double), double a,unsigned int n)
 *  \brief Evalúa la integral de a-->infinito de la función f(x), aplicando el 
 *  cambio de variable u<--1/(x+1) para integrar de 0-->1/(a+1) y ejecutar luego 
 *  la regla de Simpson con n divisiones, si n no es par internamente la 
 *  función hace n=n+1. Además es necesario que f(infinito)-->0.
 *  
 *  \f[S_n=\int_{a}^{ \infty }f(x)dx\f] 
 *  \f[u=\frac{1}{x+1}\f] 
 *  \f[S_n=\int_{0}^{ \frac{1}{a+1} }\frac{f(\frac{1}{u}-1)}{u^2}du\f] 
 *  \param[in] f La función a integrar.
 *  \param[in] a Límite inferior de la integral. a>=0.
 *  \param[in] n Es el número de divisiones.
 *  \return El valor de la integral o cero si hubo un error, ejemplo b<a o n<=0.
 *  \ingroup PdsMathGroup
 */
double pds_integration_inf(double (*f)(double), double a,unsigned int n);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSMATH_H__ */
