/*
 * pdsit.c
 * 
 * Copyright 2015 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsit.h>

#include <pds/pdsba.h>
#include <pds/pdsra.h>


#include <pds/pdsmath.h>



/** \fn int pds_symetric_ber_bsc_model(double rho,unsigned int M, double *BER)
 *  \brief Retorna la taza de erro de bit, \f$BER\f$, 
 *  en el problema CEO binario simétrico
 *  de M fontes \f$ U_m, \forall ~ m \in \{1, 2, ..., M\}\f$, donde \f$\hat{U}_0\f$
 *  representa la mejor aproximación de \f$U_0\f$ conociendo \f$U_1 U_2 ... U_M\f$. 
 *
 *  Estas fuentes son generadas pasando una fuente \f$U_0\f$, con probabilidade
 *  \f$P(U_0=1)=0.5\f$ , atraves de M canales BSC
 *  con probabilidades de error \f$P(U_m \neq U0 | U0)=\rho\f$.
 *
 *  <center>
 *  \image html msbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *
 *  Quando M es impar :
 *  \f[ BER= \sum_{j=(M+1)/2}^{M} {{M}\choose{j}} {\rho}^j (1-\rho)^{M-j} \f]
 *
 *  Quando M es par :
 *  \f[ BER= \sum_{j=(M/2)+1}^{M} {{M}\choose{j}} {\rho}^j (1-\rho)^{M-j}+0.5 {{M}\choose{M/2}} {\rho}^{M/2} (1-\rho)^{M/2} \f]
 *
 *  <b>!!!!CUIDADO ACTUAL \f$BER=P(U_m\neq U_0 | \lfloor \frac{M+1}{2} \rfloor\leq m \leq M)\f$ !!!!!</b>
 *
 *  The formula for calculus of BER is in \cite BER1 and other simplification in \cite BER2 \cite jointsbceo .
 *
 *  \param[in] rho Es la probabilidad de error de los canales BSC. \f$ P(U_m \neq U_0| U_0)=\rho \f$.
 *  \param[in] M Es el número de canales BSC.
 *  \param[out] BER es la taza de error de bit.
 *  \return TRUE si todo fue bien, o FALSE sino. 
 *  \ingroup PdsITGroup
 */
int pds_symetric_ber_bsc_model(double rho,unsigned int M, double *BER)
{
	double ber;
	int j;

    if(BER==NULL)   return FALSE;

    if ((M%2)==0)
    {
	    ber=0.5*pds_nchoosek(M,M/2)*pow(rho,M/2)*pow(1.0-rho,M/2);
	    for(j=((M/2)+1);j<=M;j++)
	    {
		    ber=ber+pds_nchoosek(M,j)*pow(rho,j)*pow(1.0-rho,M-j);
	    }

    }
    else
    {
	    ber=0;
	    for(j=((M+1)/2);j<=M;j++)
	    {
		    ber=ber+pds_nchoosek(M,j)*pow(rho,j)*pow(1.0-rho,M-j);
	    }
    }
	*BER=ber;

	return TRUE;
}

/** \fn int pds_vector_symetric_ber_bsc_model(const PdsVector* RHO,PdsRaNatural M,PdsVector* BER)
 *  \brief Retorna un vector con las tazas de erro de bit, \f$BER\f$,
 *  (para cada valor de probabilidad de error dentro del vector RHO) 
 *  en el problema CEO binario simétrico
 *  de M fontes \f$ U_m, \forall ~ m \in \{1, 2, ..., M\}\f$, donde \f$\hat{U}_0\f$
 *  representa la mejor aproximación de \f$U_0\f$ conociendo \f$U_1 U_2 ... U_M\f$. 
 *
 *  Estas fuentes son generadas pasando una fuente \f$U_0\f$, con probabilidade
 *  \f$P(U_0=1)=0.5\f$ , atraves de M canales BSC
 *  con probabilidades de error \f$P(U_m \neq U0 | U0)=\rho\f$.
 *
 *  <center>
 *  \image html msbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *
 *  Quando M es impar :
 *  \f[ BER= \sum_{j=(M+1)/2}^{M} {{M}\choose{j}} {\rho}^j (1-\rho)^{M-j} \f]
 *
 *  Quando M es par :
 *  \f[ BER= \sum_{j=(M/2)+1}^{M} {{M}\choose{j}} {\rho}^j (1-\rho)^{M-j}+0.5 {{M}\choose{M/2}} {\rho}^{M/2} (1-\rho)^{M/2} \f]
 *
 *  <b>!!!!CUIDADO ACTUAL \f$BER=P(U_m\neq U_0 | \lfloor \frac{M+1}{2} \rfloor\leq m \leq M)\f$ !!!!!</b>
 *
 *  The formula for calculus of BER is in \cite BER1 and other simplification in \cite BER2 \cite jointsbceo .
 *
 *  \param[in] RHO Es un vector con las probabilidades de error \f$\rho\f$ a ser 
 *  evaluadas. \f$ P(U_m \neq U_0| U_0)=\rho \f$.
 *  \param[in] M Es el número de canales BSC.
 *  \param[out] BER Es un vector con la taza de error de bit para cada valor de 
 *  probabilidades de error \f$\rho\f$ en el vector RHO.
 *  \return TRUE si todo fue bien, o FALSE sino. 
 *  \ingroup PdsITGroup
 */
int pds_vector_symetric_ber_bsc_model(const PdsVector* RHO,PdsRaNatural M,PdsVector* BER)
{
    int dat;
    PdsRaNatural i;
    double val;

    if(RHO==NULL)   return FALSE;
    if(BER==NULL)   return FALSE;
    if(M<1)         return FALSE;

    if(BER->Nel!=RHO->Nel)   return FALSE;

    for(i=0;i<RHO->Nel;i++)
    {
        dat=pds_symetric_ber_bsc_model (RHO->V[i], M,&val);
        if(dat==FALSE)  return FALSE;
        BER->V[i]=val;
    }
    return TRUE;
}



/** \fn int pds_joint_probability_bsc_model(PdsBVector *OmegaM,const PdsVector *Ps,double p0, double *JointProb)
 *  \brief Encuentra la probabilidad conjunta \f$P(\Omega_M)\f$ de tener en la salida 
 *  de un grupo de canales BSC un conjunto de valores binarios formando un vector 
 *  \f$\Omega_M\f$ \cite jointprobh . 
 *
 *  <center>
 *  \image html mbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Conocido un conjunto \f$S=\{U_1, U_2, ..., U_M\}\f$ de fuentes binarias,
 *  estas son construidas pasando la fuente binaria
 *  \f$U_0\f$ con \f$ P(U_0=1)=p_0 \f$, por M canales BSC con probabilidad
 *  de error \f$ P(U_m \neq U_0| U_0)={P_s}_{m}~\forall U_m \in S  \f$.
 *  Si definimos el vector \f$\Omega_M\f$ como:
 *  \f[ \Omega_M \equiv \{u_m: U_m=u_m \forall U_m \in S\} \f]
 *  Entonces este tendrá como probabilidad:
 *  \f{eqnarray*}{
 *  P(\Omega_M)& = & P(U_1=u_1|U_0=0) ... P(U_M=u_M|U_0=0) P(U_0=0) \\
 *  ~          & + & P(U_1=u_1|U_0=1) ... P(U_M=u_M|U_0=1) P(U_0=1)

 *  \f}
 *
 *  \param[in] OmegaM Es un caso de ocurrencia del vector binario.
 *  \param[in] Ps Es el vector de probabilidades de error de las fuentes BSC.
 *  Ps=[Ps0, Ps1, ..., Ps(M-1)]. \f$ {P_s}_{m} \rightarrow Ps(m-1)\f$.
 *  \param[in] p0 Probabilida Pr(U0=1)=p0 de la fuente U0.
 *  \param[out] JointProb probabilidad conjuta del caso de ocurrencia de OmegaM.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: OmegaM==NULL, Ps==NULL.
 *  \ingroup PdsITGroup
 */
int pds_joint_probability_bsc_model(PdsBVector *OmegaM,const PdsVector *Ps,double p0, double *JointProb)
{
	PdsBaNatural M,i;
	double u00,u01;
	PdsBaBit ui;

	if(OmegaM==NULL)			return FALSE;
	if(Ps==NULL)				return FALSE;
	if(OmegaM->Nel!=Ps->Nel)	return FALSE;

	M=OmegaM->Nel;

	//u00 = P(U1|U0=0) ... P(Um|U0=0) ... P(UM|U0=0)
	//u01 = P(U1|U0=1) ... P(Um|U0=1) ... P(UM|U0=1)
	for(u00=1.0,u01=1.0,i=0;i<M;i++)
	{
		pds_bvector_get_bit (OmegaM,i,&ui);
		if(ui==1)
		{
			u00=u00*Ps->V[i];
			u01=u01*(1.0-Ps->V[i]);
		}
		else
		{
			u00=u00*(1.0-Ps->V[i]);
			u01=u01*Ps->V[i];
		}
	}
	
	*JointProb=u00*(1-p0)+u01*p0;

	return TRUE;
}

/** \fn int pds_joint_entropy_bsc_model(const PdsVector *Ps,double p0, double *HJoint)
 *  \brief Encuentra la entropia conjunta \f$ H(U_1,...,U_M)\f$ para M fontes generadas
 *  pasando una fuente U0, con probabilidade Pr(U0=1)=p0, atraves de M canales BSC
 *  con probabilidades de error Ps=[Ps0, Ps1, ..., Ps(M-1)] \cite jointprobh . 
 *
 *  <center>
 *  \image html mbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  \f[ H(U_1, U_2, ..., U_M)= \sum_{U_1, U_2, ... U_M} P(U_1, U_2, ..., U_M)~log_2(P(U_1, U_2, ..., U_M)) \f]
 *   
 *  \f[ P(U_m \neq U_0| U_0)={P_s}_{m} \f]
 *
 *  \param[in] Ps Es el vector de probabilidades de error de las fuentes BSC.
 *  Ps=[Ps0, Ps1, ..., Ps(M-1)]. \f$ {P_s}_{m} \rightarrow Ps(m-1)\f$.
 *  \param[in] p0 Probabilida Pr(U0=1)=p0.
 *  \param[out] HJoint Es el resultado, la entropia conjuta.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: Ps==NULL.
 *  \ingroup PdsITGroup
 */
int pds_joint_entropy_bsc_model(const PdsVector *Ps,double p0, double *HJoint)
{
	double hjoint;
	double prob;
	PdsBaNatural L,i;
	int id;

	PdsBVector *U=NULL;

	if(Ps==NULL)		return FALSE;

	////////////////////////////////////////////////////////////////////////
	L=pow(2,Ps->Nel);
	U=pds_bvector_new(Ps->Nel);
	////////////////////////////////////////////////////////////////////////

	for(hjoint=0.0,i=0;i<L;i++)
	{
		pds_bvector_set_natural(U,i);
		prob=0;	
		id=pds_joint_probability_bsc_model(U,Ps,p0,&prob);
		if(id==FALSE)	
		{
			*HJoint=0;
			pds_bvector_free(U);
			return FALSE;
		}
		if ( (prob>0) && (prob<1.0) )
		{
			hjoint=hjoint-prob*log(prob)/PDS_LN2;
		}
	}
	*HJoint=hjoint;
	////////////////////////////////////////////////////////////////////////

	pds_bvector_free(U);

	return TRUE;
}


/** \fn int pds_probability_u0_omega_bsc_model(PdsBVector *OmegaM,const PdsVector *Ps,PdsBaBit U0,double p0, double *PcU0OmegaM, double *PjU0OmegaM)
 *  \brief Encuentra las probabilidades  condicionada \f$P(u_0 |\Omega_M)\f$ y
 *  conjunta \f$P(u_0 \Omega_M)\f$ \cite jointprobh . 
 *
 *  <center>
 *  \image html mbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Conocido un conjunto \f$S=\{U_1, U_2, ..., U_M\}\f$ de fuentes binarias,
 *  estas son construidas pasando la fuente binaria
 *  \f$U_0\f$ con \f$ P(U_0=1)=p_0 \f$, por M canales BSC con probabilidad
 *  de error \f$ P(U_m \neq U_0| U_0)={P_s}_{m}~\forall U_m \in S  \f$.
 *  Asi, definimos el vector \f$\Omega_M\f$ como:
 *  \f[ \Omega_M \equiv \{u_m: U_m=u_m \forall U_m \in S\} \f]
 *  Entonces este tendrá como probabilidad:
 *  \f[
 *  P(u_0\Omega_M) =  P(U_1=u_1|U_0=u_0) ... P(U_M=u_M|U_0=u_0) P(U_0=u_0) 
 *  \f]
 *  \f[
 *  P(u_0|\Omega_M) =  \frac{P(u_0\Omega_M) }{P(\Omega_M)}
 *  \f]
 *  Sabiendo que:
 *  \f{eqnarray*}{
 *  P(\Omega_M)& = & P(U_1=u_1|U_0=0) ... P(U_M=u_M|U_0=0) P(U_0=0) \\
 *  ~          & + & P(U_1=u_1|U_0=1) ... P(U_M=u_M|U_0=1) P(U_0=1)

 *  \f}
 *
 *  \param[in] OmegaM Es un caso de ocurrencia del vector binario.
 *  \param[in] Ps Es el vector de probabilidades de error de las fuentes BSC.
 *  Ps=[Ps0, Ps1, ..., Ps(M-1)]. \f$ {P_s}_{m} \rightarrow Ps(m-1)\f$.
 *  \param[in] U0 Valor de U0 a analiza.
 *  \param[in] p0 Probabilida Pr(U0=1)=p0 de la fuente U0.
 *  \param[out] PcU0OmegaM Retorna la probabilidad condicionada. \f$P(u_0 |\Omega_M)\f$.
 *  \param[out] PjU0OmegaM Retorna la probabilidad conjuta. \f$P(u_0 \Omega_M)\f$.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: OmegaM==NULL, Ps==NULL.
 *  \ingroup PdsITGroup
 */
int pds_probability_u0_omega_bsc_model(PdsBVector *OmegaM,const PdsVector *Ps,PdsBaBit U0,double p0, double *PcU0OmegaM, double *PjU0OmegaM)
{
	PdsBaNatural M,i;
	double u00,u01,PrOmegaM;
	PdsBaBit ui;

	if(OmegaM==NULL)			return FALSE;
	if(Ps==NULL)				return FALSE;
	if(OmegaM->Nel!=Ps->Nel)	return FALSE;

	M=OmegaM->Nel;

	//u00 = P(U1|U0=0) ... P(Um|U0=0) ... P(UM|U0=0)
	//u01 = P(U1|U0=1) ... P(Um|U0=1) ... P(UM|U0=1)
	for(u00=1.0,u01=1.0,i=0;i<M;i++)
	{
		pds_bvector_get_bit (OmegaM,i,&ui);
		if(ui==1)
		{
			u00=u00*Ps->V[i];
			u01=u01*(1.0-Ps->V[i]);
		}
		else
		{
			u00=u00*(1.0-Ps->V[i]);
			u01=u01*Ps->V[i];
		}
	}
	
	PrOmegaM=u00*(1-p0)+u01*p0;

	if(U0==0)
	{
		*PcU0OmegaM=(u00*(1-p0))/PrOmegaM;
		*PjU0OmegaM=(u00*(1-p0));
	}
	else
	{
		*PcU0OmegaM=(u01*p0)/PrOmegaM;
		*PjU0OmegaM=(u01*p0);
	}
	

	return TRUE;
}


/** \fn int pds_entropy_u0_omega_bsc_model(const PdsVector *Ps,double p0, double *HCond)
 *  \brief Encuentra la entropia condicionada. \f$ H(U_0|U_1,...,U_M)\f$ \cite jointprobh . 
 *
 *  <center>
 *  \image html mbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Dadas M fontes Um , m ={1,2,...,M}, generadas
 *  pasando una fuente U0, con probabilidade Pr(U0=1)=p0, atraves de M canales BSC
 *  con probabilidades de error Ps=[Ps0, Ps1, ..., Ps(M-1)].
 *
 *  \f[ H(U_0|U_1, U_2, ..., U_M)= \sum_{U_0,U_1, U_2, ... U_M} P(U_0,U_1, U_2, ..., U_M)~log_2(P(U_0|U_1, U_2, ..., U_M)) \f]
 *   
 *  \f[ P(U_m \neq U_0| U_0)={P_s}_{m} \f]
 *
 *  \param[in] Ps Es el vector de probabilidades de error de las fuentes BSC.
 *  Ps=[Ps0, Ps1, ..., Ps(M-1)]. \f$ {P_s}_{m} \rightarrow Ps(m-1)\f$.
 *  \param[in] p0 Probabilida Pr(U0=1)=p0.
 *  \param[out] HCond Entropia condicionada. \f$ H(U_0|U_1,...,U_M)\f$.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: Ps==NULL.
 *  \ingroup PdsITGroup
 */
int pds_entropy_u0_omega_bsc_model(const PdsVector *Ps,double p0, double *HCond)
{
	double hcond;
	double probc,probj;
	PdsBaNatural L,i,j;
	PdsBaBit U0;
	int id;

	PdsBVector *U=NULL;

	if(Ps==NULL)		return FALSE;

	////////////////////////////////////////////////////////////////////////
	L=pow(2,Ps->Nel);
	U=pds_bvector_new(Ps->Nel);
	////////////////////////////////////////////////////////////////////////

	hcond=0.0;

	for(j=0;j<2;j++)
	{
		U0=j;

		for(i=0;i<L;i++)
		{
			pds_bvector_set_natural(U,i);

			probc=0;probj=0;	
			id=pds_probability_u0_omega_bsc_model(U,Ps,U0,p0,&probc,&probj);
			if(id==FALSE)	{*HCond=0;pds_bvector_free(U);return FALSE;}
			if( (probc>0)&&(probc<1) )
			{
				hcond=hcond-probj*log(probc)/log(2.0);
			}
		}
	}
	*HCond=hcond;
	////////////////////////////////////////////////////////////////////////

	pds_bvector_free(U);

	return TRUE;
}


/** \fn int pds_symetric_entropy_u0_omega_bsc_model(double rho,PdsBaNatural M, double *HCond)
 *  \brief Encuentra la entropia condicionada HCond \cite condsbceo . \f$ HCond=h(U_0| \rho,M)=H(U_0|U_1,...,U_M)\f$.
 *  Este es el caso cuando todas las probabilidades de error son iguales a rho. \f$rho=\rho\f$.
 *
 *  <center>
 *  \image html msbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Dadas M fontes Um , m ={1,2,...,M}, generadas
 *  pasando una fuente U0, con probabilidade Pr(U0=1)=0.5, atraves de M canales BSC
 *  con probabilidades de error Ps=[rho, rho, ..., rho].
 *
 *  \f[ h(U_0| \rho,M)= \sum_{k=0}^{M} {M \choose k} \rho^k (1-\rho)^{M-k}~log_2 \left ( 1 + \{ \frac{\rho}{1-\rho} \}^{M-2 k} \right ) \f]
 *   
 *  \f[ P(U_m \neq U_0| U_0)=\rho \f]
 *
 *  \param[in] rho Es la probabilidad de error de los canales BSC. \f$ P(U_m \neq U_0| U_0)=\rho \f$.
 *  \param[in] M Es el número de canales BSC.
 *  \param[out] HCond Entropia condicionada. \f$ h(U_0| \rho,M)=H(U_0|U_1,...,U_M)\f$.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: Ps==NULL.
 *  \ingroup PdsITGroup
 */
int pds_symetric_entropy_u0_omega_bsc_model(double rho,PdsBaNatural M, double *HCond)
{
	double hcond;
	double prob1,prob2;
	int m;

	hcond=0;
	for(m=0;m<=M;m++)
	{
		prob1=pds_nchoosek(M,m)*pow(rho,m)*pow(1.0-rho,M-m);
		prob2=1.0+pow(rho/(1.0-rho),M-2.0*m);

		hcond=hcond+prob1*log(prob2)/PDS_LN2 ;
	}
	*HCond=hcond;

	return TRUE;
}


/** \fn int pds_symetric_joint_entropy_bsc_model(double rho,PdsRaNatural M, double *HJoint)
 *  \brief Encuentra la entropia conjunta HJoint, \f$ HJoint=h( \rho,M)=H(U_1,...,U_M)\f$, de las señales
 *  a la salida de un bloque de de M fuentes binarias correlacionadas \cite jointsbceo.
 * 
 *  Este es el caso cuando todas las probabilidades de error son iguales a rho \f$=\rho\f$.
 *
 *  <center>
 *  \image html msbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Dadas M fontes Um , m ={1,2,...,M}, generadas
 *  pasando una fuente U0, con probabilidade Pr(U0=1)=0.5, atraves de M canales BSC
 *  con probabilidades de error Ps=[rho, rho, ..., rho].
 *
 *  \f[ P_k(\rho,M)= 0.5 \{ \rho^k (1-\rho)^{M-k} + \rho^{M-k} (1-\rho)^{k} \} \f]

 *  \f[ h(\rho,M)= \sum_{k=0}^{M}  {M \choose k} P_k(\rho,M)~log_2 ( P_k(\rho,M)    ) \f]
 *   
 *  \f[ P(U_m \neq U_0| U_0)=\rho \f]
 *
 *  \param[in] rho Es la probabilidad de error de los canales BSC. \f$ P(U_m \neq U_0| U_0)=\rho \f$.
 *  \param[in] M Es el número de canales BSC.
 *  \param[out] HJoint Entropia conjunta. \f$ h(\rho,M)=H(U_1,...,U_M)\f$.
 *  \return Sempre retorna TRUE.
 *  \ingroup PdsITGroup
 */
int pds_symetric_joint_entropy_bsc_model(double rho,PdsRaNatural M, double *HJoint)
{
	double hjoint;
	double prob;
	PdsBaNatural m;

	hjoint=0;
	for(m=0;m<=M;m++)
	{
		prob=0.5*(pow(rho,m)*pow(1-rho,M-m)+pow(1-rho,m)*pow(rho,M-m));
		
		hjoint=hjoint-pds_nchoosek(M,m)*prob*log(prob)/PDS_LN2 ;
	}
	*HJoint=hjoint;
	return TRUE;
}


/** \fn int pds_inv_symetric_entropy_u0_omega_bsc_model(double HCond,short int M,double * rho)
 *  \brief Encuentra la inversa \f$\rho\f$ de la entropia condicionada HCond \cite condsbceo . 
 *  \f$ HCond=h(U_0| \rho,M)=H(U_0|U_1,...,U_M)\f$.
 *
 *  Esta funcion es la inversa de pds_symetric_entropy_u0_omega_bsc_model().
 *  Cuando HCond y M son entregados la funcion retorna el valor de rho.
 *  Este es el caso cuando todas las probabilidades de error son iguales a rho. \f$rho=\rho\f$.
 *
 *  <center>
 *  \image html msbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Dadas M fontes Um , m ={1,2,...,M}, generadas
 *  pasando una fuente U0, con probabilidade Pr(U0=1)=0.5, atraves de M canales BSC
 *  con probabilidades de error Ps=[rho, rho, ..., rho].
 *
 *  \f[ h(U_0| \rho,M)= \sum_{k=0}^{M} {M \choose k} \rho^k (1-\rho)^{M-k}~log_2 \left ( 1 + \{ \frac{\rho}{1-\rho} \}^{M-2 k} \right ) \f]
 *   
 *  \f[ P(U_m \neq U_0| U_0)=\rho \f]
 *
 *  \param[in] HCond Entropia condicionada. \f$ HCond=h(U_0| \rho,M)=H(U_0|U_1,...,U_M)\f$.
 *  \param[in] M Es el número de canales BSC.
 *  \param[out] rho Es la probabilidad de error de los canales BSC. \f$ P(U_m \neq U_0| U_0)=\rho \f$.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: rho==NULL.
 *  \ingroup PdsITGroup
 */
int pds_inv_symetric_entropy_u0_omega_bsc_model(double HCond,short int M,double * rho)
{
	double H;
	double ps_min=0;
	double ps_max=0.5;
	double ps;

	double percent=0.01;//0.01% de difrencia como maximo entre HCond y H(u0|ps,M)

	if(HCond<=0.0)	
	{
		*rho=ps_min;

		if(HCond<0.0)	return FALSE;

	}
	else if(HCond>=1.0)
	{
		*rho=ps_max;

		if(HCond>1.0)	return FALSE;
	}
	else
	{
		do{
			ps=(ps_min+ps_max)/2;

			H=0;
			pds_symetric_entropy_u0_omega_bsc_model(ps,M,&H);

			if(H<HCond)
			{
				ps_min=ps;
			}
			if(H>HCond)
			{
				ps_max=ps;
			}

		}while((fabs(H-HCond)/HCond)>(percent/100.0));
		*rho=ps;
	}
	return TRUE;
}


/** \fn int pds_inv_symetric_joint_entropy_bsc_model(double HJoint,short int M,double * rho)
 *  \brief Encuentra la inversa \f$\rho\f$ de la entropia conjunta \f$HJoint\f$
 *  \cite jointsbceo . \f$ HJoint=h( \rho,M)=H(U_1,...,U_M)\f$
 *
 *  Esta funcion es la inversa de pds_symetric_joint_entropy_bsc_model().
 *  Cuando HJoint y M son entregados la funcion retorna el valor de rho.
 *  Este es el caso cuando todas las probabilidades de error son iguales a rho \f$=\rho\f$.
 *
 *  <center>
 *  \image html msbscpack.png "Bloque de M fuentes binarias correlacionadas"
 *  </center>
 *
 *  Dadas M fontes Um , m ={1,2,...,M}, generadas
 *  pasando una fuente U0, con probabilidade Pr(U0=1)=0.5, atraves de M canales BSC
 *  con probabilidades de error Ps=[rho, rho, ..., rho].
 *
 *  \f[ P_k(\rho,M)= 0.5 \{ \rho^k (1-\rho)^{M-k} + \rho^{M-k} (1-\rho)^{k} \} \f]
 *
 *  \f[ h(\rho,M)= \sum_{k=0}^{M}  {M \choose k} P_k(\rho,M)~log_2 ( P_k(\rho,M)    ) \f]
 *   
 *  \f[ P(U_m \neq U_0| U_0)=\rho \f]
 *
 *  \param[in] HJoint Entropia conjunta. \f$ h(\rho,M)=H(U_1,...,U_M)\f$.
 *  \param[in] M Es el número de canales BSC.
 *  \param[out] rho Es la probabilidad de error de los canales BSC. \f$ P(U_m \neq U_0| U_0)=\rho \f$.
 *  \return Sempre retorna TRUE.
 *  \ingroup PdsITGroup
 */
int pds_inv_symetric_joint_entropy_bsc_model(double HJoint,short int M,double * rho)
{
	double H;
	double ps_min=0;
	double ps_max=0.5;
	double ps;

	double percent=0.01;//0.01% de diferencia como maximo entre HJoint y H(ps,M)

	if(HJoint<=0.0)	
	{
		*rho=ps_min;

		if(HJoint<0.0)	return FALSE;

	}
	else if(HJoint>=M)
	{
		*rho=ps_max;

		if(HJoint>M)	return FALSE;
	}
	else
	{
		do{
			ps=(ps_min+ps_max)/2;

			H=0;
			pds_symetric_joint_entropy_bsc_model(ps,M,&H);

			if(H<HJoint)
			{
				ps_min=ps;
			}
			if(H>HJoint)
			{
				ps_max=ps;
			}

		}while((fabs(H-HJoint)/HJoint)>(percent/100.0));
		*rho=ps;
	}
	return TRUE;
}


