/*
 * pdsfft.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsfft.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con Fft.
 *  
 */

#ifndef __PDSFFT_H__
#define __PDSFFT_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <stdio.h>

#include <pds/pdsftglobal.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsFftGroup Módulo PdsFft.
 *  \brief Cálculo de la FFT (Fast Fourier Transform).
 *  
 *  Estas funciones trabajan con una FFT así.<br>
 *  <center>
 *  \image html pdsfft.png "Transformada Rápida de Fourier"
 *  </center>
   \f{eqnarray*}
        X[k] &=& \sum_{i=0}^{N-1}x[i]W_N^{ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N }
   \f}
 * @{
 */


/*! \struct PdsFft
 *  \brief Estructura tipo  PdsFft.
 *  Esta estructura genera una Transformada Rápida de Fourier de N puntos.
 *  
 *  Para usar incluir pds/pdsft.h.
 *
 *  <center>
 *  \image html pdsfft.png "Transformada Rápida de Fourier"
 *  </center>
   \f{eqnarray*}
        X[k] &=& \sum_{i=0}^{N-1}x[i]W_N^{ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N }
   \f}
 *  \ingroup PdsFftGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de N/2 elementos con W[k]=CIS(-2*M_PI*k/N), k en [0,N/2). */
	PdsCVector *W;
	/*! Número de elementos, N es potencia de 2, 2^r=N  */
	PdsFtNatural N;
}PdsFft;


//@}

/** @name pds_fft_new
 *  Creando nuevas estructuras de tipo PdsFft
 * @{
 */

/** \fn PdsFft *pds_fft_new(PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsFft, para generar una FFT de N puntos.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una FFT
 *  con un  N1, que si es potencia de dos y mayor a N, (N1>=N). El valor de N mínimo
 *  es N=2. El nuevo valor N1 será cargado a N. N=N1.
 *  \param[in,out] N Es el número de elementos de la FFT.
 *  \return Un puntero a una estructura de tipo PdsFft. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsFftGroup
 */
PdsFft *pds_fft_new(PdsFtNatural *N);


//@}

/** @name pds_fft_evaluate
 *  Evaluando datos
 * @{
 */


/** \fn int pds_fft_evaluate_real(const PdsFft *FFT,PdsCVector *Out,const PdsVector *In)
 *  \brief Evalua la transformada rapida de fourier a un vetor real. El tamanho 
 *  del vector Out debe ser igual que el número de puntos de la FFT, y mayor que
 *  el número de elementos de In, La diferencia se llevanara con ceros.
 *  \param[in] FFT La estructura a una FFT de N puntos.
 *  \param[out] Out El vector complejo con la FFT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la FFT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFftGroup
 */
int pds_fft_evaluate_real(const PdsFft *FFT,PdsCVector *Out,const PdsVector *In);


/** \fn int pds_fft_evaluate_complex(const PdsFft *FFT,PdsCVector *Out,const PdsCVector *In)
 *  \brief Evalua la transformada rapida de fourier a un vetor complejo. El tamanho 
 *  del vector Out debe ser igual que el número de puntos de la FFT, y mayor que
 *  el número de elementos de In, La diferencia se llevanara con ceros.
 *  \param[in] FFT La estructura a una FFT de N puntos.
 *  \param[out] Out El vector complejo con la FFT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la FFT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFftGroup
 */
int pds_fft_evaluate_complex(const PdsFft *FFT,PdsCVector *Out,const PdsCVector *In);


//@}

/** @name pds_fft_<free>
 *  Liberando memoria
 * @{
 */


/** \fn void pds_fft_free(PdsFft *FFT)
 *  \brief Libera una estructura de tipo puntero PdsFft.
 *  \param[in,out] FFT La FFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFftGroup
 */
void pds_fft_free(PdsFft *FFT);


/** \fn void pds_fft_destroy(PdsFft **FFT)
 *  \brief Libera una estructura de tipo puntero PdsFft, y carga a la estructura con NULL.
 *  \param[in,out] FFT La FFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFftGroup
 */
void pds_fft_destroy(PdsFft **FFT);

//@}


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSFFT_H__ */ 

