/*
 * pdsgoertzel.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsgoertzel.h>
#include <pds/pdsvector.h>
#include <pds/pdscn.h>
#include <stdio.h>
#include <math.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsGoertzel                                            ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsGoertzel *pds_goertzel_new(PdsFtNatural k,PdsFtNatural N)
 *  \brief Crea una estructura de tipo PdsGoertzel, para generar una FT de N puntos.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una FT
 *  con un  N1, que si es potencia de dos y mayor a N, (N1>=N). El valor de N mínimo
 *  es N=2 .
 *  \param[in] k Es el elemento de la FT{x[n]}, X[k], que se desea encontrar.
 *  \param[in] N Es el número de elementos de la FT.
 *  \return Un puntero a una estructura de tipo PdsGoertzel. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsGoertzelGroup
 */
PdsGoertzel *pds_goertzel_new(PdsFtNatural k,PdsFtNatural N)
{
	PdsGoertzel *FT=NULL;
	PdsFtNatural i,r,n;

	if(N<=1)	return NULL;
	if(k>=N)	return NULL;


	////////// Encuentro un N potencia de 2 //////////
	for(i=0,r=(N-1);r>0;i++)	r=r/2;
	n=i;
	for(i=0,r=1;i<n;i++)		r=r*2;
	//////////////////////////////////////////////////

	FT=(PdsGoertzel *)calloc(1,sizeof(PdsGoertzel));
	if(FT==NULL)	return NULL;

	FT->k=k;
	FT->N=r;
	FT->W=pds_complex_cis(-(2.0*M_PI*k)/(FT->N*1.0));
	FT->a=2.0*cos((2.0*M_PI*k)/(FT->N*1.0));
	
	return FT;
}


/** \fn int pds_goertzel_real_evaluate(const PdsGoertzel *FT,PdsComplex *X,const PdsVector *In)
 *  \brief Evalúa la transformada de fourier X[k] a un vector real. El tamaño 
 *  del vector In debe ser igual o menor que el número de puntos de la FT, La diferencia 
 *  se llenará con ceros.
 *  \param[in] FT La estructura a una FT de Goertzel de N puntos.
 *  \param[out] X El elemento complejo en la posición k de la FT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la FT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsGoertzelGroup
 */
int pds_goertzel_real_evaluate(const PdsGoertzel *FT,PdsComplex *X,const PdsVector *In)
{
	PdsFtNatural i;
	PdsComplex y,z;
	PdsFtReal t[3],x;

	if(FT==NULL)		return FALSE;
	if(In==NULL)		return FALSE;
	if(In->Nel>FT->N)	return FALSE;

	t[0]=0.0;	t[1]=0.0;	t[2]=0.0;

	for(i=0;i<=FT->N;i++)
	{
		t[2]=t[1];
		t[1]=t[0];

		if(i<In->Nel)	x=In->V[i];
		else		x=0.0;
		t[0]=x + FT->a*t[1] - t[2];
	}
	*X=_CMULR(FT->W,-t[1]);
	*X=_CADDR(*X,t[0]);
	return TRUE;
}


/** \fn void pds_goertzel_free(PdsGoertzel *FT)
 *  \brief Libera una estructura de tipo puntero PdsGoertzel.
 *  \param[in,out] FT La FT de Goertzel a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsGoertzelGroup
 */
void pds_goertzel_free(PdsGoertzel *FT)
{
	if(FT!=NULL)
	{
		free(FT);
	}
}


/** \fn void pds_goertzel_destroy(PdsGoertzel **FT)
 *  \brief Libera una estructura de tipo puntero PdsGoertzel, y carga a la estructura con NULL.
 *  \param[in,out] FT La FT de Goertzel a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsGoertzelGroup
 */
void pds_goertzel_destroy(PdsGoertzel **FT)
{
	if((*FT)!=NULL)
	{
		free(*FT);
		(*FT)=NULL;
	}
}

