/*
 * pdswave.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdswave.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-12-2011
 *  \brief Funciones que trabajan con estructuras de tipo onda.
 *  
 *  <br><br>
 *  Se genera una onda a partir de un arreglo de datos, y se repite cíclicamente.
 *  <br>
 *  \image html test3grafico1.png "Señal de salida y[n]."
 */

#ifndef __PDSWAVE_H__
#define __PDSWAVE_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdsglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsWaveGroup Funciones del módulo PdsWave.
 *
 *  <br><br>
 *  Se genera una onda a partir de un arreglo de datos, y se repite cíclicamente.
 *  <br>
 *  \image html test3grafico1.png "Señal de salida y[n]."
 * @{
 */


/*! \struct PdsWave
 *  \brief La estructura tipo  PdsWave .
 *
 *  Esta estructura crea un generador onda repitiendo cíclicamente un arreglo.
 *  Para usar incluir pds/pdsds.h.
 *  <br><br>
 *  Se genera una onda a partir de un arreglo de datos, y se repite cíclicamente.
 *  <br>
 *  \image html test3grafico1.png "Señal de salida y[n]."
 *  \ingroup PdsWaveGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de Nel elementos. */
	PdsDsReal *V;
	/*! Número de elementos.  */
	PdsDsNatural Nel;
	/*! ID de elemento actual */
	PdsDsNatural ID;
	/*! Valor del elemento actual. */
	PdsDsReal x;
}PdsWave;


/** \fn PdsWave *pds_wave_new(PdsDsNatural Nel)
 *  \brief Crea una estructura de tipo PdsWave de Nel elementos.
 * 
 *  Todos los elementos son rellenados con ceros.
 *  \param[in] Nel Es el número de elementos por ciclo.
 *  \return Un puntero a una estructura de tipo PdsWave.
 *  \ingroup PdsWaveGroup
 */
PdsWave *pds_wave_new(PdsDsNatural Nel);


/** \fn PdsWave *pds_wave_new_vector(const PdsVector *VecSrc)
 *  \brief Crea una estructura de tipo PdsWave usando como fuente el vector VecSrc.
 *  \param[in] VecSrc Es el vector que se usará como fuente.
 *  \return Un puntero a la onda de tipo PdsWave.
 *  \ingroup PdsWaveGroup
 */
PdsWave *pds_wave_new_vector(const PdsVector *VecSrc);


/** \fn int pds_wave_set_data(PdsWave *Wave, PdsDsNatural x, PdsDsReal m)
 *  \brief Escribe el valor m en la posición (x) del generador Wave.
 *  (x) inicia con (0).
 *  \param[in,out] Wave El generador a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsWaveGroup
 */
int pds_wave_set_data(PdsWave *Wave, PdsDsNatural x, PdsDsReal m);


/** \fn int pds_wave_get_value(PdsWave *WAVE,PdsDsReal *y)
 *  \brief Obtiene un valor en la salida e itera el generador de ondas.
 *  \param[in,out] WAVE La estructura de PdsWave a consultar.
 *  \param[out] y En esta variable se carga el  nuevo valor a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsWaveGroup
 */
int pds_wave_get_value(PdsWave *WAVE,PdsDsReal *y);


/** \fn int pds_wave_get_last_value(PdsWave *WAVE,PdsDsReal *y)
 *  \brief Obtiene el último valor a la salida del generador de onda.
 *  \param[in,out] WAVE La estructura de PdsWave a consultar.
 *  \param[out] y En esta variable se carga el último valor a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsWaveGroup
 */
int pds_wave_get_last_value(PdsWave *WAVE,PdsDsReal *y);


/** \fn int pds_wave_get_vector(PdsWave *WAVE,PdsVector *Vector)
 *  \brief Carga un vector con valores obtenidos en la salida del 
 *  generador de onda. Realiza una iteración para cada elemento del vector.
 *  \param[in,out] WAVE La estructura de PdsWave a consultar.
 *  \param[out] Vector En este vector se cargará los valores a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL, Vector==NULL.
 *  \ingroup PdsWaveGroup
 */
int pds_wave_get_vector(PdsWave *WAVE,PdsVector *Vector);


/** \fn void pds_wave_free(PdsWave *WAVE)
 *  \brief Libera una estructura de tipo puntero PdsWave.
 *  \param[in,out] WAVE La estructura de PdsWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsWaveGroup
 */
void pds_wave_free(PdsWave *WAVE);


/** \fn void pds_wave_destroy(PdsWave **WAVE)
 *  \brief Libera una estructura de tipo puntero PdsWave, y lo carga con NULL.
 *  \param[in,out] WAVE La estructura de PdsWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsWaveGroup
 */
void pds_wave_destroy(PdsWave **WAVE);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSWAVE_H__ */ 

