/*
 * pdsswave.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsswave.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con estructuras Sine Wave.
 *  
 *  <br><br>
 *  La frecuencia de muestreo es 2pi, teta \f$ \theta \f$ es la frecuencia del sine wave
 *  \image html pdsswave.png "Generador sinusoidal digital"
 *  \f[ h[n]=A~sin[(n+1)\theta ]u[n] \f]
 *  \f[ H(z)=\frac{A~sin(\theta)}{(1-e^{\theta i}z^{-1})(1-e^{-\theta i}z^{-1})}
	    =\frac{A~sin(\theta)}{ 1-2cos(\theta)z^{-1}+z^{-2} } 
    \f]
 *  \f[ y[n]= A~sin(\theta)x[n]+2cos(\theta)y[n-1]-y[n-2] \f]
 *  <br>
 *  \f[ x[n]=\delta[n] \f]
 *  \f[ y[n]=A~sin[(n+1)\theta ]u[n] \f]
 *  \f[ y[0]=A~sin[\theta ]~~~y[-1]=0~~~y[-2]=0 \f]
 *  \image html test1grafico1.png "Señal de salida y[n]."
 */

#ifndef __PDSSWAVE_H__
#define __PDSSWAVE_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdsglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsSWaveGroup Funciones del módulo PdsSWave.
 *
 *  <br><br>
 *  La frecuencia de muestreo es 2pi, teta \f$ \theta \f$ es la frecuencia del sine wave
 *  \image html pdsswave.png "Generador sinusoidal digital"
 *  \f[ h[n]=A~sin[(n+1)\theta ]u[n] \f]
 *  \f[ H(z)=\frac{A~sin(\theta)}{(1-e^{\theta i}z^{-1})(1-e^{-\theta i}z^{-1})}
	    =\frac{A~sin(\theta)}{ 1-2cos(\theta)z^{-1}+z^{-2} } 
    \f]
 *  \f[ y[n]= A~sin(\theta)x[n]+2cos(\theta)y[n-1]-y[n-2] \f]
 *  <br>
 *  \f[ x[n]=\delta[n] \f]
 *  \f[ y[n]=A~sin[(n+1)\theta ]u[n] \f]
 *  \f[ y[0]=A~sin[\theta ]~~~y[-1]=0~~~y[-2]=0 \f]
 *  \image html test1grafico1.png "Señal de salida y[n]."
 * @{
 */


/*! \struct PdsSWave
 *  \brief La estructura tipo  PdsSWave .
 *
 *  Esta estructura crea un generador sinusoidal de frecuencia teta \f$\theta\f$.
 *  Para usar incluir pds/pdsds.h.
 *  <br><br>
 *  La frecuencia de muestreo es 2pi, teta \f$ \theta \f$ es la frecuencia del sine wave
 *  \image html pdsswave.png "Generador sinusoidal digital"
 *  \f[ h[n]=A~sin[(n+1)\theta ]u[n] \f]
 *  \f[ H(z)=\frac{A~sin(\theta)}{(1-e^{\theta i}z^{-1})(1-e^{-\theta i}z^{-1})}
	    =\frac{A~sin(\theta)}{ 1-2cos(\theta)z^{-1}+z^{-2} } 
    \f]
 *  \f[ y[n]= A~sin(\theta)x[n]+2cos(\theta)y[n-1]-y[n-2] \f]
 *  <br>
 *  \f[ x[n]=\delta[n] \f]
 *  \f[ y[n]=A~sin[(n+1)\theta ]u[n] \f]
 *  \f[ y[0]=A~sin[\theta ]~~~y[-1]=0~~~y[-2]=0 \f]
 *  \image html test1grafico1.png "Señal de salida y[n]."
 *  \ingroup PdsSWaveGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Salida del generador y tiempos pasados de la salida. */
	PdsDsReal Y[3];
	/*! Amplitud del generador */
	PdsDsReal A;
	/*! Frecuencia normalizada del generador. */
	PdsDsReal Teta;
	/*! a= 2*cos(Teta). */
	PdsDsReal a;
}PdsSWave;

/** \fn PdsSWave *pds_swave_new(PdsDsReal A,PdsDsReal Teta)
 *  \brief Crea una estructura de tipo PdsSWave.
 * 
 *  El primer valor de la estructura es y[0]=A*sen(Teta).
 *  \param[in] A Es la amplitud de onda sinusoidal.
 *  \param[in] Teta \f$\theta\f$ Es la frecuencia normalizada a 2pi para Fsampling, en la
 *  onda sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsSWave.
 *  \ingroup PdsSWaveGroup
 */
PdsSWave *pds_swave_new(PdsDsReal A,PdsDsReal Teta);


/** \fn PdsSWave *pds_swave_new_nos(PdsDsReal A,PdsDsNatural N)
 *  \brief Crea una estructura de tipo PdsSWave.
 * 
 *  El primer valor de la estructura es y[0]=A*sen(Teta).
 *  \param[in] A Es la amplitud de onda sinusoidal.
 *  \param[in] N Número de muestras por periodo en la onda sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsSWave.
 *  \ingroup PdsSWaveGroup
 */
PdsSWave *pds_swave_new_nos(PdsDsReal A,PdsDsNatural N);


/** \fn int pds_swave_get_value(PdsSWave *WAVE,PdsDsReal *y)
 *  \brief Obtiene un valor en la salida e itera el generador sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsSWave a consultar.
 *  \param[out] y En esta variable se carga el  nuevo valor a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsSWaveGroup
 */
int pds_swave_get_value(PdsSWave *WAVE,PdsDsReal *y);


/** \fn int pds_swave_get_last_value(PdsSWave *WAVE,PdsDsReal *y)
 *  \brief Obtiene el último valor a la salida del generador sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsSWave a consultar.
 *  \param[out] y En esta variable se carga el último valor a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsSWaveGroup
 */
int pds_swave_get_last_value(PdsSWave *WAVE,PdsDsReal *y);


/** \fn int pds_swave_get_vector(PdsSWave *WAVE,PdsVector *Vector)
 *  \brief Carga un vector con valores obtenidos en la salida del 
 *  generador sinusoidal. Realiza una iteración para cada elemento del vector.
 *  \param[in,out] WAVE La estructura de PdsSWave a consultar.
 *  \param[out] Vector En este vector se cargará los valores a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL, Vector==NULL.
 *  \ingroup PdsSWaveGroup
 */
int pds_swave_get_vector(PdsSWave *WAVE,PdsVector *Vector);


/** \fn void pds_swave_free(PdsSWave *WAVE)
 *  \brief Libera una estructura de tipo puntero PdsSWave.
 *  \param[in,out] WAVE La estructura de PdsSWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSWaveGroup
 */
void pds_swave_free(PdsSWave *WAVE);


/** \fn void pds_swave_destroy(PdsSWave **WAVE)
 *  \brief Libera una estructura de tipo puntero PdsSWave, y lo carga con NULL.
 *  \param[in,out] WAVE La estructura de PdsSWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSWaveGroup
 */
void pds_swave_destroy(PdsSWave **WAVE);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSSWAVE_H__ */ 

