/*
 * pdsiqwave.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsiqwave.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con estructuras IQ Wave.
 *  
 *  <br><br>
 *  La frecuencia de muestreo es 2pi, teta \f$ \theta \f$ es la frecuencia del IQ Wave
 *  \image html pdsiqwave.png "Generador IQ sinusoidal digital"
 *  <br>
    \f{eqnarray*}{
        y_c[n] &=& cos(\theta)~y_c[n-1]-sin(\theta)~y_s[n-1] \\ 
        y_s[n] &=& sin(\theta)~y_c[n-1]+cos(\theta)~y_s[n-1] 
    \f}
 *  <br>
 *  \f[ y_c[n]=A~cos[(n+1)\theta ] \f]
 *  \f[ y_s[n]=A~sin[(n+1)\theta ] \f]
 *  \f[ y_c[0]=A~cos[\theta ]~~~y_c[-1]=A \f]
 *  \f[ y_s[0]=A~sin[\theta ]~~~y_s[-1]=0 \f]
 */

#ifndef __PDSIQWAVE_H__
#define __PDSIQWAVE_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdsglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsIQWaveGroup Funciones del módulo PdsIQWave.
 *
 *  <br><br>
 *  La frecuencia de muestreo es 2pi, teta \f$ \theta \f$ es la frecuencia del IQ Wave
 *  \image html pdsiqwave.png "Generador IQ sinusoidal digital"
 *  <br>
    \f{eqnarray*}{
        y_c[n] &=& cos(\theta)~y_c[n-1]-sin(\theta)~y_s[n-1] \\ 
        y_s[n] &=& sin(\theta)~y_c[n-1]+cos(\theta)~y_s[n-1] 
    \f}
 *  <br>
 *  \f[ y_c[n]=A~cos[(n+1)\theta ] \f]
 *  \f[ y_s[n]=A~sin[(n+1)\theta ] \f]
 *  \f[ y_c[0]=A~cos[\theta ]~~~y_c[-1]=A \f]
 *  \f[ y_s[0]=A~sin[\theta ]~~~y_s[-1]=0 \f]
 * @{
 */


/*! \struct PdsIQWave
 *  \brief La estructura tipo  PdsIQWave .
 *
 *  Esta estructura crea un generador IQ sinusoidal de frecuencia teta \f$\theta\f$.
 *  Para usar incluir pds/pdsds.h.
 *  <br><br>
 *  La frecuencia de muestreo es 2pi, teta \f$ \theta \f$ es la frecuencia del IQ Wave
 *  \image html pdsiqwave.png "Generador IQ sinusoidal digital"
 *  <br>
    \f{eqnarray*}{
        y_c[n] &=& cos(\theta)~y_c[n-1]-sin(\theta)~y_s[n-1] \\ 
        y_s[n] &=& sin(\theta)~y_c[n-1]+cos(\theta)~y_s[n-1] 
    \f}
 *  <br>
 *  \f[ y_c[n]=A~cos[(n+1)\theta ] \f]
 *  \f[ y_s[n]=A~sin[(n+1)\theta ] \f]
 *  \f[ y_c[0]=A~cos[\theta ]~~~y_c[-1]=A \f]
 *  \f[ y_s[0]=A~sin[\theta ]~~~y_s[-1]=0 \f]
 *  \ingroup PdsIQWaveGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Salida del generador seno en las iteraciones pasadas. */
	PdsDsReal Ys[2];
	/*! Salida del generador coseno en las iteraciones pasadas. */
	PdsDsReal Yc[2];
	/*! Amplitud del generador */
	PdsDsReal A;
	/*! Frecuencia normalizada del generador. */
	PdsDsReal Teta;
	/*! a= cos(Teta). */
	PdsDsReal a;
	/*! b= sin(Teta). */
	PdsDsReal b;
}PdsIQWave;


/** \fn PdsIQWave *pds_iqwave_new(PdsDsReal A,PdsDsReal Teta)
 *  \brief Crea una estructura de tipo PdsIQWave.
 * 
 *  El primer valor de la estructura es yc[0]=A*cos(Teta) ys[0]=A*sin(Teta).
 *  \param[in] A Es la amplitud de onda IQ sinusoidal.
 *  \param[in] Teta \f$\theta\f$ Es la frecuencia normalizada a 2pi para Fsampling, en la
 *  onda IQ sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsIQWave.
 *  \ingroup PdsIQWaveGroup
 */
PdsIQWave *pds_iqwave_new(PdsDsReal A,PdsDsReal Teta);


/** \fn PdsIQWave *pds_iqwave_new_nos(PdsDsReal A,PdsDsNatural N)
 *  \brief Crea una estructura de tipo PdsIQWave.
 * 
 *  El primer valor de la estructura es yc[0]=A*cos(Teta) ys[0]=A*sin(Teta).
 *  \param[in] A Es la amplitud de onda IQ sinusoidal.
 *  \param[in] N Número de muestras por periodo en la onda IQ sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsIQWave.
 *  \ingroup PdsIQWaveGroup
 */
PdsIQWave *pds_iqwave_new_nos(PdsDsReal A,PdsDsNatural N);


/** \fn int pds_iqwave_get_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys)
 *  \brief Obtiene un valor de yc e ys en la salida e itera el generador IQ sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsIQWave a consultar.
 *  \param[out] yc En esta variable se carga el  nuevo valor a la salida de generador coseno.
 *  \param[out] ys En esta variable se carga el  nuevo valor a la salida de generador seno.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsIQWaveGroup
 */
int pds_iqwave_get_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys);


/** \fn int pds_iqwave_get_last_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys)
 *  \brief Obtiene el último valor a la salida del generador IQ sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsIQWave a consultar.
 *  \param[out] yc En esta variable se carga el  último valor a la salida de generador coseno.
 *  \param[out] ys En esta variable se carga el  último valor a la salida de generador seno.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsIQWaveGroup
 */
int pds_iqwave_get_last_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys);


/** \fn int pds_iqwave_get_vector(PdsIQWave *WAVE,PdsVector *VectorC,PdsVector *VectorS)
 *  \brief Carga un par de vectores con valores obtenidos en la salida del 
 *  generador IQ. Realiza una iteración para cada elemento "i" de los vectores.
 *  \param[in,out] WAVE La estructura de PdsIQWave a consultar.
 *  \param[out] VectorC En este vector se cargará los valores a la salida coseno de generador.
 *  \param[out] VectorS En este vector se cargará los valores a la salida seno de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL, Vector==NULL o
 *  los tamaños de los vectores no coinciden.
 *  \ingroup PdsIQWaveGroup
 */
int pds_iqwave_get_vector(PdsIQWave *WAVE,PdsVector *VectorC,PdsVector *VectorS);


/** \fn void pds_iqwave_free(PdsIQWave *WAVE)
 *  \brief Libera una estructura de tipo puntero PdsIQWave.
 *  \param[in,out] WAVE La estructura de PdsIQWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIQWaveGroup
 */
void pds_iqwave_free(PdsIQWave *WAVE);


/** \fn void pds_iqwave_destroy(PdsIQWave **WAVE)
 *  \brief Libera una estructura de tipo puntero PdsIQWave, y lo carga con NULL.
 *  \param[in,out] WAVE La estructura de PdsIQWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIQWaveGroup
 */
void pds_iqwave_destroy(PdsIQWave **WAVE);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSIQWAVE_H__ */ 

